/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Reads in a show schedule and sets up the ALLSHOWS array.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "ubs-sched.h"

/*+
  Reads in the schedule and does all the gruntwork.  This is mostly accomplished by using a rather large state machine to determine where we are in the file, and set the according values in the ALLSHOWS array.

  int read_sched Returns NO on success (same as OK), YES if invalid shows were found, and NO_MEM for memory errors

  char *config_fname The name of the schedule to parse
+*/
int read_sched(char *config_fname)
{
 int i, tmp, bad = NO;
 char buf[STRBUF];
 char *ptr1;
 ubs_table show;

 ubs_table_init(&show);

 if((ALLSHOWS = (struct show *)malloc(atoi(ubs_table_data(&SCHED, "numshows")) * sizeof(struct show))) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to allocate memory for ALLSHOWS array");
  return NO_MEM;
 }

 for(i = 1; i <= atoi(ubs_table_data(&SCHED, "numshows")); i++) {
  snprintf(buf, STRBUF, "show%d", i);
  if(read_config(config_fname, buf, &show)) {
   fprintf(stdout, "ERROR: Cannot parse show #%d\n", i);
   bad = YES;
   continue;
  }
  else {
   // if the show is invalid, then print an error message and get out
   if(ubs_table_data(&show, "name") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No name given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&show, "dir") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No directory given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&show, "start") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No start time given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&show, "end") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No stop time given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&show, "type") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No playlist type given\n", i);
    bad = YES;
   }
   else if(ubs_table_data(&show, "repeat") == NULL) {
    fprintf(stdout, "Show #%d is invalid: No repeat value given\n", i);
    bad = YES;
   }

   if(bad) {
    continue;
   }

   // show id
   ALLSHOWS[i - 1].show_id = i;
   // show name
   strncpy(ALLSHOWS[i - 1].show_name, ubs_table_data(&show, "name"), STRBUF);
   // show directory
   strncpy(ALLSHOWS[i - 1].dir, ubs_table_data(&show, "dir"), FILEBUF);

   // show type
   if(!strcasecmp(ubs_table_data(&show, "type"), "random")) {
    ALLSHOWS[i - 1].type = LIST_RANDOM;
   }
   else if(!strcasecmp(ubs_table_data(&show, "type"), "m3u")) {
    ALLSHOWS[i - 1].type = LIST_M3U;
   }
   else if(!strcasecmp(ubs_table_data(&show, "type"), "itunes")) {
    ALLSHOWS[i - 1].type = LIST_ITUNES;
   }
   else {
    fprintf(stdout, "Unknown show type '%s' in show #%d\n", ubs_table_data(&show, "type"), i);
    bad = YES;
   }

   // start time
   bzero(buf, STRBUF);
   if((ptr1 = strchr(ubs_table_data(&show, "start"), ':')) != NULL) {
    tmp = atoi(strncpy(buf, ubs_table_data(&show, "start"), ptr1 - ubs_table_data(&show, "start")));
    tmp *= 2;
    strncpy(buf, ptr1 + 1, STRBUF);
    if(strcmp(buf, "00")) {
     tmp += 1;
    }
   }
   else {
    tmp = atoi(strncpy(buf, ubs_table_data(&show, "start"), STRBUF));
    tmp *= 2;
   }
   ALLSHOWS[i - 1].start = tmp;

   // stop time
   bzero(buf, STRBUF);
   if((ptr1 = strchr(ubs_table_data(&show, "end"), ':')) != NULL) {
    tmp = atoi(strncpy(buf, ubs_table_data(&show, "end"), ptr1 - ubs_table_data(&show, "end")));
    tmp *= 2;
    strncpy(buf, ptr1 + 1, STRBUF);
    if(strcmp(buf, "00")) {
     tmp += 1;
    }
   }
   else {
    tmp = atoi(strncpy(buf, ubs_table_data(&show, "end"), STRBUF));
    tmp *= 2;
   }
   ALLSHOWS[i - 1].end = tmp;

   // frequency
   if(!strcmp(ubs_table_data(&show, "repeat"), "once")) {
    ALLSHOWS[i - 1].repeat = once;
   }
   else if(!strcmp(ubs_table_data(&show, "repeat"), "weekly")) {
    ALLSHOWS[i - 1].repeat = weekly;
   }
   else if(!strcmp(ubs_table_data(&show, "repeat"), "daily")) {
    ALLSHOWS[i - 1].repeat = daily;
   }
   else {
    fprintf(stdout, "Unknown frequency '%s' in show #%d\n", ubs_table_data(&show, "repeat"), i);
    bad = YES;
   }

   // banner event
   if(ubs_table_exists(&show, "banner")) {
    strncpy(ALLSHOWS[i - 1].banner, ubs_table_data(&show, "banner"), FILEBUF);
   }

   // whence
   if(ALLSHOWS[i - 1].repeat == weekly) {
    if(!ubs_table_exists(&show, "when")) {
     fprintf(stdout, "No day given for weekly show");
     bad = YES;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "sun")) {
     ALLSHOWS[i - 1].when.weekday = sun;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "mon")) {
     ALLSHOWS[i - 1].when.weekday = mon;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "tues")) {
     ALLSHOWS[i - 1].when.weekday = tues;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "wed")) {
     ALLSHOWS[i - 1].when.weekday = wed;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "thurs")) {
     ALLSHOWS[i - 1].when.weekday = thurs;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "fri")) {
     ALLSHOWS[i - 1].when.weekday = fri;
    }
    else if(!strcmp(ubs_table_data(&show, "when"), "sat")) {
     ALLSHOWS[i - 1].when.weekday = sat;
    }
    else {
     fprintf(stdout, "Unknown day '%s' for weekly event\n", ubs_table_data(&show, "when"));
     bad = YES;
    }
   }
   else if(ALLSHOWS[i - 1].repeat == once) {
    strncpy(ALLSHOWS[i - 1].when.day, ubs_table_data(&show, "when"), SMALLBUF);
   }
  }
 }

 log_error_msg(LOG_DEBUG, "Added show %d to ALLSHOWS array", i - 1);

 return bad;
}
