/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Displays information about known shows from ubs.conf.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <errno.h>

#include "ubs.h"
#include "ubs-main.h"

// needed for find_show()
#include "ubs-sched.h"

/*+
  Displays information about known shows.

  int cmd_shows Returns OK on success

  int cmdc Number of arguments

  char **cmdv Argument vector
+*/
int cmd_shows(int cmdc, char **cmdv) {
 char ch;
 int format = ENGLISH, now = NO, i, cur_day, cur_slot;

 while((ch = getopt(cmdc, cmdv, "hwcesnp")) != -1) {
  switch(ch) {
   case 'e':
    format = ENGLISH;
    break;
   case 'c':
    format = CSV;
    break;
   case 'w':
    format = HTML;
    break;
   case 's':
    format = SILENT;
    break;
   case 'n':
    now = YES;
    break;
   case 'p':
    return sub_now_playing(format);
    break;
   case 'h':
    sub_help_cmd("shows");
    return OK;
    break;
   default:
    sub_help_cmd("shows");
    return OK;
    break;
  }
 }

 cmdc -= optind;
 cmdv += optind;

 if(now) {
  gettime();
  cur_day = CUR_TIME->tm_wday;
  cur_slot = (CUR_TIME->tm_hour * 2) + (int)(CUR_TIME->tm_min / 30);
  i = find_show(cur_day, cur_slot);
  // default fallback show
  if(i < 0) {
   if(format == ENGLISH) {
    printf("Show ID: %d\n", i);
    printf("  Show Name: Default Show\n");
    printf("  Music Directory: %s\n", ubs_table_data(&SCHED, "miscdir"));
    printf("  Time: Unspecified\n");
    printf("  Frequency: Unspecified\n");
    printf("  Type: Random\n");
   }
   else if(format == CSV) {
    printf("%d,Default Show,%s,-1,-1,-1,%d\n", i, ubs_table_data(&SCHED, "miscdir"), LIST_RANDOM);
   }
   else if(format == HTML) {
    printf("<tr><td>%d</td><td>Default Show</td></td>%s</td><td>-1</td><td>-1</td><td>-1</td><td>%d</td></tr>\n", i, ubs_table_data(&SCHED, "miscdir"), LIST_RANDOM);
   }
  }
  else {
   sub_dump_show(i, format);
  }
 }
 else {
  for(i = 0; i < atoi(ubs_table_data(&SCHED, "numshows")); i++) {
   sub_dump_show(i, format);
  }
 }

 return OK;
}

/*+
  sub_dump_show Returns OK for success

  int show_num The show index

  int format The output mode
+*/
int sub_dump_show(int show_num, int format) {
 if(format == ENGLISH) {
  printf("Show ID: %d\n", ALLSHOWS[show_num].show_id);
  printf("  Show Name: %s\n", ALLSHOWS[show_num].show_name);
  printf("  Music Directory: %s\n", ALLSHOWS[show_num].dir);
  printf("  Time: %d:%02d to %d:%02d\n", ALLSHOWS[show_num].start / 2, (ALLSHOWS[show_num].start % 2) * 30, ALLSHOWS[show_num].end / 2, (ALLSHOWS[show_num].end % 2) * 30);
  printf("  Frequency: %d\n", ALLSHOWS[show_num].repeat);
  printf("  Type: %d\n", ALLSHOWS[show_num].type);
 }
 else if(format == CSV) {
  printf("%d,%s,%s,%d,%d,%d,%d\n", ALLSHOWS[show_num].show_id, ALLSHOWS[show_num].show_name, ALLSHOWS[show_num].dir, ALLSHOWS[show_num].start, ALLSHOWS[show_num].end, ALLSHOWS[show_num].repeat, ALLSHOWS[show_num].type);
 }
 else if(format == HTML) {
  printf("<tr><td>%d</td><td>%s</td><td>%s</td><td>%d:%02d</td><td>%d:%02d</td><td>%d</td><td>%d</td></tr>\n", ALLSHOWS[show_num].show_id, ALLSHOWS[show_num].show_name, ALLSHOWS[show_num].dir, ALLSHOWS[show_num].start / 2, (ALLSHOWS[show_num].start % 2) * 30, ALLSHOWS[show_num].end / 2, (ALLSHOWS[show_num].end % 2) * 30, ALLSHOWS[show_num].repeat, ALLSHOWS[show_num].type);
 }
 
 return OK;
}

int sub_now_playing(int format) {
 FILE *fp;
 char buf[FILEBUF];

 if((fp = fopen("var/now_playing", "r")) == NULL) {
  perror("fopen now_playing");
  return NO_FILE;
 }

 if(fgets(buf, FILEBUF, fp) == NULL) {
  perror("fgets now_playing");
  return NO_FILE;
 }

 return sub_print_tag(buf, format);
}
