/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Displays the system logfiles in a friendly format.  Also can list information about known logfiles.
****/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "ubs-main.h"

/*+
  Displays information about logfiles

  int cmd_log Returns OK on success, NO_FILE if an logfile is not given, or otherwise just returns whatever get_log says.

  int cmdc Number of arguments

  char **cmdv Argument vector
+*/
int cmd_log(int cmdc, char **cmdv) {
 char buf[FILEBUF];
 int ch, format = ENGLISH, num_lines = 10;

 while((ch = getopt(cmdc, cmdv, "hwcesn:l")) != -1) {
  switch(ch) {
   case 'e':
    format = ENGLISH;
    break;
   case 'c':
    format = CSV;
    break;
   case 'w':
    format = HTML;
    break;
   case 's':
    format = SILENT;
    break;
   case 'n':
    num_lines = atoi(optarg);
    break;
   case 'l':
    return sub_list_logs(format);
    break;
   case 'h':
    sub_help_cmd("log");
    return OK;
    break;
   default:
    sub_help_cmd("log");
    return OK;
    break;
  }
 }

 cmdc -= optind;
 cmdv += optind;

 if(cmdc == 0) {
  printf("Not enough arguments to 'log'");
  sub_help_cmd("log");
  return OK;
 }

 snprintf(buf, FILEBUF, "log/%s.log", cmdv[0]);
 return get_log(buf, num_lines, format);

 return OK;
}

/*+
  int sub_list_logs Returns OK on success, NO_DIR if the logfile directory can't be opened (which needs to be done for listing logs)

  int format Output format to use
+*/
int sub_list_logs(int format) {
 DIR *dp;
 struct dirent *ds;

 if((dp = opendir("log")) == NULL) {
  printf("Error: Unable to open logfile directory");
  return NO_DIR;
 }

 while((ds = readdir(dp)) != NULL) {
  if(strcmp(ds->d_name, ".") && strcmp(ds->d_name, "..")) {
   if(format == ENGLISH || format == CSV) {
    printf("%s\n", ds->d_name);
   }
   else if(format == HTML) {
    printf("<li>%s</li>\n", ds->d_name);
   }
  }
 }

 closedir(dp);
 return OK;
}
