/*
Copyright (C) 2003, Nik Reiman - nik@aboleo.net

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/****
  Logging mechanisms for the UBS.  This includes reading and writing logfiles, and getting the date and time, and other such things.
****/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>

#include "ubs.h"

/*+
  This sets a global variable CUR_TIME to reflect the present time

  void gettime Takes nothing, gives nothing.
+*/
void gettime(void)
{
 time_t tmptime;

 tmptime = time(NULL);
 CUR_TIME = localtime(&tmptime);
 mktime(CUR_TIME);

 return;
}

/*+
  Gets the current time and converts it to a string in the form MM/DD

  void get_date Returns nothing.  Probably a bad thing.

  char *fmt_time The character array that the current date will be written to
+*/
void get_date(char *fmt_time)
{
 time_t tmptime;
 struct tm *p_time;

 tmptime = time(NULL);
 p_time = localtime(&tmptime);
 mktime(p_time);

 // in case you were wondering, tm_mon goes from 0-11, not 1-12 :/
 snprintf(fmt_time, SMALLBUF, "%02d/%02d", p_time->tm_mon + 1, p_time->tm_mday);
 return;
}

/*+
  Gets the current time and converts it into a string of the form HH:MM

  void get_time Returns nothing

  char *fmt_time The string which will be written to
+*/
void get_time(char *fmt_time)
{
 time_t tmptime;
 struct tm *p_time;

 tmptime = time(NULL);
 p_time = localtime(&tmptime);
 mktime(p_time);

 snprintf(fmt_time, SMALLBUF, "%02d:%02d", p_time->tm_hour, p_time->tm_min);
 return;
}

/*+
  Records the current status of the daemon.  This is a bit sloppy at the moment.  Basically, whenever the program feels like it, it can record its status, which gets read by the UBS shell and other fun things.  By default, the file that will be written to is called var/NAME.status, where NAME is the process name of the daemon.  The main UBS daemons call this function after successfully starting, and at the bottom of each infinite while() loop.

  int record_status Returns OK on success, NO_FILE if the status file can't be opened

  char *msg The string to write to the status file
+*/
int record_status(char *msg) {
 FILE *fp;
 char buf[STRBUF];

 snprintf(buf, STRBUF, "var/%s.status", PROCNAME);
 if((fp = fopen(buf, "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open .status file for writing");
  return NO_FILE;
 }

 fprintf(fp, "%s", msg);
 fclose(fp);
 return OK;
}

/*+
  Record the process ID so that other daemons know where to find us.  By default, this file is var/NAME.pid, where NAME is the PROCNAME of the daemon

  int record_pid Returns OK on success, NO_FILE if the pid file can't be opened.

  int pid The current proccess ID (pid)
+*/
int record_pid(int pid) {
 FILE *fp;
 char buf[STRBUF];

 snprintf(buf, STRBUF, "var/%s.pid", PROCNAME);
 if((fp = fopen(buf, "w")) == NULL) {
  log_error_msg(LOG_ERROR, "Unable to open .pid file for writing");
  return NO_FILE;
 }

 fprintf(fp, "%d", pid);
 fclose(fp);
 return OK;
}

/*+
  Frontend to the error logging mechanism.  This function can be accomplished by writing to a character buffer, and simply using log_event, but it's kind of a pain that way, especially when most calls to that function are trying to do the same thing.

  int log_error_msg Returns whatever write_log() returns, FAIL if the priority is insufficient, FAIL if an unknown control character is used (ie, %x or something), and FAIL if an unknown debug priority is given.

  int priority Logging priority

  char *msg The format string to use.

  ... Arguments to the string
+*/
int log_error_msg(int priority, char *msg, ...)
{
 va_list ap;
 char *p, str_out[BIGBUF], num_str[SMALLBUF];
 struct log_entry entry;

 if(priority > atoi(ubs_table_data(&GLOBAL, "loglevel"))) {
  return FAIL;
 }

 memset(str_out, 0x0, BIGBUF);
 va_start(ap, msg);
 for(p = msg; *p; p++) {
  if(*p != '%') {
   strncat(str_out, p, 1);
   continue;
  }
  switch(*++p) {
   case 's':
    strncat(str_out, va_arg(ap, char *), BIGBUF);
    break;
   case 'd':
    snprintf(num_str, SMALLBUF, "%d", va_arg(ap, int));
    strncat(str_out, num_str, SMALLBUF);
    break;
   default:
    return FAIL;
  }
 }
 va_end(ap);

 get_date(entry.date);
 get_time(entry.time);
 strncpy(entry.field1, PROCNAME, BIGBUF);
 switch(priority) {
  case LOG_EMERG:
   strncpy(entry.field2, "emerg", BIGBUF);
   break;
  case LOG_STATUS:
   strncpy(entry.field2, "status", BIGBUF);
   break;
  case LOG_ERROR:
   strncpy(entry.field2, "error", BIGBUF);
   break;
  case LOG_DEBUG:
   strncpy(entry.field2, "debug", BIGBUF);
   break;
  default:
   return FAIL;
 }
 strncpy(entry.field3, str_out, BIGBUF);

 return write_log(&entry, ubs_table_data(&GLOBAL, "errorlog"));
}

/*+
  This function is the frontend to write_log.  It allows you to basically write to either the playlist log or the error log with the same set of data structures.   For the playlist log, the format is generally as follows:
  [date][time][filename (full path)][artist name][song title]
  For the error logs, the format is as such:
  [date][time][process name][severity][error message]

  int log_event Returns OK on success, FAIL if the loglevel indicates not to log this message, and FAIL if one of the fields contains a bracket character (will screw up parsing)

  char *logfile The target filename to log to

  char *field1 A character array for the first field

  char *field2 A character array for the second field

  char *field3 A character array for the third field
  
  int priority The priority for logging information
+*/
int log_event(char *logfile, char *field1, char *field2, char *field3, int priority)
{
 struct log_entry entry;

 if(priority > atoi(ubs_table_data(&GLOBAL, "loglevel"))) {
  return FAIL;
 }
 else {
  if(strchr(field1, '[') || strchr(field1, ']') ||
    strchr(field2, '[') || strchr(field2, ']') ||
    strchr(field3, '[') || strchr(field3, ']')) {
   return FAIL;
  }
  get_date(entry.date);
  get_time(entry.time);
  strncpy(entry.field1, field1, BIGBUF);
  strncpy(entry.field2, field2, BIGBUF);
  strncpy(entry.field3, field3, BIGBUF);

  write_log(&entry, logfile);
  return OK;
 }
}

/*+
  The ugly side of log_event.  This takes in a log_entry structure and appends it to the end of a logfile.  This is rather cumbersome to access directly, and is usually only called via log_entry.

  int write log Returns OK on success, NO_FILE if the logfile cannot be opened

  struct log_entry *entry The log entry structure (three character values, a date, and a time) to be written to the file

  char *logfile The filename to log to
+*/
int write_log(struct log_entry *entry, char *logfile)
{
 FILE *fp;

 fp = fopen(logfile, "a+");
 if(fp == NULL) {
  return NO_FILE;
 }

 fprintf(fp, "[%s][%s][%s][%s][%s]\n", entry->date, entry->time, entry->field1,
   entry->field2, entry->field3);
 fclose(fp);
 return OK;
}

/*+
  Given a single line from a logfile (bracket delimited), it will break it up into the respective components and place this data in a structure.

  int read_log Returns OK on success, FAIL if any single parse fails

  struct log_entry *entry The structure to be filled in

  char *logfile A single line from a properly formatted logfile
+*/
int read_log(struct log_entry *entry, char *logbuf) {
 char *p1 = NULL, *p2 = NULL;
 int ret = OK;

 // get the date
 if((p1 = strchr(logbuf, '[')) != NULL && (p2 = strchr(p1, ']')) != NULL) {
  if(p2 <= p1 || (p1 + 1) == p2) {
   strncpy(entry->date, "(null)", 6);
  }
  else {
   strncpy(entry->date, p1 + 1, p2 - p1 - 1);
  }
 }
 else {
  strncpy(entry->date, "(null)", 6);
  ret = FAIL;
 }
 // get the time
 if((p1 = strchr(p2, '[')) != NULL && (p2 = strchr(p1, ']')) != NULL) {
  if(p2 <= p1 || (p1 + 1) == p2) {
   strncpy(entry->time, "(null)", 6);
  }
  else {
   strncpy(entry->time, p1 + 1, p2 - p1 - 1);
  }
 }
 else {
  strncpy(entry->time, "(null)", 6);
  ret = FAIL;
 }
 // get field1 (filename or process name)
 if((p1 = strchr(p2, '[')) != NULL && (p2 = strchr(p1, ']')) != NULL) {
  if(p2 <= p1 || (p1 + 1) == p2) {
   strncpy(entry->field1, "(null)", BIGBUF);
  }
  else {
   strncpy(entry->field1, p1 + 1, p2 - p1 - 1);
  }
 }
 else {
  strncpy(entry->field1, "(null)", BIGBUF);
  ret = FAIL;
 }
 // get field2 (artist name or warning level)
 if((p1 = strchr(p2, '[')) != NULL && (p2 = strchr(p1, ']')) != NULL) {
  if(p2 <= p1 || (p1 + 1) == p2) {
   strncpy(entry->field2, "(null)", BIGBUF);
  }
  else {
   strncpy(entry->field2, p1 + 1, p2 - p1 - 1);
  }
 }
 else {
  strncpy(entry->field2, "(null)", BIGBUF);
  ret = FAIL;
 }
 // get field3 (song name or warning message
 if((p1 = strchr(p2, '[')) != NULL && (p2 = strchr(p1, ']')) != NULL) {
  if(p2 <= p1 || (p1 + 1) == p2) {
   strncpy(entry->field3, "(null)", BIGBUF);
  }
  else {
   strncpy(entry->field3, p1 + 1, p2 - p1 - 1);
  }
 }
 else {
  strncpy(entry->field3, "(null)", BIGBUF);
  ret = FAIL;
 }

 return ret;
}

/*+
  Prints a log_entry structure to standard output.

  int print_log Returns OK on success

  struct log_entry *entry The entry to print out

  int mode The output mode
+*/
int print_log(struct log_entry *entry, int mode) {
 if(mode == ENGLISH) {
  printf("%s at %s: \"%s\" by %s\n", entry->date, entry->time,
    entry->field3, entry->field2);
 }
 else if(mode == CSV) {
  printf("%s,%s,%s,%s,%s\n", entry->date, entry->time, entry->field1,
    entry->field2, entry->field3);
 }
 else if(mode == HTML) {
  printf("<tr><td>%s</td><td>%s</td><td>%s</td><td>%s</td><td>%s</td></tr>\n",
    entry->date, entry->time, entry->field1, entry->field2, entry->field3);
 }

 return OK;
}

/*+
  Prints out an error message to standard error.

  void console_error Returns nothing

  char *message Message to print

  int signal Exit with signal
+*/
void console_error(char *message, int signal) {
 fprintf(stderr, "FATAL: %s\n", message);
 exit(signal);
}

void console_warning(char *message) {
 fprintf(stderr, "ERROR: %s\n", message);
}
