// Package log provides a global logger for zlog.
package log

import (
	"context"
	"fmt"
	"io"
	"os"
	"time"

	"github.com/rs/zerolog"
	"tuxpa.in/a/zlog"
)

// Logger is the global logger.
var Logger = zlog.New(nil).Output(
	zerolog.ConsoleWriter{
		Out:        os.Stderr,
		TimeFormat: time.RFC3339,
	},
).With().Timestamp().Logger()

// Output duplicates the global logger and sets w as its output.
func Output(w io.Writer) zlog.Logger {
	return Logger.Output(w)
}

// With creates a child logger with the field added to its context.
func With() zlog.Context {
	return Logger.With()
}

// Level creates a child logger with the minimum accepted level set to level.
func Level(level zlog.Level) zlog.Logger {
	return Logger.Level(level)
}

// Sample returns a logger with the s sampler.
func Sample(s zlog.Sampler) zlog.Logger {
	return Logger.Sample(s)
}

// Hook returns a logger with the h Hook.
func Hook(h zlog.Hook) zlog.Logger {
	return Logger.Hook(h)
}

// Err starts a new message with error level with err as a field if not nil or
// with info level if err is nil.
//
// You must call Msg on the returned event in order to send the event.
func Err(err error) *zlog.Event {
	return Logger.Err(err)
}

// Trace starts a new message with trace level.
//
// You must call Msg on the returned event in order to send the event.
func Trace() *zlog.Event {
	return Logger.Trace()
}

// Debug starts a new message with debug level.
//
// You must call Msg on the returned event in order to send the event.
func Debug() *zlog.Event {
	return Logger.Debug()
}

// Info starts a new message with info level.
//
// You must call Msg on the returned event in order to send the event.
func Info() *zlog.Event {
	return Logger.Info()
}

// Warn starts a new message with warn level.
//
// You must call Msg on the returned event in order to send the event.
func Warn() *zlog.Event {
	return Logger.Warn()
}

// Error starts a new message with error level.
//
// You must call Msg on the returned event in order to send the event.
func Error() *zlog.Event {
	return Logger.Error()
}

// Errorf sends a log event using debug level and no extra field.
// Arguments are handled in the manner of fmt.Errorf.
func Errorf(format string, v ...interface{}) {
	Logger.Error().CallerSkipFrame(1).Msgf(format, v...)
}

func Errorln(args ...interface{}) {
	Logger.Error().Msg(fmt.Sprintln(args...))
}

// Fatal starts a new message with fatal level. The os.Exit(1) function
// is called by the Msg method.
//
// You must call Msg on the returned event in order to send the event.
func Fatal() *zlog.Event {
	return Logger.Fatal()
}

func Fatalf(format string, args ...interface{}) {
	Logger.Fatal().Msgf(format, args...)
}

func Fatalln(args ...interface{}) {
	Logger.Fatal().Msg(fmt.Sprintln(args...))
}

// Panic starts a new message with panic level. The message is also sent
// to the panic function.
//
// You must call Msg on the returned event in order to send the event.
func Panic() *zlog.Event {
	return Logger.Panic()
}

func Panicf(format string, args ...interface{}) {
	Logger.Panic().Msgf(format, args...)
}

func Panicln(args ...interface{}) {
	Logger.Panic().Msg(fmt.Sprintln(args...))
}

// WithLevel starts a new message with level.
//
// You must call Msg on the returned event in order to send the event.
func WithLevel(level zlog.Level) *zlog.Event {
	return Logger.WithLevel(level)
}

// Log starts a new message with no level. Setting zlog.GlobalLevel to
// zlog.Disabled will still disable events produced by this method.
//
// You must call Msg on the returned event in order to send the event.
func Log() *zlog.Event {
	return Logger.Log()
}

// Print sends a log event using debug level and no extra field.
// Arguments are handled in the manner of fmt.Print.
func Print(v ...interface{}) {
	Logger.Debug().CallerSkipFrame(1).Msg(fmt.Sprint(v...))
}

// Printf sends a log event using debug level and no extra field.
// Arguments are handled in the manner of fmt.Printf.
func Printf(format string, v ...interface{}) {
	Logger.Debug().CallerSkipFrame(1).Msgf(format, v...)
}

func Println(args ...interface{}) {
	Logger.Debug().Msg(fmt.Sprintln(args...))
}

// Ctx returns the Logger associated with the ctx. If no logger
// is associated, a disabled logger is returned.
func Ctx(ctx context.Context) *zlog.Logger {
	return zlog.Ctx(ctx)
}
