/*	$Id: rtelnet.c,v 1.5 2010/01/17 13:32:47 steve Exp $	*/

/*-
 * Copyright (c) 2006 Steve C. Woodford.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Steve C. Woodford.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <paths.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include <netinet/in.h>
#define	TELOPTS
#define	TELCMDS
#include <arpa/telnet.h>
#undef TELOPTS
#undef TELCMDS

#include "context.h"
#include "buffer.h"
#include "config.h"
#include "rtelnet.h"
#include "tty.h"	/* XXX: For TTY_IOCTL_SENDBREAK */


static const char *cf_rtelnet_init(void *, char **, int, int, void *);
static int	rtelnet_init(struct client_ctx *, const void *);
static void	rtelnet_destroy(struct client_ctx *);
static int	rtelnet_event(struct client_ctx *, int);
static int	rtelnet_output(struct client_ctx *, const char *, size_t,
		    struct client_ctx *);
static int	rtelnet_read_pending(struct client_ctx *);
static int	rtelnet_write_pending(struct client_ctx *);
static int	rtelnet_ioctl(struct client_ctx *, int, void *,
		    struct client_ctx *);
static int	rtelnet_connect_host(struct client_ctx *);
static void	rtelnet_disconnect_host(struct client_ctx *, int);
static void	rtelnet_peer_disconnected(struct client_ctx *);
static void	rtelnet_send_break(struct client_ctx *);

struct client_ops rtelnet_ops = {
	"rtelnet",
	cf_rtelnet_init, rtelnet_init, rtelnet_destroy, rtelnet_event,
	rtelnet_output, rtelnet_read_pending, rtelnet_write_pending,
	rtelnet_ioctl
};


struct rtelnet_opt {
	char		*ro_host;
	int		ro_port;
	int		ro_closeidle;
	int		ro_raw;
};

struct rtelnet_ctx {
	struct rtelnet_opt	rc_ro;
	struct buffer_ctx	*rc_buff;
	int			rc_open;
	void		     (*rc_handler)(struct rtelnet_ctx *, unsigned char);
	int			rc_sawiac;
	int			rc_skipcnt;
	struct client_qhead	rc_children;
	int			rc_client_cnt;
	int			rc_connect_pending;
};

static void	rtelnet_opt_dont(struct rtelnet_ctx *, unsigned char);
static void	rtelnet_opt_do(struct rtelnet_ctx *, unsigned char);
static void	rtelnet_opt_wont(struct rtelnet_ctx *, unsigned char);
static void	rtelnet_opt_will(struct rtelnet_ctx *, unsigned char);
static void	rtelnet_command(struct rtelnet_ctx *, unsigned char);

static const char *cf_rtelnet_host(void *, char **, int, int, void *);
static const char *cf_rtelnet_port(void *, char **, int, int, void *);
static const char *cf_rtelnet_closeidle(void *, char **, int, int, void *);
static const char *cf_rtelnet_raw(void *, char **, int, int, void *);

static struct config_tokens cf_rtelnet_tokens[] = {
 {"host",      1, cf_rtelnet_host},
 {"port",      1, cf_rtelnet_port},
 {"closeidle", 1, cf_rtelnet_closeidle},
 {"raw",       1, cf_rtelnet_raw},
 {NULL, 0, NULL}
};

#ifndef MIN
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#endif

static int rtelnet_verbose = 0;

/* ARGSUSED */
static const char *
cf_rtelnet_init(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	static struct rtelnet_opt ro;
	const char *errstr;

	ro.ro_host = NULL;
	ro.ro_port = -1;
	ro.ro_closeidle = 0;
	ro.ro_raw = 0;

	errstr = config_parse(cs, cf_rtelnet_tokens, (void *)&ro);
	if (errstr) {
		if (ro.ro_host)
			(void) free(ro.ro_host);
		return (errstr);
	}

	if (ro.ro_host == NULL || ro.ro_port == -1)
		return (config_err(cs, "You must specify a host & port"));

	*((void **) arg) = &ro;
	return (NULL);
}

static int
rtelnet_init(struct client_ctx *cc, const void *arg)
{
	struct rtelnet_ctx *rc;

	if ((rc = calloc(1, sizeof(*rc))) == NULL)
		return (-1);

	cc->cc_data = rc;
	rc->rc_ro = *((const struct rtelnet_opt *) arg);
	rc->rc_client_cnt = 0;
	TAILQ_INIT(&rc->rc_children);

	cc->cc_fd = open(_PATH_DEVNULL, O_RDWR);
	if (cc->cc_fd < 0) {
		(void) free(rc->rc_ro.ro_host);
		(void) free(rc);
		return (-1);
	}

	if ((rc->rc_ro.ro_closeidle == 0 && rtelnet_connect_host(cc) < 0) ||
	    buffer_init(&rc->rc_buff, cc->cc_fd) < 0) {
		(void) close(cc->cc_fd);
		(void) free(rc->rc_ro.ro_host);
		(void) free(rc);
		return (-1);
	}

	return (0);
}

static void
rtelnet_destroy(struct client_ctx *cc)
{
	struct rtelnet_ctx *rc = cc->cc_data;

	rtelnet_disconnect_host(cc, 1);
	(void) close(cc->cc_fd);
	(void) free(rc->rc_ro.ro_host);
	(void) free(rc);
}

static int
rtelnet_event(struct client_ctx *cc, int events)
{
	struct rtelnet_ctx *rc = cc->cc_data;
	unsigned char buff[128], outb[128], *in, *out;
	ssize_t rv, i;
	int nbytes;

	if (rc->rc_open == 0)
		return (0);

	if (events & POLLRDNORM) {
		/*
		 * Read data from the remote host
		 */
		if (ioctl(cc->cc_fd, FIONREAD, &nbytes) < 0)
			return (-1);

		if (nbytes == 0)
			rtelnet_peer_disconnected(cc);

		while (nbytes) {
			rv = read(cc->cc_fd, buff, MIN(nbytes, sizeof(buff)));
			if (rv == 0) {
				/* Remote peer disconnected? */
				rtelnet_peer_disconnected(cc);
				break;
			} else
			if (rv < 0)
				return (-1);

			/*
			 * We need to handle the 'telnet' protocol here
			 * on behalf of clients.
			 */
			if (rc->rc_ro.ro_raw == 0) {
				for (i = 0, in = buff, out = outb;
				    i < rv; i++, in++) {
					if (rc->rc_handler)
						(rc->rc_handler)(rc, *in);
					else
					if (rc->rc_sawiac)
						rtelnet_command(rc, *in);
					else
					if (rc->rc_skipcnt)
						rc->rc_skipcnt--;
					else
					if (*in == IAC)
						rc->rc_sawiac = 1;
					else
					if (*in != '\0')
						*out++ = *in;
				}

				if (out != outb) {
					context_client_output(cc->cc_ctx,
					    (char *)outb,
					    (size_t)(out - outb), cc);
				}
			} else {
				context_client_output(cc->cc_ctx, (char *)buff,
				    rv, cc);
			}
			nbytes -= (int)rv;
		}
	}

	if (events & POLLWRNORM) {
		if (rc->rc_connect_pending == 0)
			return ((int) buffer_drain(rc->rc_buff));
		else {
			socklen_t sl;
			int e;

			rc->rc_connect_pending = 0;
			sl = sizeof(e);
			if (getsockopt(cc->cc_fd, SOL_SOCKET, SO_ERROR, &e,
			    &sl) < 0) {
				syslog(LOG_ALERT, "rtelnet_event(): getsockopt:"
				    " %s", strerror(errno));
				rtelnet_peer_disconnected(cc);
			} else
			if (e) {
				syslog(LOG_ALERT, "rtelnet_event(): failed to "
				    "connect to remote host: %s", strerror(e));
				rtelnet_peer_disconnected(cc);
			}
		}
	}

	return (0);
}

/* ARGSUSED */
static int
rtelnet_output(struct client_ctx *cc, const char *buff, size_t len,
	   struct client_ctx *sender)
{
	struct rtelnet_ctx *rc = cc->cc_data;

	if (sender->cc_options.co_readonly || rc->rc_open == 0)
		return (0);

	return buffer_fill(rc->rc_buff, buff, len);
}

static int
rtelnet_read_pending(struct client_ctx *cc)
{
	struct rtelnet_ctx *rc = cc->cc_data;

	return (rc->rc_open);
}

static int
rtelnet_write_pending(struct client_ctx *cc)
{
	struct rtelnet_ctx *rc = cc->cc_data;

	if (rc->rc_open == 0)
		return (0);

	return (rc->rc_connect_pending || buffer_length(rc->rc_buff) != 0);
}

/* ARGSUSED */
static int
rtelnet_ioctl(struct client_ctx *cc, int cmd, void *arg,
    struct client_ctx *sender)
{
	struct rtelnet_ctx *rc = cc->cc_data;

	switch (cmd) {
	case TTY_IOCTL_SENDBREAK:
		if (rc->rc_open &&
		    (sender->cc_options.co_readonly == 0 ||
		     sender->cc_options.co_allowbreak))
			rtelnet_send_break(cc);
		break;

	case CONTEXT_IOCTL_ADD_CLIENT:
		if (rc->rc_open == 0 && rtelnet_connect_host(cc) < 0)
			return (-1);
		TAILQ_INSERT_TAIL(&rc->rc_children, sender, cc_qent_parent);
		rc->rc_client_cnt++;
		break;

	case CONTEXT_IOCTL_NEW_SERVER:
		break;

	case CONTEXT_IOCTL_DEL_CLIENT:
		TAILQ_REMOVE(&rc->rc_children, sender, cc_qent_parent);
		rc->rc_client_cnt--;
		if (rc->rc_ro.ro_closeidle && cc->cc_ctx->c_client_count == 0)
			rtelnet_disconnect_host(cc, 0);
		break;
	}

	return (0);
}

static void
rtelnet_opt_reply(struct rtelnet_ctx *rc, unsigned char opt, int resp)
{
	unsigned char buff[3];

	buff[0] = IAC;
	buff[1] = (unsigned char)resp;
	buff[2] = opt;
	(void) buffer_fill(rc->rc_buff, (char *)buff, sizeof(buff));
}

static void
rtelnet_opt_dont(struct rtelnet_ctx *rc, unsigned char opt)
{
	int resp = WONT;

	/*
	 * Telnet server is telling us that we should not use a particular
	 * option.
	 */
	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_dont(): <- DONT %s",
		    TELOPT(opt));
	}

	switch (opt) {
	case TELOPT_BINARY:
	case TELOPT_ECHO:
	case TELOPT_RCP:
	case TELOPT_SGA:
	case TELOPT_NAMS:
	case TELOPT_STATUS:
	case TELOPT_TM:
	case TELOPT_RCTE:
	case TELOPT_NAOL:
	case TELOPT_NAOP:
	case TELOPT_NAOCRD:
	case TELOPT_NAOHTS:
	case TELOPT_NAOHTD:
	case TELOPT_NAOFFD:
	case TELOPT_NAOVTS:
	case TELOPT_NAOVTD:
	case TELOPT_NAOLFD:
	case TELOPT_XASCII:
	case TELOPT_LOGOUT:
	case TELOPT_BM:
	case TELOPT_DET:
	case TELOPT_SUPDUP:
	case TELOPT_SUPDUPOUTPUT:
	case TELOPT_SNDLOC:
	case TELOPT_TTYPE:
	case TELOPT_EOR:
	case TELOPT_TUID:
	case TELOPT_OUTMRK:
	case TELOPT_TTYLOC:
	case TELOPT_3270REGIME:
	case TELOPT_X3PAD:
	case TELOPT_NAWS:
	case TELOPT_TSPEED:
	case TELOPT_LFLOW:
	case TELOPT_LINEMODE:
	case TELOPT_XDISPLOC:
	case TELOPT_OLD_ENVIRON:
	case TELOPT_AUTHENTICATION:
	case TELOPT_ENCRYPT:
	case TELOPT_NEW_ENVIRON:
	case TELOPT_TN3270E:
	case TELOPT_CHARSET:
	case TELOPT_COMPORT:
	case TELOPT_KERMIT:
		break;

	case TELOPT_EXOPL:
		syslog(LOG_ALERT, "rtelnet_opt_dont(): <- DONT TELOPT_EXOPL");
		break;

	default:
		syslog(LOG_ALERT, "rtelnet_opt_dont(): <- DONT ? (option %d)",
		    opt);
		break;
	}

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_dont(): -> %s %s", TELCMD(resp),
		    TELOPT(opt));
	}

	rc->rc_handler = NULL;
	rtelnet_opt_reply(rc, opt, resp);
}

static void
rtelnet_opt_do(struct rtelnet_ctx *rc, unsigned char opt)
{
	int resp = WONT;

	/*
	 * Telnet server would like us to use a particular option
	 */

	if (rtelnet_verbose && TELOPT_OK(opt))
		syslog(LOG_ALERT, "rtelnet_opt_do(): <- DO %s", TELOPT(opt));

	switch (opt) {
	case TELOPT_BINARY:
		resp = WILL;
		break;

	case TELOPT_ECHO:
	case TELOPT_RCP:
	case TELOPT_SGA:
	case TELOPT_NAMS:
	case TELOPT_STATUS:
	case TELOPT_TM:
	case TELOPT_RCTE:
	case TELOPT_NAOL:
	case TELOPT_NAOP:
	case TELOPT_NAOCRD:
	case TELOPT_NAOHTS:
	case TELOPT_NAOHTD:
	case TELOPT_NAOFFD:
	case TELOPT_NAOVTS:
	case TELOPT_NAOVTD:
	case TELOPT_NAOLFD:
	case TELOPT_XASCII:
	case TELOPT_LOGOUT:
	case TELOPT_BM:
	case TELOPT_DET:
	case TELOPT_SUPDUP:
	case TELOPT_SUPDUPOUTPUT:
	case TELOPT_SNDLOC:
	case TELOPT_TTYPE:
	case TELOPT_EOR:
	case TELOPT_TUID:
	case TELOPT_OUTMRK:
	case TELOPT_TTYLOC:
	case TELOPT_3270REGIME:
	case TELOPT_X3PAD:
	case TELOPT_NAWS:
	case TELOPT_TSPEED:
	case TELOPT_LFLOW:
	case TELOPT_LINEMODE:
	case TELOPT_XDISPLOC:
	case TELOPT_OLD_ENVIRON:
	case TELOPT_AUTHENTICATION:
	case TELOPT_ENCRYPT:
	case TELOPT_NEW_ENVIRON:
	case TELOPT_TN3270E:
	case TELOPT_CHARSET:
	case TELOPT_COMPORT:
	case TELOPT_KERMIT:
		break;

	case TELOPT_EXOPL:
		syslog(LOG_ALERT, "rtelnet_opt_do(): <- DO TELOPT_EXOPL");
		break;

	default:
		syslog(LOG_ALERT, "rtelnet_opt_do(): <- DO ? (option %d)",
		    opt);
		break;
	}

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_do(): -> %s %s", TELCMD(resp),
		    TELOPT(opt));
	}

	rc->rc_handler = NULL;
	rtelnet_opt_reply(rc, opt, resp);
}

static void
rtelnet_opt_wont(struct rtelnet_ctx *rc, unsigned char opt)
{
	int resp = DONT;

	/*
	 * Telnet server is telling us that it won't use a particular option
	 */

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_wont(): <- WONT %s",
		    TELOPT(opt));
	}

	switch (opt) {
	case TELOPT_BINARY:
	case TELOPT_ECHO:
	case TELOPT_RCP:
	case TELOPT_SGA:
	case TELOPT_NAMS:
	case TELOPT_STATUS:
	case TELOPT_TM:
	case TELOPT_RCTE:
	case TELOPT_NAOL:
	case TELOPT_NAOP:
	case TELOPT_NAOCRD:
	case TELOPT_NAOHTS:
	case TELOPT_NAOHTD:
	case TELOPT_NAOFFD:
	case TELOPT_NAOVTS:
	case TELOPT_NAOVTD:
	case TELOPT_NAOLFD:
	case TELOPT_XASCII:
	case TELOPT_LOGOUT:
	case TELOPT_BM:
	case TELOPT_DET:
	case TELOPT_SUPDUP:
	case TELOPT_SUPDUPOUTPUT:
	case TELOPT_SNDLOC:
	case TELOPT_TTYPE:
	case TELOPT_EOR:
	case TELOPT_TUID:
	case TELOPT_OUTMRK:
	case TELOPT_TTYLOC:
	case TELOPT_3270REGIME:
	case TELOPT_X3PAD:
	case TELOPT_NAWS:
	case TELOPT_TSPEED:
	case TELOPT_LFLOW:
	case TELOPT_LINEMODE:
	case TELOPT_XDISPLOC:
	case TELOPT_OLD_ENVIRON:
	case TELOPT_AUTHENTICATION:
	case TELOPT_ENCRYPT:
	case TELOPT_NEW_ENVIRON:
	case TELOPT_TN3270E:
	case TELOPT_CHARSET:
	case TELOPT_COMPORT:
	case TELOPT_KERMIT:
		break;

	case TELOPT_EXOPL:
		syslog(LOG_ALERT, "rtelnet_opt_wont(): <- WONT TELOPT_EXOPL");
		break;

	default:
		syslog(LOG_ALERT, "rtelnet_opt_wont(): <- WONT ? (option %d)",
		    opt);
		break;
	}

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_wont(): -> %s %s", TELCMD(resp),
		    TELOPT(opt));
	}

	rc->rc_handler = NULL;
	rtelnet_opt_reply(rc, opt, resp);
}

static void
rtelnet_opt_will(struct rtelnet_ctx *rc, unsigned char opt)
{
	int resp = DONT;

	/*
	 * Telnet server is telling us that it will use a particular option
	 */

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_will(): <- WILL %s",
		    TELOPT(opt));
	}

	switch (opt) {
	case TELOPT_ECHO:
	case TELOPT_SGA:
		resp = DO;
		break;

	case TELOPT_BINARY:
	case TELOPT_RCP:
	case TELOPT_NAMS:
	case TELOPT_STATUS:
	case TELOPT_TM:
	case TELOPT_RCTE:
	case TELOPT_NAOL:
	case TELOPT_NAOP:
	case TELOPT_NAOCRD:
	case TELOPT_NAOHTS:
	case TELOPT_NAOHTD:
	case TELOPT_NAOFFD:
	case TELOPT_NAOVTS:
	case TELOPT_NAOVTD:
	case TELOPT_NAOLFD:
	case TELOPT_XASCII:
	case TELOPT_LOGOUT:
	case TELOPT_BM:
	case TELOPT_DET:
	case TELOPT_SUPDUP:
	case TELOPT_SUPDUPOUTPUT:
	case TELOPT_SNDLOC:
	case TELOPT_TTYPE:
	case TELOPT_EOR:
	case TELOPT_TUID:
	case TELOPT_OUTMRK:
	case TELOPT_TTYLOC:
	case TELOPT_3270REGIME:
	case TELOPT_X3PAD:
	case TELOPT_NAWS:
	case TELOPT_TSPEED:
	case TELOPT_LFLOW:
	case TELOPT_LINEMODE:
	case TELOPT_XDISPLOC:
	case TELOPT_OLD_ENVIRON:
	case TELOPT_AUTHENTICATION:
	case TELOPT_ENCRYPT:
	case TELOPT_NEW_ENVIRON:
	case TELOPT_TN3270E:
	case TELOPT_CHARSET:
	case TELOPT_COMPORT:
	case TELOPT_KERMIT:
		break;

	case TELOPT_EXOPL:
		syslog(LOG_ALERT, "rtelnet_opt_will(): <- WILL TELOPT_EXOPL");
		break;

	default:
		syslog(LOG_ALERT, "rtelnet_opt_will(): <- WILL ? (option %d)",
		    opt);
		break;
	}

	if (rtelnet_verbose && TELOPT_OK(opt)) {
		syslog(LOG_ALERT, "rtelnet_opt_will(): -> %s %s", TELCMD(resp),
		    TELOPT(opt));
	}

	rc->rc_handler = NULL;
	rtelnet_opt_reply(rc, opt, resp);
}

static void
rtelnet_command(struct rtelnet_ctx *rc, unsigned char cmd)
{

	rc->rc_sawiac = 0;

	switch (cmd) {
	case DONT:
		rc->rc_handler = rtelnet_opt_dont;
		break;

	case DO:
		rc->rc_handler = rtelnet_opt_do;
		break;

	case WONT:
		rc->rc_handler = rtelnet_opt_wont;
		break;

	case WILL:
		rc->rc_handler = rtelnet_opt_will;
		break;

	default:
		syslog(LOG_ALERT, "rtelnet_command(): unhandled command: %d",
		    cmd);
		rc->rc_handler = NULL;
		break;
	}
}

static int
rtelnet_connect_host(struct client_ctx *cc)
{
	struct rtelnet_ctx *rc = cc->cc_data;
	struct addrinfo hints, *res, *res0;
	const char *cause;
	char port[16];
	int rv, fd, fd2, eno;

	if (rc->rc_open)
		return (0);

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	snprintf(port, sizeof(port), "%d", rc->rc_ro.ro_port);

	if ((rv = getaddrinfo(rc->rc_ro.ro_host, port, &hints, &res0)) != 0) {
		syslog(LOG_ALERT, "rtelnet_connect_host(): getaddrinfo(%s:%s) "
		    "failed: %s", rc->rc_ro.ro_host, port, gai_strerror(rv));
		return (-1);
	}

	rv = 0;
	for (cause = NULL, fd = -1, res = res0; res; res = res->ai_next) {
		if (res->ai_protocol != IPPROTO_TCP ||
		    res->ai_family != PF_INET || res->ai_family == PF_INET6)
			continue;

		if ((fd = socket(res->ai_family, res->ai_socktype,
		    res->ai_protocol)) < 0) {
			cause = "socket";
			continue;
		}

		if (fcntl(fd, F_SETFL, O_NONBLOCK) < 0) {
			cause = "fcntl";
			close(fd);
			fd = -1;
			continue;
		}

		if ((rv = connect(fd, res->ai_addr, res->ai_addrlen)) < 0 &&
		    errno != EINPROGRESS) {
			cause = "connect";
			close(fd);
			fd = -1;
			continue;
		}

		break;
	}

	eno = errno;
	freeaddrinfo(res0);

	if (fd < 0) {
		syslog(LOG_ALERT, "rtelnet_connect_host(): failed to connect "
		    "to %s:%s: %s: %s", rc->rc_ro.ro_host, port, cause,
		    strerror(eno));
		return (-1);
	}

	fd2 = dup2(fd, cc->cc_fd);
	(void) close(fd);

	if (fd2 != cc->cc_fd) {
		syslog(LOG_ALERT, "rtelnet_connect_host(): dup2() failed (%s)",
		    strerror(errno));
		return (-1);
	}

	rc->rc_open = 1;

	if (rv < 0) {
		/*
		 * The connect(2) is in-progress. It will complete
		 * asynchronously.
		 */
		rc->rc_connect_pending = 1;
	}

	return (0);
}

static void
rtelnet_disconnect_host(struct client_ctx *cc, int kill_clients)
{
	struct rtelnet_ctx *rc = cc->cc_data;
	struct client_ctx *ccc;
	int fd, fd2;

	if (rc->rc_open == 0)
		return;
	rc->rc_open = 0;
	rc->rc_connect_pending = 0;

	if (kill_clients) {
		volatile struct client_qhead *qq = &rc->rc_children; 

		while ((ccc = TAILQ_FIRST(qq)) != NULL)
			context_del_client(cc->cc_ctx, ccc);
	}

	if ((fd = open(_PATH_DEVNULL, O_RDWR)) < 0) {
		syslog(LOG_ALERT, "rtelnet_disconnect_host(): failed to open "
		    "%s (%s)", _PATH_DEVNULL, strerror(errno));
		return;
	}

	fd2 = dup2(fd, cc->cc_fd);
	close(fd);

	if (fd2 != cc->cc_fd) {
		syslog(LOG_ALERT, "rtelnet_disconnect_host(): dup2 failed (%s)",
		    strerror(errno));
		return;
	}
}

static void
rtelnet_peer_disconnected(struct client_ctx *cc)
{

	rtelnet_disconnect_host(cc, 0);
	rtelnet_connect_host(cc);
}

static void
rtelnet_send_break(struct client_ctx *cc)
{
	struct rtelnet_ctx *rc = cc->cc_data;
	unsigned char buff[2];

	buff[0] = IAC;
	buff[1] = BREAK;
	(void) buffer_fill(rc->rc_buff, (char *)buff, sizeof(buff));
}

/* ARGSUSED */
static const char *
cf_rtelnet_host(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	struct rtelnet_opt *ro = arg;
	struct addrinfo hints, *res, *res0;
	int rv;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;

	if ((rv = getaddrinfo(argv[1], NULL, &hints, &res0)) != 0)
		return (config_err(cs, "%s - %s", argv[1], gai_strerror(rv)));

	for (res = res0, rv = 0; res; res = res->ai_next) {
		if (res->ai_protocol == IPPROTO_TCP &&
		    (res->ai_family == PF_INET || res->ai_family == PF_INET6))
			rv++;
	}

	freeaddrinfo(res0);

	if (rv == 0)
		return (config_err(cs,"No suitable address for '%s'", argv[1]));

	if ((ro->ro_host = strdup(argv[1])) == NULL)
		return (config_err(cs, "%s", strerror(errno)));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_rtelnet_port(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	struct rtelnet_opt *ro = arg;
	struct servent *se;

	if (isdigit((unsigned char)argv[1][0])) {
		ro->ro_port = atoi(argv[1]);
		if (ro->ro_port < 0 || ro->ro_port > IPPORT_ANONMAX)
			return (config_err(cs, "Invalid port number '%s'",
			    argv[1]));
	} else
	if ((se = getservbyname(argv[1], "tcp")) != NULL)
		ro->ro_port = se->s_port;
	else
		return (config_err(cs, "Service not found '%s'", argv[1]));

	return (NULL);
}

static const char *
cf_rtelnet_closeidle(void *cs, char **argv, int argc, int is_compound,
    void *arg)
{
	struct rtelnet_opt *ro = arg;
	const char *errstr;

	if ((errstr = config_boolean(cs, argv[1], &ro->ro_closeidle)) != NULL)
		return (errstr);

	return (NULL);
}

static const char *
cf_rtelnet_raw(void *cs, char **argv, int argc, int is_compound,
    void *arg)
{
	struct rtelnet_opt *ro = arg;
	const char *errstr;

	if ((errstr = config_boolean(cs, argv[1], &ro->ro_raw)) != NULL)
		return (errstr);

	return (NULL);
}
