/*	$Id: masterpty.c,v 1.3 2006/10/06 23:38:24 steve Exp $	*/

/*-
 * Copyright (c) 2004 Steve C. Woodford.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Steve C. Woodford.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>

#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <paths.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "context.h"
#include "buffer.h"
#include "config.h"
#include "masterpty.h"

static const char *cf_masterpty_init(void *, char **, int, int, void *);
static int	masterpty_init(struct client_ctx *, const void *);
static void	masterpty_destroy(struct client_ctx *);
static int	masterpty_event(struct client_ctx *, int);
static int	masterpty_output(struct client_ctx *, const char *, size_t,
		    struct client_ctx *);
static int	masterpty_write_pending(struct client_ctx *);
static int	masterpty_open_pt(struct client_ctx *);

struct client_ops masterpty_ops = {
	"masterpty",
	cf_masterpty_init, masterpty_init, masterpty_destroy, masterpty_event,
	masterpty_output, NULL, masterpty_write_pending, NULL
};

static const char *cf_masterpty_symlinkdir(void *, char **, int, int, void *);
static const char *cf_masterpty_symlinkname(void *, char **, int, int, void *);
static const char *cf_masterpty_readonly(void *, char **, int, int, void *);
static const char *cf_masterpty_allowbreak(void *, char **, int, int, void *);

static struct config_tokens cf_masterpty_tokens[] = {
	{"symlinkdir",	1,	cf_masterpty_symlinkdir},
	{"symlinkname",	1,	cf_masterpty_symlinkname},
	{"readonly",	1,	cf_masterpty_readonly},
	{"allowbreak",	1,	cf_masterpty_allowbreak},
	{NULL, 0, NULL}
};

struct masterpty_args {
	char	ma_linkdir[FILENAME_MAX];
	char	ma_linkname[FILENAME_MAX];
	char	*ma_name;
	struct client_options ma_copts;
};

struct masterpty_ctx {
	struct masterpty_args mc_ma;
	struct buffer_ctx *mc_bc;
	char *mc_slavename;
	char *mc_symlink;
};


static int
masterpty_init(struct client_ctx *cc, const void *arg)
{
	const struct masterpty_args *ma = arg;
	struct masterpty_ctx *mc;

	if ((mc = calloc(1, sizeof(*mc))) == NULL) {
		close(cc->cc_fd);
		return (-1);
	}
	cc->cc_data = (void *) mc;
	cc->cc_name = NULL;
	mc->mc_ma = *ma;

	if (masterpty_open_pt(cc) < 0) {
		free(mc);
		return (-1);
	}

	return (0);
}

static void
masterpty_destroy(struct client_ctx *cc)
{
	struct masterpty_ctx *mc = cc->cc_data;

	buffer_destroy(mc->mc_bc);
	(void) close(cc->cc_fd);
	(void) unlink(mc->mc_symlink);
	(void) close(cc->cc_fd);
	(void) free(mc->mc_symlink);
	(void) free(mc->mc_slavename);
	(void) free(mc);
}

static int
masterpty_event(struct client_ctx *cc, int events)
{
	struct masterpty_ctx *mc = cc->cc_data;
	char buff[256];
	ssize_t rv = 0;

	if ((events & POLLHUP) != 0) {
		/* Slave has been closed */
		return (masterpty_open_pt(cc));
	}

	if ((events & POLLRDNORM) != 0) {
		while ((rv = read(cc->cc_fd, buff, sizeof(buff))) != 0) {
			if (rv < 0) {
				if (errno == EINTR || errno == EWOULDBLOCK ||
				    errno == EAGAIN) {
					rv = 0;
					break;
				}
				return (-1);
			}

			context_server_output(cc->cc_ctx, buff, (size_t)rv, cc);
		}
	}

	if (rv >= 0 && (events & POLLWRNORM) != 0)
		rv = buffer_drain(mc->mc_bc);

	return (rv);
}

static int
masterpty_open_pt(struct client_ctx *cc)
{
	struct masterpty_ctx *mc = cc->cc_data;
	char *slavename;
	char buff[1024];
	int fd, fd2;

	cc->cc_name = NULL;

	if (mc->mc_bc) {
		buffer_destroy(mc->mc_bc);
		mc->mc_bc = NULL;
	}

	if (mc->mc_symlink) {
		(void) unlink(mc->mc_symlink);
		(void) free(mc->mc_symlink);
		mc->mc_symlink = NULL;
	}

	if (mc->mc_slavename) {
		(void) free(mc->mc_slavename);
		mc->mc_slavename = NULL;
	}

	if (cc->cc_fd >= 0) {
		fd = open(_PATH_DEVNULL, O_RDWR);
		if (fd < 0)
			return (-1);
		fd2 = dup2(fd, cc->cc_fd);
		(void) close(fd);

		if (fd2 != cc->cc_fd)
			return (-1);
	}

	if ((fd = posix_openpt(O_RDWR | O_NONBLOCK)) < 0)
		return (-1);

	if (unlockpt(fd) < 0 || grantpt(fd) < 0) {
		close(fd);
		return (-1);
	}

	if ((slavename = ptsname(fd)) == NULL) {
		close(fd);
		return (-1);
	}

	if ((mc->mc_slavename = strdup(slavename)) == NULL) {
		close(fd);
		return (-1);
	}

	if (strlen(mc->mc_ma.ma_linkname) == 0)
		snprintf(buff, sizeof(buff), "%s/%s", mc->mc_ma.ma_linkdir,
		    basename(mc->mc_slavename));
	else
		snprintf(buff, sizeof(buff), "%s/%s", mc->mc_ma.ma_linkdir,
		    mc->mc_ma.ma_linkname);

	if ((mc->mc_symlink = strdup(buff)) == NULL) {
		(void) free(mc->mc_slavename);
		mc->mc_slavename = NULL;
		close(fd);
		return (-1);
	}

	if (symlink(mc->mc_slavename, mc->mc_symlink) < 0) {
		(void) free(mc->mc_slavename);
		(void) free(mc->mc_symlink);
		mc->mc_slavename = NULL;
		mc->mc_symlink = NULL;
		close(fd);
		return (-1);
	}

	if (cc->cc_fd >= 0) {
		fd2 = dup2(fd, cc->cc_fd);
		(void) close(fd);

		if (fd2 != cc->cc_fd) {
			(void) unlink(mc->mc_symlink);
			(void) free(mc->mc_slavename);
			(void) free(mc->mc_symlink);
			mc->mc_slavename = NULL;
			mc->mc_symlink = NULL;
			return (-1);
		}
	} else
		cc->cc_fd = fd;

	if (buffer_init(&mc->mc_bc, cc->cc_fd) < 0) {
		(void) unlink(mc->mc_symlink);
		(void) free(mc->mc_slavename);
		(void) free(mc->mc_symlink);
		mc->mc_slavename = NULL;
		mc->mc_symlink = NULL;
		return (-1);
	}

	cc->cc_name = mc->mc_slavename;

	return (0);
}

/* ARGSUSED */
static int
masterpty_output(struct client_ctx *cc, const char *buff, size_t len,
    struct client_ctx *sender)
{
	struct masterpty_ctx *mc = cc->cc_data;

	return buffer_fill(mc->mc_bc, buff, len);
}

static int
masterpty_write_pending(struct client_ctx *cc)
{
	struct masterpty_ctx *mc = cc->cc_data;

	return (buffer_length(mc->mc_bc) != 0);
}

/* ARGSUSED */
static const char *
cf_masterpty_init(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	struct context *ctx = arg;
	struct client_ctx *cc;
	struct masterpty_args ma;
	const char *errstr;

	memset(&ma, 0, sizeof(ma));
	ma.ma_copts.co_allowbreak = 1;
	strcpy(ma.ma_linkdir, TITS_DEFAULT_LINKDIR);

	errstr = config_parse(cs, cf_masterpty_tokens, (void *)&ma);
	if (errstr)
		return (errstr);

	if (client_init(&cc, NULL, &masterpty_ops, &ma.ma_copts, &ma) < 0)
		errstr = config_err(cs, "masterpty: %s", strerror(errno));
	else
	if (context_add_client(ctx, cc) < 0) {
		errstr = config_err(cs, "masterpty: %s", strerror(errno));
		client_destroy(cc);
	}

	return (errstr);
}

/* ARGSUSED */
static const char *
cf_masterpty_symlinkdir(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	strncpy(ma->ma_linkdir, argv[1], sizeof(ma->ma_linkdir));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_masterpty_symlinkname(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	strncpy(ma->ma_linkname, argv[1], sizeof(ma->ma_linkname));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_masterpty_readonly(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	return (config_boolean(cs, argv[1], &ma->ma_copts.co_readonly));
}

/* ARGSUSED */
static const char *
cf_masterpty_allowbreak(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct masterpty_args *ma = arg;

	return (config_boolean(cs, argv[1], &ma->ma_copts.co_allowbreak));
}
