/*	$Id: main.c,v 1.48 2015/02/25 15:15:18 kristaps Exp $ */
/*
 * Copyright (c) 2015 Kristaps Dzonsons <kristaps@bsd.lv>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#if defined(__linux__) || defined(__MINT__)
# define _GNU_SOURCE /* memmem */
#endif
#include <assert.h>
#include <ctype.h>
#include <getopt.h>
#include <libgen.h>
#include <limits.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "extern.h"

#define	SECTSZ 4
static	const char *const sects[SECTSZ] = {
	"Sh",
	"Ss",
	"Em",
	"No",
};

static	void doaccent(struct texi *, enum texicmd, size_t *);
static	void doblock(struct texi *, enum texicmd, size_t *);
static	void dobracket(struct texi *, enum texicmd, size_t *);
static	void dobye(struct texi *, enum texicmd, size_t *);
static	void dodefindex(struct texi *, enum texicmd, size_t *);
static	void dodefn(struct texi *, enum texicmd, size_t *);
static	void dodisplay(struct texi *, enum texicmd, size_t *);
static	void doend(struct texi *, enum texicmd, size_t *);
static	void doenumerate(struct texi *, enum texicmd, size_t *);
static	void doexample(struct texi *, enum texicmd, size_t *);
static	void doignargn(struct texi *, enum texicmd, size_t *);
static	void doignblock(struct texi *, enum texicmd, size_t *);
static	void doignbracket(struct texi *, enum texicmd, size_t *);
static	void doignline(struct texi *, enum texicmd, size_t *);
static	void doinline(struct texi *, enum texicmd, size_t *);
static	void doinclude(struct texi *, enum texicmd, size_t *);
static	void doitem(struct texi *, enum texicmd, size_t *);
static	void doitemize(struct texi *, enum texicmd, size_t *);
static	void dolink(struct texi *, enum texicmd, size_t *);
static	void domacro(struct texi *, enum texicmd, size_t *);
static	void domath(struct texi *, enum texicmd, size_t *);
static	void domultitable(struct texi *, enum texicmd, size_t *);
static	void doquotation(struct texi *, enum texicmd, size_t *);
static	void dotable(struct texi *, enum texicmd, size_t *);
static	void dotop(struct texi *, enum texicmd, size_t *);
static	void dosecoffs(struct texi *, enum texicmd, size_t *);
static	void dosection(struct texi *, enum texicmd, size_t *);
static	void dosp(struct texi *, enum texicmd, size_t *);
static	void dosubsection(struct texi *, enum texicmd, size_t *);
static	void dosubsubsection(struct texi *, enum texicmd, size_t *);
static	void dosymbol(struct texi *, enum texicmd, size_t *);
static	void dotab(struct texi *, enum texicmd, size_t *);
static	void dotitle(struct texi *, enum texicmd, size_t *);
static	void dovalue(struct texi *, enum texicmd, size_t *);
static	void doverb(struct texi *, enum texicmd, size_t *);
static	void doverbatim(struct texi *, enum texicmd, size_t *);
static	void doverbinclude(struct texi *, enum texicmd, size_t *);

static	const struct texitok __texitoks[TEXICMD__MAX] = {
	/* TEXICMD__BEGIN */
	{ doignline, "afourpaper", 10 }, /* TEXICMD_A4PAPER */
	{ dosymbol, "AA", 2 }, /* TEXICMD_AA */
	{ dosymbol, "aa", 2 }, /* TEXICMD_AASMALL */
	{ doignargn, "acronym", 7 }, /* TEXICMD_ACRONYM */
	{ doaccent, "'", 1 }, /* TEXICMD_ACUTE */
	{ dosymbol, "AE", 2 }, /* TEXICMD_AE */
	{ dosymbol, "ae", 2 }, /* TEXICMD_AESMALL */
	{ doignbracket, "anchor", 6 }, /* TEXICMD_ANCHOR */
	{ dosection, "appendix", 8 }, /* TEXICMD_APPENDIX */
	{ dosection, "appendixsec", 11 }, /* TEXICMD_APPENDIXSEC */
	{ dosubsection, "appendixsubsec", 14 }, /* TEXICMD_APPENDIXSUBSEC */
	{ dosubsubsection, "appendixsubsubsec", 17 }, /* TEXICMD_APPENDIXSUBSUBSEC */
	{ doinline, "asis", 4 }, /* TEXICMD_ASIS */
	{ dosymbol, "*", 1 }, /* TEXICMD_ASTERISK */
	{ dosymbol, "@", 1 }, /* TEXICMD_AT */
	{ doignline, "author", 6 }, /* TEXICMD_AUTHOR */
	{ doinline, "b", 1 }, /* TEXICMD_BOLD */
	{ dosymbol, "!", 1 }, /* TEXICMD_BANG */
	{ dosymbol, "bullet", 6 }, /* TEXICMD_BULLET */
	{ dobye, "bye", 3 }, /* TEXICMD_BYE */
	{ doblock, "cartouche", 9 }, /* TEXICMD_CARTOUCHE */
	{ doaccent, ",", 1 }, /* TEXICMD_CEDILLA */
	{ doignline, "center", 6 }, /* TEXICMD_CENTER */
	{ dosection, "chapter", 7 }, /* TEXICMD_CHAPTER */
	{ doignline, "cindex", 6 }, /* TEXICMD_CINDEX */
	{ doaccent, "^", 1 }, /* TEXICMD_CIRCUMFLEX */
	{ doinline, "cite", 4 }, /* TEXICMD_CITE */
	{ dovalue, "clear", 5 }, /* TEXICMD_CLEAR */
	{ doinline, "code", 4 }, /* TEXICMD_CODE */
	{ dosymbol, ":", 1 }, /* TEXICMD_COLON */
	{ NULL, "columnfractions", 15 }, /* TEXICMD_COLUMNFRACTIONS */
	{ dosymbol, "comma", 5 }, /* TEXICMD_COMMA */
	{ doinline, "command", 7 }, /* TEXICMD_COMMAND */
	{ doignline, "c", 1 }, /* TEXICMD_COMMENT */
	{ doignline, "comment", 7 }, /* TEXICMD_COMMENT_LONG */
	{ doignline, "contents", 8 }, /* TEXICMD_CONTENTS */
	{ doignblock, "copying", 7 }, /* TEXICMD_COPYING */
	{ dosymbol, "copyright", 9 }, /* TEXICMD_COPYRIGHT */
	{ dodefindex, "defcodeindex", 12 }, /* TEXICMD_DEFCODEINDEX */
	{ dodefn, "deffn", 5 }, /* TEXICMD_DEFFN */
	{ dodefn, "deffnx", 6 }, /* TEXICMD_DEFFNX */
	{ dodefindex, "defindex", 8 }, /* TEXICMD_DEFINDEX */
	{ dodefn, "defmac", 6 }, /* TEXICMD_DEFMAC */
	{ dodefn, "defmacx", 7 }, /* TEXICMD_DEFMACX */
	{ dodefn, "deftp", 5 }, /* TEXICMD_DEFTP */
	{ dodefn, "deftpx", 6 }, /* TEXICMD_DEFTPX */
	{ dodefn, "deftypefn", 9 }, /* TEXICMD_DEFTYPEFN */
	{ dodefn, "deftypefnx", 10 }, /* TEXICMD_DEFTYPEFNX */
	{ dodefn, "deftypefun", 10 }, /* TEXICMD_DEFTYPEFUN */
	{ dodefn, "deftypefunx", 11 }, /* TEXICMD_DEFTYPEFUNX */
	{ dodefn, "deftypemethod", 13 }, /* TEXICMD_DEFTYPEMETHOD */
	{ dodefn, "deftypemethodx", 14 }, /* TEXICMD_DEFTYPEMETHODX */
	{ dodefn, "deftypevar", 10 }, /* TEXICMD_DEFTYPEVAR */
	{ dodefn, "deftypevarx", 11 }, /* TEXICMD_DEFTYPEVARX */
	{ dodefn, "deftypevr", 9 }, /* TEXICMD_DEFTYPEVR */
	{ dodefn, "deftypevrx", 10 }, /* TEXICMD_DEFTYPEVRX */
	{ dodefn, "defun", 5 }, /* TEXICMD_DEFUN */
	{ dodefn, "defunx", 6 }, /* TEXICMD_DEFUNX */
	{ dodefn, "defvar", 6 }, /* TEXICMD_DEFVAR */
	{ dodefn, "defvarx", 7 }, /* TEXICMD_DEFVARX */
	{ dodefn, "defvr", 5 }, /* TEXICMD_DEFVR */
	{ dodefn, "defvrx", 6 }, /* TEXICMD_DEFVRX */
	{ doignblock, "detailmenu", 10 }, /* TEXICMD_DETAILMENU */
	{ doinline, "dfn", 3 }, /* TEXICMD_DFN */
	{ dosymbol, "DH", 2 }, /* TEXICMD_DH */
	{ dosymbol, "dh", 2 }, /* TEXICMD_DHSMALL */
	{ doignline, "dircategory", 11 }, /* TEXICMD_DIRCATEGORY */
	{ doignblock, "direntry", 8 }, /* TEXICMD_DIRENTRY */
	{ dodisplay, "display", 7 }, /* TEXICMD_DISPLAY */
	{ doignbracket, "dmn", 3 }, /* TEXICMD_DMN */
	{ doignblock, "documentdescription", 19 }, /* TEXICMD_DOCUMENTDESCRIPTION */
	{ doignline, "documentencoding", 16 }, /* TEXICMD_DOCUMENTENCODING */
	{ doignline, "documentlanguage", 16 }, /* TEXICMD_DOCUMENTLANGUAGE */
	{ doaccent, "dotaccent", 9 }, /* TEXICMD_DOTACCENT */
	{ doaccent, "dotless", 7 }, /* TEXICMD_DOTLESS */
	{ dosymbol, "dots", 4 }, /* TEXICMD_DOTS */
	{ dolink, "email", 5 }, /* TEXICMD_EMAIL */
	{ doinline, "emph", 4 }, /* TEXICMD_EMPH */
	{ doend, "end", 3 }, /* TEXICMD_END */
	{ dosymbol, "enddots", 7 }, /* TEXICMD_ENDDOTS */
	{ doenumerate, "enumerate", 9 }, /* TEXICMD_ENUMERATE */
	{ doinline, "env", 3 }, /* TEXICMD_ENV */
	{ dosymbol, "equiv", 5 }, /* TEXICMD_EQUIV */
	{ dosymbol, "error", 5 }, /* TEXICMD_ERROR */
	{ dosymbol, "euro", 4 }, /* TEXICMD_EURO */
	{ doexample, "example", 7 }, /* TEXICMD_EXAMPLE */
	{ dosymbol, "exclamdown", 10 }, /* TEXICMD_EXCLAMDOWN */
	{ doignline, "exdent", 6 }, /* TEXICMD_EXDENT */
	{ dosymbol, "expansion", 9 }, /* TEXICMD_EXPANSION */
	{ doinline, "file", 4 }, /* TEXICMD_FILE */
	{ doignline, "finalout", 8 }, /* TEXICMD_FINALOUT */
	{ doignline, "findex", 6 }, /* TEXICMD_FINDEX */
	{ doblock, "flushleft", 9 }, /* TEXICMD_FLUSHLEFT */
	{ doblock, "flushright", 10 }, /* TEXICMD_FLUSHRIGHT */
	{ doignline, "firstparagraphindent", 20 }, /* TEXICMD_FIRSTPARAGRAPHINDENT */
	{ doignbracket, "footnote", 8 }, /* TEXICMD_FOOTNOTE */
	{ doignline, "footnotestyle", 13 }, /* TEXICMD_FOOTNOTESTYLE */
	{ dotable, "ftable", 6 }, /* TEXICMD_FTABLE */
	{ dodisplay, "format", 6 }, /* TEXICMD_FORMAT */
	{ dosymbol, "geq", 3 }, /* TEXICMD_GEQ */
	{ doaccent, "`", 1 }, /* TEXICMD_GRAVE */
	{ doblock, "group", 5 }, /* TEXICMD_GROUP */
	{ dosymbol, "guillemetleft", 13 }, /* TEXICMD_GUILLEMETLEFT */
	{ dosymbol, "guillemetright", 14 }, /* TEXICMD_GUILLEMETRIGHT */
	{ dosymbol, "guillemotleft", 13 }, /* TEXICMD_GUILLEMOTLEFT */
	{ dosymbol, "guillemotright", 14 }, /* TEXICMD_GUILLEMOTRIGHT */
	{ dosymbol, "guilsinglleft", 13 }, /* TEXICMD_GUILSINGLLEFT */
	{ dosymbol, "guilsinglright", 14 }, /* TEXICMD_GUILSINGLRIGHT */
	{ doaccent, "H", 1 }, /* TEXICMD_H */
	{ dosection, "heading", 7 }, /* TEXICMD_HEADING */
	{ doignline, "headings", 8 }, /* TEXICMD_HEADINGS */
	{ doitem, "headitem", 8 }, /* TEXICMD_HEADITEM */
	{ doignblock, "html", 4 }, /* TEXICMD_HTML */
	{ dosymbol, "-", 1 }, /* TEXICMD_HYPHEN */
	{ doinline, "i", 1 }, /* TEXICMD_I */
	{ dovalue, "ifclear", 7 }, /* TEXICMD_IFCLEAR */
	{ doignblock, "ifdocbook", 9 }, /* TEXICMD_IFDOCBOOK */
	{ doignblock, "ifhtml", 6 }, /* TEXICMD_IFHTML */
	{ doblock, "ifinfo", 6 }, /* TEXICMD_IFINFO */
	{ doblock, "ifnotdocbook", 12 }, /* TEXICMD_IFNOTDOCBOOK */
	{ doblock, "ifnothtml", 9 }, /* TEXICMD_IFNOTHTML */
	{ doblock, "ifnotinfo", 9 }, /* TEXICMD_IFNOTINFO */
	{ doignblock, "ifnotplaintext", 14 }, /* TEXICMD_IFNOTPLAINTEXT */
	{ doblock, "ifnottex", 8 }, /* TEXICMD_IFNOTTEX */
	{ doblock, "ifnotxml", 8 }, /* TEXICMD_IFNOTXML */
	{ doblock, "ifplaintext", 11 }, /* TEXICMD_IFPLAINTEXT */
	{ doignblock, "iftex", 5 }, /* TEXICMD_IFTEX */
	{ dovalue, "ifset", 5 }, /* TEXICMD_IFSET */
	{ doignblock, "ifxml", 5 }, /* TEXICMD_IFXML */
	{ doignblock, "ignore", 6 }, /* TEXICMD_IGNORE */
	{ doignbracket, "image", 5 }, /* TEXICMD_IMAGE */
	{ doinclude, "include", 7 }, /* TEXICMD_INCLUDE */
	{ dodisplay, "indentblock", 11 }, /* TEXICMD_INDENTBLOCK */
	{ dolink, "indicateurl", 11 }, /* TEXICMD_INDICATEURL */
	{ dolink, "inforef", 7 }, /* TEXICMD_INFOREF */
	{ doignline, "insertcopying", 13 }, /* TEXICMD_INSERTCOPYING */
	{ doitem, "item", 4 }, /* TEXICMD_ITEM */
	{ doitemize, "itemize", 7 }, /* TEXICMD_ITEMIZE */
	{ doitem, "itemx", 5 }, /* TEXICMD_ITEMX */
	{ doinline, "kbd", 3 }, /* TEXICMD_KBD */
	{ dobracket, "key", 3 }, /* TEXICMD_KEY */
	{ doignline, "kindex", 6 }, /* TEXICMD_KINDEX */
	{ dosymbol, "L", 1 }, /* TEXICMD_L */
	{ dosymbol, "LaTeX", 5 }, /* TEXICMD_LATEX */
	{ dosymbol, "leq", 3 }, /* TEXICMD_LEQ */
	{ dosecoffs, "lowersections", 13 }, /* TEXICMD_LOWERSECTIONS */
	{ dosymbol, "l", 1 }, /* TEXICMD_LSMALL */
	{ domacro, "macro", 5 }, /* TEXICMD_MACRO */
	{ doaccent, "=", 1 }, /* TEXICMD_MACRON */
	{ domath, "math", 4 }, /* TEXICMD_MATH */
	{ doignblock, "menu", 4 }, /* TEXICMD_MENU */
	{ dosymbol, "minus", 5 }, /* TEXICMD_MINUS */
	{ domultitable, "multitable", 10 }, /* TEXICMD_MULTITABLE */
	{ doignline, "need", 4 }, /* TEXICMD_NEED */
	{ dosymbol, "\n", 1 }, /* TEXICMD_NEWLINE */
	{ doignline, "node", 4 }, /* TEXICMD_NODE */
	{ doignline, "noindent", 8 }, /* TEXICMD_NOINDENT */
	{ dosymbol, "O", 1 }, /* TEXICMD_O */
	{ dosymbol, "OE", 2 }, /* TEXICMD_OE */
	{ dosymbol, "oe", 2 }, /* TEXICMD_OESMALL */
	{ doaccent, "ogonek", 6 }, /* TEXICMD_OGONEK */
	{ doinline, "option", 6 }, /* TEXICMD_OPTION */
	{ dosymbol, "ordf", 4 }, /* TEXICMD_ORDF */
	{ dosymbol, "ordm", 4 }, /* TEXICMD_ORDM */
	{ dosymbol, "o", 1 }, /* TEXICMD_OSMALL */
	{ doignline, "page", 4 }, /* TEXICMD_PAGE */
	{ doignline, "paragraphindent", 15 }, /* TEXICMD_PARINDENT */
	{ dosymbol, ".", 1 }, /* TEXICMD_PERIOD */
	{ doignline, "pindex", 6 }, /* TEXICMD_PINDEX */
	{ dosymbol, "pounds", 6 }, /* TEXICMD_POUNDS */
	{ doignline, "printindex", 10 }, /* TEXICMD_PRINTINDEX */
	{ dolink, "pxref", 5 }, /* TEXICMD_PXREF */
	{ dosymbol, "questiondown", 12 }, /* TEXICMD_QUESTIONDOWN */
	{ dosymbol, "?", 1 }, /* TEXICMD_QUESTIONMARK */
	{ doquotation, "quotation", 9 }, /* TEXICMD_QUOTATION */
	{ dosymbol, "quotedblbase", 12 }, /* TEXICMD_QUOTEDBLBASE */
	{ dosymbol, "quotedblleft", 12 }, /* TEXICMD_QUOTEDBLLEFT */
	{ dosymbol, "quotedblright", 13 }, /* TEXICMD_QUOTEDBLRIGHT */
	{ dosymbol, "quotesinglbase", 14 }, /* TEXICMD_QUOTESINGLBASE */
	{ dosymbol, "quoteleft", 9 }, /* TEXICMD_QUOTELEFT */
	{ dosymbol, "quoteright", 10 }, /* TEXICMD_QUOTERIGHT */
	{ doinline, "r", 1 }, /* TEXICMD_R */
	{ dosecoffs, "raisesections", 13 }, /* TEXICMD_RAISESECTIONS */
	{ dobracket, "ref", 3 }, /* TEXICMD_REF */
	{ doignline, "refill", 6 }, /* TEXICMD_REFILL */
	{ dosymbol, "registeredsymbol", 16 }, /* TEXICMD_REGISTEREDSYMBOL */
	{ dosymbol, "result", 6 }, /* TEXICMD_RESULT */
	{ doaccent, "ringaccent", 10 }, /* TEXICMD_RINGACCENT */
	{ doinline, "samp", 4 }, /* TEXICMD_SAMP */
	{ doinline, "sansserif", 9 }, /* TEXICMD_SANSSERIF */
	{ dobracket, "sc", 2 }, /* TEXICMD_SC */
	{ dosection, "section", 7 }, /* TEXICMD_SECTION */
	{ dovalue, "set", 3 }, /* TEXICMD_SET */
	{ doignline, "setchapternewpage", 17 }, /* TEXICMD_SETCHAPNEWPAGE */
	{ doignline, "setcontentsaftertitlepage", 25 }, /* TEXICMD_SETCONTENTSAFTER */
	{ doignline, "setfilename", 11 }, /* TEXICMD_SETFILENAME */
	{ dotitle, "settitle", 8 }, /* TEXICMD_SETTITLE */
	{ doignline, "shortcontents", 13 }, /* TEXICMD_SHORTCONTENTS */
	{ doinline, "slanted", 7 }, /* TEXICMD_SLANTED */
	{ dosymbol, "/", 1 }, /* TEXICMD_SLASH */
	{ dosp, "sp", 2 }, /* TEXICMD_SP */
	{ dosymbol, " ", 1 }, /* TEXICMD_SPACE */
	{ doignline, "smallbook", 9 }, /* TEXICMD_SMALLBOOK */
	{ dodisplay, "smalldisplay", 12 }, /* TEXICMD_SMALLDISPLAY */
	{ doexample, "smallexample", 12 }, /* TEXICMD_SMALLEXAMPLE */
	{ dodisplay, "smallformat", 11 }, /* TEXICMD_SMALLFORMAT */
	{ dodisplay, "smallindentblock", 16 }, /* TEXICMD_SMALLINDENTBLOCK */
	{ dosymbol, "{", 1 }, /* TEXICMD_SQUIGGLE_LEFT */
	{ dosymbol, "}", 1 }, /* TEXICMD_SQUIGGLE_RIGHT */
	{ dosymbol, "ss", 2 }, /* TEXICMD_SS */
	{ doinline, "strong", 6 }, /* TEXICMD_STRONG */
	{ dosubsection, "subheading", 10 }, /* TEXICMD_SUBHEADING */
	{ dosubsection, "subsection", 10 }, /* TEXICMD_SUBSECTION */
	{ dosubsubsection, "subsubheading", 13 }, /* TEXICMD_SUBSUBHEADING */
	{ dosubsubsection, "subsubsection", 13 }, /* TEXICMD_SUBSUBSECTION */
	{ doignline, "subtitle", 8 }, /* TEXICMD_SUBTITLE */
	{ doignline, "summarycontents", 15 }, /* TEXICMD_SUMMARYCONTENTS */
	{ doignline, "synindex", 8 }, /* TEXICMD_SYNINDEX */
	{ doignline, "syncodeindex", 12 }, /* TEXICMD_SYNCODEINDEX */
	{ doinline, "t", 1 }, /* TEXICMD_T */
	{ dotab, "tab", 3 }, /* TEXICMD_TAB */
	{ dosymbol, "\t", 1 }, /* TEXICMD_TABSYM */
	{ dotable, "table", 5 }, /* TEXICMD_TABLE */
	{ doignblock, "tex", 3 }, /* TEXICMD_TEX */
	{ dosymbol, "TeX", 3 }, /* TEXICMD_TEXSYM */
	{ dosymbol, "textdegree", 10 }, /* TEXICMD_TEXTDEGREE */
	{ dosymbol, "TH", 2 }, /* TEXICMD_TH */
	{ dosymbol, "th", 2 }, /* TEXICMD_THSMALL */
	{ dosymbol, "tie", 3 }, /* TEXICMD_TIE */
	{ doaccent, "tieaccent", 9 }, /* TEXICMD_TIEACCENT */
	{ doaccent, "~", 1 }, /* TEXICMD_TILDE */
	{ doignline, "tindex", 6 }, /* TEXICMD_TINDEX */
	{ doignline, "title", 5 }, /* TEXICMD_TITLE */
	{ dobracket, "titlefont", 9 }, /* TEXICMD_TITLEFONT */
	{ doignblock, "titlepage", 9 }, /* TEXICMD_TITLEPAGE */
	{ dotop, "top", 3 }, /* TEXICMD_TOP */
	{ doaccent, "u", 1 }, /* TEXICMD_U */
	{ doaccent, "ubaraccent", 10 }, /* TEXICMD_UBARACCENT */
	{ doaccent, "udotaccent", 10 }, /* TEXICMD_UDOTACCENT */
	{ doaccent, "\"", 1 }, /* TEXICMD_UMLAUT */
	{ dosection, "unnumbered", 10 }, /* TEXICMD_UNNUMBERED */
	{ dosection, "unnumberedsec", 13 }, /* TEXICMD_UNNUMBEREDSEC */
	{ dosubsection, "unnumberedsubsec", 16 }, /* TEXICMD_UNNUMBEREDSUBSEC */
	{ dosubsubsection, "unnumberedsubsubsec", 19 }, /* TEXICMD_UNNUMBEREDSUBSUBSEC */
	{ dolink, "uref", 4 }, /* TEXICMD_UREF */
	{ dolink, "url", 3 }, /* TEXICMD_URL */
	{ doignline, "", 0 }, /* TEXICMD_USER_INDEX */
	{ doaccent, "v", 1 }, /* TEXICMD_V */
	{ dovalue, "value", 5 }, /* TEXICMD_VALUE */
	{ doinline, "var", 3 }, /* TEXICMD_VAR */
	{ doverb, "verb", 4 }, /* TEXICMD_VERB */
	{ doverbatim, "verbatim", 8 }, /* TEXICMD_VERBATIM */
	{ doverbinclude, "verbatiminclude", 15 }, /* TEXICMD_VERBATIMINCLUDE */
	{ doignline, "vindex", 6 }, /* TEXICMD_VINDEX */
	{ dosp, "vskip", 5 }, /* TEXICMD_VSKIP */
	{ dotable, "vtable", 6 }, /* TEXICMD_VTABLE */
	{ dobracket, "w", 1 }, /* TEXICMD_W */
	{ dolink, "xref", 4 }, /* TEXICMD_XREF */
	/* TEXICMD__END */
};

const	struct texitok *const texitoks = __texitoks;

static void
dodefindex(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t	 start, end;
	char	*cp;

	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);

	start = end = *pos;
	while (end < BUFSZ(p) && ! ismspace(BUF(p)[end]))
		end++;

	if (start == end) {
		advanceeoln(p, pos, 1);
		return;
	} else if (NULL == (cp = malloc(end - start + 1)))
		texiabort(p, NULL);

	memcpy(cp, &BUF(p)[start], end - start);
	cp[end - start] = '\0';

	p->indexs = realloc(p->indexs, 
		sizeof(char *) * (p->indexsz + 1));

	if (NULL == p->indexs)
		texiabort(p, NULL);
	p->indexs[p->indexsz++] = cp;
}

static void
dodefn(struct texi *p, enum texicmd cmd, size_t *pos)
{
	const char	*blk;

	blk = NULL;
	switch (cmd) {
	case (TEXICMD_DEFFN):
	case (TEXICMD_DEFMAC):
	case (TEXICMD_DEFTP):
	case (TEXICMD_DEFTYPEFN):
	case (TEXICMD_DEFTYPEFUN):
	case (TEXICMD_DEFTYPEMETHOD):
	case (TEXICMD_DEFTYPEVAR):
	case (TEXICMD_DEFTYPEVR):
	case (TEXICMD_DEFUN):
	case (TEXICMD_DEFVAR):
	case (TEXICMD_DEFVR):
		blk = texitoks[cmd].tok;
		break;
	default:
		break;
	}

	if (p->ign) {
		NULL != blk ?
			parseto(p, pos, blk) :
			parseeoln(p, pos);
		return;
	}

	if (NULL != blk)
		texivspace(p);

	switch (cmd) {
	case (TEXICMD_DEFTYPEMETHOD):
	case (TEXICMD_DEFTYPEMETHODX):
		texiputchars(p, "Method");
		break;
	case (TEXICMD_DEFMAC):
	case (TEXICMD_DEFMACX):
		texiputchars(p, "Macro");
		break;
	case (TEXICMD_DEFTYPEVAR):
	case (TEXICMD_DEFTYPEVARX):
	case (TEXICMD_DEFVAR):
	case (TEXICMD_DEFVARX):
		texiputchars(p, "Variable");
		break;
	case (TEXICMD_DEFTYPEFUN):
	case (TEXICMD_DEFTYPEFUNX):
	case (TEXICMD_DEFUN):
	case (TEXICMD_DEFUNX):
		texiputchars(p, "Function");
		break;
	default:
		parselinearg(p, pos);
		break;
	}

	texiputchar(p, ':');
	texiputchar(p, '\n');

	switch (cmd) {
	case (TEXICMD_DEFMAC):
	case (TEXICMD_DEFMACX):
		teximacroopen(p, "Dv");
		while (parselinearg(p, pos)) 
			/* Spin. */ ;
		teximacroclose(p);
		break;
	case (TEXICMD_DEFFN):
	case (TEXICMD_DEFFNX):
	case (TEXICMD_DEFUN):
	case (TEXICMD_DEFUNX):
		teximacroopen(p, "Fo");
		parselinearg(p, pos);
		teximacroclose(p);
		teximacroopen(p, "Fa");
		while (parselinearg(p, pos)) 
			/* Spin. */ ;
		teximacroclose(p);
		teximacro(p, "Fc");
		break;
	case (TEXICMD_DEFTYPEFUN):
	case (TEXICMD_DEFTYPEFUNX):
	case (TEXICMD_DEFTYPEFN):
	case (TEXICMD_DEFTYPEFNX):
	case (TEXICMD_DEFTYPEMETHOD):
	case (TEXICMD_DEFTYPEMETHODX):
		teximacroopen(p, "Ft");
		parselinearg(p, pos);
		teximacroclose(p);
		teximacroopen(p, "Fo");
		parselinearg(p, pos);
		teximacroclose(p);
		teximacroopen(p, "Fa");
		while (parselinearg(p, pos)) 
			/* Spin. */ ;
		teximacroclose(p);
		teximacro(p, "Fc");
		break;
	case (TEXICMD_DEFTP):
	case (TEXICMD_DEFTPX):
	case (TEXICMD_DEFTYPEVAR):
	case (TEXICMD_DEFTYPEVARX):
	case (TEXICMD_DEFTYPEVR):
	case (TEXICMD_DEFTYPEVRX):
		teximacroopen(p, "Vt");
		while (parselinearg(p, pos)) 
			/* Spin. */ ;
		teximacroclose(p);
		break;
	case (TEXICMD_DEFVAR):
	case (TEXICMD_DEFVARX):
	case (TEXICMD_DEFVR):
	case (TEXICMD_DEFVRX):
		teximacroopen(p, "Va");
		while (parselinearg(p, pos)) 
			/* Spin. */ ;
		teximacroclose(p);
		break;
	default:
		abort();
	}

	texivspace(p);
	if (NULL != blk)
		parseto(p, pos, blk);
}

static void
domacro(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t		  start, end, endtoksz, len;
	struct teximacro  m;
	const char	 *endtok, *blk;

	memset(&m, 0, sizeof(struct teximacro));

	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);

	for (start = end = *pos; end < BUFSZ(p); end++)
		if (ismspace(BUF(p)[end]) || '{' == BUF(p)[end])
			break;

	if (start == end)
		texierr(p, "zero-length macro name");

	advanceto(p, pos, end);

	m.key = malloc(end - start + 1);
	if (NULL == m.key)
		texiabort(p, NULL);
	memcpy(m.key, &BUF(p)[start], end - start);
	m.key[end - start] = '\0';

	m.args = argparse(p, pos, &m.argsz, 0);

	/* Note: we advance to the beginning of the macro. */
	advanceeoln(p, pos, 1);

	/*
	 * According to the Texinfo manual, the macro ends on the
	 * newline subsequent the @end macro.
	 * That's COMPLETELY FUCKING WRONG.
	 * It ends inclusive the newline, which is why so many macros
	 * say things like @r{hello}@c, where the subsequent @c swallows
	 * the newline.
	 * However, it does swallow the leading newline, so look for the
	 * @end macro without the leading newline else we might look
	 * past empty macros.
	 */
	start = *pos;
	endtok = "@end macro\n";
	endtoksz = strlen(endtok);
	blk = memmem(&BUF(p)[start], BUFSZ(p) - start, endtok, endtoksz);
	if (NULL == blk)
		texierr(p, "unterminated macro body");
	/* Roll us back one character. */
	while (&BUF(p)[*pos] != blk)
		advance(p, pos);
	assert('@' == BUF(p)[*pos]);
	if ('\n' != BUF(p)[*pos - 1]) 
		texierr(p, "cannot handle @end macro in-line");

	len = blk - &BUF(p)[start];
	m.value = malloc(len + 1);
	if (NULL == m.value)
		texiabort(p, NULL);
	memcpy(m.value, &BUF(p)[start], len);
	m.value[len] = '\0';

	p->macros = realloc
		(p->macros,
		(p->macrosz + 1) * 
		sizeof(struct teximacro));
	if (NULL == p->macros)
		texiabort(p, NULL);

	p->macros[p->macrosz++] = m;
	advanceeoln(p, pos, 1);
}

static void
doignblock(struct texi *p, enum texicmd cmd, size_t *pos)
{
	char		 end[32], start[32];
	const char	*endt, *startt;
	size_t		 esz, ssz, newpos, stack;

	/* 
	 * FIXME: this is cheating.
	 * These tokens are supposed to begin on a newline.
	 * However, if we do that, then we would need to check within
	 * the loop for trailer (or leading, as the case may be)
	 * newline, and that's just a bit too complicated right now.
	 * This is becasue
	 * 	@ifset BAR
	 * 	@ifset FOO
	 * 	@end ifset
	 * 	@end ifset
	 * won't work right now: we'd read after the first "@end ifset"
	 * to the next line, then look for the next line after that.
	 */
	ssz = snprintf(start, sizeof(start), 
		"@%s", texitoks[cmd].tok);
	assert(ssz < sizeof(start));
	esz = snprintf(end, sizeof(end), 
		"@end %s\n", texitoks[cmd].tok);
	assert(esz < sizeof(end));
	stack = 1;

	/*
	 * Here we look for the end token "end" somewhere in the file in
	 * front of us.
	 * It's not that easy, of course: if we have a nested block,
	 * then there'll be an "end" token of the same kind between us.
	 * Thus, we keep track of scopes for matching "end" blocks.
	 */
	while (stack > 0 && *pos < BUFSZ(p)) {
		if (stack > 10)
			abort();
		endt = memmem(&BUF(p)[*pos], BUFSZ(p) - *pos, end, esz);
		startt = memmem(&BUF(p)[*pos], BUFSZ(p) - *pos, start, ssz);
		if (NULL == endt) {
			texiwarn(p, "unterminated \"%s\" "
				"block", texitoks[cmd].tok);
			*pos = BUFSZ(p);
			break;
		} 

		newpos = *pos;
		if (NULL == startt || startt > endt) {
			newpos += esz + (size_t)(endt - &BUF(p)[*pos]);
			stack--;
		} else {
			newpos += ssz + (size_t)(startt - &BUF(p)[*pos]);
			stack++;
		}

		assert(newpos <= BUFSZ(p));
		while (*pos < newpos)
			advance(p, pos);
	}
}

static void
doblock(struct texi *p, enum texicmd cmd, size_t *pos)
{
	
	parseto(p, pos, texitoks[cmd].tok);
}

static void
doinline(struct texi *p, enum texicmd cmd, size_t *pos)
{
	const char	*macro = NULL;

	switch (cmd) {
	case (TEXICMD_CODE):
	case (TEXICMD_KBD):
	case (TEXICMD_SAMP):
	case (TEXICMD_T):
		macro = "Li";
		break;
	case (TEXICMD_CITE):
	case (TEXICMD_DFN):
	case (TEXICMD_EMPH):
	case (TEXICMD_I):
	case (TEXICMD_SLANTED):
		macro = "Em";
		break;
	case (TEXICMD_B):
	case (TEXICMD_STRONG):
		macro = "Sy";
		break;
	case (TEXICMD_COMMAND):
		macro = "Xr";
		break;
	case (TEXICMD_ENV):
		macro = "Ev";
		break;
	case (TEXICMD_FILE):
		macro = "Pa";
		break;
	case (TEXICMD_OPTION):
		macro = "Op";
		break;
	case (TEXICMD_VAR):
		macro = "Va";
		break;
	default:
		break;
	}

	if (NULL == macro || p->literal || TEXILIST_TABLE == p->list) {
		parsebracket(p, pos);
		return;
	}

	teximacroopen(p, macro);
	p->seenws = 0;
	parsebracket(p, pos);
	texipunctuate(p, pos);
	teximacroclose(p);
}

static void
doverb(struct texi *p, enum texicmd cmd, size_t *pos)
{
	char	 delim;
	size_t	 start;

	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);
	if (*pos == BUFSZ(p) || '{' != BUF(p)[*pos])
		return;
	advance(p, pos);
	if (*pos == BUFSZ(p))
		return;

	delim = BUF(p)[*pos];
	advance(p, pos);
	/* Make sure we flush out our initial whitespace... */
	if (p->seenws && p->outcol && 0 == p->literal)
		texiputchar(p, ' ');
	p->seenws = 0;
	start = *pos;
	/* Read until we see the delimiter then end-brace. */
	while (*pos < BUFSZ(p) - 1) {
		if (BUF(p)[*pos] == delim && BUF(p)[*pos + 1] == '}') 
			break;
		advance(p, pos);
	}
	if (*pos == BUFSZ(p) - 1)
		return;
	texiputbuf(p, start, *pos);

	/* Make sure we read after the end-brace. */
	assert(delim == BUF(p)[*pos]);
	advance(p, pos);
	assert('}' == BUF(p)[*pos]);
	advance(p, pos);
}

static void
doverbatim(struct texi *p, enum texicmd cmd, size_t *pos)
{
	const char	*end, *term;
	size_t		 endsz, endpos;

	/* We read from the @verbatim\n newline inclusive! */
	end = "\n@end verbatim\n";
	endsz = strlen(end);
	advanceeoln(p, pos, 0);
	if (*pos == BUFSZ(p)) {
		texiwarn(p, "unexpected end of file");
		return;
	}

	term = memmem(&BUF(p)[*pos], BUFSZ(p) - *pos, end, endsz);
	if (NULL == term) {
		texiwarn(p, "unterminated verbatim block");
		endpos = BUFSZ(p);
	} else
		endpos = *pos + (size_t)(term - &BUF(p)[*pos]);

	assert(endpos <= BUFSZ(p));
	assert('\n' == BUF(p)[*pos]);
	advance(p, pos);
	teximacro(p, "Bd -literal -offset indent");
	while (*pos < endpos) {
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
	}
	teximacro(p, "Ed");
	if (*pos < BUFSZ(p))
		advanceto(p, pos, endpos + endsz);
}

static void
doverbinclude(struct texi *p, enum texicmd cmd, size_t *pos)
{
	char	 	 fname[PATH_MAX], path[PATH_MAX];
	int	 	 rc;
	size_t	 	 i, end;
	const char	*v;
	enum texicmd	 type;

	while (*pos < BUFSZ(p) && ' ' == BUF(p)[*pos])
		advance(p, pos);

	for (i = 0; *pos < BUFSZ(p) && '\n' != BUF(p)[*pos]; ) {
		if (i == sizeof(fname) - 1)
			break;
		if ('@' != BUF(p)[*pos]) {
			fname[i++] = BUF(p)[*pos];
			advance(p, pos);
			continue;
		}
		type = texicmd(p, *pos, &end, NULL);
		advanceto(p, pos, end);
		if (TEXICMD_VALUE != type) 
			texierr(p, "unknown verbatiminclude command");
		v = valueblookup(p, pos);
		if (NULL == v)
			continue;
		while ('\0' != *v) {
			if (i == sizeof(fname) - 1)
				break;
			fname[i++] = *v++;
		}
		if ('\0' != *v)
			break;
	}

	if (i == 0)
		texierr(p, "path too short");
	else if ('\n' != BUF(p)[*pos])
		texierr(p, "path too long");
	else if ('/' == fname[0])
		texierr(p, "no absolute paths");
	fname[i] = '\0';

	if (strstr(fname, "../") || strstr(fname, "/.."))
		texierr(p, "insecure path");

	rc = snprintf(path, sizeof(path), 
		"%s/%s", p->dirs[0], fname);
	if (rc < 0) 
		texierr(p, "couldn't format path");
	else if ((size_t)rc >= sizeof(path))
		texierr(p, "path too long");

	parsefile(p, path, 0);
}

static void
doinclude(struct texi *p, enum texicmd cmd, size_t *pos)
{
	char	 	 fname[PATH_MAX], path[PATH_MAX];
	size_t	 	 i, end;
	int	 	 rc;
	const char	*v;
	enum texicmd	 type;

	while (*pos < BUFSZ(p) && ' ' == BUF(p)[*pos])
		advance(p, pos);

	/* Read in the filename. */
	for (i = 0; *pos < BUFSZ(p) && '\n' != BUF(p)[*pos]; ) {
		if (i == sizeof(fname) - 1)
			break;
		if ('@' != BUF(p)[*pos]) {
			fname[i++] = BUF(p)[*pos];
			advance(p, pos);
			continue;
		}
		type = texicmd(p, *pos, &end, NULL);
		advanceto(p, pos, end);
		if (TEXICMD_VALUE != type) 
			texierr(p, "unknown include command");
		v = valueblookup(p, pos);
		if (NULL == v)
			continue;
		while ('\0' != *v) {
			if (i == sizeof(fname) - 1)
				break;
			fname[i++] = *v++;
		}
		if ('\0' != *v)
			break;
	}

	if (i == 0)
		texierr(p, "path too short");
	else if ('\n' != BUF(p)[*pos])
		texierr(p, "path too long");
	else if ('/' == fname[0])
		texierr(p, "no absolute paths");
	fname[i] = '\0';

	if (strstr(fname, "../") || strstr(fname, "/.."))
		texierr(p, "insecure path");

	for (i = 0; i < p->dirsz; i++) {
		rc = snprintf(path, sizeof(path), 
			"%s/%s", p->dirs[i], fname);
		if (rc < 0) 
			texierr(p, "couldn't format path");
		else if ((size_t)rc >= sizeof(path))
			texierr(p, "path too long");
		else if (-1 == access(path, R_OK))
			continue;

		parsefile(p, path, 1);
		return;
	}

	texierr(p, "couldn't find %s in includes", fname);
}

static void
dobracket(struct texi *p, enum texicmd cmd, size_t *pos)
{

	parsebracket(p, pos);
}

static void
dodisplay(struct texi *p, enum texicmd cmd, size_t *pos)
{

	switch (cmd) {
	case (TEXICMD_FORMAT):
	case (TEXICMD_SMALLFORMAT):
		teximacro(p, "Bd -filled");
		break;
	default:
		teximacro(p, "Bd -filled -offset indent");
		break;
	}

	p->seenvs = 1;
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
	parseto(p, pos, texitoks[cmd].tok);
	teximacro(p, "Ed");
}

static void
doexample(struct texi *p, enum texicmd cmd, size_t *pos)
{

	teximacro(p, "Bd -literal -offset indent");
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
	p->literal++;
	parseto(p, pos, texitoks[cmd].tok);
	p->literal--;
	teximacro(p, "Ed");
}

static void
dobye(struct texi *p, enum texicmd cmd, size_t *pos)
{

	texiexit(p);
	exit(EXIT_SUCCESS);
}

static void
dotitle(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t	 start, end;

	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);
	start = end = *pos;
	while (end < BUFSZ(p) && '\n' != BUF(p)[end])
		end++;
	advanceeoln(p, pos, 1);
	free(p->subtitle);
	p->subtitle = malloc(end - start + 1);
	if (NULL == p->subtitle)
		texiabort(p, NULL);
	memcpy(p->subtitle, &BUF(p)[start], end - start);
	p->subtitle[end - start] = '\0';
}

static void
doaccent(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int	 brace = 0;

	if (*pos == BUFSZ(p)) {
		texiwarn(p, "truncated: @%s", texitoks[cmd].tok);
		return;
	}

	/* Pad us with space, if necessary. */
	if (p->seenws && p->outcol && 0 == p->literal) {
		texiputchar(p, ' ');
		p->seenws = 0;
	}

	/*
	 * If we're braced, then that's that.
	 * Otherwise, in a special Texinfo case: if we're a non
	 * alphabetic command of one letter, then the next character is
	 * the critical one.
	 * Otherwise, space can sit between us and our argument.
	 */
	if ('{' == BUF(p)[*pos]) {
		brace = 1;
		advance(p, pos);
	} else if (isalpha(texitoks[cmd].tok[0])) 
		while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
			advance(p, pos);

	if (*pos == BUFSZ(p)) {
		texiwarn(p, "truncated: @%s", texitoks[cmd].tok);
		return;
	}

	switch (cmd) {
	case (TEXICMD_ACUTE):
		switch (BUF(p)[*pos]) {
		case ('a'): case ('A'):
		case ('e'): case ('E'):
		case ('i'): case ('I'):
		case ('o'): case ('O'):
		case ('u'): case ('U'):
			texiputchars(p, "\\(\'");
			break;
		default:
			texiwarn(p, "ignoring accent");
			break;
		}
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_CEDILLA):
		if ('c' == BUF(p)[*pos] || 'C' == BUF(p)[*pos])
			texiputchars(p, "\\(,");
		else
			texiwarn(p, "ignoring accent");
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_CIRCUMFLEX):
		switch (BUF(p)[*pos]) {
		case ('a'): case ('A'):
		case ('e'): case ('E'):
		case ('i'): case ('I'):
		case ('o'): case ('O'):
		case ('u'): case ('U'):
			texiputchars(p, "\\(^");
			break;
		default:
			texiwarn(p, "ignoring accent");
			break;
		}
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_DOTLESS):
		if ('i' == BUF(p)[*pos] || 'j' == BUF(p)[*pos])
			texiputchars(p, "\\(.");
		else
			texiwarn(p, "ignoring accent");
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_GRAVE):
		switch (BUF(p)[*pos]) {
		case ('a'): case ('A'):
		case ('e'): case ('E'):
		case ('i'): case ('I'):
		case ('o'): case ('O'):
		case ('u'): case ('U'):
			texiputchars(p, "\\(`");
			break;
		default:
			texiwarn(p, "ignoring accent");
			break;
		}
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_TILDE):
		switch (BUF(p)[*pos]) {
		case ('a'): case ('A'):
		case ('n'): case ('N'):
		case ('o'): case ('O'):
			texiputchars(p, "\\(~");
			break;
		default:
			texiwarn(p, "ignoring accent");
			break;
		}
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	case (TEXICMD_UMLAUT):
		switch (BUF(p)[*pos]) {
		case ('a'): case ('A'):
		case ('e'): case ('E'):
		case ('i'): case ('I'):
		case ('o'): case ('O'):
		case ('u'): case ('U'):
		case ('y'):
			texiputchars(p, "\\(:");
			break;
		default:
			texiwarn(p, "ignoring accent");
			break;
		}
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	default:
		texiputchar(p, BUF(p)[*pos]);
		advance(p, pos);
		break;
	}

	if (brace) {
		while (*pos < BUFSZ(p) && '}' != BUF(p)[*pos]) {
			texiputchar(p, BUF(p)[*pos]);
			advance(p, pos);
		}
		if (*pos < BUFSZ(p))
			advance(p, pos);
	} 

	switch (cmd) {
	case (TEXICMD_TIEACCENT):
		texiputchar(p, ']');
		break;
	case (TEXICMD_DOTACCENT):
		texiputchar(p, '*');
		break;
	default:
		break;
	}
}

static void
dosymbol(struct texi *p, enum texicmd cmd, size_t *pos)
{

	/* Remember to pad us. */
	if (p->seenws && p->outcol && 0 == p->literal)
		texiputchar(p, ' ');

	p->seenws = 0;

	switch (cmd) {
	case (TEXICMD_AA):
		texiputchars(p, "\\(oA");
		break;
	case (TEXICMD_AASMALL):
		texiputchars(p, "\\(oa");
		break;
	case (TEXICMD_AE):
		texiputchars(p, "\\(AE");
		break;
	case (TEXICMD_AESMALL):
		texiputchars(p, "\\(ae");
		break;
	case (TEXICMD_ASTERISK):
	case (TEXICMD_NEWLINE):
	case (TEXICMD_SPACE):
	case (TEXICMD_TABSYM):
		texiputchar(p, ' ');
		break;
	case (TEXICMD_AT):
		texiputchar(p, '@');
		break;
	case (TEXICMD_BANG):
		texiputchar(p, '!');
		break;
	case (TEXICMD_BULLET):
		texiputchars(p, "\\(bu");
		break;
	case (TEXICMD_COMMA):
		texiputchar(p, ',');
		break;
	case (TEXICMD_COPYRIGHT):
		texiputchars(p, "\\(co");
		break;
	case (TEXICMD_DH):
		texiputchars(p, "\\(-D");
		break;
	case (TEXICMD_DHSMALL):
		texiputchars(p, "\\(Sd");
		break;
	case (TEXICMD_DOTS):
	case (TEXICMD_ENDDOTS):
		texiputchars(p, "...");
		break;
	case (TEXICMD_EQUIV):
		texiputchars(p, "\\(==");
		break;
	case (TEXICMD_ERROR):
		texiputchars(p, "error\\(->");
		break;
	case (TEXICMD_EURO):
		texiputchars(p, "\\(Eu");
		break;
	case (TEXICMD_EXCLAMDOWN):
		texiputchars(p, "\\(r!");
		break;
	case (TEXICMD_EXPANSION):
		texiputchars(p, "\\(->");
		break;
	case (TEXICMD_GEQ):
		texiputchars(p, "\\(>=");
		break;
	case (TEXICMD_GUILLEMETLEFT):
	case (TEXICMD_GUILLEMOTLEFT):
		texiputchars(p, "\\(Fo");
		break;
	case (TEXICMD_GUILLEMETRIGHT):
	case (TEXICMD_GUILLEMOTRIGHT):
		texiputchars(p, "\\(Fc");
		break;
	case (TEXICMD_GUILSINGLLEFT):
		texiputchars(p, "\\(fo");
		break;
	case (TEXICMD_GUILSINGLRIGHT):
		texiputchars(p, "\\(fc");
		break;
	case (TEXICMD_L):
		texiputchars(p, "\\(/L");
		break;
	case (TEXICMD_LATEX):
		texiputchars(p, "LaTeX");
		break;
	case (TEXICMD_LEQ):
		texiputchars(p, "\\(<=");
		break;
	case (TEXICMD_LSMALL):
		texiputchars(p, "\\(/l");
		break;
	case (TEXICMD_MINUS):
		texiputchars(p, "\\-");
		break;
	case (TEXICMD_O):
		texiputchars(p, "\\(/O");
		break;
	case (TEXICMD_OE):
		texiputchars(p, "\\(OE");
		break;
	case (TEXICMD_OESMALL):
		texiputchars(p, "\\(oe");
		break;
	case (TEXICMD_ORDF):
		texiputchars(p, "a");
		break;
	case (TEXICMD_ORDM):
		texiputchars(p, "o");
		break;
	case (TEXICMD_OSMALL):
		texiputchars(p, "\\(/o");
		break;
	case (TEXICMD_PERIOD):
		texiputchar(p, '.');
		break;
	case (TEXICMD_POUNDS):
		texiputchars(p, "\\(Po");
		break;
	case (TEXICMD_QUESTIONDOWN):
		texiputchars(p, "\\(r?");
		break;
	case (TEXICMD_QUESTIONMARK):
		texiputchar(p, '?');
		break;
	case (TEXICMD_QUOTEDBLBASE):
		texiputchars(p, "\\(Bq");
		break;
	case (TEXICMD_QUOTEDBLLEFT):
		texiputchars(p, "\\(lq");
		break;
	case (TEXICMD_QUOTEDBLRIGHT):
		texiputchars(p, "\\(rq");
		break;
	case (TEXICMD_QUOTESINGLBASE):
		texiputchars(p, "\\(bq");
		break;
	case (TEXICMD_QUOTELEFT):
		texiputchars(p, "\\(oq");
		break;
	case (TEXICMD_QUOTERIGHT):
		texiputchars(p, "\\(cq");
		break;
	case (TEXICMD_REGISTEREDSYMBOL):
		texiputchars(p, "\\(rg");
		break;
	case (TEXICMD_RESULT):
		texiputchars(p, "\\(rA");
		break;
	case (TEXICMD_SLASH):
		texiputchar(p, '/');
		break;
	case (TEXICMD_SS):
		texiputchars(p, "\\(ss");
		break;
	case (TEXICMD_SQUIGGLE_LEFT):
		texiputchars(p, "{");
		break;
	case (TEXICMD_SQUIGGLE_RIGHT):
		texiputchars(p, "}");
		break;
	case (TEXICMD_TEXSYM):
		texiputchars(p, "TeX");
		break;
	case (TEXICMD_TEXTDEGREE):
		texiputchars(p, "\\(de");
		break;
	case (TEXICMD_TH):
		texiputchars(p, "\\(TP");
		break;
	case (TEXICMD_THSMALL):
		texiputchars(p, "\\(Tp");
		break;
	case (TEXICMD_TIE):
		texiputchars(p, "\\ ");
		break;
	case (TEXICMD_COLON):
	case (TEXICMD_HYPHEN):
		break;
	default:
		abort();
	}

	/* Alphabetic commands have braces we ignore. */
	if (isalpha(texitoks[cmd].tok[0]))
		doignbracket(p, cmd, pos);
}

static void
doquotation(struct texi *p, enum texicmd cmd, size_t *pos)
{
	
	teximacro(p, "Qo");
	parseto(p, pos, "quotation");
	teximacro(p, "Qc");
}

static void
domath(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t	 nest, start;

	/*
	 * Math handling is different from everything else.
	 * We don't allow any subcomponents, and we ignore the rules in
	 * terms of @-commands.
	 * This departs from GNU's rules, but whatever.
	 */
	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);
	if (*pos == BUFSZ(p) || '{' != BUF(p)[*pos])
		return;
	advance(p, pos);
	if (p->seenws && p->outcol && 0 == p->literal)
		texiputchar(p, ' ');
	p->seenws = 0;
	for (nest = 1, start = *pos; *pos < BUFSZ(p) && nest > 0; ) {
		if ('{' == BUF(p)[*pos])
			nest++;
		else if ('}' == BUF(p)[*pos])
			if (0 == --nest)
				continue;
		advance(p, pos);
	}
	if (*pos == BUFSZ(p))
		return;
	assert('}' == BUF(p)[*pos]);
	texiputbuf(p, start, *pos);
	advance(p, pos);
}

static void
dovalue(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t		 start, end;
	char		*key, *val;
	const char	*cp;

	if (TEXICMD_SET == cmd) {
		while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
			advance(p, pos);
		for (start = end = *pos; end < BUFSZ(p); end++)
			if (ismspace(BUF(p)[end]))
				break;
		/* We don't allow empty keys. */
		if (start == end)
			return;
		advanceto(p, pos, end);

		key = malloc(end - start + 1);
		if (NULL == key)
			texiabort(p, NULL);
		memcpy(key, &BUF(p)[start], end - start);
		key[end - start] = '\0';

		while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
			advance(p, pos);
		for (start = end = *pos; end < BUFSZ(p); end++)
			if ('\n' == BUF(p)[end])
				break;
		/* We do allow empty values. */
		advanceeoln(p, pos, 1);

		val = malloc(end - start + 1);
		if (NULL == val)
			texiabort(p, NULL);
		memcpy(val, &BUF(p)[start], end - start);
		val[end - start] = '\0';
		valueadd(p, key, val);
	} else if (TEXICMD_VALUE == cmd) {
		if (p->seenws)
			texiputchar(p, ' ');
		p->seenws = 0;
		if (NULL != (cp = valueblookup(p, pos)))
			texisplice(p, cp, strlen(cp), pos);
		else
			texiputchars(p, "{No value}");
	} else if (TEXICMD_IFCLEAR == cmd) {
		if (NULL != valuellookup(p, pos))
			doignblock(p, cmd, pos);
		else
			parseto(p, pos, texitoks[cmd].tok);
	} else if (TEXICMD_IFSET == cmd) {
		if (NULL == valuellookup(p, pos))
			doignblock(p, cmd,  pos);
		else
			parseto(p, pos, texitoks[cmd].tok);
	} else if (TEXICMD_CLEAR == cmd)
		valuelclear(p, pos);
}

static void
dolink(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int	 c;

	switch (cmd) {
	case (TEXICMD_EMAIL):
		teximacroopen(p, "Mt");
		break;
	case (TEXICMD_UREF):
	case (TEXICMD_URL):
	case (TEXICMD_INDICATEURL):
		teximacroopen(p, "Lk");
		break;
	case (TEXICMD_XREF):
		texiputchars(p, "See Section");
		teximacroopen(p, "Dq");
		break;
	case (TEXICMD_PXREF):
		texiputchars(p, "see Section");
		teximacroopen(p, "Dq");
		break;
	case (TEXICMD_INFOREF):
		texiputchars(p, "See Info file node");
		teximacroopen(p, "Dq");
		break;
	default:
		abort();
	}

	c = parsearg(p, pos, 0);
	p->ign++;
	while (c > 0)
		c = parsearg(p, pos, 1);
	p->ign--;

	texipunctuate(p, pos);
	teximacroclose(p);
}

static void
doignargn(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int	 c;

	c = parsearg(p, pos, 0);
	p->ign++;
	while (c > 0)
		c = parsearg(p, pos, 1);
	p->ign--;
}

/*
 * Sections can be made subsections and so on by way of the
 * @raiseections and @lowersections commands.
 * Perform this check here and return the actual section number adjusted
 * to the raise level.
 */
static int
sectioner(struct texi *p, int sec)
{

	if ((sec -= p->secoffs) < 0) {
		texiwarn(p, "section below minimum, clamping");
		return(0);
	} else if (sec >= SECTSZ) {
		texiwarn(p, "section above maximum, clamping");
		return(SECTSZ - 1);
	}
	return(sec);
}

static void
dosubsubsection(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int	 sec;

	sec = sectioner(p, 3);

	/* We don't have a subsubsubsection, so make one up. */
	texivspace(p);
	teximacroopen(p, sects[sec]);
	parseeoln(p, pos);
	teximacroclose(p);
	texivspace(p);
}

static void
dosubsection(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int	 sec;

	sec = sectioner(p, 2);

	if (p->outmacro)
		texierr(p, "\"%s\" in open line scope!?", sects[sec]);
	else if (p->literal)
		texierr(p, "\"%s\" in a literal scope!?", sects[sec]);

	/* We don't have a subsubsection, so make one up. */
	if (sec > 1)
		texivspace(p);
	teximacroopen(p, sects[sec]);
	parseeoln(p, pos);
	teximacroclose(p);
	if (sec > 1)
		texivspace(p);
}

static void
dosecoffs(struct texi *p, enum texicmd cmd, size_t *pos)
{

	if (TEXICMD_RAISESECTIONS == cmd)
		p->secoffs++;
	else
		p->secoffs--;
}

static void
dosection(struct texi *p, enum texicmd cmd, size_t *pos)
{
	int		 sec;

	switch (cmd) {
	case (TEXICMD_APPENDIX):
	case (TEXICMD_CHAPTER):
	case (TEXICMD_TOP):
	case (TEXICMD_UNNUMBERED):
		sec = sectioner(p, 0);
		break;
	case (TEXICMD_APPENDIXSEC):
	case (TEXICMD_HEADING):
	case (TEXICMD_SECTION):
	case (TEXICMD_UNNUMBEREDSEC):
		sec = sectioner(p, 1);
		break;
	default:
		abort();
	}

	if (p->outmacro)
		texierr(p, "\"%s\" in open line scope!?", sects[sec]);
	else if (p->literal)
		texierr(p, "\"%s\" in a literal scope!?", sects[sec]);

	teximacroopen(p, sects[sec]);
	parseeoln(p, pos);
	teximacroclose(p);
	p->seenvs = 1;
}

static void
dosp(struct texi *p, enum texicmd cmd, size_t *pos)
{

	if (p->literal)
		texiputchar(p, '\n');
	else
		texivspace(p);
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
}

static void
dotop(struct texi *p, enum texicmd cmd, size_t *pos)
{
	const char	*cp;
	time_t		 t;
	char		 date[32];

	if (--p->ign)
		texierr(p, "@top command while ignoring");

	/*
	 * Here we print our standard mdoc(7) prologue.
	 * We use the title set with @settitle for the `Nd' description
	 * and the source document filename (the first one as invoked on
	 * the command line) for the title.
	 * The date is set to the current date.
	 */
	t = time(NULL);
	strftime(date, sizeof(date), "%F", localtime(&t));

	teximacroopen(p, "Dd");
	texiputchars(p, date);
	teximacroclose(p);
	teximacroopen(p, "Dt");
	for (cp = p->title; '\0' != *cp; cp++)
		texiputchar(p, toupper(*cp));
	texiputchars(p, " 7");
	teximacroclose(p);
	teximacro(p, "Os");
	teximacro(p, "Sh NAME");
	teximacroopen(p, "Nm");
	for (cp = p->title; '\0' != *cp; cp++)
		texiputchar(p, *cp);
	teximacroclose(p);
	teximacroopen(p, "Nd");
	if (NULL != p->subtitle)
		for (cp = p->subtitle; '\0' != *cp; cp++)
			texiputchar(p, *cp);
	else
		texiputchars(p, "Unknown description");
	teximacroclose(p);
	p->seenvs = 1;
	dosection(p, cmd, pos);
}

static void
doitem(struct texi *p, enum texicmd cmd, size_t *pos)
{

	/* Multitable is using raw tbl(7). */
	if (TEXILIST_TABLE == p->list) {
		texiputchar(p, '\n');
		return;
	} 
	
	if (p->outmacro)
		texierr(p, "item in open line scope!?");
	else if (p->literal)
		texierr(p, "item in a literal scope!?");

	switch (p->list) {
	case (TEXILIST_ITEM):
		teximacroopen(p, "It");
		break;
	case (TEXILIST_NOITEM):
		teximacro(p, "It");
		break;
	default:
		texivspace(p);
		break;
	}

	/* Trick so we don't start with Pp. */
	p->seenvs = 1;
	parseeoln(p, pos);

	if (TEXILIST_ITEM == p->list)
		teximacroclose(p);
	else if (p->outcol > 0)
		texiputchar(p, '\n');
}

static void
dotab(struct texi *p, enum texicmd cmd, size_t *pos)
{

	/* This command is only useful in @multitable. */
	if (TEXILIST_TABLE == p->list)
		texiputchar(p, '\t');
}

static void
domultitable(struct texi *p, enum texicmd cmd, size_t *pos)
{
	enum texilist	sv = p->list;
	int		svliteral = p->literal;
	enum texicmd	type;
	size_t		i, end, columns;

	p->list = TEXILIST_TABLE;
	/* 
	 * TS/TE blocks aren't "in mdoc(7)", so we can disregard the
	 * fact that we're in literal mode right now.
	 */
	p->literal = 0;
	teximacro(p, "TS");
	columns = 0;

	/* Advance to the first argument... */
	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);

	/* Make sure we don't print anything when scanning. */
	p->ign++;
	if ('@' == BUF(p)[*pos]) {
		/* 
		 * Look for @columnfractions.
		 * We ignore these, but we do use the number of
		 * arguments to set the number of columns that we'll
		 * have.
		 */
		type = texicmd(p, *pos, &end, NULL);
		advanceto(p, pos, end);
		if (TEXICMD_COLUMNFRACTIONS != type) 
			texierr(p, "unknown multitable command");
		while (*pos < BUFSZ(p) && '\n' != BUF(p)[*pos]) {
			while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
				advance(p, pos);
			while (*pos < BUFSZ(p) && ! isws(BUF(p)[*pos])) {
				if ('\n' == BUF(p)[*pos])
					break;
				advance(p, pos);
			}
			columns++;
		}
	} else 
		/*
		 * We have arguments.
		 * We could parse these, but it's easier to just let
		 * tbl(7) figure it out.
		 * So use this only to count arguments.
		 */
		while (parselinearg(p, pos) > 0)
			columns++;
	p->ign--;

	/* Left-justify each table entry. */
	for (i = 0; i < columns; i++) {
		if (i > 0)
			texiputchar(p, ' ');
		texiputchar(p, 'l');
	}
	texiputchars(p, ".\n");
	p->outmacro++;
	parseto(p, pos, texitoks[cmd].tok);
	p->outmacro--;
	teximacro(p, "TE");
	p->literal = svliteral;
	p->list = sv;
}

static void
dotable(struct texi *p, enum texicmd cmd, size_t *pos)
{
	enum texilist	sv = p->list;

	p->list = TEXILIST_ITEM;
	teximacro(p, "Bl -tag -width Ds");
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
	p->seenvs = 1;
	parseto(p, pos, texitoks[cmd].tok);
	teximacro(p, "El");
	p->list = sv;
}

static void
doend(struct texi *p, enum texicmd cmd, size_t *pos)
{
	size_t	 start;

	while (*pos < BUFSZ(p) && isws(BUF(p)[*pos]))
		advance(p, pos);
	start = *pos;
	while (*pos < BUFSZ(p) && '\n' != BUF(p)[*pos])
		advance(p, pos);

	texiwarn(p, "unexpected \"end\": %.*s", (int)(*pos - start), &BUF(p)[start]);
	advanceeoln(p, pos, 1);
}

static void
doenumerate(struct texi *p, enum texicmd cmd, size_t *pos)
{
	enum texilist	 sv = p->list;

	p->list = TEXILIST_NOITEM;
	teximacro(p, "Bl -enum");
	p->seenvs = 1;
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
	parseto(p, pos, "enumerate");
	teximacro(p, "El");
	p->list = sv;
}

static void
doitemize(struct texi *p, enum texicmd cmd, size_t *pos)
{
	enum texilist	sv = p->list;

	p->list = TEXILIST_NOITEM;
	teximacro(p, "Bl -bullet");
	p->seenvs = 1;
	/* FIXME: ignore and parseeoln. */
	advanceeoln(p, pos, 1);
	parseto(p, pos, "itemize");
	teximacro(p, "El");
	p->list = sv;
}

static void
doignbracket(struct texi *p, enum texicmd cmd, size_t *pos)
{

	p->ign++;
	parsebracket(p, pos);
	p->ign--;
}

static void
doignline(struct texi *p, enum texicmd cmd, size_t *pos)
{

	/* FIXME: ignore and parseeoln. */
	advanceeoln(p,  pos, 1);
}

/*
 * Parse colon-separated directories from "cp" (if not NULL) and returns
 * the array of pointers.
 * Prepends "base" to the array, if found.
 * This does NOT sanitise the directories!
 */
static char **
parsedirs(struct texi *p, const char *base, const char *cp, size_t *sz)
{
	char	   	 *tok, *str, *tofree;
	const char 	 *cpp;
	size_t	    	  i = 0;
	char		**dirs;

	/* Count up our expected arguments. */
	*sz = NULL != base;
	if (NULL != (cpp = cp))
		for ((*sz)++; NULL != (cpp = strchr(cpp, ':')); (*sz)++)
			cpp++;

	if (0 == *sz)
		return(NULL);
	if (NULL == (dirs = calloc(*sz, sizeof(char *))))
		texiabort(p, NULL);
	if (NULL != base && NULL == (dirs[i++] = strdup(base))) 
		texiabort(p, NULL);
	if (NULL == cp)
		return(dirs);
	if (NULL == (tofree = tok = str = strdup(cp)))
		texiabort(p, NULL);

	for ( ; NULL != (tok = strsep(&str, ":")); i++) 
		if (NULL == (dirs[i] = strdup(tok))) 
			texiabort(p, NULL);

	free(tofree);
	return(dirs);
}

int
main(int argc, char *argv[])
{
	struct texi	 texi;
	int		 c;
	char		*dirpath, *dir, *ccp;
	const char	*progname, *Idir, *cp;

	progname = strrchr(argv[0], '/');
	if (progname == NULL)
		progname = argv[0];
	else
		++progname;

	memset(&texi, 0, sizeof(struct texi));
	texi.ign = 1;
	Idir = NULL;

	while (-1 != (c = getopt(argc, argv, "I:"))) 
		switch (c) {
		case ('I'):
			Idir = optarg;
			break;
		default:
			goto usage;
		}

	argv += optind;
	argc -= optind;

	if (argc > 0) {
		if (NULL == (dirpath = strdup(argv[0])))
			texiabort(&texi, NULL);
		if (NULL == (dir = dirname(dirpath)))
			texiabort(&texi, NULL);
		if (NULL != (cp = strrchr(argv[0], '/'))) 
			texi.title = strdup(cp + 1);
		else
			texi.title = strdup(argv[0]);
		if (NULL == texi.title)
			texiabort(&texi, NULL);
		else if (NULL != (ccp = strchr(texi.title, '.')))
			*ccp = '\0';
		texi.dirs = parsedirs(&texi, dir, Idir, &texi.dirsz);
		free(dirpath);
		parsefile(&texi, argv[0], 1);
	} else {
		texi.title = strdup("Unknown Manual");
		texi.dirs = parsedirs(&texi, NULL, Idir, &texi.dirsz);
		parsestdin(&texi);
	}

	texiexit(&texi);
	return(EXIT_FAILURE);
usage:
	fprintf(stderr, "usage: %s [-Idirs] [file]\n", progname);
	return(EXIT_FAILURE);
}
