package main

import (
	"encoding/json"
	"os"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/stretchr/testify/require"
)

func TestReplaceDeprecatedLicenses(t *testing.T) {
	results := LicenseList{
		Licenses: []License{
			license1,
			license2,
			license3,
			license4,
			license5,
			license6,
		},
	}

	expected := map[string]string{
		"abc-1":              "ABC-1.0-only",
		"abc-1-only":         "ABC-1.0-only",
		"abc-1.0":            "ABC-1.0-only",
		"abc-1.0.0":          "ABC-1.0-only",
		"abc-1.0-only":       "ABC-1.0-only",
		"abc-1.0.0-only":     "ABC-1.0-only",
		"abc-1+":             "ABC-1.0-or-later",
		"abc-1.0+":           "ABC-1.0-or-later",
		"abc-1.0.0+":         "ABC-1.0-or-later",
		"abc-1-or-later":     "ABC-1.0-or-later",
		"abc-1.0-or-later":   "ABC-1.0-or-later",
		"abc-1.0.0-or-later": "ABC-1.0-or-later",
		"duh-1":              "Duh-1.0",
		"duh-1.0":            "Duh-1.0",
		"duh-1.0.0":          "Duh-1.0",
		"duh-1-duh":          "Duh-1.0-duh",
		"duh-1.0-duh":        "Duh-1.0-duh",
		"duh-1.0.0-duh":      "Duh-1.0-duh",
	}

	licenses := processSPDXLicense(results)
	for k, v := range licenses {
		e := expected[k]
		require.Equal(t, e, v, k)
	}
}

func Test_processSPDXLicense(t *testing.T) {
	tests := []struct {
		fixture string
		want    map[string]string
	}{
		{
			fixture: "test-fixtures/licenses.json",
			want: map[string]string{
				"0bsd":                                   "0BSD",
				"aal":                                    "AAL",
				"abstyles":                               "Abstyles",
				"adobe-2006":                             "Adobe-2006",
				"adobe-2006.0":                           "Adobe-2006",
				"adobe-2006.0.0":                         "Adobe-2006",
				"adobe-glyph":                            "Adobe-Glyph",
				"adsl":                                   "ADSL",
				"afl-1":                                  "AFL-1.1",
				"afl-1.1":                                "AFL-1.1",
				"afl-1.1.0":                              "AFL-1.1",
				"afl-1.2":                                "AFL-1.2",
				"afl-1.2.0":                              "AFL-1.2",
				"afl-2":                                  "AFL-2.0",
				"afl-2.0":                                "AFL-2.0",
				"afl-2.0.0":                              "AFL-2.0",
				"afl-2.1":                                "AFL-2.1",
				"afl-2.1.0":                              "AFL-2.1",
				"afl-3":                                  "AFL-3.0",
				"afl-3.0":                                "AFL-3.0",
				"afl-3.0.0":                              "AFL-3.0",
				"afmparse":                               "Afmparse",
				"agpl-1":                                 "AGPL-1.0-only",
				"agpl-1-only":                            "AGPL-1.0-only",
				"agpl-1-or-later":                        "AGPL-1.0-or-later",
				"agpl-1.0":                               "AGPL-1.0-only",
				"agpl-1.0-only":                          "AGPL-1.0-only",
				"agpl-1.0-or-later":                      "AGPL-1.0-or-later",
				"agpl-1.0.0":                             "AGPL-1.0-only",
				"agpl-1.0.0-only":                        "AGPL-1.0-only",
				"agpl-1.0.0-or-later":                    "AGPL-1.0-or-later",
				"agpl-3":                                 "AGPL-3.0-only",
				"agpl-3-only":                            "AGPL-3.0-only",
				"agpl-3-or-later":                        "AGPL-3.0-or-later",
				"agpl-3.0":                               "AGPL-3.0-only",
				"agpl-3.0-only":                          "AGPL-3.0-only",
				"agpl-3.0-or-later":                      "AGPL-3.0-or-later",
				"agpl-3.0.0":                             "AGPL-3.0-only",
				"agpl-3.0.0-only":                        "AGPL-3.0-only",
				"agpl-3.0.0-or-later":                    "AGPL-3.0-or-later",
				"aladdin":                                "Aladdin",
				"amdplpa":                                "AMDPLPA",
				"aml":                                    "AML",
				"ampas":                                  "AMPAS",
				"antlr-pd":                               "ANTLR-PD",
				"antlr-pd-fallback":                      "ANTLR-PD-fallback",
				"apache-1":                               "Apache-1.0",
				"apache-1.0":                             "Apache-1.0",
				"apache-1.0.0":                           "Apache-1.0",
				"apache-1.1":                             "Apache-1.1",
				"apache-1.1.0":                           "Apache-1.1",
				"apache-2":                               "Apache-2.0",
				"apache-2.0":                             "Apache-2.0",
				"apache-2.0.0":                           "Apache-2.0",
				"apafml":                                 "APAFML",
				"apl-1":                                  "APL-1.0",
				"apl-1.0":                                "APL-1.0",
				"apl-1.0.0":                              "APL-1.0",
				"app-s2p":                                "App-s2p",
				"apsl-1":                                 "APSL-1.0",
				"apsl-1.0":                               "APSL-1.0",
				"apsl-1.0.0":                             "APSL-1.0",
				"apsl-1.1":                               "APSL-1.1",
				"apsl-1.1.0":                             "APSL-1.1",
				"apsl-1.2":                               "APSL-1.2",
				"apsl-1.2.0":                             "APSL-1.2",
				"apsl-2":                                 "APSL-2.0",
				"apsl-2.0":                               "APSL-2.0",
				"apsl-2.0.0":                             "APSL-2.0",
				"arphic-1999":                            "Arphic-1999",
				"arphic-1999.0":                          "Arphic-1999",
				"arphic-1999.0.0":                        "Arphic-1999",
				"artistic-1":                             "Artistic-1.0",
				"artistic-1-cl8":                         "Artistic-1.0-cl8",
				"artistic-1-perl":                        "Artistic-1.0-Perl",
				"artistic-1.0":                           "Artistic-1.0",
				"artistic-1.0-cl8":                       "Artistic-1.0-cl8",
				"artistic-1.0-perl":                      "Artistic-1.0-Perl",
				"artistic-1.0.0":                         "Artistic-1.0",
				"artistic-1.0.0-cl8":                     "Artistic-1.0-cl8",
				"artistic-1.0.0-perl":                    "Artistic-1.0-Perl",
				"artistic-2":                             "Artistic-2.0",
				"artistic-2.0":                           "Artistic-2.0",
				"artistic-2.0.0":                         "Artistic-2.0",
				"baekmuk":                                "Baekmuk",
				"bahyph":                                 "Bahyph",
				"barr":                                   "Barr",
				"beerware":                               "Beerware",
				"bitstream-vera":                         "Bitstream-Vera",
				"bittorrent-1":                           "BitTorrent-1.0",
				"bittorrent-1.0":                         "BitTorrent-1.0",
				"bittorrent-1.0.0":                       "BitTorrent-1.0",
				"bittorrent-1.1":                         "BitTorrent-1.1",
				"bittorrent-1.1.0":                       "BitTorrent-1.1",
				"blessing":                               "blessing",
				"blueoak-1":                              "BlueOak-1.0.0",
				"blueoak-1.0":                            "BlueOak-1.0.0",
				"blueoak-1.0.0":                          "BlueOak-1.0.0",
				"borceux":                                "Borceux",
				"bsd-1-clause":                           "BSD-1-Clause",
				"bsd-1.0-clause":                         "BSD-1-Clause",
				"bsd-1.0.0-clause":                       "BSD-1-Clause",
				"bsd-2-clause":                           "BSD-2-Clause",
				"bsd-2-clause-freebsd":                   "BSD-2-Clause-Views",
				"bsd-2-clause-netbsd":                    "BSD-2-Clause",
				"bsd-2-clause-patent":                    "BSD-2-Clause-Patent",
				"bsd-2-clause-views":                     "BSD-2-Clause-Views",
				"bsd-2.0-clause":                         "BSD-2-Clause",
				"bsd-2.0-clause-freebsd":                 "BSD-2-Clause-Views",
				"bsd-2.0-clause-netbsd":                  "BSD-2-Clause",
				"bsd-2.0-clause-patent":                  "BSD-2-Clause-Patent",
				"bsd-2.0-clause-views":                   "BSD-2-Clause-Views",
				"bsd-2.0.0-clause":                       "BSD-2-Clause",
				"bsd-2.0.0-clause-freebsd":               "BSD-2-Clause-Views",
				"bsd-2.0.0-clause-netbsd":                "BSD-2-Clause",
				"bsd-2.0.0-clause-patent":                "BSD-2-Clause-Patent",
				"bsd-2.0.0-clause-views":                 "BSD-2-Clause-Views",
				"bsd-3-clause":                           "BSD-3-Clause",
				"bsd-3-clause-attribution":               "BSD-3-Clause-Attribution",
				"bsd-3-clause-clear":                     "BSD-3-Clause-Clear",
				"bsd-3-clause-lbnl":                      "BSD-3-Clause-LBNL",
				"bsd-3-clause-modification":              "BSD-3-Clause-Modification",
				"bsd-3-clause-no-military-license":       "BSD-3-Clause-No-Military-License",
				"bsd-3-clause-no-nuclear-license":        "BSD-3-Clause-No-Nuclear-License",
				"bsd-3-clause-no-nuclear-license-2014":   "BSD-3-Clause-No-Nuclear-License-2014",
				"bsd-3-clause-no-nuclear-warranty":       "BSD-3-Clause-No-Nuclear-Warranty",
				"bsd-3-clause-open-mpi":                  "BSD-3-Clause-Open-MPI",
				"bsd-3.0-clause":                         "BSD-3-Clause",
				"bsd-3.0-clause-attribution":             "BSD-3-Clause-Attribution",
				"bsd-3.0-clause-clear":                   "BSD-3-Clause-Clear",
				"bsd-3.0-clause-lbnl":                    "BSD-3-Clause-LBNL",
				"bsd-3.0-clause-modification":            "BSD-3-Clause-Modification",
				"bsd-3.0-clause-no-military-license":     "BSD-3-Clause-No-Military-License",
				"bsd-3.0-clause-no-nuclear-license":      "BSD-3-Clause-No-Nuclear-License",
				"bsd-3.0-clause-no-nuclear-license-2014": "BSD-3-Clause-No-Nuclear-License-2014",
				"bsd-3.0-clause-no-nuclear-warranty":     "BSD-3-Clause-No-Nuclear-Warranty",
				"bsd-3.0-clause-open-mpi":                "BSD-3-Clause-Open-MPI",
				"bsd-3.0.0-clause":                       "BSD-3-Clause",
				"bsd-3.0.0-clause-attribution":           "BSD-3-Clause-Attribution",
				"bsd-3.0.0-clause-clear":                 "BSD-3-Clause-Clear",
				"bsd-3.0.0-clause-lbnl":                  "BSD-3-Clause-LBNL",
				"bsd-3.0.0-clause-modification":          "BSD-3-Clause-Modification",
				"bsd-3.0.0-clause-no-military-license":   "BSD-3-Clause-No-Military-License",
				"bsd-3.0.0-clause-no-nuclear-license":    "BSD-3-Clause-No-Nuclear-License",
				"bsd-3.0.0-clause-no-nuclear-license-2014": "BSD-3-Clause-No-Nuclear-License-2014",
				"bsd-3.0.0-clause-no-nuclear-warranty":     "BSD-3-Clause-No-Nuclear-Warranty",
				"bsd-3.0.0-clause-open-mpi":                "BSD-3-Clause-Open-MPI",
				"bsd-4-clause":                             "BSD-4-Clause",
				"bsd-4-clause-shortened":                   "BSD-4-Clause-Shortened",
				"bsd-4-clause-uc":                          "BSD-4-Clause-UC",
				"bsd-4.0-clause":                           "BSD-4-Clause",
				"bsd-4.0-clause-shortened":                 "BSD-4-Clause-Shortened",
				"bsd-4.0-clause-uc":                        "BSD-4-Clause-UC",
				"bsd-4.0.0-clause":                         "BSD-4-Clause",
				"bsd-4.0.0-clause-shortened":               "BSD-4-Clause-Shortened",
				"bsd-4.0.0-clause-uc":                      "BSD-4-Clause-UC",
				"bsd-protection":                           "BSD-Protection",
				"bsd-source-code":                          "BSD-Source-Code",
				"bsl-1":                                    "BSL-1.0",
				"bsl-1.0":                                  "BSL-1.0",
				"bsl-1.0.0":                                "BSL-1.0",
				"busl-1":                                   "BUSL-1.1",
				"busl-1.1":                                 "BUSL-1.1",
				"busl-1.1.0":                               "BUSL-1.1",
				"bzip2-1":                                  "bzip2-1.0.6",
				"bzip2-1.0":                                "bzip2-1.0.6",
				"bzip2-1.0.5":                              "bzip2-1.0.6",
				"bzip2-1.0.6":                              "bzip2-1.0.6",
				"c-uda-1":                                  "C-UDA-1.0",
				"c-uda-1.0":                                "C-UDA-1.0",
				"c-uda-1.0.0":                              "C-UDA-1.0",
				"cal-1":                                    "CAL-1.0",
				"cal-1-combined-work-exception":            "CAL-1.0-Combined-Work-Exception",
				"cal-1.0":                                  "CAL-1.0",
				"cal-1.0-combined-work-exception":          "CAL-1.0-Combined-Work-Exception",
				"cal-1.0.0":                                "CAL-1.0",
				"cal-1.0.0-combined-work-exception":        "CAL-1.0-Combined-Work-Exception",
				"caldera":                                  "Caldera",
				"catosl-1":                                 "CATOSL-1.1",
				"catosl-1.1":                               "CATOSL-1.1",
				"catosl-1.1.0":                             "CATOSL-1.1",
				"cc-by-1":                                  "CC-BY-1.0",
				"cc-by-1.0":                                "CC-BY-1.0",
				"cc-by-1.0.0":                              "CC-BY-1.0",
				"cc-by-2":                                  "CC-BY-2.0",
				"cc-by-2-au":                               "CC-BY-2.5-AU",
				"cc-by-2.0":                                "CC-BY-2.0",
				"cc-by-2.0.0":                              "CC-BY-2.0",
				"cc-by-2.5":                                "CC-BY-2.5",
				"cc-by-2.5-au":                             "CC-BY-2.5-AU",
				"cc-by-2.5.0":                              "CC-BY-2.5",
				"cc-by-2.5.0-au":                           "CC-BY-2.5-AU",
				"cc-by-3":                                  "CC-BY-3.0",
				"cc-by-3-at":                               "CC-BY-3.0-AT",
				"cc-by-3-de":                               "CC-BY-3.0-DE",
				"cc-by-3-igo":                              "CC-BY-3.0-IGO",
				"cc-by-3-nl":                               "CC-BY-3.0-NL",
				"cc-by-3-us":                               "CC-BY-3.0-US",
				"cc-by-3.0":                                "CC-BY-3.0",
				"cc-by-3.0-at":                             "CC-BY-3.0-AT",
				"cc-by-3.0-de":                             "CC-BY-3.0-DE",
				"cc-by-3.0-igo":                            "CC-BY-3.0-IGO",
				"cc-by-3.0-nl":                             "CC-BY-3.0-NL",
				"cc-by-3.0-us":                             "CC-BY-3.0-US",
				"cc-by-3.0.0":                              "CC-BY-3.0",
				"cc-by-3.0.0-at":                           "CC-BY-3.0-AT",
				"cc-by-3.0.0-de":                           "CC-BY-3.0-DE",
				"cc-by-3.0.0-igo":                          "CC-BY-3.0-IGO",
				"cc-by-3.0.0-nl":                           "CC-BY-3.0-NL",
				"cc-by-3.0.0-us":                           "CC-BY-3.0-US",
				"cc-by-4":                                  "CC-BY-4.0",
				"cc-by-4.0":                                "CC-BY-4.0",
				"cc-by-4.0.0":                              "CC-BY-4.0",
				"cc-by-nc-1":                               "CC-BY-NC-1.0",
				"cc-by-nc-1.0":                             "CC-BY-NC-1.0",
				"cc-by-nc-1.0.0":                           "CC-BY-NC-1.0",
				"cc-by-nc-2":                               "CC-BY-NC-2.0",
				"cc-by-nc-2.0":                             "CC-BY-NC-2.0",
				"cc-by-nc-2.0.0":                           "CC-BY-NC-2.0",
				"cc-by-nc-2.5":                             "CC-BY-NC-2.5",
				"cc-by-nc-2.5.0":                           "CC-BY-NC-2.5",
				"cc-by-nc-3":                               "CC-BY-NC-3.0",
				"cc-by-nc-3-de":                            "CC-BY-NC-3.0-DE",
				"cc-by-nc-3.0":                             "CC-BY-NC-3.0",
				"cc-by-nc-3.0-de":                          "CC-BY-NC-3.0-DE",
				"cc-by-nc-3.0.0":                           "CC-BY-NC-3.0",
				"cc-by-nc-3.0.0-de":                        "CC-BY-NC-3.0-DE",
				"cc-by-nc-4":                               "CC-BY-NC-4.0",
				"cc-by-nc-4.0":                             "CC-BY-NC-4.0",
				"cc-by-nc-4.0.0":                           "CC-BY-NC-4.0",
				"cc-by-nc-nd-1":                            "CC-BY-NC-ND-1.0",
				"cc-by-nc-nd-1.0":                          "CC-BY-NC-ND-1.0",
				"cc-by-nc-nd-1.0.0":                        "CC-BY-NC-ND-1.0",
				"cc-by-nc-nd-2":                            "CC-BY-NC-ND-2.0",
				"cc-by-nc-nd-2.0":                          "CC-BY-NC-ND-2.0",
				"cc-by-nc-nd-2.0.0":                        "CC-BY-NC-ND-2.0",
				"cc-by-nc-nd-2.5":                          "CC-BY-NC-ND-2.5",
				"cc-by-nc-nd-2.5.0":                        "CC-BY-NC-ND-2.5",
				"cc-by-nc-nd-3":                            "CC-BY-NC-ND-3.0",
				"cc-by-nc-nd-3-de":                         "CC-BY-NC-ND-3.0-DE",
				"cc-by-nc-nd-3-igo":                        "CC-BY-NC-ND-3.0-IGO",
				"cc-by-nc-nd-3.0":                          "CC-BY-NC-ND-3.0",
				"cc-by-nc-nd-3.0-de":                       "CC-BY-NC-ND-3.0-DE",
				"cc-by-nc-nd-3.0-igo":                      "CC-BY-NC-ND-3.0-IGO",
				"cc-by-nc-nd-3.0.0":                        "CC-BY-NC-ND-3.0",
				"cc-by-nc-nd-3.0.0-de":                     "CC-BY-NC-ND-3.0-DE",
				"cc-by-nc-nd-3.0.0-igo":                    "CC-BY-NC-ND-3.0-IGO",
				"cc-by-nc-nd-4":                            "CC-BY-NC-ND-4.0",
				"cc-by-nc-nd-4.0":                          "CC-BY-NC-ND-4.0",
				"cc-by-nc-nd-4.0.0":                        "CC-BY-NC-ND-4.0",
				"cc-by-nc-sa-1":                            "CC-BY-NC-SA-1.0",
				"cc-by-nc-sa-1.0":                          "CC-BY-NC-SA-1.0",
				"cc-by-nc-sa-1.0.0":                        "CC-BY-NC-SA-1.0",
				"cc-by-nc-sa-2":                            "CC-BY-NC-SA-2.0",
				"cc-by-nc-sa-2-fr":                         "CC-BY-NC-SA-2.0-FR",
				"cc-by-nc-sa-2-uk":                         "CC-BY-NC-SA-2.0-UK",
				"cc-by-nc-sa-2.0":                          "CC-BY-NC-SA-2.0",
				"cc-by-nc-sa-2.0-fr":                       "CC-BY-NC-SA-2.0-FR",
				"cc-by-nc-sa-2.0-uk":                       "CC-BY-NC-SA-2.0-UK",
				"cc-by-nc-sa-2.0.0":                        "CC-BY-NC-SA-2.0",
				"cc-by-nc-sa-2.0.0-fr":                     "CC-BY-NC-SA-2.0-FR",
				"cc-by-nc-sa-2.0.0-uk":                     "CC-BY-NC-SA-2.0-UK",
				"cc-by-nc-sa-2.5":                          "CC-BY-NC-SA-2.5",
				"cc-by-nc-sa-2.5.0":                        "CC-BY-NC-SA-2.5",
				"cc-by-nc-sa-3":                            "CC-BY-NC-SA-3.0",
				"cc-by-nc-sa-3-de":                         "CC-BY-NC-SA-3.0-DE",
				"cc-by-nc-sa-3-igo":                        "CC-BY-NC-SA-3.0-IGO",
				"cc-by-nc-sa-3.0":                          "CC-BY-NC-SA-3.0",
				"cc-by-nc-sa-3.0-de":                       "CC-BY-NC-SA-3.0-DE",
				"cc-by-nc-sa-3.0-igo":                      "CC-BY-NC-SA-3.0-IGO",
				"cc-by-nc-sa-3.0.0":                        "CC-BY-NC-SA-3.0",
				"cc-by-nc-sa-3.0.0-de":                     "CC-BY-NC-SA-3.0-DE",
				"cc-by-nc-sa-3.0.0-igo":                    "CC-BY-NC-SA-3.0-IGO",
				"cc-by-nc-sa-4":                            "CC-BY-NC-SA-4.0",
				"cc-by-nc-sa-4.0":                          "CC-BY-NC-SA-4.0",
				"cc-by-nc-sa-4.0.0":                        "CC-BY-NC-SA-4.0",
				"cc-by-nd-1":                               "CC-BY-ND-1.0",
				"cc-by-nd-1.0":                             "CC-BY-ND-1.0",
				"cc-by-nd-1.0.0":                           "CC-BY-ND-1.0",
				"cc-by-nd-2":                               "CC-BY-ND-2.0",
				"cc-by-nd-2.0":                             "CC-BY-ND-2.0",
				"cc-by-nd-2.0.0":                           "CC-BY-ND-2.0",
				"cc-by-nd-2.5":                             "CC-BY-ND-2.5",
				"cc-by-nd-2.5.0":                           "CC-BY-ND-2.5",
				"cc-by-nd-3":                               "CC-BY-ND-3.0",
				"cc-by-nd-3-de":                            "CC-BY-ND-3.0-DE",
				"cc-by-nd-3.0":                             "CC-BY-ND-3.0",
				"cc-by-nd-3.0-de":                          "CC-BY-ND-3.0-DE",
				"cc-by-nd-3.0.0":                           "CC-BY-ND-3.0",
				"cc-by-nd-3.0.0-de":                        "CC-BY-ND-3.0-DE",
				"cc-by-nd-4":                               "CC-BY-ND-4.0",
				"cc-by-nd-4.0":                             "CC-BY-ND-4.0",
				"cc-by-nd-4.0.0":                           "CC-BY-ND-4.0",
				"cc-by-sa-1":                               "CC-BY-SA-1.0",
				"cc-by-sa-1.0":                             "CC-BY-SA-1.0",
				"cc-by-sa-1.0.0":                           "CC-BY-SA-1.0",
				"cc-by-sa-2":                               "CC-BY-SA-2.0",
				"cc-by-sa-2-jp":                            "CC-BY-SA-2.1-JP",
				"cc-by-sa-2-uk":                            "CC-BY-SA-2.0-UK",
				"cc-by-sa-2.0":                             "CC-BY-SA-2.0",
				"cc-by-sa-2.0-uk":                          "CC-BY-SA-2.0-UK",
				"cc-by-sa-2.0.0":                           "CC-BY-SA-2.0",
				"cc-by-sa-2.0.0-uk":                        "CC-BY-SA-2.0-UK",
				"cc-by-sa-2.1-jp":                          "CC-BY-SA-2.1-JP",
				"cc-by-sa-2.1.0-jp":                        "CC-BY-SA-2.1-JP",
				"cc-by-sa-2.5":                             "CC-BY-SA-2.5",
				"cc-by-sa-2.5.0":                           "CC-BY-SA-2.5",
				"cc-by-sa-3":                               "CC-BY-SA-3.0",
				"cc-by-sa-3-at":                            "CC-BY-SA-3.0-AT",
				"cc-by-sa-3-de":                            "CC-BY-SA-3.0-DE",
				"cc-by-sa-3.0":                             "CC-BY-SA-3.0",
				"cc-by-sa-3.0-at":                          "CC-BY-SA-3.0-AT",
				"cc-by-sa-3.0-de":                          "CC-BY-SA-3.0-DE",
				"cc-by-sa-3.0.0":                           "CC-BY-SA-3.0",
				"cc-by-sa-3.0.0-at":                        "CC-BY-SA-3.0-AT",
				"cc-by-sa-3.0.0-de":                        "CC-BY-SA-3.0-DE",
				"cc-by-sa-4":                               "CC-BY-SA-4.0",
				"cc-by-sa-4.0":                             "CC-BY-SA-4.0",
				"cc-by-sa-4.0.0":                           "CC-BY-SA-4.0",
				"cc-pddc":                                  "CC-PDDC",
				"cc0-1":                                    "CC0-1.0",
				"cc0-1.0":                                  "CC0-1.0",
				"cc0-1.0.0":                                "CC0-1.0",
				"cddl-1":                                   "CDDL-1.0",
				"cddl-1.0":                                 "CDDL-1.0",
				"cddl-1.0.0":                               "CDDL-1.0",
				"cddl-1.1":                                 "CDDL-1.1",
				"cddl-1.1.0":                               "CDDL-1.1",
				"cdl-1":                                    "CDL-1.0",
				"cdl-1.0":                                  "CDL-1.0",
				"cdl-1.0.0":                                "CDL-1.0",
				"cdla-permissive-1":                        "CDLA-Permissive-1.0",
				"cdla-permissive-1.0":                      "CDLA-Permissive-1.0",
				"cdla-permissive-1.0.0":                    "CDLA-Permissive-1.0",
				"cdla-permissive-2":                        "CDLA-Permissive-2.0",
				"cdla-permissive-2.0":                      "CDLA-Permissive-2.0",
				"cdla-permissive-2.0.0":                    "CDLA-Permissive-2.0",
				"cdla-sharing-1":                           "CDLA-Sharing-1.0",
				"cdla-sharing-1.0":                         "CDLA-Sharing-1.0",
				"cdla-sharing-1.0.0":                       "CDLA-Sharing-1.0",
				"cecill-1":                                 "CECILL-1.0",
				"cecill-1.0":                               "CECILL-1.0",
				"cecill-1.0.0":                             "CECILL-1.0",
				"cecill-1.1":                               "CECILL-1.1",
				"cecill-1.1.0":                             "CECILL-1.1",
				"cecill-2":                                 "CECILL-2.0",
				"cecill-2.0":                               "CECILL-2.0",
				"cecill-2.0.0":                             "CECILL-2.0",
				"cecill-2.1":                               "CECILL-2.1",
				"cecill-2.1.0":                             "CECILL-2.1",
				"cecill-b":                                 "CECILL-B",
				"cecill-c":                                 "CECILL-C",
				"cern-ohl-1":                               "CERN-OHL-1.1",
				"cern-ohl-1.1":                             "CERN-OHL-1.1",
				"cern-ohl-1.1.0":                           "CERN-OHL-1.1",
				"cern-ohl-1.2":                             "CERN-OHL-1.2",
				"cern-ohl-1.2.0":                           "CERN-OHL-1.2",
				"cern-ohl-p-2":                             "CERN-OHL-P-2.0",
				"cern-ohl-p-2.0":                           "CERN-OHL-P-2.0",
				"cern-ohl-p-2.0.0":                         "CERN-OHL-P-2.0",
				"cern-ohl-s-2":                             "CERN-OHL-S-2.0",
				"cern-ohl-s-2.0":                           "CERN-OHL-S-2.0",
				"cern-ohl-s-2.0.0":                         "CERN-OHL-S-2.0",
				"cern-ohl-w-2":                             "CERN-OHL-W-2.0",
				"cern-ohl-w-2.0":                           "CERN-OHL-W-2.0",
				"cern-ohl-w-2.0.0":                         "CERN-OHL-W-2.0",
				"checkmk":                                  "checkmk",
				"clartistic":                               "ClArtistic",
				"cnri-jython":                              "CNRI-Jython",
				"cnri-python":                              "CNRI-Python",
				"cnri-python-gpl-compatible":               "CNRI-Python-GPL-Compatible",
				"coil-1":                                   "COIL-1.0",
				"coil-1.0":                                 "COIL-1.0",
				"coil-1.0.0":                               "COIL-1.0",
				"community-spec-1":                         "Community-Spec-1.0",
				"community-spec-1.0":                       "Community-Spec-1.0",
				"community-spec-1.0.0":                     "Community-Spec-1.0",
				"condor-1":                                 "Condor-1.1",
				"condor-1.1":                               "Condor-1.1",
				"condor-1.1.0":                             "Condor-1.1",
				"copyleft-next-0.3":                        "copyleft-next-0.3.0",
				"copyleft-next-0.3.0":                      "copyleft-next-0.3.0",
				"copyleft-next-0.3.1":                      "copyleft-next-0.3.1",
				"cpal-1":                                   "CPAL-1.0",
				"cpal-1.0":                                 "CPAL-1.0",
				"cpal-1.0.0":                               "CPAL-1.0",
				"cpl-1":                                    "CPL-1.0",
				"cpl-1.0":                                  "CPL-1.0",
				"cpl-1.0.0":                                "CPL-1.0",
				"cpol-1":                                   "CPOL-1.02",
				"cpol-1.02":                                "CPOL-1.02",
				"cpol-1.02.0":                              "CPOL-1.02",
				"crossword":                                "Crossword",
				"crystalstacker":                           "CrystalStacker",
				"cua-opl-1":                                "CUA-OPL-1.0",
				"cua-opl-1.0":                              "CUA-OPL-1.0",
				"cua-opl-1.0.0":                            "CUA-OPL-1.0",
				"cube":                                     "Cube",
				"curl":                                     "curl",
				"d-fsl-1":                                  "D-FSL-1.0",
				"d-fsl-1.0":                                "D-FSL-1.0",
				"d-fsl-1.0.0":                              "D-FSL-1.0",
				"diffmark":                                 "diffmark",
				"dl-de-by-2":                               "DL-DE-BY-2.0",
				"dl-de-by-2.0":                             "DL-DE-BY-2.0",
				"dl-de-by-2.0.0":                           "DL-DE-BY-2.0",
				"doc":                                      "DOC",
				"dotseqn":                                  "Dotseqn",
				"drl-1":                                    "DRL-1.0",
				"drl-1.0":                                  "DRL-1.0",
				"drl-1.0.0":                                "DRL-1.0",
				"dsdp":                                     "DSDP",
				"dvipdfm":                                  "dvipdfm",
				"ecl-1":                                    "ECL-1.0",
				"ecl-1.0":                                  "ECL-1.0",
				"ecl-1.0.0":                                "ECL-1.0",
				"ecl-2":                                    "ECL-2.0",
				"ecl-2.0":                                  "ECL-2.0",
				"ecl-2.0.0":                                "ECL-2.0",
				"ecos-2":                                   "eCos-2.0",
				"ecos-2.0":                                 "eCos-2.0",
				"ecos-2.0.0":                               "eCos-2.0",
				"efl-1":                                    "EFL-1.0",
				"efl-1.0":                                  "EFL-1.0",
				"efl-1.0.0":                                "EFL-1.0",
				"efl-2":                                    "EFL-2.0",
				"efl-2.0":                                  "EFL-2.0",
				"efl-2.0.0":                                "EFL-2.0",
				"egenix":                                   "eGenix",
				"elastic-2":                                "Elastic-2.0",
				"elastic-2.0":                              "Elastic-2.0",
				"elastic-2.0.0":                            "Elastic-2.0",
				"entessa":                                  "Entessa",
				"epics":                                    "EPICS",
				"epl-1":                                    "EPL-1.0",
				"epl-1.0":                                  "EPL-1.0",
				"epl-1.0.0":                                "EPL-1.0",
				"epl-2":                                    "EPL-2.0",
				"epl-2.0":                                  "EPL-2.0",
				"epl-2.0.0":                                "EPL-2.0",
				"erlpl-1":                                  "ErlPL-1.1",
				"erlpl-1.1":                                "ErlPL-1.1",
				"erlpl-1.1.0":                              "ErlPL-1.1",
				"etalab-2":                                 "etalab-2.0",
				"etalab-2.0":                               "etalab-2.0",
				"etalab-2.0.0":                             "etalab-2.0",
				"eudatagrid":                               "EUDatagrid",
				"eupl-1":                                   "EUPL-1.0",
				"eupl-1.0":                                 "EUPL-1.0",
				"eupl-1.0.0":                               "EUPL-1.0",
				"eupl-1.1":                                 "EUPL-1.1",
				"eupl-1.1.0":                               "EUPL-1.1",
				"eupl-1.2":                                 "EUPL-1.2",
				"eupl-1.2.0":                               "EUPL-1.2",
				"eurosym":                                  "Eurosym",
				"fair":                                     "Fair",
				"fdk-aac":                                  "FDK-AAC",
				"frameworx-1":                              "Frameworx-1.0",
				"frameworx-1.0":                            "Frameworx-1.0",
				"frameworx-1.0.0":                          "Frameworx-1.0",
				"freebsd-doc":                              "FreeBSD-DOC",
				"freeimage":                                "FreeImage",
				"fsfap":                                    "FSFAP",
				"fsful":                                    "FSFUL",
				"fsfullr":                                  "FSFULLR",
				"fsfullrwd":                                "FSFULLRWD",
				"ftl":                                      "FTL",
				"gd":                                       "GD",
				"gfdl-1":                                   "GFDL-1.1-only",
				"gfdl-1-invariants-only":                   "GFDL-1.1-invariants-only",
				"gfdl-1-invariants-or-later":               "GFDL-1.1-invariants-or-later",
				"gfdl-1-no-invariants-only":                "GFDL-1.1-no-invariants-only",
				"gfdl-1-no-invariants-or-later":            "GFDL-1.1-no-invariants-or-later",
				"gfdl-1-only":                              "GFDL-1.1-only",
				"gfdl-1-or-later":                          "GFDL-1.1-or-later",
				"gfdl-1.1":                                 "GFDL-1.1-only",
				"gfdl-1.1-invariants-only":                 "GFDL-1.1-invariants-only",
				"gfdl-1.1-invariants-or-later":             "GFDL-1.1-invariants-or-later",
				"gfdl-1.1-no-invariants-only":              "GFDL-1.1-no-invariants-only",
				"gfdl-1.1-no-invariants-or-later":          "GFDL-1.1-no-invariants-or-later",
				"gfdl-1.1-only":                            "GFDL-1.1-only",
				"gfdl-1.1-or-later":                        "GFDL-1.1-or-later",
				"gfdl-1.1.0":                               "GFDL-1.1-only",
				"gfdl-1.1.0-invariants-only":               "GFDL-1.1-invariants-only",
				"gfdl-1.1.0-invariants-or-later":           "GFDL-1.1-invariants-or-later",
				"gfdl-1.1.0-no-invariants-only":            "GFDL-1.1-no-invariants-only",
				"gfdl-1.1.0-no-invariants-or-later":        "GFDL-1.1-no-invariants-or-later",
				"gfdl-1.1.0-only":                          "GFDL-1.1-only",
				"gfdl-1.1.0-or-later":                      "GFDL-1.1-or-later",
				"gfdl-1.2":                                 "GFDL-1.2-only",
				"gfdl-1.2-invariants-only":                 "GFDL-1.2-invariants-only",
				"gfdl-1.2-invariants-or-later":             "GFDL-1.2-invariants-or-later",
				"gfdl-1.2-no-invariants-only":              "GFDL-1.2-no-invariants-only",
				"gfdl-1.2-no-invariants-or-later":          "GFDL-1.2-no-invariants-or-later",
				"gfdl-1.2-only":                            "GFDL-1.2-only",
				"gfdl-1.2-or-later":                        "GFDL-1.2-or-later",
				"gfdl-1.2.0":                               "GFDL-1.2-only",
				"gfdl-1.2.0-invariants-only":               "GFDL-1.2-invariants-only",
				"gfdl-1.2.0-invariants-or-later":           "GFDL-1.2-invariants-or-later",
				"gfdl-1.2.0-no-invariants-only":            "GFDL-1.2-no-invariants-only",
				"gfdl-1.2.0-no-invariants-or-later":        "GFDL-1.2-no-invariants-or-later",
				"gfdl-1.2.0-only":                          "GFDL-1.2-only",
				"gfdl-1.2.0-or-later":                      "GFDL-1.2-or-later",
				"gfdl-1.3":                                 "GFDL-1.3-only",
				"gfdl-1.3-invariants-only":                 "GFDL-1.3-invariants-only",
				"gfdl-1.3-invariants-or-later":             "GFDL-1.3-invariants-or-later",
				"gfdl-1.3-no-invariants-only":              "GFDL-1.3-no-invariants-only",
				"gfdl-1.3-no-invariants-or-later":          "GFDL-1.3-no-invariants-or-later",
				"gfdl-1.3-only":                            "GFDL-1.3-only",
				"gfdl-1.3-or-later":                        "GFDL-1.3-or-later",
				"gfdl-1.3.0":                               "GFDL-1.3-only",
				"gfdl-1.3.0-invariants-only":               "GFDL-1.3-invariants-only",
				"gfdl-1.3.0-invariants-or-later":           "GFDL-1.3-invariants-or-later",
				"gfdl-1.3.0-no-invariants-only":            "GFDL-1.3-no-invariants-only",
				"gfdl-1.3.0-no-invariants-or-later":        "GFDL-1.3-no-invariants-or-later",
				"gfdl-1.3.0-only":                          "GFDL-1.3-only",
				"gfdl-1.3.0-or-later":                      "GFDL-1.3-or-later",
				"giftware":                                 "Giftware",
				"gl2ps":                                    "GL2PS",
				"glide":                                    "Glide",
				"glulxe":                                   "Glulxe",
				"glwtpl":                                   "GLWTPL",
				"gnuplot":                                  "gnuplot",
				"gpl-1":                                    "GPL-1.0-only",
				"gpl-1+":                                   "GPL-1.0-or-later",
				"gpl-1-only":                               "GPL-1.0-only",
				"gpl-1-or-later":                           "GPL-1.0-or-later",
				"gpl-1.0":                                  "GPL-1.0-only",
				"gpl-1.0+":                                 "GPL-1.0-or-later",
				"gpl-1.0-only":                             "GPL-1.0-only",
				"gpl-1.0-or-later":                         "GPL-1.0-or-later",
				"gpl-1.0.0":                                "GPL-1.0-only",
				"gpl-1.0.0+":                               "GPL-1.0-or-later",
				"gpl-1.0.0-only":                           "GPL-1.0-only",
				"gpl-1.0.0-or-later":                       "GPL-1.0-or-later",
				"gpl-2":                                    "GPL-2.0-only",
				"gpl-2+":                                   "GPL-2.0-or-later",
				"gpl-2-only":                               "GPL-2.0-only",
				"gpl-2-or-later":                           "GPL-2.0-or-later",
				"gpl-2-with-autoconf-exception":            "GPL-2.0-with-autoconf-exception",
				"gpl-2-with-bison-exception":               "GPL-2.0-with-bison-exception",
				"gpl-2-with-classpath-exception":           "GPL-2.0-with-classpath-exception",
				"gpl-2-with-font-exception":                "GPL-2.0-with-font-exception",
				"gpl-2-with-gcc-exception":                 "GPL-2.0-with-GCC-exception",
				"gpl-2.0":                                  "GPL-2.0-only",
				"gpl-2.0+":                                 "GPL-2.0-or-later",
				"gpl-2.0-only":                             "GPL-2.0-only",
				"gpl-2.0-or-later":                         "GPL-2.0-or-later",
				"gpl-2.0-with-autoconf-exception":          "GPL-2.0-with-autoconf-exception",
				"gpl-2.0-with-bison-exception":             "GPL-2.0-with-bison-exception",
				"gpl-2.0-with-classpath-exception":         "GPL-2.0-with-classpath-exception",
				"gpl-2.0-with-font-exception":              "GPL-2.0-with-font-exception",
				"gpl-2.0-with-gcc-exception":               "GPL-2.0-with-GCC-exception",
				"gpl-2.0.0":                                "GPL-2.0-only",
				"gpl-2.0.0+":                               "GPL-2.0-or-later",
				"gpl-2.0.0-only":                           "GPL-2.0-only",
				"gpl-2.0.0-or-later":                       "GPL-2.0-or-later",
				"gpl-2.0.0-with-autoconf-exception":        "GPL-2.0-with-autoconf-exception",
				"gpl-2.0.0-with-bison-exception":           "GPL-2.0-with-bison-exception",
				"gpl-2.0.0-with-classpath-exception":       "GPL-2.0-with-classpath-exception",
				"gpl-2.0.0-with-font-exception":            "GPL-2.0-with-font-exception",
				"gpl-2.0.0-with-gcc-exception":             "GPL-2.0-with-GCC-exception",
				"gpl-3":                                    "GPL-3.0-only",
				"gpl-3+":                                   "GPL-3.0-or-later",
				"gpl-3-only":                               "GPL-3.0-only",
				"gpl-3-or-later":                           "GPL-3.0-or-later",
				"gpl-3-with-autoconf-exception":            "GPL-3.0-with-autoconf-exception",
				"gpl-3-with-gcc-exception":                 "GPL-3.0-with-GCC-exception",
				"gpl-3.0":                                  "GPL-3.0-only",
				"gpl-3.0+":                                 "GPL-3.0-or-later",
				"gpl-3.0-only":                             "GPL-3.0-only",
				"gpl-3.0-or-later":                         "GPL-3.0-or-later",
				"gpl-3.0-with-autoconf-exception":          "GPL-3.0-with-autoconf-exception",
				"gpl-3.0-with-gcc-exception":               "GPL-3.0-with-GCC-exception",
				"gpl-3.0.0":                                "GPL-3.0-only",
				"gpl-3.0.0+":                               "GPL-3.0-or-later",
				"gpl-3.0.0-only":                           "GPL-3.0-only",
				"gpl-3.0.0-or-later":                       "GPL-3.0-or-later",
				"gpl-3.0.0-with-autoconf-exception":        "GPL-3.0-with-autoconf-exception",
				"gpl-3.0.0-with-gcc-exception":             "GPL-3.0-with-GCC-exception",
				"gsoap-1.3.0b":                             "gSOAP-1.3b",
				"gsoap-1.3b":                               "gSOAP-1.3b",
				"gsoap-1b":                                 "gSOAP-1.3b",
				"haskellreport":                            "HaskellReport",
				"hippocratic-2":                            "Hippocratic-2.1",
				"hippocratic-2.1":                          "Hippocratic-2.1",
				"hippocratic-2.1.0":                        "Hippocratic-2.1",
				"hpnd":                                     "HPND",
				"hpnd-sell-variant":                        "HPND-sell-variant",
				"htmltidy":                                 "HTMLTIDY",
				"ibm-pibs":                                 "IBM-pibs",
				"icu":                                      "ICU",
				"ijg":                                      "IJG",
				"imagemagick":                              "ImageMagick",
				"imatix":                                   "iMatix",
				"imlib2":                                   "Imlib2",
				"info-zip":                                 "Info-ZIP",
				"intel":                                    "Intel",
				"intel-acpi":                               "Intel-ACPI",
				"interbase-1":                              "Interbase-1.0",
				"interbase-1.0":                            "Interbase-1.0",
				"interbase-1.0.0":                          "Interbase-1.0",
				"ipa":                                      "IPA",
				"ipl-1":                                    "IPL-1.0",
				"ipl-1.0":                                  "IPL-1.0",
				"ipl-1.0.0":                                "IPL-1.0",
				"isc":                                      "ISC",
				"jam":                                      "Jam",
				"jasper-2":                                 "JasPer-2.0",
				"jasper-2.0":                               "JasPer-2.0",
				"jasper-2.0.0":                             "JasPer-2.0",
				"jpnic":                                    "JPNIC",
				"json":                                     "JSON",
				"knuth-ctan":                               "Knuth-CTAN",
				"lal-1":                                    "LAL-1.2",
				"lal-1.2":                                  "LAL-1.2",
				"lal-1.2.0":                                "LAL-1.2",
				"lal-1.3":                                  "LAL-1.3",
				"lal-1.3.0":                                "LAL-1.3",
				"latex2e":                                  "Latex2e",
				"leptonica":                                "Leptonica",
				"lgpl-2":                                   "LGPL-2.0-only", // This is an important line in testing! lgpl-2.1 would typically override this.
				"lgpl-2+":                                  "LGPL-2.0-or-later",
				"lgpl-2-only":                              "LGPL-2.0-only",
				"lgpl-2-or-later":                          "LGPL-2.0-or-later",
				"lgpl-2.0":                                 "LGPL-2.0-only",
				"lgpl-2.0+":                                "LGPL-2.0-or-later",
				"lgpl-2.0-only":                            "LGPL-2.0-only",
				"lgpl-2.0-or-later":                        "LGPL-2.0-or-later",
				"lgpl-2.0.0":                               "LGPL-2.0-only",
				"lgpl-2.0.0+":                              "LGPL-2.0-or-later",
				"lgpl-2.0.0-only":                          "LGPL-2.0-only",
				"lgpl-2.0.0-or-later":                      "LGPL-2.0-or-later",
				"lgpl-2.1":                                 "LGPL-2.1-only",
				"lgpl-2.1+":                                "LGPL-2.1-or-later",
				"lgpl-2.1-only":                            "LGPL-2.1-only",
				"lgpl-2.1-or-later":                        "LGPL-2.1-or-later",
				"lgpl-2.1.0":                               "LGPL-2.1-only",
				"lgpl-2.1.0+":                              "LGPL-2.1-or-later",
				"lgpl-2.1.0-only":                          "LGPL-2.1-only",
				"lgpl-2.1.0-or-later":                      "LGPL-2.1-or-later",
				"lgpl-3":                                   "LGPL-3.0-only",
				"lgpl-3+":                                  "LGPL-3.0-or-later",
				"lgpl-3-only":                              "LGPL-3.0-only",
				"lgpl-3-or-later":                          "LGPL-3.0-or-later",
				"lgpl-3.0":                                 "LGPL-3.0-only",
				"lgpl-3.0+":                                "LGPL-3.0-or-later",
				"lgpl-3.0-only":                            "LGPL-3.0-only",
				"lgpl-3.0-or-later":                        "LGPL-3.0-or-later",
				"lgpl-3.0.0":                               "LGPL-3.0-only",
				"lgpl-3.0.0+":                              "LGPL-3.0-or-later",
				"lgpl-3.0.0-only":                          "LGPL-3.0-only",
				"lgpl-3.0.0-or-later":                      "LGPL-3.0-or-later",
				"lgpllr":                                   "LGPLLR",
				"libpng":                                   "Libpng",
				"libpng-2":                                 "libpng-2.0",
				"libpng-2.0":                               "libpng-2.0",
				"libpng-2.0.0":                             "libpng-2.0",
				"libselinux-1":                             "libselinux-1.0",
				"libselinux-1.0":                           "libselinux-1.0",
				"libselinux-1.0.0":                         "libselinux-1.0",
				"libtiff":                                  "libtiff",
				"libutil-david-nugent":                     "libutil-David-Nugent",
				"liliq-p-1":                                "LiLiQ-P-1.1",
				"liliq-p-1.1":                              "LiLiQ-P-1.1",
				"liliq-p-1.1.0":                            "LiLiQ-P-1.1",
				"liliq-r-1":                                "LiLiQ-R-1.1",
				"liliq-r-1.1":                              "LiLiQ-R-1.1",
				"liliq-r-1.1.0":                            "LiLiQ-R-1.1",
				"liliq-rplus-1":                            "LiLiQ-Rplus-1.1",
				"liliq-rplus-1.1":                          "LiLiQ-Rplus-1.1",
				"liliq-rplus-1.1.0":                        "LiLiQ-Rplus-1.1",
				"linux-man-pages-copyleft":                 "Linux-man-pages-copyleft",
				"linux-openib":                             "Linux-OpenIB",
				"lpl-1":                                    "LPL-1.0",
				"lpl-1.0":                                  "LPL-1.0",
				"lpl-1.0.0":                                "LPL-1.0",
				"lpl-1.02":                                 "LPL-1.02",
				"lpl-1.02.0":                               "LPL-1.02",
				"lppl-1":                                   "LPPL-1.0",
				"lppl-1.0":                                 "LPPL-1.0",
				"lppl-1.0.0":                               "LPPL-1.0",
				"lppl-1.1":                                 "LPPL-1.1",
				"lppl-1.1.0":                               "LPPL-1.1",
				"lppl-1.2":                                 "LPPL-1.2",
				"lppl-1.2.0":                               "LPPL-1.2",
				"lppl-1.3.0a":                              "LPPL-1.3a",
				"lppl-1.3.0c":                              "LPPL-1.3c",
				"lppl-1.3a":                                "LPPL-1.3a",
				"lppl-1.3c":                                "LPPL-1.3c",
				"lppl-1a":                                  "LPPL-1.3a",
				"lppl-1c":                                  "LPPL-1.3c",
				"lzma-sdk-9":                               "LZMA-SDK-9.22",
				"lzma-sdk-9-to-9.20":                       "LZMA-SDK-9.11-to-9.20",
				"lzma-sdk-9.11-to-9.20":                    "LZMA-SDK-9.11-to-9.20",
				"lzma-sdk-9.11.0-to-9.20":                  "LZMA-SDK-9.11-to-9.20",
				"lzma-sdk-9.22":                            "LZMA-SDK-9.22",
				"lzma-sdk-9.22.0":                          "LZMA-SDK-9.22",
				"makeindex":                                "MakeIndex",
				"minpack":                                  "Minpack",
				"miros":                                    "MirOS",
				"mit":                                      "MIT",
				"mit-0":                                    "MIT-0",
				"mit-advertising":                          "MIT-advertising",
				"mit-cmu":                                  "MIT-CMU",
				"mit-enna":                                 "MIT-enna",
				"mit-feh":                                  "MIT-feh",
				"mit-modern-variant":                       "MIT-Modern-Variant",
				"mit-open-group":                           "MIT-open-group",
				"mitnfa":                                   "MITNFA",
				"motosoto":                                 "Motosoto",
				"mpi-permissive":                           "mpi-permissive",
				"mpich2":                                   "mpich2",
				"mpl-1":                                    "MPL-1.0",
				"mpl-1.0":                                  "MPL-1.0",
				"mpl-1.0.0":                                "MPL-1.0",
				"mpl-1.1":                                  "MPL-1.1",
				"mpl-1.1.0":                                "MPL-1.1",
				"mpl-2":                                    "MPL-2.0",
				"mpl-2-no-copyleft-exception":              "MPL-2.0-no-copyleft-exception",
				"mpl-2.0":                                  "MPL-2.0",
				"mpl-2.0-no-copyleft-exception":            "MPL-2.0-no-copyleft-exception",
				"mpl-2.0.0":                                "MPL-2.0",
				"mpl-2.0.0-no-copyleft-exception":          "MPL-2.0-no-copyleft-exception",
				"mplus":                                    "mplus",
				"ms-lpl":                                   "MS-LPL",
				"ms-pl":                                    "MS-PL",
				"ms-rl":                                    "MS-RL",
				"mtll":                                     "MTLL",
				"mulanpsl-1":                               "MulanPSL-1.0",
				"mulanpsl-1.0":                             "MulanPSL-1.0",
				"mulanpsl-1.0.0":                           "MulanPSL-1.0",
				"mulanpsl-2":                               "MulanPSL-2.0",
				"mulanpsl-2.0":                             "MulanPSL-2.0",
				"mulanpsl-2.0.0":                           "MulanPSL-2.0",
				"multics":                                  "Multics",
				"mup":                                      "Mup",
				"naist-2003":                               "NAIST-2003",
				"naist-2003.0":                             "NAIST-2003",
				"naist-2003.0.0":                           "NAIST-2003",
				"nasa-1":                                   "NASA-1.3",
				"nasa-1.3":                                 "NASA-1.3",
				"nasa-1.3.0":                               "NASA-1.3",
				"naumen":                                   "Naumen",
				"nbpl-1":                                   "NBPL-1.0",
				"nbpl-1.0":                                 "NBPL-1.0",
				"nbpl-1.0.0":                               "NBPL-1.0",
				"ncgl-uk-2":                                "NCGL-UK-2.0",
				"ncgl-uk-2.0":                              "NCGL-UK-2.0",
				"ncgl-uk-2.0.0":                            "NCGL-UK-2.0",
				"ncsa":                                     "NCSA",
				"net-snmp":                                 "Net-SNMP",
				"netcdf":                                   "NetCDF",
				"newsletr":                                 "Newsletr",
				"ngpl":                                     "NGPL",
				"nicta-1":                                  "NICTA-1.0",
				"nicta-1.0":                                "NICTA-1.0",
				"nicta-1.0.0":                              "NICTA-1.0",
				"nist-pd":                                  "NIST-PD",
				"nist-pd-fallback":                         "NIST-PD-fallback",
				"nlod-1":                                   "NLOD-1.0",
				"nlod-1.0":                                 "NLOD-1.0",
				"nlod-1.0.0":                               "NLOD-1.0",
				"nlod-2":                                   "NLOD-2.0",
				"nlod-2.0":                                 "NLOD-2.0",
				"nlod-2.0.0":                               "NLOD-2.0",
				"nlpl":                                     "NLPL",
				"nokia":                                    "Nokia",
				"nosl":                                     "NOSL",
				"noweb":                                    "Noweb",
				"npl-1":                                    "NPL-1.0",
				"npl-1.0":                                  "NPL-1.0",
				"npl-1.0.0":                                "NPL-1.0",
				"npl-1.1":                                  "NPL-1.1",
				"npl-1.1.0":                                "NPL-1.1",
				"nposl-3":                                  "NPOSL-3.0",
				"nposl-3.0":                                "NPOSL-3.0",
				"nposl-3.0.0":                              "NPOSL-3.0",
				"nrl":                                      "NRL",
				"ntp":                                      "NTP",
				"ntp-0":                                    "NTP-0",
				"nunit":                                    "Nunit",
				"o-uda-1":                                  "O-UDA-1.0",
				"o-uda-1.0":                                "O-UDA-1.0",
				"o-uda-1.0.0":                              "O-UDA-1.0",
				"occt-pl":                                  "OCCT-PL",
				"oclc-2":                                   "OCLC-2.0",
				"oclc-2.0":                                 "OCLC-2.0",
				"oclc-2.0.0":                               "OCLC-2.0",
				"odbl-1":                                   "ODbL-1.0",
				"odbl-1.0":                                 "ODbL-1.0",
				"odbl-1.0.0":                               "ODbL-1.0",
				"odc-by-1":                                 "ODC-By-1.0",
				"odc-by-1.0":                               "ODC-By-1.0",
				"odc-by-1.0.0":                             "ODC-By-1.0",
				"ofl-1":                                    "OFL-1.0",
				"ofl-1-no-rfn":                             "OFL-1.0-no-RFN",
				"ofl-1-rfn":                                "OFL-1.0-RFN",
				"ofl-1.0":                                  "OFL-1.0",
				"ofl-1.0-no-rfn":                           "OFL-1.0-no-RFN",
				"ofl-1.0-rfn":                              "OFL-1.0-RFN",
				"ofl-1.0.0":                                "OFL-1.0",
				"ofl-1.0.0-no-rfn":                         "OFL-1.0-no-RFN",
				"ofl-1.0.0-rfn":                            "OFL-1.0-RFN",
				"ofl-1.1":                                  "OFL-1.1",
				"ofl-1.1-no-rfn":                           "OFL-1.1-no-RFN",
				"ofl-1.1-rfn":                              "OFL-1.1-RFN",
				"ofl-1.1.0":                                "OFL-1.1",
				"ofl-1.1.0-no-rfn":                         "OFL-1.1-no-RFN",
				"ofl-1.1.0-rfn":                            "OFL-1.1-RFN",
				"ogc-1":                                    "OGC-1.0",
				"ogc-1.0":                                  "OGC-1.0",
				"ogc-1.0.0":                                "OGC-1.0",
				"ogdl-taiwan-1":                            "OGDL-Taiwan-1.0",
				"ogdl-taiwan-1.0":                          "OGDL-Taiwan-1.0",
				"ogdl-taiwan-1.0.0":                        "OGDL-Taiwan-1.0",
				"ogl-canada-2":                             "OGL-Canada-2.0",
				"ogl-canada-2.0":                           "OGL-Canada-2.0",
				"ogl-canada-2.0.0":                         "OGL-Canada-2.0",
				"ogl-uk-1":                                 "OGL-UK-1.0",
				"ogl-uk-1.0":                               "OGL-UK-1.0",
				"ogl-uk-1.0.0":                             "OGL-UK-1.0",
				"ogl-uk-2":                                 "OGL-UK-2.0",
				"ogl-uk-2.0":                               "OGL-UK-2.0",
				"ogl-uk-2.0.0":                             "OGL-UK-2.0",
				"ogl-uk-3":                                 "OGL-UK-3.0",
				"ogl-uk-3.0":                               "OGL-UK-3.0",
				"ogl-uk-3.0.0":                             "OGL-UK-3.0",
				"ogtsl":                                    "OGTSL",
				"oldap-1":                                  "OLDAP-1.1",
				"oldap-1.1":                                "OLDAP-1.1",
				"oldap-1.1.0":                              "OLDAP-1.1",
				"oldap-1.2":                                "OLDAP-1.2",
				"oldap-1.2.0":                              "OLDAP-1.2",
				"oldap-1.3":                                "OLDAP-1.3",
				"oldap-1.3.0":                              "OLDAP-1.3",
				"oldap-1.4":                                "OLDAP-1.4",
				"oldap-1.4.0":                              "OLDAP-1.4",
				"oldap-2":                                  "OLDAP-2.0",
				"oldap-2.0":                                "OLDAP-2.0",
				"oldap-2.0.0":                              "OLDAP-2.0",
				"oldap-2.0.1":                              "OLDAP-2.0.1",
				"oldap-2.1":                                "OLDAP-2.1",
				"oldap-2.1.0":                              "OLDAP-2.1",
				"oldap-2.2":                                "OLDAP-2.2",
				"oldap-2.2.0":                              "OLDAP-2.2",
				"oldap-2.2.1":                              "OLDAP-2.2.1",
				"oldap-2.2.2":                              "OLDAP-2.2.2",
				"oldap-2.3":                                "OLDAP-2.3",
				"oldap-2.3.0":                              "OLDAP-2.3",
				"oldap-2.4":                                "OLDAP-2.4",
				"oldap-2.4.0":                              "OLDAP-2.4",
				"oldap-2.5":                                "OLDAP-2.5",
				"oldap-2.5.0":                              "OLDAP-2.5",
				"oldap-2.6":                                "OLDAP-2.6",
				"oldap-2.6.0":                              "OLDAP-2.6",
				"oldap-2.7":                                "OLDAP-2.7",
				"oldap-2.7.0":                              "OLDAP-2.7",
				"oldap-2.8":                                "OLDAP-2.8",
				"oldap-2.8.0":                              "OLDAP-2.8",
				"oml":                                      "OML",
				"openssl":                                  "OpenSSL",
				"opl-1":                                    "OPL-1.0",
				"opl-1.0":                                  "OPL-1.0",
				"opl-1.0.0":                                "OPL-1.0",
				"opubl-1":                                  "OPUBL-1.0",
				"opubl-1.0":                                "OPUBL-1.0",
				"opubl-1.0.0":                              "OPUBL-1.0",
				"oset-pl-2":                                "OSET-PL-2.1",
				"oset-pl-2.1":                              "OSET-PL-2.1",
				"oset-pl-2.1.0":                            "OSET-PL-2.1",
				"osl-1":                                    "OSL-1.0",
				"osl-1.0":                                  "OSL-1.0",
				"osl-1.0.0":                                "OSL-1.0",
				"osl-1.1":                                  "OSL-1.1",
				"osl-1.1.0":                                "OSL-1.1",
				"osl-2":                                    "OSL-2.0",
				"osl-2.0":                                  "OSL-2.0",
				"osl-2.0.0":                                "OSL-2.0",
				"osl-2.1":                                  "OSL-2.1",
				"osl-2.1.0":                                "OSL-2.1",
				"osl-3":                                    "OSL-3.0",
				"osl-3.0":                                  "OSL-3.0",
				"osl-3.0.0":                                "OSL-3.0",
				"parity-6":                                 "Parity-6.0.0",
				"parity-6.0":                               "Parity-6.0.0",
				"parity-6.0.0":                             "Parity-6.0.0",
				"parity-7":                                 "Parity-7.0.0",
				"parity-7.0":                               "Parity-7.0.0",
				"parity-7.0.0":                             "Parity-7.0.0",
				"pddl-1":                                   "PDDL-1.0",
				"pddl-1.0":                                 "PDDL-1.0",
				"pddl-1.0.0":                               "PDDL-1.0",
				"php-3":                                    "PHP-3.0",
				"php-3.0":                                  "PHP-3.0",
				"php-3.0.0":                                "PHP-3.0",
				"php-3.01":                                 "PHP-3.01",
				"php-3.01.0":                               "PHP-3.01",
				"plexus":                                   "Plexus",
				"polyform-noncommercial-1":                 "PolyForm-Noncommercial-1.0.0",
				"polyform-noncommercial-1.0":               "PolyForm-Noncommercial-1.0.0",
				"polyform-noncommercial-1.0.0":             "PolyForm-Noncommercial-1.0.0",
				"polyform-small-business-1":                "PolyForm-Small-Business-1.0.0",
				"polyform-small-business-1.0":              "PolyForm-Small-Business-1.0.0",
				"polyform-small-business-1.0.0":            "PolyForm-Small-Business-1.0.0",
				"postgresql":                               "PostgreSQL",
				"psf-2":                                    "PSF-2.0",
				"psf-2.0":                                  "PSF-2.0",
				"psf-2.0.0":                                "PSF-2.0",
				"psfrag":                                   "psfrag",
				"psutils":                                  "psutils",
				"python-2":                                 "Python-2.0",
				"python-2.0":                               "Python-2.0",
				"python-2.0.0":                             "Python-2.0",
				"python-2.0.1":                             "Python-2.0.1",
				"qhull":                                    "Qhull",
				"qpl-1":                                    "QPL-1.0",
				"qpl-1.0":                                  "QPL-1.0",
				"qpl-1.0.0":                                "QPL-1.0",
				"rdisc":                                    "Rdisc",
				"rhecos-1":                                 "RHeCos-1.1",
				"rhecos-1.1":                               "RHeCos-1.1",
				"rhecos-1.1.0":                             "RHeCos-1.1",
				"rpl-1":                                    "RPL-1.1",
				"rpl-1.1":                                  "RPL-1.1",
				"rpl-1.1.0":                                "RPL-1.1",
				"rpl-1.5":                                  "RPL-1.5",
				"rpl-1.5.0":                                "RPL-1.5",
				"rpsl-1":                                   "RPSL-1.0",
				"rpsl-1.0":                                 "RPSL-1.0",
				"rpsl-1.0.0":                               "RPSL-1.0",
				"rsa-md":                                   "RSA-MD",
				"rscpl":                                    "RSCPL",
				"ruby":                                     "Ruby",
				"sax-pd":                                   "SAX-PD",
				"saxpath":                                  "Saxpath",
				"scea":                                     "SCEA",
				"schemereport":                             "SchemeReport",
				"sendmail":                                 "Sendmail",
				"sendmail-8":                               "Sendmail-8.23",
				"sendmail-8.23":                            "Sendmail-8.23",
				"sendmail-8.23.0":                          "Sendmail-8.23",
				"sgi-b-1":                                  "SGI-B-1.0",
				"sgi-b-1.0":                                "SGI-B-1.0",
				"sgi-b-1.0.0":                              "SGI-B-1.0",
				"sgi-b-1.1":                                "SGI-B-1.1",
				"sgi-b-1.1.0":                              "SGI-B-1.1",
				"sgi-b-2":                                  "SGI-B-2.0",
				"sgi-b-2.0":                                "SGI-B-2.0",
				"sgi-b-2.0.0":                              "SGI-B-2.0",
				"shl-0.5":                                  "SHL-0.5",
				"shl-0.5.0":                                "SHL-0.5",
				"shl-0.51":                                 "SHL-0.51",
				"shl-0.51.0":                               "SHL-0.51",
				"simpl-2":                                  "SimPL-2.0",
				"simpl-2.0":                                "SimPL-2.0",
				"simpl-2.0.0":                              "SimPL-2.0",
				"sissl":                                    "SISSL",
				"sissl-1":                                  "SISSL-1.2",
				"sissl-1.2":                                "SISSL-1.2",
				"sissl-1.2.0":                              "SISSL-1.2",
				"sleepycat":                                "Sleepycat",
				"smlnj":                                    "SMLNJ",
				"smppl":                                    "SMPPL",
				"snia":                                     "SNIA",
				"spencer-86":                               "Spencer-86",
				"spencer-86.0":                             "Spencer-86",
				"spencer-86.0.0":                           "Spencer-86",
				"spencer-94":                               "Spencer-94",
				"spencer-94.0":                             "Spencer-94",
				"spencer-94.0.0":                           "Spencer-94",
				"spencer-99":                               "Spencer-99",
				"spencer-99.0":                             "Spencer-99",
				"spencer-99.0.0":                           "Spencer-99",
				"spl-1":                                    "SPL-1.0",
				"spl-1.0":                                  "SPL-1.0",
				"spl-1.0.0":                                "SPL-1.0",
				"ssh-openssh":                              "SSH-OpenSSH",
				"ssh-short":                                "SSH-short",
				"sspl-1":                                   "SSPL-1.0",
				"sspl-1.0":                                 "SSPL-1.0",
				"sspl-1.0.0":                               "SSPL-1.0",
				"standardml-nj":                            "SMLNJ",
				"sugarcrm-1":                               "SugarCRM-1.1.3",
				"sugarcrm-1.1":                             "SugarCRM-1.1.3",
				"sugarcrm-1.1.3":                           "SugarCRM-1.1.3",
				"swl":                                      "SWL",
				"tapr-ohl-1":                               "TAPR-OHL-1.0",
				"tapr-ohl-1.0":                             "TAPR-OHL-1.0",
				"tapr-ohl-1.0.0":                           "TAPR-OHL-1.0",
				"tcl":                                      "TCL",
				"tcp-wrappers":                             "TCP-wrappers",
				"tmate":                                    "TMate",
				"torque-1":                                 "TORQUE-1.1",
				"torque-1.1":                               "TORQUE-1.1",
				"torque-1.1.0":                             "TORQUE-1.1",
				"tosl":                                     "TOSL",
				"tu-berlin-1":                              "TU-Berlin-1.0",
				"tu-berlin-1.0":                            "TU-Berlin-1.0",
				"tu-berlin-1.0.0":                          "TU-Berlin-1.0",
				"tu-berlin-2":                              "TU-Berlin-2.0",
				"tu-berlin-2.0":                            "TU-Berlin-2.0",
				"tu-berlin-2.0.0":                          "TU-Berlin-2.0",
				"ucl-1":                                    "UCL-1.0",
				"ucl-1.0":                                  "UCL-1.0",
				"ucl-1.0.0":                                "UCL-1.0",
				"unicode-dfs-2015":                         "Unicode-DFS-2015",
				"unicode-dfs-2015.0":                       "Unicode-DFS-2015",
				"unicode-dfs-2015.0.0":                     "Unicode-DFS-2015",
				"unicode-dfs-2016":                         "Unicode-DFS-2016",
				"unicode-dfs-2016.0":                       "Unicode-DFS-2016",
				"unicode-dfs-2016.0.0":                     "Unicode-DFS-2016",
				"unicode-tou":                              "Unicode-TOU",
				"unlicense":                                "Unlicense",
				"upl-1":                                    "UPL-1.0",
				"upl-1.0":                                  "UPL-1.0",
				"upl-1.0.0":                                "UPL-1.0",
				"vim":                                      "Vim",
				"vostrom":                                  "VOSTROM",
				"vsl-1":                                    "VSL-1.0",
				"vsl-1.0":                                  "VSL-1.0",
				"vsl-1.0.0":                                "VSL-1.0",
				"w3c":                                      "W3C",
				"w3c-19980720":                             "W3C-19980720",
				"w3c-19980720.0":                           "W3C-19980720",
				"w3c-19980720.0.0":                         "W3C-19980720",
				"w3c-20150513":                             "W3C-20150513",
				"w3c-20150513.0":                           "W3C-20150513",
				"w3c-20150513.0.0":                         "W3C-20150513",
				"watcom-1":                                 "Watcom-1.0",
				"watcom-1.0":                               "Watcom-1.0",
				"watcom-1.0.0":                             "Watcom-1.0",
				"wsuipa":                                   "Wsuipa",
				"wtfpl":                                    "WTFPL",
				"wxwindows":                                "wxWindows",
				"x11":                                      "X11",
				"x11-distribute-modifications-variant":     "X11-distribute-modifications-variant",
				"xerox":                                    "Xerox",
				"xfree86-1":                                "XFree86-1.1",
				"xfree86-1.1":                              "XFree86-1.1",
				"xfree86-1.1.0":                            "XFree86-1.1",
				"xinetd":                                   "xinetd",
				"xnet":                                     "Xnet",
				"xpp":                                      "xpp",
				"xskat":                                    "XSkat",
				"ypl-1":                                    "YPL-1.0",
				"ypl-1.0":                                  "YPL-1.0",
				"ypl-1.0.0":                                "YPL-1.0",
				"ypl-1.1":                                  "YPL-1.1",
				"ypl-1.1.0":                                "YPL-1.1",
				"zed":                                      "Zed",
				"zend-2":                                   "Zend-2.0",
				"zend-2.0":                                 "Zend-2.0",
				"zend-2.0.0":                               "Zend-2.0",
				"zimbra-1":                                 "Zimbra-1.3",
				"zimbra-1.3":                               "Zimbra-1.3",
				"zimbra-1.3.0":                             "Zimbra-1.3",
				"zimbra-1.4":                               "Zimbra-1.4",
				"zimbra-1.4.0":                             "Zimbra-1.4",
				"zlib":                                     "Zlib",
				"zlib-acknowledgement":                     "zlib-acknowledgement",
				"zpl-1":                                    "ZPL-1.1",
				"zpl-1.1":                                  "ZPL-1.1",
				"zpl-1.1.0":                                "ZPL-1.1",
				"zpl-2":                                    "ZPL-2.0",
				"zpl-2.0":                                  "ZPL-2.0",
				"zpl-2.0.0":                                "ZPL-2.0",
				"zpl-2.1":                                  "ZPL-2.1",
				"zpl-2.1.0":                                "ZPL-2.1",
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.fixture, func(t *testing.T) {
			f, err := os.Open(tt.fixture)
			require.NoError(t, err)

			var list LicenseList
			require.NoError(t, json.NewDecoder(f).Decode(&list))

			d := cmp.Diff(tt.want, processSPDXLicense(list))
			if d != "" {
				t.Log("differing license list:\n" + d)
				t.Fail()
			}
		})
	}
}
