(* SwiftSurf
 * Sebastien Ailleret *)

open Activebuffer
open Conf
open Types
open Utils

(* lecture des arguments de la ligne de commande *)
let conf_file = ref ""
let daemon = ref false

let usage = "usage: " ^ Sys.argv.(0) ^ " [-d] [-c config_file]"
let speclist =
  [("-d", Arg.Unit (fun () -> daemon := true),
    ": daemon mode");
   ("-c", Arg.String (fun c -> conf_file:=c),
    ": name of the configuration file (default $HOME/.swiftsurf /etc/swiftsurf.conf ./swiftsurf.conf /dev/null")]

(* is it allowed to connect from this addr ? *)
let allowed addr =
  match addr with
  | Unix.ADDR_UNIX _ -> assert false
  | Unix.ADDR_INET (addr, _) ->
      let str_addr = Unix.string_of_inet_addr addr in
      match !connect_from with
      | None -> true, str_addr
      | Some reg -> Str.string_match reg str_addr 0, str_addr

(* default profile (just used for initialising) *)
let def_prof = {
  auth_pass = "";
  forbidden_urls = [];
  allowed_urls = [], "";
  methods = [];
  canCONNECT = true;
  allCONNECT = false;
  req_1 = false;
  req_2 = false;
  req_in = false;
  req_out = false;
  ans_1 = false;
  ans_2 = false;
  ans_in = false;
  ans_out = false }

(* find the new timeout *)
let rec before_select time conns =
  let new_timeout x y =
    if x = -1.
    then y
    else min x y in
  let rec aux fread fwrite timeout nb_conns = function
    | [] ->
        let ok_conns =
          !Types.max_conns == -1 || nb_conns < !Types.max_conns in
        let res_to =
          if timeout = -1.
          then -1.
          else max 0. (ceil (timeout -. time)) in
        fread, fwrite, res_to, nb_conns, ok_conns
    | conn::l ->
        let new_timeout = new_timeout timeout conn.timeout in
        match conn.state with
        | STARTING | DNS | DNSDONE _ ->
            if conn.size_req > 0
            then aux (conn.client::fread) fwrite new_timeout (nb_conns+1) l
            else aux fread fwrite new_timeout (nb_conns+1) l
        | CONNECTING ->
            let new_fwrite = conn.server::fwrite in
            if conn.size_req > 0
            then aux (conn.client::fread) new_fwrite new_timeout (nb_conns+1) l
            else aux fread new_fwrite new_timeout (nb_conns+1) l
        | ALIVE ->
            let ref_read = ref fread in
            let ref_write = ref fwrite in
            if conn.size_req > 0 then ref_read:= conn.client :: !ref_read;
            if conn.size_ans > 0 then ref_read:= conn.server :: !ref_read;
            if length conn.write_req <> 0 then
              ref_write:= conn.server :: !ref_write;
            if length conn.write_ans <> 0 then
              ref_write:= conn.client :: !ref_write;
            aux !ref_read !ref_write new_timeout (nb_conns+1) l
        | FINISHING ->
            aux fread (conn.client::fwrite) new_timeout (nb_conns+1) l in
  aux [] [] (-1.) 0 conns

(* Main loop of the program *)
let schedule fd_serv conns =
  let fread_tmp, fwrite, timeout, nb_conns, ok_conns =
    before_select (Unix.time ()) !conns in
  if !Types.debug > 1 then
    (Printf.printf "%d, %d, %d" nb_conns
       (List.length fread_tmp) (List.length fwrite);
     flush stdout);
  let fread = if ok_conns then fd_serv :: fread_tmp else fread_tmp in
  let active_read, active_write, _ =
    Unix.select ((Dns.give_fds ()) @ fread) fwrite [] timeout in
  if !Types.debug > 1 then
    (Printf.printf ", %d, %d\n"
       (List.length active_read)
       (List.length active_write);
     flush stdout);
  (* Are there new connexions *)
  if ok_conns && List.mem fd_serv active_read then
    (let fds, addr = Unix.accept fd_serv in
    let ok, str_addr = allowed addr in
    if ok then
      (Unix.set_nonblock fds;
       conns := { client = fds;
                  from = str_addr;
                  timeout = Unix.time () +. !Types.timeout;
                  server = Unix.stdin;
                  state = STARTING;
                  auth = "";
                  prof = def_prof;
                  host = ("", 80);
                  need_proxy = None;
                  url = "";
                  proto_str = "http://";
                  meth = "";
                  read_req = Activebuffer.create buf_size;
                  write_req = Activebuffer.create buf_size;
                  size_req = buf_size; state_ans = CMD_LINE;
                  read_ans = Activebuffer.create buf_size;
                  write_ans = Activebuffer.create buf_size;
                  size_ans = buf_size; state_req = CMD_LINE;
                  len_post = 0 }
         ::!conns)
    else
      (if !Types.stats >= 1 then
        (let t = Unix.localtime (Unix.time ()) in
        let s2_of_int i =
          (string_of_int (i / 10)) ^ (string_of_int (i mod 10)) in
        Printf.printf "%d-%d-%d, %d:%s:%s : connection refused from %s\n"
          t.Unix.tm_mday (t.Unix.tm_mon + 1) (1900 + t.Unix.tm_year)
          t.Unix.tm_hour (s2_of_int t.Unix.tm_min)
          (s2_of_int t.Unix.tm_sec) str_addr;
        flush stdout);
      try Unix.close fds with _ -> ()));
  (* Read and write what can be *)
  Dns.update active_read;
  let time = Unix.time () in
  conns := Request.gere_conns time active_read active_write !conns;
  conns := Answer.gere_conns time active_read !conns

let main () =
  (* the garbage collector must compact from time to time *)
  Gc.set { (Gc.get()) with Gc.max_overhead = 400 };
  (* Read the arguments *)
  Arg.parse
    speclist
    (fun  x -> raise (Arg.Bad ("Bad argument : " ^ x)))
    usage;
  (* read configuration file *)
  Conf.init !conf_file !daemon;
  (* ignore SIG_PIPE *)
  Sys.set_signal Sys.sigpipe Sys.Signal_ignore;
  (* Bind the socket *)
  let fd_serv = bind_server !Conf.port in
  let conns = ref [] in
  if (!Types.debug + !Types.stats) > 0 then
    (print_string "Swiftsurf is waiting for request\n"; flush stdout);
  while true do
    schedule fd_serv conns
  done;;

main ()
