(* SwiftSurf
 * Sebastien Ailleret *)

open Unix
open Pervasives

open Activebuffer
open Conf
open Types
open Utils

(**************************************************************)
(* understand *)
(**************************************************************)

(* Try to understand what we received *)
let rec compute_read conn =
  match conn.state_ans with
  | WAIT_DNS | HEADERSCONNECT | END ->
      assert false
  | CMD_LINE ->
      (try
        let pos = index conn.read_ans '\n' in
        let cmd = String.sub conn.read_ans.buffer 0 (pos+1) in
        conn.read_ans.pos_deb <- pos+1;
        Activebuffer.add_string conn.write_ans cmd;
        conn.state_ans <- HEADERS;
        if conn.prof.ans_1 then
          (print_string cmd; flush stdout);
        compute_read conn
      with Not_found ->
        (* the command line is not finished *)
        ())
  | HEADERS ->
      (try
        let pos = index conn.read_ans '\n' in
        let header = String.sub conn.read_ans.buffer
            conn.read_ans.pos_deb (pos+1-conn.read_ans.pos_deb) in
        conn.read_ans.pos_deb <- pos+1;
        if String.length header <= 2 then
          (* last line of headers *)
          (if conn.prof.ans_1 then
            (print_string header; flush stdout);
           conn.state_ans <- CONTENT;
           Activebuffer.add_string conn.write_ans header)
        else
          (if ok_ans_header header then
            (if conn.prof.ans_1 then
              (print_string header; flush stdout);
             Activebuffer.add_string conn.write_ans header));
        compute_read conn
      with Not_found ->
        (* this line of headers is not finished *)
        ())
  | CONTENT ->
      if conn.prof.ans_2 then
        (print_string (Activebuffer.contents conn.read_ans); flush stdout);
      Activebuffer.add_buffer conn.write_ans conn.read_ans;
      Activebuffer.clear conn.read_ans

(**************************************************************)
(* read and write *)
(**************************************************************)

(* buffer used for read *)
let buf = String.create buf_size

(* Update every connexions *)
let gere_conns time active_read conns =

  let rec gere_aux conns = function
    | [] -> conns
    | conn::l ->
        match conn.state with
        | STARTING
        | DNS
        | DNSDONE _
        | CONNECTING ->
            (* nothing yet to read *)
            gere_aux (conn::conns) l
        | ALIVE ->
            if List.mem conn.server active_read then
              (let cont = manage_read conn in
              if Activebuffer.length conn.read_ans <> 0 then compute_read conn;
              if not cont then finish conn "";
              manage_write conn conns l)
            else
              (* try to write what must be *)
              manage_write conn conns l
        | FINISHING ->
            (* we just have to finish writing data  to the client *)
            manage_write conn conns l

  (* write what can be and is ready *)
  and manage_write conn conns l =
    let fin = ref false in
    (try
      let len = length conn.write_ans in
      let str, pos = buffer conn.write_ans in
      let nb = Unix.write conn.client str pos len in
      if nb > 0 then
        (if conn.prof.ans_out then
           print_string (String.sub str pos nb);
         sub conn.write_ans nb (len - nb);
         conn.size_ans <- min (conn.size_ans + nb) buf_size;
         conn.timeout <- time +. !Types.timeout)
    with
    | Unix_error (EAGAIN, "write", "") ->
        ()
    | Unix_error (_, "write", "") ->
        fin := true);
    if !fin then
      (* Connexion closed by client *)
      (close_connexion conn;
       gere_aux conns l)
    else
      (* Unable to write any more now *)
      if conn.state = FINISHING && length conn.write_ans = 0 then
        (* That's finished... *)
        (close_connexion conn;
         gere_aux conns l)
      else
         gere_aux (conn::conns) l

  (* read all available data
   * return true if the connexion is still alive
   *        false if the connexion is closed *)
  and manage_read conn =
    try
      conn.size_ans == 0 || (* just read if allowed *)
      let str, pos = before_read conn.read_ans conn.size_ans in
      match Unix.read conn.server str pos conn.size_ans with
      | 0 -> (* connexion closed *)
          false
      | nb ->
          if conn.prof.ans_in then
            print_string (String.sub str pos nb);
          after_read conn.read_ans nb;
          conn.timeout <- time +. !Types.timeout;
          conn.size_ans <- conn.size_ans - nb;
          true
    with
    | Unix_error (EAGAIN, "read", "") -> (* Nothing more to read now *)
        true
    | Unix_error (_, "read", "") -> (* connexion closed *)
        false
  in

  gere_aux [] conns
