/*
 * pcapf.c
 *
 * Functions involved in interfacing with libpcap, mostly in opening
 * and closing pcap_t handles. Much of the actual yanking of packets
 * is back in packetp.c. packetp_pcap_init and packetp_pcap_end
 * are the only functions exposed/used by other code, the rest of
 * the functions here are support for those two.
 *
 * Copyright (c) 2002, 2003 Todd MacDermid <tmacd@synacklabs.net>
 *
 */

#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <dnet.h>

#include "packetp.h"

/* zero_low_net_bits will copy one struct addr into another, but make sure 
 * that any bits that are not included in the netmask are zeroed out. This
 * will make libpcap happier when it gets them as part of a filter.
 * (mini-rant: Why can't libpcap do this for itself?)
 */

int 
zero_low_net_bits(struct addr *dest, struct addr *src) 
{
  int i;
  uint8_t bytemask;
  
  if((dest == NULL) || (src == NULL)) {
    fprintf(stderr, "zero_low_net_bits: NULL struct addr pointer passed\n");
    return(-1);
  }
  dest->addr_type = src->addr_type;
  dest->addr_bits = src->addr_bits;
  for(i = 0; ((i+1)*8) <= src->addr_bits; i ++) 
    dest->addr_data8[i] = src->addr_data8[i];
  
  if((src->addr_bits % 8) != 0) {
    bytemask = (~0 << (8 - (src->addr_bits % 8)));
    dest->addr_data8[i] = (src->addr_data8[i] & bytemask);
    i++;
  }
  for(; i < 16; i++) {
    dest->addr_data8[i] = 0;
  }
  return(0);
}

/* get_pcap_handle actually does the grunt work of opening pcap_t handles. 
 * It gets called from packetp_pcap_init().
 * Returns NULL on failure, or a pointer to the pcap_t on success
 */

pcap_t *
get_pcap_handle(const char *interface, char *filter) 
{
  pcap_t *pcap_handle;
  bpf_u_int32 my_addr, my_netmask;
  struct bpf_program filter_code;
  char errbuf[PCAP_ERRBUF_SIZE];
  char localintf[INTCHAR];
  int ret;

  strncpy(localintf, interface, INTCHAR-1); /* Because interface is const, 
					     * and intf_loop needs it to
					     * be const, and the pcap
					     * opening functions aren't
					     * const, even though they
					     * don't change it. Gets rid
					     * of a warning. 
					     */

  if((pcap_lookupnet(localintf, &my_addr, &my_netmask, errbuf)) < 0) {   
    fprintf(stderr,"pcap_lookupnet failed: %s\n", errbuf);
    return(NULL);
  }

  if((pcap_handle = pcap_open_live(localintf, PACKETSIZE, 1, PACKETWAIT,
                               errbuf)) == NULL) {
    fprintf(stderr,"get_pcap_handle: pcap_open_live failed:%s\n", errbuf);
    return(NULL);
  }

  if ((pcap_compile(pcap_handle, &filter_code, filter, 1, my_netmask)) < 0) {
    fprintf(stderr, "get_pcap_handle: pcap_compile of \"%s\" failed\n",
	    filter);
    pcap_close(pcap_handle);
    return(NULL);
  }

  ret = pcap_setfilter(pcap_handle, &filter_code);
  pcap_freecode(&filter_code);

  if(ret < 0) {
    fprintf(stderr, "pcap_setfilter failed: %s\n", pcap_geterr(pcap_handle));
    pcap_close(pcap_handle);
    return(NULL);
  }
  return(pcap_handle);
}

/* add_pcap_node allocates and adds the given pcap handle to the linked list
 * of all pcap handles within the program. 0 on success, -1 on failure
 */

int 
add_pcap_node(struct packetp_ctx *ctx, pcap_t *pcap_handle, 
		  const struct intf_entry *entry) 
{
  struct pcap_node *new_node;

  if((new_node = 
      (struct pcap_node *)calloc(1, sizeof(struct pcap_node))) == NULL) 
    {
      fprintf(stderr, "add_pcap_node: Error allocating memory\n");
      return(-1);
    }
  
  new_node->pcap_handle = pcap_handle;
  new_node->pcap_fd = pcap_fileno(pcap_handle);
  memcpy(&(new_node->interface), entry, sizeof(struct intf_entry));
  new_node->next = ctx->pcap_head;
  ctx->pcap_head = new_node;

  return(0);
}

/* loop_pcap is the intf_loop handler used to set up pcap handles on
 * all interfaces when IP_ADDR_ANY is used as the target IP on a TUN_LOOPFW
 * style tunnel. 0 on success, -1 on failure
 */

int 
loop_pcap(const struct intf_entry *entry, void *ctx_arg) 
{
  struct packetp_ctx *ctx;
  pcap_t *pcap_handle;
  char filter[MAX_FILTER_SIZE];
  struct addr my_addr;

  ctx = ctx_arg;

  if(entry->intf_addr.addr_type == ADDR_TYPE_IP) {
    if(entry->intf_type == INTF_TYPE_LOOPBACK) {
      snprintf(filter, MAX_FILTER_SIZE-1, "");
    } else if (entry->intf_type == INTF_TYPE_ETH) {
      memcpy(&my_addr, &(entry->intf_addr), sizeof(struct addr));
      my_addr.addr_bits = IP_ADDR_BITS;
      snprintf(filter, MAX_FILTER_SIZE-1, 
  	       "not src host %s and ( arp or dst host %s )", 
	       addr_ntoa(&(my_addr)),  
	       addr_ntoa(&(my_addr)));
    } else {
      return(0);
    }
    if((pcap_handle = get_pcap_handle(entry->intf_name, filter)) == NULL) {
      fprintf(stderr, "loop_pcap: get_pcap_handle returned NULL\n");
      return(-1);
    }
    if(add_pcap_node(ctx, pcap_handle, entry) < 0) {
      fprintf(stderr, "loop_pcap: add_pcap_node failed\n");
      pcap_close(pcap_handle);
      return(-1);
    }
  } /* Skip non-IPv4 interfaces, or unaddressed interfaces */ 
  return(0);
}


/* packetp_pcap_end closes all pcap handles and frees memory associated 
 * with the linked list of pcap_nodes in the given context. 
 */

void 
packetp_pcap_end(struct packetp_ctx *ctx) 
{
  struct pcap_node *walk_node, *free_node;
  
  walk_node = ctx->pcap_head;
  while(walk_node != NULL) {
    free_node = walk_node;
    pcap_close(walk_node->pcap_handle);
    walk_node = walk_node->next;
    free(free_node);
  }
}

/* packetp_pcap_init opens up all the pcap_t pointers required by the settings
 * given. Does the best it can with what it has. Sets the
 * filter strings, then calls get_pcap_handle to actually do most of the
 * interfacing with libpcap. packetp_route_save must have been called prior
 * to calling this function, as it uses interface information set in
 * that function. Returns 0 on success, -1 on failure
 */

int 
packetp_pcap_init(struct packetp_ctx *ctx) 
{
  char filter[MAX_FILTER_SIZE];
  struct addr net_addr, loopback_addr, my_addr;
  struct route_node *route_info;
  struct intf_entry loopback_intf;
  pcap_t *pcap_handle;

  if(ctx->wedge_type == PP_FAKEIP) {

    if((route_info = packetp_route_find(ctx, &(ctx->proxy_ip))) == NULL) {
      fprintf(stderr, 
	      "packetp_pcap_init: Could not find interface for proxy %s\n", 
	      addr_ntoa(&(ctx->proxy_ip)));
      return(-1);
    }

    snprintf(filter, MAX_FILTER_SIZE-1, 
	     "dst host %s", addr_ntoa(&(ctx->proxy_ip)));

    memcpy(&(ctx->my_ip), &(route_info->interface.intf_addr), 
	   sizeof(struct addr));
    ctx->my_ip.addr_bits = IP_ADDR_BITS;

    if((pcap_handle = 
	get_pcap_handle(route_info->interface.intf_name, filter)) == NULL) {
      fprintf(stderr, 
	      "packetp_pcap_init: Failed to open pcap with filter \"%s\" on interface \"%s\"\n", 
	      filter, 
	      route_info->interface.intf_name);
      return(-1);
    }

    if(add_pcap_node(ctx, pcap_handle, &(route_info->interface)) < 0) {
      fprintf(stderr, "packetp_pcap_init: add_pcap_node failed\n");
      pcap_close(pcap_handle);
      return(-1);
    }

  } else if (ctx->wedge_type == PP_LOOPFW) {
    if(ctx->target_ip.addr_ip == IP_ADDR_ANY) {
      if((intf_loop(ctx->intf_handle, loop_pcap, ctx)) < 0) {
	fprintf(stderr, "packetp_pcap_init: Error in intf_loop\n");
	return(-1);
      }

    } else { /* If we've only been given a single CIDR block of target_ip */

      /* First, open a pcap pointer on the interface corresponding to the
       * target IP
       */

      if((route_info = packetp_route_find(ctx, &(ctx->target_ip))) == NULL) {
	fprintf(stderr, 
		"packetp_pcap_init: Could not find interface for target %s\n", 
		addr_ntoa(&(ctx->target_ip)));
	return(-1);
      } 

      memcpy(&my_addr, &(route_info->interface.intf_addr), 
	     sizeof(struct addr));
      my_addr.addr_bits = IP_ADDR_BITS;

      if(ctx->target_ip.addr_bits == IP_ADDR_BITS) {
	snprintf(filter, MAX_FILTER_SIZE-1, 
		 "not src host %s and ( arp or (src host %s and dst host %s ))", 
		 addr_ntoa(&(my_addr)),
		 addr_ntoa(&(net_addr)),
		 addr_ntoa(&(my_addr)));
      } else {
	zero_low_net_bits(&net_addr, &(ctx->target_ip));
	snprintf(filter, MAX_FILTER_SIZE-1, 
		 "not src host %s and ( arp or src net %s and dst host %s ))", 
		 addr_ntoa(&(my_addr)),
		 addr_ntoa(&(net_addr)),
		 addr_ntoa(&(my_addr)));
      }

      if((pcap_handle = 
	  get_pcap_handle(route_info->interface.intf_name, filter)) == NULL) 
	{
	  fprintf(stderr, "packetp_pcap_init: Failed to open pcap with filter \"%s\" on interface \"%s\"\n", filter, route_info->interface.intf_name);
	  return(-1);
	}
      
      if(add_pcap_node(ctx, pcap_handle, &(route_info->interface)) < 0) {
	fprintf(stderr, "packetp_pcap_init: add_pcap_node failed\n");
	pcap_close(pcap_handle);
	return(-1);
      }
      
      /* Next, open a pcap pointer to the loopback interface */

      loopback_intf.intf_len = sizeof(struct intf_entry);

      addr_aton("127.0.0.1", &loopback_addr);
      if((intf_get_src(ctx->intf_handle, 
		       &loopback_intf, &loopback_addr)) < 0) 
	{
	  fprintf(stderr, 
		  "packetp_pcap_init: Error getting loopback interface\n");
	  return(-1);
	}

      if(ctx->target_ip.addr_bits == IP_ADDR_BITS) {
	snprintf(filter, MAX_FILTER_SIZE-1, 
		 "dst host %s", addr_ntoa(&(net_addr)));
      } else {
	snprintf(filter, MAX_FILTER_SIZE-1, 
		 "dst net %s", addr_ntoa(&(net_addr)));
      }
      if((pcap_handle = 
	  get_pcap_handle(loopback_intf.intf_name, filter)) == NULL) 
	{
	  fprintf(stderr, "packetp_pcap_init: Failed to open pcap with filter \"%s\" on interface \"%s\"\n", filter, loopback_intf.intf_name);
	  return(-1);
	}

      if(add_pcap_node(ctx, pcap_handle, &(loopback_intf)) < 0) {
	fprintf(stderr, "packetp_pcap_init: add_pcap_node failed\n");
	pcap_close(pcap_handle);
	return(-1);
      }
    } /* End of non-IP_ADDR_ANY */
  }   /* End of TUN_LOOPFW */
  return(0);
}
