/* This file is part of statist
**
** It is distributed under the GNU General Public License.
** See the file COPYING for details.
** 
** (c) 1997 Dirk Melcher
** old email address: Dirk.Melcher@usf.Uni-Osnabrueck.DE
**
** adapted for statistX: Andreas Beyer, 1999, abeyer@usf.uni-osnabrueck.de
**
**  published by Bernhard Reiter  http://www.usf.Uni-Osnabrueck.DE/~breiter
**  $Id: data.c,v 1.8 2001/11/15 20:24:59 breiter Exp breiter $
***************************************************************/

/* data.c fuer STATIST */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <stdlib.h>

#include "statist.h"
#include "data.h"
#include "funcs.h"

#include "gettext.h"

/* ==================================================================== */

typedef struct {
  PREAL ptr;
  char  *str;
} LABEL_INDEX;


static LABEL_INDEX label_tab[MCOL];
static short int labelcol[MCOL];
static short int n_lab;


/* ==================================================================== */




char *makefilename(int i, char *string) {
#ifndef MSDOS
  typedef struct {
    char name[L_tmpnam];
    BOOLEAN flag;
  } FNAM;

  static FNAM fnam[MCOL];
  static BOOLEAN very_first=TRUE;
  int j;
    
  if (very_first) {
    /*    printf("makefilename called very_first!\n"); */
    very_first = FALSE;
    for (j=0; j<MCOL; j++) {
      fnam[j].flag = FALSE;
    }
  }
  
  if (fnam[i].flag==FALSE) {
    tmpnam(fnam[i].name);
    fnam[i].flag = TRUE;
  }
  strcpy(string, fnam[i].name);
#else
  sprintf(string, MSDOS_FIXED_TMP_FILE , i);
#endif
  /*  printf("tmpnam %i = %s\n", i, string); */ /* for debugging */
  return string;
}

/* ==================================================================== */


void erasetempfiles() {
   int i;
   char filename[255];
   out_d(_("Removing temporary files ...",
   		"Loesche temporaere Dateien ...") );
   for (i=0; i<ncol; i++) {
     if (x_read[i]) {
       /*       printf("i=%i , myfree xx\n", i); */
      myfree(xx[i]);
     }
     /*     printf("i=%i, myfree alias\n", i); */
     myfree(alias[i]);
     /*     printf("rufe makefilename\n"); */
     makefilename(i, filename);
     /*     printf("filename = |%s|, cal remove\n", filename); */
     remove(filename);
     /*     printf("remove ausgefuehrt\n"); */
   }
   out_d(_(" done\n", " fertig\n") );
   /* printf("Temporaere Dateien geloescht!\n"); */
 }

/* =================================================================== */


int parsecomment(char *comment) {
  int n;
  char *s, *token;
  const char ignore[] = " ,;\n\t";
  const char var_id[] = "#%";/* this char indicates, the line contains labels */

  if (strstr(comment, var_id)!=comment) { /* no valid Var - comment */
    if(strstr(comment, "#!")==comment && strcmp(var_id,"#!")!=0 )
        out_err(WAR, ERR_FILE, 0, 
		_("'#!' is an illegal indicator of a column of labels.\n",
		"'#!' ist ungueltig als Indikator fuer eine Zeile mit Spaltennamen.\n") );
    return -1;
  }

  s = comment+strlen(var_id);  /* jump over var_id */
  n = 0;  
  while ((token = strtok(s, ignore))!= NULL) {
    s = NULL;
    if (token[0]=='$') {
      labelcol[n_lab] = n;
      n_lab++;
      out_d(_("Label in column %i='%s'\n",
		  "Label in Spalte %i='%s'\n") , n, token);
    }
    else {
      myfree(alias[n]);
      alias[n] = (char*) mycalloc(strlen(token)+1, sizeof(char));
      strcpy(alias[n], token);
      out_d(_("Column %i = %s\n", "Spalte %i = %s\n"), 
		n+1, alias[n]);
      n++;
    }
  }
 
  if (n==0) {
    out_err(FAT, ERR_FILE, ERR_LINE, 
    	_("No variables found in comment!", 
	"Keine Variablen in Kommentarzeile gefunden!") );
  }

  return n;
}

/* ==================================================================== */



void readsourcefile(char *sourcename){
   FILE *ttempfile[MCOL], *source;
   char  aline[512], oldline[512], filename[255];
   int   j, newcol=0, actcol, colread, lread=0, n_alias=0, i_lab=0;
   REAL  test;
   char ignore[]= " \n\t\0", *ptr, *token;

   n_lab = 0;
   FOPEN(sourcename, "rt", source);

					/* create column labels		*/
   if (ncol == 0) {
     for (j=0; j<MCOL; j++) {
       alias[j] = (char*)mycalloc(2, sizeof(char));
       sprintf(alias[j], "%c", (char)(j+CHAR_OFFSET));
       label_tab[j].ptr = NULL;
     }
   }
					/* get first non empty line	*/
   FGETS(aline, 512, source);
   lread ++;
   /*   while ((aline[0] == COMMENT) || (aline[0]=='\n')) { */
   while ((aline[0] == COMMENT) || (emptyline(aline))) {
     if ((j = parsecomment(aline))> 0) {
       n_alias = j;
     }
     FGETS(aline, 511, source);
     lread ++;
   }
   actcol = ncol + newcol;

      
   strncpy(oldline, aline, 511);
   ptr = aline;
   while ((token = strtok(ptr, ignore))!= NULL) {
     ptr = NULL;
     while ((i_lab<n_lab) && (newcol==labelcol[i_lab])) {
       i_lab++;
       token = strtok(ptr, ignore);
     }
     if (token==NULL) {
       break;
     }
     if ((token[0]== NODATA) && (strlen(token)==1))  {
       FOPEN(makefilename(ncol, filename), "wb", ttempfile[ncol]);
     }
     else if ((sscanf(token, "%lf", &test)==1)) {
       FOPEN(makefilename(ncol, filename), "wb", ttempfile[ncol]);
     }
     else {
       out_err(FAT, ERR_FILE, ERR_LINE, 
         _("Illegal format of value '%s' in column %i!\n"
	     "Couldn't read file %s!\n",
	   "Ungueltiges Eingabeformat des Wertes \"%s\" in Zeile %i!\n"
         "Datei %s konnte nicht eingelesen werden!\n"), 
	token, lread, sourcename);
       return;
     }
     newcol ++ ;
     ncol ++;
     if (ncol > MCOL) {
       out_err(FAT, ERR_FILE, ERR_LINE, 
	 	_("More than %i columns not allowed!\n",
		"Mehr als %i Spalten erreicht!\n") ,MCOL);
     }
   }
   
   if ((n_alias !=0) && (n_alias != ncol)) {
     out_err(FAT, ERR_FILE, ERR_LINE,
	   _("Number of columns does not equal number of labels!\n",
	     "Anzahl der Spalten ungleich Anzahl der Variablennamen!\n") );
     out_err(FAT, ERR_FILE, ERR_LINE,
	     "n_alias=%i  ncol=%i newcol=%i\n", n_alias, ncol, newcol);
   }
   out_d(_("Reading %i columns ...\n",
   		"lese %i Spalten ein ...\n"), newcol);

   
   strncpy(aline, oldline, 511);
   do {
     if (delrow) {	/* delete uncomplete rows, if this option is set */
       if (strchr(aline, (int)NODATA) != NULL) {
        lread ++;
         continue;
       }
     }
     if ((!emptyline(aline)) && (aline[0]!=COMMENT)) {
       colread = 0;
       j = 0;
       i_lab = 0;
       ptr = aline;

       while ((token = strtok(ptr, ignore))!= NULL) {
	 ptr = NULL;
	 actcol = j + (ncol-newcol);
	 while ((i_lab<n_lab) && (j==labelcol[i_lab])) {
	   i_lab++;
	   token = strtok(ptr, ignore);
	 }
	 if (token==NULL) {
	   break;
	 }
	 if (j>=newcol) {
	   out_err(FAT, ERR_FILE, ERR_LINE, 
	   	_("Too many columns in row %i\n",
		"Zu viele Spalten in Zeile %i\n"), lread);
	 }
	 
	 if ((token[0] == NODATA) && (strlen(token)==1)) {
	   colread ++;
	 }
	 else if (sscanf(token, "%lf", &test)==1) {
	   FWRITE(&test, sizeof(REAL), 1, ttempfile[actcol]);
	   nn[actcol] ++;
	   colread ++;
	 }
	 else {
	   out_err(FAT, ERR_FILE, ERR_LINE,
		 _("Illegal format of value '%s' in column %i!\n",
		   "Ungueltiges Eingabeformat des Wertes \"%s\" in Zeile %i!"),
		   token, lread);
	 }
	 j++;
       } 
     }
     lread ++;
     
     if (colread != newcol) {
       out_err(FAT, ERR_FILE, ERR_LINE, 
       	_("Row %i contains just %i instead of %i columns!",
		"Zeile %i enthaelt nur %i statt %i Spalten!"),
	(lread-1), colread, newcol);
     }
   }
   while (fgets(aline, 512, source) != NULL);
   
   for (j=0; j<newcol; j++) {
      actcol = j + (ncol-newcol);
      FCLOSE(ttempfile[actcol]);
   }
   FCLOSE(source);
   out_d(_("\nRead data sets: \n",
   		"\nGelesene Datensaetze: \n") );
   for (j=0; j<newcol; j++) {
     actcol = j + (ncol-newcol);
     out_d(_("Column %s: %i\n", "Spalte %s: %i\n"), 
     			alias[actcol], nn[actcol]);
   }
}

/* ==================================================================== */


void newsourcefile() {
   char answer[3], sourcename[80];
   FILE *source;
   int i;

   out_d(_("Name of the data file: ", "Name der Datei: ") );
   GETRLINE;
   sscanf(line, "%s", sourcename);
   out_d("\n\n");

   while ((source = fopen(sourcename,"rt")) == NULL)
   {
      out_d(_("File %s not found!\n",
		  "Datei %s nicht gefunden! \n") ,sourcename);
      out_d(_("Please enter new file name: ",
		  "Bitte neuen Dateinamen eingeben: ") );
      GETRLINE;
      sscanf(line, "%s", sourcename);
      out_d("\n");
   }
   FCLOSE(source);
   out_d(_("Shall the old data be removed? (y/n) ",
   	"Sollen die alten Daten geloescht werden? (j/n) ") );
   GETRLINE;
   sscanf(line, "%s", answer);
   if (answer[0]==_('y', 'j') ) {
      erasetempfiles();
      for (i=0; i<ncol; i++) {
	x_read[i] = FALSE;
	nn[i] = 0;
      }
      ncol = 0;
   }
   readsourcefile(sourcename);
   if (log_set) {
     fprintf(logfile, "-----------------------------------------------------\n");
     fprintf(logfile, 
     	_("\nNew source file: %s\n\n",
	  "\nNeue Quelldatei: %s\n\n") , sourcename);
   }
 }


/* =================================================================== */

void printcol(REAL x[], int n) {
   int i, k;
   out_r(_("Data from column %s:\n",
   		"Daten aus Spalte %s:\n"), get_label(x));
   for (i=0; i<n; i++) {
     k=i+1;
     out_r("%5i.)  %g\n", k, x[i]);
     if ((i+1) % SCREENSIZE == 0) {
	mywait();
	if (!empty) {
	  return;
	}
     }
   }
   out_r("-------------------------------------------\n\n");
}

/* =================================================================== */


PREAL readcol(int i) {
  PREAL px;
  char  filename[255];
  FILE  *tempfile;

  if (nn[i] == 0) {
    out_err(FAT, ERR_FILE, ERR_LINE, 
    	_("Column %i does not exist!",
	"Spalte %i existiert nicht!"), i+1);
  }
  FOPEN(makefilename(i, filename),"rb", tempfile);
  px = (REAL*)mycalloc(nn[i], sizeof(REAL));
  FREAD(px, sizeof(REAL), nn[i], tempfile);
  FCLOSE(tempfile);
  x_read[i] = TRUE;
  return px;
}

/* ==================================================================== */


void getcols(int i) {
  char salias[80];
  int k, j, this_col;
  BOOLEAN inputok, found;

  out_d(_("Columns: ", "Spalten: ") );
  for (k=0; k<ncol; k++) {
    out_d("%s ", alias[k]);
  }
  out_d("\n\n");


  for (k=0; k<i; k++) {
    inputok = FALSE;
    found  = FALSE;
    while (!inputok) {
      out_d(_("Column for variable %i: ",
		"Spalte fuer Variable %i: "), (k+1));
      GETRLINE;
      sscanf(line, "%s", salias);
				/* check if column name is matched exactly */
      for (j=0; j<ncol; j++) {
       if (strcmp(alias[j], salias)==0) {
         this_col = j;
         inputok = TRUE;
         break;
       }
      }
      if (inputok) {		
       break;		/* exact column alias entered -> go on 	*/
      }

				/* try to complete entered column alias	*/
      for (j=0; j<ncol; j++) {
	if (str_in_str(alias[j], salias)) {
	  if (found) {
	    out_err(ERR, ERR_FILE, ERR_LINE,
		  _("Column name '%s' is not unique!\n",
		    "Spaltenname `%s' ist mehrdeutig!\n"), salias);
	    inputok = FALSE;
	    break;
	  }
	  else {
	    found = TRUE;
	    inputok = TRUE;
	    this_col = j;
	  }
	}
      }
      if ((!inputok) && (!found)) {
	out_err(ERR, ERR_FILE, ERR_LINE, 
		_("Column %s does not exist!",
		"Spalte %s existiert nicht!"), salias);
      }
      else if ((!inputok) && (found)) {
	found = FALSE;
      }
    }
    acol[k] = this_col;
   /* out_r("%i alias %s\n", k, alias[acol[k]]);  */
  }

  out_d("\n");
  if (log_set) {
    fprintf(logfile, "-----------------------------------------------------------\n\n");
  }

  alloc_cols(i);  
}



void alloc_cols(int n_alloc) {
  int k, j;
  BOOLEAN inlist;
  
  for (k=0; k<MCOL; k++) {
    inlist = FALSE;
    for (j=0; j<n_alloc; j++) {
      if (k == acol[j]) {
	inlist = TRUE;
      }
    }
    if ((x_read[k]) && (!inlist))  {
      myfree(xx[k]);
      x_read[k] = FALSE;
      label_tab[k].ptr = NULL; 
    }
  }

  for (k=0; k<n_alloc; k++) {
    if (!x_read[acol[k]]) {
      xx[acol[k]] = readcol(acol[k]);
      label_tab[acol[k]].ptr = xx[acol[k]];
      label_tab[acol[k]].str = alias[acol[k]];
    }
    if (log_set) {
      fprintf(logfile, 
		_("Variable %i = Column %s\n",
		  "Variable %i = Spalte %s\n"), (k+1), alias[acol[k]] );
    }
  }

}


FILE *make_new_col(char *analias, int n) {
  FILE *fp;
  int i;
  char filename[255];

  if (ncol>(MCOL-1)) {
    out_err(ERR, ERR_FILE, ERR_LINE,
	  _("Sorry, maximum number of columns reached!\n",
	    "Sorry, maximale Anzahl von Spalten erreicht!\n") );
    return NULL;
  }
  
  for (i=0; i<ncol; i++) {
    if (strcmp(analias, alias[i])==0) {
      out_err(ERR, ERR_FILE, ERR_LINE, 
		_("Column %s exists already!\n",
		  "Spalte %s existiert bereits!\n"), 
      analias);
      return NULL;
    }
  }
  FOPEN(makefilename(ncol, filename), "wb", fp);
  myfree(alias[ncol]);
  alias[ncol] = (char*)mymalloc((strlen(analias)+1));  
  strcpy(alias[ncol], analias);
  out_r(_("New column %s created!\n",
	    "Neue Spalte %s erzeugt!\n"), alias[ncol]);
  nn[ncol] = n;
  ncol ++;
  return fp;
}
    
 /* =================================================================== */
 
int col_exist(char *analias) {
       int i;

       for ( i=0; i<MCOL; i++ ) {
               if ( alias[i] && strcmp(analias, alias[i])==0)  {
                       return i;
               }
       }
       return -1;
}

/* =================================================================== */

/* =================================================================== */

#ifndef STATIST_X
char *get_label(PREAL x) {
  int i;
  
  for (i=0; i<ncol; i++) {
    if (x == label_tab[i].ptr) {
      return label_tab[i].str;
    }
  }   
  out_err(ERR, ERR_FILE, ERR_LINE, 
  	_("No label found for column!\n",
	"Kein Label fuer Spalte gefunden!\n") );
  return NULL;
}    
#endif


void log_transform() {
  FILE *logfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column for log-transformation\n",
  	"Bitte Spalte waehlen, die logarithmiert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
    if (xx[acol[0]][i] > 0.0) {
      y[i] = log10(xx[acol[0]][i]);
    }
    else {
      out_err(MAT, ERR_FILE, ERR_LINE, 
		_("Column contains values <= 0!",
		"Die Spalte enthaelt Werte <= 0!") );
      return;
    }
  }

  strncpy(analias, "log_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((logfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], logfile); 
  FCLOSE(logfile);
}


void ln_transform() {
  FILE *logfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column for log-transformation\n",
  	"Bitte Spalte waehlen, die logarithmiert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
    if (xx[acol[0]][i] > 0.0) {
      y[i] = log(xx[acol[0]][i]);
    }
    else {
      out_err(MAT, ERR_FILE, ERR_LINE, 
		_("Column contains values <= 0!",
		"Die Spalte enthaelt Werte <= 0!") );
      return;
    }
  }

  strncpy(analias, "ln_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((logfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], logfile); 
  FCLOSE(logfile);
}
/* =================================================================== */


void power_10_transform() {
  FILE *logfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column for exponentiation\n",
  	   "Bitte Spalte waehlen, die zur Basis 10 exponenziert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
      y[i] = pow(10.0, xx[acol[0]][i]);
  }

  strncpy(analias, "10^_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((logfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], logfile);
  FCLOSE(logfile);
}

void power_e_transform() {
  FILE *logfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column for exponentiation\n",
  		"Bitte Spalte waehlen, die zur Basis 10 exponenziert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
      y[i] = exp(xx[acol[0]][i]);
  }

  strncpy(analias, "e^_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((logfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], logfile);
  FCLOSE(logfile);
}


/* =================================================================== */


void inv_transform() {
  FILE *invfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column for inversion\n",
  		"Bitte Spalte waehlen, die invertiert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
    y[i] = 1./xx[acol[0]][i];
  }

  strncpy(analias, "inv_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((invfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], invfile);
  FCLOSE(invfile);
}


/* =================================================================== */


void z_transform() {
  FILE *zfile;
  char analias[80];
  PREAL y;
  REAL mean, sdv;
  int i;

  out_d(_("Please select column for z-transformation\n",
  		"Bitte Spalte waehlen, die z-tranformiert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  sdv = get_sdv(xx[acol[0]], nn[acol[0]]);
  mean = get_mean(xx[acol[0]], nn[acol[0]]);
  for (i=0; i<nn[acol[0]]; i++) {
    y[i] = (xx[acol[0]][i]-mean)/sdv;
  }

  strncpy(analias, "z_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((zfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], zfile);
  FCLOSE(zfile);
}


/* =================================================================== */

void sort_col() {
  FILE *sortfile;
  char analias[80];
  PREAL y;
  int i;

  out_d(_("Please select column to be sorted\n",
  	"Bitte Spalte waehlen, die sortiert werden soll\n") );
  getcols(1);
  if (empty) return;
  y = (REAL*)m_calloc(nn[acol[0]], sizeof(REAL));
  for (i=0; i<nn[acol[0]]; i++) {
    y[i] = xx[acol[0]][i];
  }
  qsort(y, nn[acol[0]], sizeof(REAL), real_compar_up);  
  strncpy(analias, "sort_", 79);
  strncat(analias, alias[acol[0]], 79-strlen(analias));  
  if ((sortfile = make_new_col(analias, nn[acol[0]]))==NULL) {
    return;
  }
  FWRITE(y, sizeof(REAL), nn[acol[0]], sortfile);
  FCLOSE(sortfile);
}



/* =================================================================== */


void readcol_from_term() {
  FILE *tempfile;
  char filename[255], aline[80], answer[10];
  int i, n=0;
  REAL temp;
  BOOLEAN ok, stop=FALSE;

  if (ncol > 0) {
    out_d(_("Shall all data be deleted? (y/n) ",
    		"Sollen die alten Daten geloescht werden? (j/n) ") );
    GETRLINE;
    sscanf(line, "%s", answer);
    if (answer[0]==_('y', 'j') ) {
      erasetempfiles();
      for (i=0; i<ncol; i++) {
	x_read[i] = FALSE;
	nn[i] = 0;
      }
      ncol = 0;
      for (i=0; i<MCOL; i++) {
	alias[i] = (char*)mycalloc(2, sizeof(char));
	sprintf(alias[i], "%c", (char)(i+CHAR_OFFSET));
      }
    }
  }

  out_d(_("Column %i is being read, stop input with '.'\n",
  	"Spalte %i wird eingelesen, Eingabe mit '.' beenden\n") ,
	(ncol+1));
  aline[0] = '1';
  FOPEN(makefilename(ncol, filename), "wb", tempfile);

  while (!stop) {
    ok = FALSE;
    while (!ok) {
      out_d(_("Value %i: ", "Wert %i: "), (n+1));
      fgets(aline, 79, stdin);
      /* printf("aline: |%s|\n", aline); */
      if ( (aline[0]=='.') && (strlen(aline)==2) ) {
	stop = TRUE;
      }
      if ( (sscanf(aline, "%lf", &temp)==1) || (stop) ) {
	ok = TRUE;
      }
      else {
	out_err(ERR, ERR_FILE, ERR_LINE, 
	  _("Illegal input, please repeat: ",
	    "Ungueltige Eingabe, bitte wiederholen: ") );
      }

      if ( (ok) && (!stop) ) {
	n++;
	FWRITE(&temp, sizeof(REAL), 1, tempfile);
      }
    }
  }
  FCLOSE(tempfile);
  if (n>0) {
    nn[ncol] = n;
    ncol++;
  }
}



BOOLEAN str_in_str(const char *s1, const char *s2) {
  int i, n = strlen(s2);
  
  for (i=0; i<n; i++) {
    if (s1[i] != s2[i]) {
      return FALSE;
    }
  }
  return TRUE;
}


BOOLEAN emptyline(const char *s) {
  int i, n = strlen(s);

  for (i=0; i<n; i++) {
    if (!isspace((int)s[i])) {
      return FALSE;
    }
  }
  return TRUE;
}
      
	
	
   
