/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include <sys/types.h>
#include <signal.h>  // for kill() and UNIX signals
#include <sys/wait.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "tkMdp.h"
#include "debug.h"   // for generic debugging routines

#ifndef NSIG
#define NSIG _sys_nsig
#endif // !NSIG

// For some reason, this isn't in the Solaris C++ headers
#if defined(SOLARIS) || defined(SUNOS)
extern "C"
{
    int gethostname (char *name, int namelen);
}
#endif // SOLARIS   

// Or this for SUNOS
#ifdef SUNOS
char *strerror(int err)
{
   char text[16];
   sprintf(text, "Error %d", err);
   return text;
}
#endif // SUNOS

#include "mdp.xbm"
extern char TclTkLibs[];  // C-compiled Tcl/Tk library scripts
extern char InitTkMdp[];  // C-compiled MDP specific Tcl scripts

/******************************
 * TkMdpApp class implementation
 */

TkMdpApp::TkMdpApp()
	: MdpApp(), interp(NULL), 
      window_id(0), use_netscape(false),
      tx_interval_timer_id(NULL), post_processor_id(0)
{
 
}

TkMdpApp::~TkMdpApp()
{
   StopTxIntervalTimer();
}

void TkMdpApp::OnExit()
{
    MdpApp::OnExit();
    exit(0);
}



// IMPORTANT: For this to work, Tcl/Tk must be initialized before
// any MDPv2 sessions are opened!
bool TkMdpApp::MdpSocketInstaller(MdpSocketInstallCmd  cmd,
                                  MdpSocketHandle      socketHandle,
                                  MdpInstanceHandle    instanceHandle)
{
    if (MDP_SOCKET_INSTALL == cmd)
    {       
        Tcl_CreateFileHandler(MdpSocketGetDescriptor(socketHandle), 
                              TCL_READABLE, 
                              MdpSocketHandler, 
                              (ClientData)socketHandle);
    }
    else  // MDP_SOCKET_REMOVE
    {
        Tcl_DeleteFileHandler(MdpSocketGetDescriptor(socketHandle));
    }
    return true;
}  // end TkMdpApp::MdpSocketInstaller()

bool TkMdpApp::MdpTimerInstaller(MdpTimerInstallCmd    cmd, 
                                 double                delay,
                                 MdpTimerHandle        timerHandle, 
                                 MdpInstanceHandle     /*instanceHandle*/)
{
    if (MDP_TIMER_MODIFY == cmd)
    {
        Tcl_TimerToken timerId = (Tcl_TimerToken)MdpTimerGetUserData(timerHandle); 
        ASSERT(timerId);
        Tcl_DeleteTimerHandler(timerId);
        int msec = (int) (delay * 1000.0);
        timerId = Tk_CreateTimerHandler(msec, MdpTimerHandler,
                                        (ClientData)timerHandle);
        MdpTimerSetUserData(timerHandle, (void*)timerId);
        return (timerId ? true : false);
    }
    else if (MDP_TIMER_INSTALL == cmd)
    {
        int msec = (int) (delay * 1000.0);
        Tcl_TimerToken timerId = Tk_CreateTimerHandler(msec, MdpTimerHandler,
                                                       (ClientData)timerHandle);
        MdpTimerSetUserData(timerHandle, (void*)timerId);
        return (timerId ? true : false);
    }
    else  // MDP_TIMER_REMOVE
    {
        Tcl_TimerToken timerId = (Tcl_TimerToken)MdpTimerGetUserData(timerHandle);
        ASSERT(timerId);
        Tcl_DeleteTimerHandler(timerId);
        MdpTimerSetUserData(timerHandle, (void*)NULL);
        return true;
    }
}  // end TkMdpApp::MdpTimerInstaller()


void TkMdpApp::MdpSocketHandler(ClientData data, int /*mask*/)
{
    MdpSocketOnDataReady((MdpSocketHandle)data);
}  // end TkMdpSocketHandler()

void TkMdpApp::MdpTimerHandler(ClientData data)
{
	MdpTimerOnTimeout((MdpTimerHandle)data);
} // end TkMdpAppTimerHandler()

void TkMdpApp::StartTxIntervalTimer(double delay)
{
    ASSERT(!tx_interval_timer_id);
    tx_interval_timer_id = 
        Tcl_CreateTimerHandler((int)(1000 * delay),
                               DoTxIntervalTimeout, 
                               (ClientData) this);
}  // end TkMdpApp::StartTxIntervalTimer()

void TkMdpApp::StopTxIntervalTimer()
{
    if (tx_interval_timer_id)  
        Tcl_DeleteTimerHandler(tx_interval_timer_id);
    tx_interval_timer_id = NULL; 
}  // end TkMdpApp::StopTxIntervalTimer()

void TkMdpApp::DoTxIntervalTimeout(ClientData data)
{
    ((TkMdpApp *)data)->tx_interval_timer_id = NULL;
    ((MdpApp *)data)->OnTxIntervalTimeout();
}  // end TkMdpApp::DoTxIntervalTimeout()


void TkMdpApp::SetStatusText(const char* text)
{
	char cmd[256 + 16];
	sprintf(cmd, "SetMdpStatusText {%.256s}", text);
    if(TCL_OK != Tcl_Eval(interp, cmd))
        fprintf(stderr, "Error executing command: %s\n", cmd);
}  // end SetMdpStatusText() 

void TkMdpApp::SetProgressMeter(unsigned int percent)
{
	char cmd[64];
	if (percent > 100) percent = 100;
	sprintf(cmd, "SetMdpProgressMeter %d", percent);
	if(TCL_OK != Tcl_Eval(interp, cmd))
        fprintf(stderr, "Error executing command: %s\n", cmd);
}  // end SetMdpProgressMeter()
         
int TkMdpApp::DoShutdown(ClientData clientData,
                         Tcl_Interp* /*interp*/,
                         int /*argc*/, char** /*argv*/)
{
    ((TkMdpApp*)clientData)->OnExit();
    return TCL_OK;
}  // end TkMdpApp::DoShutdown()

int TkMdpApp::GetDebugLevel(ClientData /*clientData*/,
                            Tcl_Interp *interp,
                            int /*argc*/, char** /*argv*/)
{
#ifdef PROTO_DEBUG
    sprintf(interp->result, "%u", DebugLevel());
#else
    sprintf(interp->result, "0");
#endif // PROTO_DEBUG
    return TCL_OK;
}  // end TkMdpApp::GetDebugLevel()

int TkMdpApp::SetDebugLevel(ClientData /*clientData*/,
                            Tcl_Interp */*interp*/,
                            int /*argc*/, char** argv)
{
#ifdef PROTO_DEBUG
    ::SetDebugLevel(atoi(argv[1]));
#endif // PROTO_DEBUG
    return TCL_OK;
}  // end TkMdpApp::SetDebugLevel()

int TkMdpApp::GetPostProcessor(ClientData  clientData,
                               Tcl_Interp* interp,
                               int /*argc*/, char** /*argv*/)
{
    Tcl_AppendResult(interp, ((TkMdpApp*)clientData)->PostProcessor(), NULL);
    int i = 0;
    const char* ptr;
    while ((ptr = ((TkMdpApp*)clientData)->ProcessorOpt(i++)))
        Tcl_AppendResult(interp, " ", ptr, NULL);
    return TCL_OK;
}  // end TkMdpApp::GetPostProcessor()

int TkMdpApp::SetPostProcessor(ClientData clientData,
                               Tcl_Interp */*interp*/,
                               int argc, char** argv)
{    
    if (argc < 2)  // no post processor spec'd
        ((TkMdpApp*)clientData)->SetPostProcessor("\0");  
    else
        ((TkMdpApp*)clientData)->SetPostProcessor(argv[1]);
    return TCL_OK;
}  // end int TkMdpApp::SetPostProcessor()

void TkMdpApp::SetPostProcessor(char* theCmd)
{
    MdpApp::SetPostProcessor(theCmd);      
    // See if it's Netscape post processing 
    // for which we provide special support
    CheckForNetscape();
    
}  // end int TkMdpApp::SetPostProcessor()


void TkMdpApp::CheckForNetscape()
{
    // See if it's Netscape post processing 
    // for which we provide special support
    char txt[8];
    const char *ptr = strrchr(PostProcessor(), DIR_DELIMITER);
    if (ptr)
        ptr++;
    else
        ptr = PostProcessor();
    strncpy(txt, ptr, 8);
    for (int i=0; i<8; i++) txt[i] = toupper(txt[i]);
    if (!strncmp(txt, "NETSCAPE", 8)) 
        SetNetscape(true);
    else
        SetNetscape(false);
}  // end TkMdpApp::CheckForNetscape()


#define USE_ACTIVE_WINDOW 0xffffffff

void TkMdpApp::PostProcess(const char *path)
{
    const char* myArgs[32];
    char wid_text[32], url_text[PATH_MAX+64];
    
    if (use_netscape)
    {
        int i = 0;
        myArgs[i++] = PostProcessor();
        Window w = 0;
        // (TBD) We could put in a hack to look for the old
        // window if we made "w" static when "window_id" = 0xffffffff
        // This would help us capture the proper window for file display
        // It's still a hack but it might work a little better without
        // too much trouble
        if (NetscapeIsRunning(&w))
        {       
            // Use -remote command to display file in current
            // or new window
            if (window_id)
            {
                if (NetscapeCheckWindow(window_id))
                {
                    // Use the same window as last time
                    myArgs[i++] = "-id";
                    sprintf(wid_text, "0x%lx", window_id);
                    myArgs[i++] = wid_text;
                    myArgs[i++] = "-noraise";
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s)", path);
                    myArgs[i++] = url_text;
                }
                else if (USE_ACTIVE_WINDOW == window_id)
                {
                    // Capture the open window we found
                    myArgs[i++] = "-id";
                    sprintf(wid_text, "0x%lx", w);
                    myArgs[i++] = wid_text;
                    window_id = w;
                    myArgs[i++] = "-noraise"; 
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s)", path);
                    myArgs[i++] = url_text;         
                }
                else  // user must have closed old window so open another
                {
                    window_id = USE_ACTIVE_WINDOW;
                    //myArgs[i++] = "-raise";
                    myArgs[i++] = "-remote";
                    sprintf(url_text, "openURL(file://%s,new-window)", path);
                    myArgs[i++] = url_text;
                }                
                
            }
            else
            {
                // We're starting fresh, open a new window 
                window_id = USE_ACTIVE_WINDOW;
                //myArgs[i++] = "-raise";
                myArgs[i++] = "-remote";
                sprintf(url_text, "openURL(file://%s,new-window)", path);
                myArgs[i++] = url_text;
            }
        }
        else
        {
            window_id = USE_ACTIVE_WINDOW;
            myArgs[i++] = path;
        }       
        myArgs[i] = NULL;
    }
    else
    {
        if (post_processor_id) KillPostProcessor();
        int i=0, j=0;
        myArgs[i++] = PostProcessor();
        while (ProcessorOpt(j))
            myArgs[i++] = ProcessorOpt(j++);
        myArgs[i++] = path;
        myArgs[i++] = NULL;
    }
        
    switch((post_processor_id = fork()))
    {
        case -1:    // error
            fprintf(stderr, "tkMdp: PostProcess fork() error: %s\n",
                strerror(errno));
            post_processor_id = 0;
            return;

        case 0:     // child
            //for (int i = 0; i < NSIG; i++) signal(i, SIG_DFL); 
            if (execvp((char*)myArgs[0], (char**)myArgs) < 0)
            {
		        fprintf(stderr, "tkMdp:  PostProcess execvp(%s) error: %s\n",
                        myArgs[0], strerror(errno));
                exit(-1);
            }
            break;

        default:    // parent
            // SIGCHLD handler to prevent zombie post processor processes
            if(!InstallSignalHandler(SIGCHLD))
            {
                fprintf(stderr, "tkMdp:: Error installing SIGCHLD handler!\n");
                exit(-1);
            }
            break;
    }
    //SafeSIGCHLDHandler((ClientData)this,  (Tcl_Interp*)NULL, (int)NULL);
}  // end TkMdpApp::PostProcess()


void TkMdpApp::KillPostProcessor()
{
    int status;
    int count = 0;
    if (!post_processor_id) return;
    RemoveSignalHandler(SIGCHLD);
    while((kill(post_processor_id, SIGTERM) !=0) && count < 10)
	{ 
        count++;
        if (EINTR == errno) continue;
	    if (ESRCH == errno) break;
	    fprintf(stderr, "tkMdp: KillPostProcessor kill() error: %s\n",
                strerror(errno));
	}
	count = 0;
	while((waitpid(post_processor_id, &status, 0) 
	         != post_processor_id) && count < 10)
	{
        count++;
        if (EINTR == errno) continue;
	    if (ECHILD == errno) break;
	    fprintf(stderr, "tkMdp: KillPostProcessor wait() error: %s\n",
                strerror(errno));
	}
	post_processor_id = 0;
}  // end MdpSessionMgr::KillPostProcessor()

// Our application instance (needs to be global for our SignalHandler)
TkMdpApp theApp;

Tcl_AsyncHandler TkMdpApp::SIGCHLD_token = NULL;
Tcl_AsyncHandler TkMdpApp::SIGINT_token = NULL;

bool TkMdpApp::InstallSignalHandler(int signum)
{
    SignalHandlerFunc oldHandler;
    switch(signum)
    {
        case SIGCHLD:
            oldHandler = signal(signum, SignalHandler);
            if (SIG_ERR == oldHandler)
            {
                perror("tkMdp: signal() error");
                return false;   
            }
            if (SIGCHLD_token) Tcl_AsyncDelete(SIGCHLD_token);
            SIGCHLD_token = Tcl_AsyncCreate(SafeSIGCHLDHandler, (ClientData)this);  
            break;
            
        case SIGINT:
        case SIGTERM:
            oldHandler = signal(signum, SignalHandler);            
            if (SIG_ERR == oldHandler)
            {
                perror("tkMdp: signal() error");
                return false;   
            }
            if (SIGINT_token) Tcl_AsyncDelete(SIGINT_token);
            SIGINT_token = Tcl_AsyncCreate(SafeSIGINTHandler, (ClientData)this);  
            break;
            
        default:
            fprintf(stderr, "tkMdp: Attempted to install invalid signal handler!\n");
            return false;        
    }
    return true;    
}  // end TkMdpApp::InstallSignalHandler

void TkMdpApp::RemoveSignalHandler(int signum)
{
    switch (signum)
    {
        case SIGCHLD:
            if (SIGCHLD_token) Tcl_AsyncDelete(SIGCHLD_token);
            SIGCHLD_token = NULL;
            break;
            
        case SIGINT:
        case SIGTERM:
            if (SIGINT_token) Tcl_AsyncDelete(SIGINT_token);
            SIGINT_token = NULL;
            break;
            
        default:
            fprintf(stderr, "tkMdp: Attempted to remove invalid signal handler!\n");
            break;             
    }    
    signal(signum, SIG_DFL);
}  // end TkMdpApp::RemoveSignalHandler()

void TkMdpApp::SignalHandler(int signum)
{
    switch(signum)
    {
        case SIGCHLD:
            if (SIGCHLD_token)
                Tcl_AsyncMark(SIGCHLD_token);
            else
                fprintf(stderr, "tkMdp: SIGCHLD with no safe handler!\n");
            break;   
            
        case SIGINT:
        case SIGTERM:
            if (SIGINT_token)
                Tcl_AsyncMark(SIGINT_token);
            else
                fprintf(stderr, "tkMdp: SIGCHLD with no safe handler!\n");
            break;
            
        default:
            fprintf(stderr, "tkMdp signal(%d) with no safe handler!\n", signum);
            break;
    }
}  // end TkMdpApp::SignalHandler()

int TkMdpApp::SafeSIGCHLDHandler(ClientData     clientData,
                                 Tcl_Interp*    /*interp*/,
                                 int            code)
{
    TkMdpApp* theApp = (TkMdpApp*)clientData;   
    if (theApp->post_processor_id)
    {                        
        int count = 0;
        do
        {
            int status = 0;
            int result = waitpid(theApp->post_processor_id, &status, WNOHANG);
            if (result < 0)
            {
                if (ECHILD == errno) break;
                if (EINTR != errno) 
                    perror("tkMdp: wait(PostProcessor) error");
            }
            else
            {
                if (WIFEXITED(status))
                    theApp->post_processor_id = 0;
                break;
            }
            if (status)
            {
                DMSG(0, "tkMdp: PostProcessor returned non-zero status: %d\n", status);   
            }
            count++;
        } while (count < 10);
    } 
    // Re-install signal handler (some systems require it)
    if(!theApp->InstallSignalHandler(SIGCHLD))
        fprintf(stderr, "tkMdp: Error installing SIGCHLD handler!\n");
    return code;
}  // end TkMdpApp::SafeSIGCHLDHandler()

int TkMdpApp::SafeSIGINTHandler(ClientData     clientData,
                                Tcl_Interp*    /*interp*/,
                                int            code)
{
    ((TkMdpApp*)clientData)->OnExit();
    return code;
}  // end TkMdApp::SafeSIGINTHandler()

int main(int argc, char *argv[])
{	    
    // Initialize MDP file push application base class
    theApp.Init(TkMdpApp::MdpTimerInstaller,
                TkMdpApp::MdpSocketInstaller);
    
    // Set some default post processor for unix 
    //(TBD - use "BROWSER" environment variable?)
    theApp.SetPostProcessor("xv -rmode 5 -root -quit");
        
    // Parse command line options (use "getopt()" to pick up option flags)
    extern char *optarg;
    extern int optind;
    int c;
    while ((c = getopt(argc, argv, MDPAPP_CMD_LINE_FLAGS)) != EOF) 
    {
        if (!theApp.ProcessCommand(c, optarg))
        {
            exit(-1);
        }                    
    }
    argc -= optind;
    char **orig_argv = argv;
    argv += optind;
    
    // Assume remaining arguments are the server's tx file list
    for (int i = 0; i < argc; i++)
        theApp.ProcessCommand(0, argv[i]); 
    
    // See if the command-line set Netscape as post-processor
    // to invoke special post processor handling
    theApp.CheckForNetscape();
   
    
    // SIGTERM/SIGINT (CTRL-C) handler to allow for (hopefully) graceful shutdown
    if(!theApp.InstallSignalHandler(SIGINT))
    {
        fprintf(stderr, "tkMdp:: Error installing SIGINT handler!\n");
        exit(-1);
    }   
    if(!theApp.InstallSignalHandler(SIGTERM))
    {
        fprintf(stderr, "tkMdp:: Error installing SIGTERM handler!\n");
        exit(-1);
    }
    // Set Tcl/Tk for standalone applications (i.e. Create an interpreter,
    // init Tcl and Tk, load the Tcl/Tk Library scripts we need (TclTkLibs)
    
    // Also, don't send any command line options to Tk_Main() for now
    // MDPv2 command line options mess this up right now (other than
    // the launching program name)
    argc = 1;
    argv = orig_argv;

    Tcl_Interp* interpreter =  Tcl_CreateInterp();  
    theApp.SetInterpreter(interpreter);   
    
    Tcl_FindExecutable(argv[0]); 
    
    //Tcl_MakeSafe(interpreter);
    
    // The only thing Tcl_Init() does (version 8.0) is search for 
    // tcl files which our standalone app doesn't use anyway
    /*if (TCL_OK != Tcl_Init(interpreter))
    {
        
        fprintf(stderr, "tkMdp: Warning! Possible error initing Tcl.\n");
        fprintf(stderr, "       (%s)\n", interpreter->result);
		//exit(-1);      
    } */ 
            
    if (TCL_OK != Tk_Init(interpreter))
    {
        fprintf(stderr, "tkMdp: Warning! Possible error initing Tk.\n");
        fprintf(stderr, "       (%s)\n", interpreter->result);
		//exit(-1);   
    }
    
    // Init the Tcl/Tk Library routine scripts
  	if (TCL_OK != Tcl_VarEval(interpreter, TclTkLibs, NULL))
	{
		fprintf(stderr, "tkMdp: Error loading Tcl/Tk library scripts!\n");
		exit(-1);
	}
    
    
    // Call Tcl_CreateCommand() for any app specific commands the user
    // interface Tcl scripts need. (Tcl <-> C++ code interface)
    
	// Set routine called when user exits application
	Tcl_CreateCommand(interpreter, "shutdown", 
					  (Tcl_CmdProc *) TkMdpApp::DoShutdown, 
					   &theApp, NULL);
    
    // Set/Get current MDPv2 debug level & post processor
    Tcl_CreateCommand(interpreter, "SetDebugLevel", 
					  (Tcl_CmdProc *) TkMdpApp::SetDebugLevel, 
					   NULL, NULL);
    Tcl_CreateCommand(interpreter, "GetDebugLevel", 
					  (Tcl_CmdProc *) TkMdpApp::GetDebugLevel, 
					   NULL, NULL);
    Tcl_CreateCommand(interpreter, "SetPostProcessor", 
					  (Tcl_CmdProc *) &TkMdpApp::SetPostProcessor, 
					   &theApp, NULL);
    Tcl_CreateCommand(interpreter, "GetPostProcessor", 
					  (Tcl_CmdProc *) TkMdpApp::GetPostProcessor, 
					   &theApp, NULL);
    
    // Load our icon bitmap
    Tk_DefineBitmap(interpreter, Tk_GetUid("mdp_logo"),
			        mdp_bits, mdp_width, mdp_height);
	
	// Execute initialization Tcl script
	if (TCL_OK != Tcl_Eval(interpreter, InitTkMdp))
    {
        fprintf(stderr, "tkMdp: Error initing Mdp Tcl/Tk scripts!\n");
		exit(-1);
	}
    
    // Init the main TkMdpApp dialog window
	if (TCL_OK != Tcl_VarEval(interpreter, "InitMdpMainWindow ", MdpVersion(), " {", theApp.SessionName(), "}", NULL))
    {
	    fprintf(stderr, "tkMdp: Error initing main window!\n");
        exit(-1);
    }
    
    // Start mdpApp server/client participation
    if (!theApp.Start()) exit(-1);

    // Enter the Tk (or dispatcher command line) main loop
    theApp.Run();
    exit(0);
}  // end main()




