/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#ifndef _SETTINGS_FILE
#define _SETTINGS_FILE

#include "debug.h"
#include <stdio.h>
#include <string.h>

// enum SettingsItemType 
// {
//     SETTINGS_TYPE_META, 
//     SETTINGS_TYPE_STRING,
//     SETTINGS_TYPE_INT,
//     SETTINGS_TYPE_FLOAT
// };
// 
// 
// class SettingsItem
// {
//     public:
//         SettingsItem(SettingsItemType theType);
//             
//     private:
//         char*           key;
//         char*           data;
//         unsigned short  max_len;
//         void*           binding;
//         SettingsItem*   prev;
//         SettingsItem*   next;
//             
// };  // end class SettingsItem
// 
// class SettingsMgr
// {
//     public:
//             
//             
//     private:
//             
// };  // end class SettingsMgr
    
    
class SettingsFile
{
    public:
        SettingsFile();
        ~SettingsFile();
        bool OpenForRead(const char* path);
        bool OpenForWrite(const char* path);
        bool SettingsFile::WriteSetting(const char* key, const char* value);
        int GetNextSetting(const char** key, const char** value);
        void Close();
        
    private:
        FILE* file_ptr;
        char* line_buf;
        unsigned int line_buf_len;
        char* key_buf;
        unsigned int key_buf_len;
        char* value_buf;
        unsigned int value_buf_len;
        unsigned int group_level;
        unsigned int group_len;
        int ReadLine();
};  // end class SettingsFile

SettingsFile::SettingsFile()
    : file_ptr(NULL), line_buf(NULL), line_buf_len(0),
      key_buf(NULL), key_buf_len(0),
      value_buf(NULL), value_buf_len(0),
      group_level(0), group_len(0)
{
}

SettingsFile::~SettingsFile()
{
    Close();
    if (line_buf) delete []line_buf;
    if (key_buf) delete []key_buf;    
    if (value_buf) delete []value_buf;
}

bool SettingsFile::OpenForRead(const char* path)
{
    ASSERT(NULL == file_ptr);
    if (file_ptr) return false;
    return (NULL != (file_ptr = fopen(path, "r")));
}  // end SettingsFile::OpenForRead()

bool SettingsFile::OpenForWrite(const char* path)
{
    ASSERT(!file_ptr);
    if (file_ptr) return false;
    return (NULL != (file_ptr = fopen(path, "w+")));
}  // end SettingsFile::OpenForWrite()

void SettingsFile::Close()
{
    if (file_ptr) fclose(file_ptr);
    file_ptr = NULL;
    if (line_buf) delete []line_buf;
    line_buf = NULL;
}  // end SettingsFile::Close()

bool SettingsFile::WriteSetting(const char* key, const char* value)
{
    int goal_len = strlen(key) + strlen(value);
    int len;
    if (strchr(value, '\n') || strchr(value, '\r'))
    {
        goal_len += 4;
        len = fprintf(file_ptr, "%s={%s}\n", key, value);  
    }  
    else
    {
        goal_len += 2;
        len = fprintf(file_ptr, "%s=%s\n", key, value); 
    }
    return (len >= goal_len);  
}  // end SettingsFile::WriteSetting()

int SettingsFile::GetNextSetting(const char** key, const char** value)
{
    enum SeekMode{SEEKING_KEY, SEEKING_VALUE, SEEKING_GROUP};
    
    
    if (!line_buf)
    {
        if (!(line_buf = new char[256]))
        {
            line_buf_len = 0;
            return -1;
        }
        line_buf_len = 256;
    }
    
    SeekMode mode = SEEKING_KEY;
    // This loop will return on success, file end, or error
    while (1)
    {
        *line_buf = '\0';
        int result = ReadLine();
        if (result < 0) return -1;
        if (0 == result)
        {
            if (SEEKING_KEY == mode)
            {
                return 0;  // no more settings, done
            }
            else
            {
                fprintf(stderr, "SettingsFile: Error reading multi-line setting value!\n");
                return -1;
            }
        }
        else if (result < 0)
        {
            perror("SettingsFile::ReadLine() error");
            return -1; // abruptly ended settings file
        }
        char* ptr = line_buf;
        // Skip blank and commented lines (leading '#')
        while ((' ' == *ptr) || ('\t' == *ptr)) ptr++;
        if (('#' == *ptr) || ('\0' == *ptr))
            continue;
        else
            ptr = line_buf;
        bool scanning_line = true;
        while (scanning_line)
        {
            switch (mode)
            {
                case SEEKING_KEY:
                {
                    while ((' ' == *ptr) || ('\t' == *ptr)) ptr++;
                    char* end_ptr = strchr(ptr, '=');
                    
                    if (end_ptr)
                    {
                        char* value_ptr = end_ptr + 1;
                        *end_ptr-- = '\0';
                        // Strip trailing white space from key field
                        while ((' ' == *end_ptr) || 
                               ('\t' == *end_ptr) ||
                               ('\0' == *end_ptr))
                            *end_ptr-- = '\0';
                            
                        unsigned int key_len = strlen(ptr) + 1;
                        if (key_len > key_buf_len)
                        {
                            if (key_buf) delete key_buf;
                            if(!(key_buf = new char[key_len]))
                            {
                                perror("SettingsFile: Error allocating key_buf");
                                return -1; 
                            }
                            else
                            {
                                key_buf_len = key_len;
                            }
                        }
                        strcpy(key_buf, ptr);
                        mode = SEEKING_VALUE;    
                        ptr = value_ptr;                        
                    }
                    else
                    {
                        // Bad settings file line
                        fprintf(stderr, "SettingsFile: Error seeking key ...\n");
                        return -1;   
                    }
                }
                break;

                case SEEKING_VALUE:
                {
                    // Skip any leading white space
                    while ((' ' == *ptr) || ('\t' == *ptr)) ptr++;
                    
                    // Make sure we have room for this one
                    unsigned int value_len = strlen(ptr)+ 1;
                    if (value_len > value_buf_len)
                    {
                        if (value_buf) delete []value_buf;
                        if (!(value_buf = new char[value_len]))
                        {
                            perror("SettingsFile: Error allocating value_buf");
                            return -1;
                        }
                        else
                        {
                            value_buf_len = value_len;
                        }
                    }                     
                    // Is it the start of a group "\{" ?
                    if ('{' == ptr[0])
                    {
                        ptr++;
                        // Skip any leading white space
                        while ((' ' == *ptr) || ('\t' == *ptr)) ptr++;
                        group_level = 1;
                        group_len = 0;
                        value_buf[0] = '\0';
                        mode = SEEKING_GROUP;
                    }
                    else
                    {
                        char* end_ptr = ptr + value_len - 1;
                        // Strip trailing white space from key field
                        while ((' ' == *end_ptr) || 
                               ('\t' == *end_ptr) ||
                               ('\0' == *end_ptr))
                        {
                            *end_ptr-- = '\0';
                        }
                        strcpy(value_buf, ptr);
                        *key = key_buf;
                        *value = value_buf;
                        return 1;
                    }
                }
                break;

                case SEEKING_GROUP: 
                { 
                    unsigned int value_len = group_len + strlen(ptr)+ 2;
                    if ((value_len+1) > value_buf_len)
                    {
                        char* new_value_buf = new char[value_len];
                        if (!new_value_buf)
                            return -1;
                        else
                            value_buf_len = value_len;
                        memcpy(new_value_buf, value_buf, group_len+1);
                        delete []value_buf;
                        value_buf = new_value_buf;
                    }   
                    // Check for level increase/decrease or end of group
                    char* temp_ptr = ptr;
                    while('\0' != *temp_ptr)
                    {
                        if ('{' == *temp_ptr)
                            group_level++;
                        else if ('}' == *temp_ptr)
                            group_level--;
                        temp_ptr++;   
                    }
                    if (group_level)
                    {
                        strcat(value_buf, ptr);
                        strcat(value_buf, "\n");
                        group_len += strlen(ptr) + 1;
                        scanning_line = false;  // Get next line of group
                    }
                    else
                    {
                        // End of group
                        temp_ptr = strrchr(ptr, '}');
                        ASSERT(temp_ptr);
                        *temp_ptr = '\0';
                        strcat(value_buf, ptr);
                        *key = key_buf;
                        *value = value_buf;
                        return 1;
                    }
                }
                break; 
            }  // end switch(mode)
        }  // end while(scanning_line)
    }  // end while(1);
}  // end GetNextSetting()

// Put a NULL-terminated line from file into the "line_buf"
int SettingsFile::ReadLine()
{
    ASSERT(file_ptr);
    if (!line_buf)
    {
        if(!(line_buf = new char[256]))
        {
             return -1;  
        }
        line_buf_len = 256;
    }
    char* ptr = line_buf;
    unsigned int count = 0;
    while (fread(ptr, sizeof(char), 1, file_ptr))
    {
        count++;
        if (('\n' == *ptr) || ('\r' == *ptr))
        {
            *ptr = '\0';
            // Check for '\' at end-of-line to indicate continuation
            if ((count > 1) && ('\\' == line_buf[count-2]))
            {
                ptr--;
                count--;
                continue;
            }
            else
            {
                return count;  
            } 
        }
        // Make sure we have more room in our line buffer
        if (count == (line_buf_len))
        {
            // Bigger line buffer needed  
            char* new_buf = new char[2*line_buf_len];
            if (!new_buf)  return -1;
            memcpy(new_buf, line_buf, count);
            delete []line_buf;
            line_buf = new_buf;
            ptr = new_buf + count;
        }
        else
        {
            ptr++;
        }
    }  // end while fread()
    // Properly terminate at EOF      
    if (count)
    {
        if (count < line_buf_len) 
            line_buf[count] = '\0';
    }
    return count;
}  // end SettingsFile::ReadLine()


int main(int argc, char** argv)
{
    if (argc != 2)
    {
        fprintf(stderr, "Usage: set <settingsFile>\n");
        exit(-1);
    }
    
    SettingsFile theFile;
    
    if (!theFile.OpenForRead(argv[1]))
    {
        perror("set: Error opening file for read");
        exit(-1);
    }
    
    
    const char* key;
    const char* value;
    int result;
    do
    {
        result = theFile.GetNextSetting(&key, &value);
        if (result > 0)
        {
            fprintf(stdout, "set: Found setting key=\"%s\", value = \"%s\"\n",
                    key, value);
        }
        else if (result < 0)
        {
            fprintf(stderr, "set: Error parsing settings file!\n");   
        }
    } while (result > 0);
           
    theFile.Close();
    printf("Done.\n");
    exit(0);
    
}  // end main()

#endif // _SETTINGS_FILE
