/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
 
#ifndef _MDP_SIM_AGENT
#define _MDP_SIM_AGENT

#include "debug.h"
#include "protoSim.h"
#include "mdpSession.h"

extern void OpnetBreak(char* text);

class MdpSimAgent : public ProtoSimAgent
{
    public:
        virtual ~MdpSimAgent();
    
        // Methods to support SimApi 
        void HandleRecvPacket(char* buffer, unsigned int len, 
                              SIMADDR srcAddr, unsigned short srcPort,
                              bool isUnicast)
        {
            ASSERT(session);
            NetworkAddress src;
            src.SimSetAddress(srcAddr);
            src.SetPort(srcPort);
            total_recvd_low += len;
            if (total_recvd_low < len) total_recvd_high++;
            session->HandleRecvPacket(buffer, len, &src, isUnicast);   
        }
        
        void DoTimeout() {session_mgr.DoTimers();}
        
        // MDPSession parameter settings
        // Call these _BEFORE_ session is started as client or servers
        bool SetSessionTTL(unsigned char theTTL);
        bool SetSessionSegmentSize(unsigned short theSize);
        bool SetSessionBlockSize(unsigned char theSize);
        bool SetSessionNumParity(unsigned char numParity);
        bool SetSessionTxBufferSize(unsigned long theSize);
        bool SetSessionRxBufferSize(unsigned long theSize);
        bool SetSessionBaseObjectId(unsigned long theId);
        bool SetSessionEmcon(bool state);
        
        // Call these to start/stop protocol operation
        // (An agent can only be a server _or_ a client, not both)
        bool StartServer(SIMADDR theAddr, unsigned short thePort);
        bool QueueTxObject(unsigned long dataSize);
        bool StartClient(SIMADDR theAddr, unsigned short thePort);
        void Stop();
        
        bool IsClient() {return session ? session->IsClient() : false;}
                
        // Call these _ANYTIME_ a change is desired
        double TxRate() {return ((double)tx_rate*8.0);}
        void SetSessionTxRate(unsigned long theRate);
        void SetSessionAutoParity(unsigned char autoParity);
        void SetSessionCongestionControl(bool theState);
        void SetSessionFastStart(bool theState);
        void SetSessionEcnEnable(bool theState);
        void SetSessionEcnStatus(bool theState);
        void SetSessionUnicastNacks(bool theState);
        void SetSendDropRate(double percent);
        void SetRecvDropRate(double percent);
        void SetGrttEstimate(double theTime);
        void SetTxRateBounds(double txRateMin, double txRateMax);
        
        void SetNumRepeats(int theCount) {num_repeats = theCount;}
        bool StartLogging(const char* fileName)
            {return (logging = OpenDebugLog(fileName));}
        void StopLogging()
        {
            if (logging)
            {
                CloseDebugLog();
                logging = false;
            }
        }
        
        void SetMdpBackoffWindow(double value) {MDP_BACKOFF_WINDOW = value;}
        void SetMdpGroupSize(unsigned int value) {MDP_GROUP_SIZE = value;}
        
        /*// Simulation socket Mgmt & I/O functions
        virtual int OpenSocket(UdpSocket* theSocket, unsigned short thePort) = 0;
        virtual void CloseSocket(UdpSocket* theSocket) = 0;       
        virtual void SendTo(UdpSocket* theSocket, 
                            SIMADDR, unsigned short thePort,
                            char* buffer, unsigned int len) = 0;
        virtual bool JoinGroup(UdpSocket* theSocket, SIMADDR theGroup) = 0;
        virtual void LeaveGroup(UdpSocket* theSocket, SIMADDR theGroup) = 0;
        virtual void SetSocketTTL(UdpSocket* theSocket, unsigned char theTTL) = 0;
        virtual void SetSocketLoopback(UdpSocket* theSocket, bool state) = 0;
        */
        // Misc. simulator agent routines
        virtual unsigned long GetAgentId() = 0;
                
        
    protected:
        // Construction
        MdpSimAgent(); 
        
        // Simulation timer mgmt functions (for a single timer)
        /*virtual void InstallTimer(double delay) = 0;
        virtual void ModifyTimer(double delay) = 0;
        virtual void RemoveTimer() = 0;*/
        
        void Notify(MdpNotifyCode       notifyCode,
				    MdpSessionHandle    sessionhandle,
                    MdpNodeHandle       nodeHandle,
                    MdpObjectHandle     objectHandle,
                    MdpError            errorCode);
        
        
        unsigned long   total_recvd_low;
        unsigned long   total_recvd_high;
        unsigned long   goodput_recvd_low;
        unsigned long   goodput_recvd_high;
        
               
    
        MdpSessionMgr   session_mgr;  // (TBD) Change MDP API to allow multiple session mgrs
    
        // When MDP API is upgraded we can do all this via API
        MdpSession*     session;    // our single MdpSession
        
    private:    
        // Session parameters
        unsigned char   ttl;
        unsigned long   tx_rate;
        unsigned short  segment_size;
        unsigned char   block_size, num_parity, auto_parity;
        unsigned long   tx_buffer_size, rx_buffer_size;
        unsigned long   base_object_id;
        bool            congestion_control;
        bool            fast_start;
        bool            ecn_enable;
        double          tx_rate_min, tx_rate_max;
        bool		    unicast_nacks;
        bool            emcon;
        double          tx_drop_rate, rx_drop_rate;
        double          initial_grtt;
        
        unsigned long   current_tx_object_size;
        int 			num_repeats;
		bool			logging;
            
        static bool TimerInstaller(ProtocolTimerInstallCmd cmd,
                                   double                  delay,
                                   ProtocolTimerMgr*       timerMgr, 
                                   const void*             installData);
        
        static bool SocketInstaller(UdpSocketCmd        cmd,
                                    class UdpSocket*    theSocket,
                                    const void*         installData);
        
        static bool Notifier(MdpNotifyCode      notifyCode,
                             MdpInstanceHandle  instanceHandle,
                             MdpSessionHandle   sessionHandle,
                             MdpNodeHandle      nodeHandle,
                             MdpObjectHandle    objectHandle,
                             MdpError           errorCode);
};

#endif // _MDP_SIM_AGENT
