/*
 * Copyright (c) 1997-2012 Motonori Nakamura <motonori@wide.ad.jp>
 * Copyright (c) 1997-2012 WIDE Project
 * Copyright (c) 1997-2003 Kyoto University
 * Copyright (c) 2012 National Institute of Informatics
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by WIDE Project and
 *      its contributors.
 * 4. Neither the name of the Project, the University nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char *_id_ = "$Id: lmtp.c,v 1.64 2012/06/07 07:51:35 motonori Exp $";
#endif

# include "common.h"
# include "extern.h"
# include "lmtp.h"

static	struct cmd_table commands[] = {
	{"LHLO", lmtp_lhlo},
	{"MAIL", lmtp_mail},
	{"RCPT", lmtp_rcpt},
	{"DATA", lmtp_data},
	{"QUIT", lmtp_quit},
	{"RSET", lmtp_rset},
	{"NOOP", lmtp_noop},
	{"HELP", lmtp_help},
	{NULL, NULL},
};

static	char *help_msg[] = {
	"Available commands are:",
	"  LHLO  MAIL  RCPT  DATA",
	"  RSET  NOOP  HELP  QUIT",
	NULL
};

static	FILE	*LmtpIn;
static	FILE	*LmtpOut;

static	int	state;
static	int	save_rcpts_trans;

static void
sig_quit()
{
	exit(EX_TEMPFAIL);
}

void
lmtp()
{
	char lmtpbuf[MAXLINE];
	char *p;
	struct cmd_table *cmdp;
	int len, ret;
#if 0
	struct stat st;

	if (fstat(0, &st) < 0 && errno == EBADF) {
		LmtpIn = fdopen(0, "r");
	}
	if (fstat(1, &st) < 0 && errno == EBADF) {
		LmtpOut = fdopen(1, "w");
	}
#endif
	LmtpIn = stdin;
	LmtpOut = stdout;

	signal(SIGINT, sig_quit);
	signal(SIGTERM, sig_quit);

	save_rcpts_trans = cnf.rcpts_trans;

	setproctitle("LMTP: waiting greeting", NULL);
	lmtp_greeting();
	fflush (LmtpOut);
	state = LMTP_INIT;
	while (fgets(lmtpbuf, sizeof(lmtpbuf), LmtpIn) != NULL) {
		len = strlen(lmtpbuf);
		if (len > 0 && lmtpbuf[len-1] == '\n')
			lmtpbuf[--len] = '\0';
		if (len > 0 && lmtpbuf[len-1] == '\r')
			lmtpbuf[--len] = '\0';
		if ((p = strchr(lmtpbuf, ' ')) != NULL) {
			*p++ = '\0';
		}
		ret = 0;
		for (cmdp = commands; cmdp->cmd != NULL; cmdp++) {
			if (strcasecmp(lmtpbuf, cmdp->cmd) == 0) {

				if (cnf.debug & DEBUG_LMTP)
				logg(LOG_DEBUG, "LMTP input: %s %s", lmtpbuf,
					(p == NULL)?"":p);
				ret = (*cmdp->func)(p);
				fflush (LmtpOut);
				break;
			}
		}
		if (cmdp->cmd == NULL) {
			ret = lmtp_nocmd();
			fflush (LmtpOut);
		}
		if (ret < 0)
			break;
	}
}

static void
lmtp_greeting()
{
#ifdef INET6
	fprintf (LmtpOut, "220 %s LMTP server ready (%s with IPv6, PID %d)\n",
		myname, version, getpid());
#else
	fprintf (LmtpOut, "220 %s LMTP server ready (%s, PID %d)\n",
		myname, version, getpid());
#endif
}

static int
lmtp_lhlo(param)
char *param;
{
	if (state != LMTP_INIT) {
		fprintf (LmtpOut, "503 %s Duplicated\n", myname);
	}
	else if (param == NULL){
		fprintf (LmtpOut, "501 LHLO requires domain address\n");
	}
	else {
		fprintf (LmtpOut, "220-%s Hello %s\n", myname, param);
		fprintf (LmtpOut, "220-8BITMIME\n");
		fprintf (LmtpOut, "220-DSN\n");
		fprintf (LmtpOut, "220 SIZE\n");
		state = LMTP_MAIL;
	}
	return 0;
}

static int
lmtp_mail(param)
char *param;
{
	char *p, *q, *err, *addr;

	if (state == LMTP_INIT) {
		fprintf (LmtpOut, "503 Polite people say Hello first\n");
		return 0;
	} else if (state == LMTP_SENT) {
		fprintf (LmtpOut, "503 Need RSET before next transaction\n");
		return 0;
	} else if (state != LMTP_MAIL) {
		fprintf (LmtpOut, "503 Sender already specified\n");
		return 0;
	}
	if (strncasecmp (param, "from:", 5) != 0) {
		fprintf (LmtpOut, "501 Syntax error\n");
		return 0;
	}
	p = parse_address(param+5, &err, &addr, NULL);
	if (p == NULL) {
		fprintf (LmtpOut, "553 %s\n", err);
		return 0;
	}
	env.sender = newstr(addr);
	while (*p) {
		if (isspace(*p)) {
			p++;
		}
		else if (strncasecmp("size=", p, 5) == 0) {
			q = p + 5;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (env.size == NULL && p[5] != '\0')
				env.size = newstr(p + 5);
			p = q +1;
		}
		else if (strncasecmp("body=", p, 5) == 0) {
			q = p + 5;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (env.body == NULL && p[5] != '\0')
				env.body = newstr(p + 5);
			p = q +1;
		}
		else if (strncasecmp("envid=", p, 6) == 0) {
			q = p + 6;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (env.envid == NULL && p[6] != '\0')
				env.envid = newstr(p + 6);
			p = q +1;
		}
		else if (strncasecmp("ret=", p, 4) == 0) {
			q = p + 4;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (env.ret == NULL && p[4] != '\0')
				env.ret = newstr(p + 4);
			p = q +1;
		}
		else {
			fprintf (LmtpOut, "501 parameter problem: %s\n", p);
			return 0;
		}
	}
	if (cnf.data_max > 0 && env.size != NULL
	 && atoi(env.size) > cnf.data_max) {
		/* to fallback next mailer (sendmail internal SMTP) */
		fprintf (LmtpOut, "451 message size to large\n");
		return 0;
	}
	if (cnf.debug & DEBUG_LMTP)
	logg(LOG_DEBUG, "MAIL FROM: %s size=%s body=%s envid=%s ret=%s", addr,
		(env.size == NULL)?"":env.size, (env.body == NULL)?"":env.body,
		(env.envid == NULL)?"":env.envid, (env.ret == NULL)?"":env.ret);
	fprintf (LmtpOut, "250 %s Sender ok\n", addr);
	state = LMTP_RCPT;
	return 0;
}

static int
lmtp_rcpt(param)
char *param;
{
	char *p, *q, *err, *addr, *domain, *notify, *orcpt;

	if (state == LMTP_INIT) {
		fprintf (LmtpOut, "503 Polite people say Hello first\n");
		return 0;
	} else if (state == LMTP_SENT) {
		fprintf (LmtpOut, "503 Need RSET before next transaction\n");
		return 0;
	} else if (state == LMTP_MAIL) {
		fprintf (LmtpOut, "503 Sender not specified\n");
		return 0;
	}
	if (strncasecmp (param, "to:", 3) != 0) {
		fprintf (LmtpOut, "501 Syntax error\n");
		return 0;
	}
	p = parse_address(param+3, &err, &addr, &domain);
	if (p == NULL) {
		fprintf (LmtpOut, "553 %s\n", err);
		return 0;
	}
	if (strchr(addr, '@') == NULL || *domain == '\0') {
		fprintf (LmtpOut, "501 Not a internet domain address\n");
		return 0;
	}
	notify = NULL;
	orcpt = NULL;
	while (*p) {
		if (isspace(*p)) {
			p++;
		}
		else if (strncasecmp("notify=", p, 7) == 0) {
			q = p + 7;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (notify == NULL && p[7] != '\0')
				notify = p + 7;
			p = q +1;
		}
		else if (strncasecmp("orcpt=", p, 6) == 0) {
			q = p + 6;
			while (*q && !isspace(*q))
				q++;
			*q = '\0';
			if (orcpt == NULL && p[6] != '\0')
				orcpt = p + 6;
			p = q +1;
		}
		else {
			fprintf (LmtpOut, "501 parameter problem: %s\n", p);
			return 0;
		}
	}
	if (addrecipient(addr, domain, notify, orcpt) < 0) {
		/* to fallback next mailer (sendmail internal SMTP) */
		fprintf (LmtpOut, "421 too many recipients (no more memory)\n");
		return 0;
	}
	sti.nrcpt++;
	fprintf (LmtpOut, "250 %s Recipient ok\n", addr);
	state = LMTP_DATA;
	return 0;
}

static int
lmtp_data(param)
char *param;
{
	if (state == LMTP_INIT) {
		fprintf (LmtpOut, "503 Polite people say Hello first\n");
		return 0;
	} else if (state == LMTP_SENT) {
		fprintf (LmtpOut, "503 Need RSET before next transaction\n");
		return 0;
	} else if (state == LMTP_MAIL) {
		fprintf (LmtpOut, "503 Sender not specified\n");
		return 0;
	} else if (state != LMTP_DATA) {
		fprintf (LmtpOut, "503 Recipients not specified\n");
		return 0;
	} 
	if (sti.nrcpt < cnf.rcpt_min) {
		/* to fallback next mailer (sendmail internal SMTP) */
		fprintf (LmtpOut, "451 too few recipients\n");
		return 0;
	}
	fprintf (LmtpOut,
		"354 Enter mail, end with \".\" on a line by itself\n");
	fflush (LmtpOut);
	setproctitle("LMTP: reading a message", NULL);
	if (read_message() < 0) {
		/* to fallback next mailer (sendmail internal SMTP) */
		fprintf (LmtpOut, "421 message too large (no more memory)\n");
		return 0;
	}
	if (cnf.data_max > 0 && env.realsize > cnf.data_max) {
		/* to fallback next mailer (sendmail internal SMTP) */
		fprintf (LmtpOut, "421 message size too large\n");
		return 0;
	}
	deliver();		/* delivery with SMTP */
	lmtp_data_response(1);

	logg(LOG_INFO, "nquery=%d/%d nconnect=%d/%d ntimeout=%d ntrans=%d/%d nsent=%d/%d ndeferred=%d nerror=%d+%d dnsmaxcq=%d maxsock=%d tmx=%d tdelivery=%d/%d maxdelay=%d tdelay=%d tsendmail=%d dns_i/o=%d/%d smtp_i/o=%d/%d mem=%d",
		sti.nanswers, sti.nqueries, sti.nquitok, sti.nconnect,
		sti.ntimeout, sti.noktrans, sti.ntrans, sti.nsent, sti.nrcpt,
		sti.ndeferred, sti.nnsfailed, sti.nsmtpfailed,
		sti.maxcquery, sti.maxsock,
		sti.time_mxgot - sti.time_start,
		sti.time_almostfinish - sti.time_mxgot,
		sti.time_finish - sti.time_mxgot,
		sti.max_delay, sti.time_finish - sti.time_start,
		(sti.time_sm_start?(sti.time_start - sti.time_sm_start):-1),
		sti.dns_inb, sti.dns_outb, sti.inbounds, sti.outbounds,
		sti.mem);

	sti.time_sm_start = sti.time_finish;	/* for next transaction */

	state = LMTP_SENT;
	setproctitle("LMTP: waiting next command", NULL);

	signal(SIGINT, sig_quit);
	signal(SIGTERM, sig_quit);

	return 0;
}

static int
lmtp_quit()
{
	fprintf (LmtpOut, "221 %s closing connection\n", myname);
	fclose (LmtpIn);
	fclose (LmtpOut);
	return -1;
}

static int
lmtp_rset()
{
	struct recipient *rcptp, *frcptp;
	struct domain *domp;
	struct message *msgp, *fmsgp;
	char *save_queueid = NULL;
	time_t save_sm_start;

	rcptp = env.rcpt_list;
	while (rcptp != NULL)
	{
		free(rcptp->address);
		if (rcptp->notify != NULL)
			free(rcptp->notify);
		if (rcptp->orcpt != NULL)
			free(rcptp->orcpt);
		/* do not free memory pointed by response */
		frcptp = rcptp;
		rcptp = rcptp->next;
		free(frcptp);
	}
	env.rcpt_list = NULL;
	env.rcpt_tail = NULL;
	env.resp_ptr = NULL;
	for (domp = domain_list; domp != NULL; domp = domp->next)
	{
		/* remove recipient information from domain structure */
		domp->rcpt_top = NULL;
		domp->rcpt_tail = NULL;
	}
	env.work_domain = NULL;
	env.work_dom_ptr = NULL;
	msgp = env.msg;
	while (msgp != NULL)
	{
		free(msgp->data);
		fmsgp = msgp;
		msgp = msgp->next;
		free(fmsgp);
	}
	env.msg = NULL;
	if (env.sender != NULL)
		free(env.sender);
	if (env.envid != NULL)
		free(env.envid);
	if (env.size != NULL)
		free(env.size);
	if (env.body != NULL)
		free(env.body);
	if (env.ret != NULL)
		free(env.ret);
	if (env.mid != NULL)
		free(env.mid);
	if (env.queueid != NULL)
	{
		if (env.keepqueueid)
			save_queueid = env.queueid;
		else
			free(env.queueid);
	}
	bzero(&env, sizeof(env));
	if (save_queueid)
	{
		env.queueid = save_queueid;
		env.keepqueueid = 1;
	}
	save_sm_start = sti.time_sm_start;
	bzero(&sti, sizeof(sti));
	sti.time_sm_start = save_sm_start;

	/* reset configuration */
	cnf.rcpts_trans = save_rcpts_trans;

	fprintf (LmtpOut, "250 Reset state\n");
	state = LMTP_MAIL;

	return 0;
}

static int
lmtp_noop()
{
	fprintf (LmtpOut, "250 OK\n");
	return 0;
}

static int
lmtp_help(param)
char *param;
{
	char **p;
	for (p = help_msg; *p != NULL; p++)
	{
		if (p[1] == NULL)
			fprintf (LmtpOut, "214 %s\n", *p);
		else
			fprintf (LmtpOut, "214-%s\n", *p);
	}
	return 0;
}

static int
lmtp_nocmd()
{
	fprintf (LmtpOut, "500 Command unrecognized\n");
	return 0;
}

void
lmtp_data_response(finished)
int finished;	/* response ahead request if finished == 0 */
{
	struct recipient *rcptp;
	char *msg;

	for (rcptp = env.resp_ptr; rcptp != NULL; rcptp = rcptp->next)
	{
		if (!finished && rcptp->stat != RCPT_DONE)
		{
			env.resp_ptr = rcptp;	/* set top for next checking */
			fflush (LmtpOut);
			return;
		}

		if (rcptp->result >= 100)
		{
			if (strstr(rcptp->response, rcptp->address) == NULL)
			{
				if (cnf.debug & DEBUG_LMTP)
				logg(LOG_DEBUG, "%d %s... %s",
					rcptp->result, rcptp->address,
					rcptp->response);
				fprintf (LmtpOut, "%d %s... %s\n",
					rcptp->result, rcptp->address,
					rcptp->response);
			} else {
				if (cnf.debug & DEBUG_LMTP)
				logg(LOG_DEBUG, "%d %s",
					rcptp->result, rcptp->response);
				fprintf (LmtpOut, "%d %s\n",
					rcptp->result, rcptp->response);
			}
		} else {
			if (cnf.debug & DEBUG_QUERYONLY)
				msg = "Forced SMTP surpression";
			else
				msg = "Service unavailable by system error";

			sti.ndeferred++;
			if (cnf.debug & DEBUG_LMTP)
			logg(LOG_DEBUG, "%d %s... %s",
				SMTP_TEMPFAIL(51), rcptp->address, msg);
			fprintf (LmtpOut, "%d %s... %s\n",
				SMTP_TEMPFAIL(51), rcptp->address, msg);
		}
	}
	if (!finished)
		env.resp_ptr = NULL;	/* set top for next checking */
	fflush (LmtpOut);
}

static int
read_message()
{
	char lmtpbuf[MAXLINE+1], *p, *q, save;
	struct message *msgp = NULL;	/* for warning suppression */
	int len, len2;
	int in_header = 1;
	int skip = 0;
	int in_received = 0;
	int received_hacked = 0;
	int outofmemory = 0;
	int chunk_num = 0;
	int version_inserted = 0;

	env.realsize = 0;

	while (fgets(lmtpbuf, sizeof(lmtpbuf)-1, LmtpIn) != NULL)
	{
		len = strlen(lmtpbuf);
		if (len == 0 || (len > 0 && lmtpbuf[len-1] != '\n'))
		{
			lmtpbuf[len++] = '\r';
			lmtpbuf[len++] = '\n';
			lmtpbuf[len] = '\0';
		}
		else if ((len == 1 && lmtpbuf[0] == '\n')
			|| (len > 1 && lmtpbuf[len-1] == '\n'
				    && lmtpbuf[len-2] != '\r'))
		{
			lmtpbuf[len-1] = '\r';
			lmtpbuf[len++] = '\n';
			lmtpbuf[len] = '\0';
		}
#if 1
		if (lmtpbuf[0] == '.'
		 && (lmtpbuf[1] == '\r' || lmtpbuf[1] == '\n'))
			break;
#endif

		if (lmtpbuf[0] == '\r' || lmtpbuf[0] == '\n')
			in_header = 0;

		if (in_header && skip && isspace(lmtpbuf[0]))
			continue;
		skip = 0;

		/* configuration control by X-smtpfeed: header field */
		if (in_header && strncasecmp(lmtpbuf, "X-smtpfeed:", 11) == 0)
		{
			if (strchr(lmtpbuf + 11, '1') != NULL)
			{
				cnf.rcpts_trans = 1;
			}

			skip = 1;
			continue;
		}

		if (in_received && !isspace(lmtpbuf[0]))
			/* not continuous line */
			in_received = 0;
		if (in_header && !received_hacked
		 && strncasecmp(lmtpbuf, "Received:", 9) == 0)
		{
			received_hacked = 1;
			in_received = 1;
		}
		if (cnf.showversion && in_received && !version_inserted
		 && (p = strstr(lmtpbuf, "by ")) != NULL && isspace(p[-1])
		 && (strlen(lmtpbuf) + strlen(version) < MAXLINE - 2))
		{
			len2 = strlen(version) + 1;
			if ((p = strchr(p, ')')) != NULL)
			{
				bcopy(p, p + len2,
					strlen(lmtpbuf) - (p-lmtpbuf) + 1);
				*p++ = '/';
				strncpy(p, version, len2 - 1);
				len += len2;
				version_inserted = 1;
			}
		}
		if (cnf.parsequeueid && env.queueid == NULL && in_received
		 && (p = strstr(lmtpbuf, "id ")) != NULL && isspace(p[-1]))
		{
			p = p + 3;
			while (*p != '\0' && isspace(*p))
				p++;
			if (*p != '\0')
			{
				q = p + 1;
				while (*q != '\0' && *q != ';'
				    && *q != '\r' && *q != '\n')
					q++;
				if (*q != '\0')
				{
					save = *q;
					*q = '\0';
					env.queueid = newstr(p);
					*q = save;
				}
			}
		}

		if (in_header && strncasecmp(lmtpbuf, "Message-Id:", 11) == 0)
		{
			char save, *p, *q;

			p = lmtpbuf + 11;
			while (*p != '\0' && isspace(*p))
				p++;
			if (*p != '\0')
			{
				q = p + 1;
				while (*q != '\0' && *q != '\r' && *q != '\n')
					q++;
				save = *q;
				*q = '\0';
				env.mid = newstr(p);
				*q = save;
			}
		}

		if (outofmemory)
			continue;

		env.realsize += len;

		if (env.msg == NULL)
		{
			env.msg = msgp = (struct message *)
				MALLOC(sizeof(struct message));
			if (msgp == NULL)
			{
				outofmemory = 1;
				continue;
			}
			bzero(msgp, sizeof(struct message));
		}

		if (msgp->offset + len >= CHUNKSIZE)
		{
			msgp->next = (struct message *)
				MALLOC(sizeof(struct message));
			if (msgp->next == NULL)
			{
				outofmemory = 1;
				continue;
			}
			msgp = msgp->next;
			bzero(msgp, sizeof(struct message));
		}
		if (msgp->data == NULL)
		{
			msgp->data = (char *)MALLOC(CHUNKSIZE);
			if (cnf.debug & DEBUG_LMTP)
			logg(LOG_DEBUG, "new chunk %d", ++chunk_num);
			if (msgp->data == NULL)
			{
				outofmemory = 1;
				continue;
			}
			*(msgp->data) = '\0';
			msgp->offset = 0;
			msgp->next = NULL;
		}
		strcpy (&(msgp->data[msgp->offset]), lmtpbuf);
		msgp->offset += len;
	}

	if (outofmemory)
	{
		logg(LOG_NOTICE, "out of memory (LMTP data buffer allocation)");
		return -1;
	}
	return 0;
}
