/*
 * tnmMibTcl.c --
 *
 *	The Tcl interface to the MIB parser.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * Forward declarations for procedures defined later in this file:
 */

static int	
MibCmd		_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));
static int
LoadFileList	_ANSI_ARGS_((Tcl_Interp *interp, char *fileList));

static int
LoadFile	_ANSI_ARGS_((Tcl_Interp *interp, char *file));

static int
WalkTree	_ANSI_ARGS_((Tcl_Interp *interp, char *name, 
			     char *label, char *body, int exact,
			     Tnm_MibNode* nodePtr, Tnm_Oid *oid, int len));

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibInit --
 *
 *	This procedure registers the mib command in a Tcl interpreter.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_MibInit(interp)
    Tcl_Interp *interp;
{
    Tcl_CreateCommand(interp, "mib", MibCmd, (ClientData *) NULL,
		      (Tcl_CmdDeleteProc *) NULL);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * MibCmd --
 *
 *	This procedure is invoked to process the "mib" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
MibCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int	argc;
    char **argv;
{
    int exact = 0;
    char *cmd, *name, *arg;
    char *result = NULL;
    static int initialized = 0;

    if (argc > 1 && strcmp(argv[1], "-exact") == 0) {
	exact = 1;
	argc--;
	argv++;
    }

    cmd  = argc > 1 ? argv[1] : NULL;
    name = argc > 2 ? argv[2] : NULL;
    arg  = argc > 3 ? argv[3] : NULL;

    if (argc < 2) {
      wrongArgs:
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 "mib ?-exact? option args\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Auto-load the default set of MIB definitions if not initialized
     * yet. This makes use of the global Tcl variable tnm(mibs).
     */

    if (! initialized) {
	char *mibFileList;

	initialized = 1;
	mibFileList = Tcl_GetVar2(interp, "tnm", "mibs:core", TCL_GLOBAL_ONLY);
	if (LoadFileList(interp, mibFileList) != TCL_OK) {
	    return TCL_ERROR;
	}

	if (strcmp(cmd, "load") != 0) {
	    mibFileList = Tcl_GetVar2(interp, "tnm", "mibs", TCL_GLOBAL_ONLY);
	    if (mibFileList == NULL) {
		mibFileList = Tcl_GetVar(interp, "scotty_mibs", 
					 TCL_GLOBAL_ONLY);
	    }
	    if (LoadFileList(interp, mibFileList) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }

    if (strcmp(cmd, "walk") == 0) {
	Tnm_Oid *oid = NULL;
	Tnm_Oid _oid[TNM_OIDMAXLEN];
	int len, code;
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     "mib ?-exact? walk varName label command\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tnm_IsOid(arg)) {

	    int i;
	    oid = Tnm_StrToOid(arg, &len);

	    /*
	     * Make a private copy so that we do not run into problem
	     * if the buffer returned by Tnm_StrToOid() is reused for
	     * other tasks.
	     */

	    for (i = 0; i < len; i++) {
		_oid[i] = oid[i];
	    }
	    oid = _oid;
	}
	code = WalkTree(interp, name, arg, argv[4], exact, NULL, oid, len);
	return (code == TCL_BREAK) ? TCL_OK : code;
    }

    if (strcmp(cmd, "format") == 0) {
	if (argc == 4) {
	    result = Tnm_MibFormat(name, exact, arg);
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"mib ",
			     "?-exact? format name value\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (! result) goto notFound;
        Tcl_SetResult(interp, result, TCL_VOLATILE);
        return TCL_OK;
    }

    if (strcmp(cmd, "scan") == 0) {
	if (argc == 4) {
	    result = Tnm_MibScan(name, exact, arg);
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"mib ",
			     "?-exact? scan name value\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (! result) goto notFound;
        Tcl_SetResult(interp, result, TCL_VOLATILE);
        return TCL_OK;
    }

    if (argc != 3) {
	goto wrongArgs;
    }
    
    if (strcmp(cmd, "load") == 0) {
	return LoadFile(interp, name);

    } else if (strcmp(cmd, "oid") == 0) {
	result = Tnm_MibGetOid(name, exact);

    } else if (strcmp(cmd, "name") == 0) {
	result = Tnm_MibGetName(name, exact);

    } else if (strcmp(cmd, "syntax") == 0) {
	result = Tnm_MibGetSyntax(name, exact);

    } else if (strcmp(cmd, "description") == 0) {
	result = Tnm_MibGetDescription(name, exact);

    } else if (strcmp(cmd, "access") == 0) {
	result = Tnm_MibGetAccess(name, exact);

    } else if (strcmp(cmd, "macro") == 0) {
	result = Tnm_MibGetMacro(name, exact);

    } else if (strcmp(cmd, "module") == 0) {
	result = Tnm_MibGetModule(name, exact);

    } else if (strcmp(cmd, "successor") == 0) {
	result = Tnm_MibGetSucc(name);

    } else if (strcmp(cmd, "tc") == 0) {
	result = Tnm_MibGetTC(name, exact);

    } else if (strcmp(cmd, "file") == 0) {
	result = Tnm_MibGetFile(name, exact);

    } else if (strcmp(cmd, "index") == 0) {
	result = Tnm_MibGetIndex(name, exact);

    } else if (strcmp(cmd, "parent") == 0) {
	result = Tnm_MibGetParent(name, exact);

    } else if (strcmp(cmd, "defval") == 0) {
	result = Tnm_MibGetDefault(name, exact);

    } else {

	Tcl_AppendResult(interp, "bad option \"", cmd, "\": should be ",
			 "access, description, format, index, load, macro, ",
			 "module, name, oid, scan, successor, syntax, tc, ",
			 "or walk", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (! result) goto notFound;
    
    Tcl_SetResult(interp, result, TCL_VOLATILE);
    return TCL_OK;
    
notFound:
    Tcl_AppendResult(interp, "no object \"", name, "\"", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * LoadFileList --
 *
 *	This procedure reads MIB definitions from a list of files and
 *	adds the objects to the internal MIB tree.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
LoadFileList(interp, fileList)
    Tcl_Interp *interp;
    char *fileList;
{
    int i, code, argc;
    char **argv;

    if (fileList == NULL) {
	return TCL_OK;
    }

    code = Tcl_SplitList(interp, fileList, &argc, &argv);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < argc; i++) {
	if (LoadFile(interp, argv[i]) != TCL_OK) {
	    ckfree((char *) argv);
	    return TCL_ERROR;
	}
    }
    ckfree((char *) argv);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * LoadFile --
 *
 *	This procedure reads MIB definitions from a file and adds the
 *	objects to the internal MIB tree. This function expands ~
 *	filenames and searches $tnm(library)/site and $tnm(library)/mibs
 *	for the given filename. The frozen image of MIB definitions is 
 *	written into a machine specific directory.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	New frozen MIB files may be created.
 *
 *----------------------------------------------------------------------
 */

static int
LoadFile(interp, file)
    Tcl_Interp *interp;
    char *file;
{
    Tcl_DString fileBuffer, frozenFileBuffer;
    char *library, *cache, *arch, *fileName, *frozenFileName = NULL;
    char **fileArgv = NULL;
    int fileArgc;
    int code = TCL_OK;

    Tcl_DStringInit(&fileBuffer);
    Tcl_DStringInit(&frozenFileBuffer);

    /*
     * Split the file argument into a path and the file name. Also,
     * get the Tnm library and cache paths and the architecture string.
     */

    Tcl_SplitPath(file, &fileArgc, &fileArgv);

    library = Tcl_GetVar2(interp, "tnm", "library", TCL_GLOBAL_ONLY);
    cache = Tcl_GetVar2(interp, "tnm", "cache", TCL_GLOBAL_ONLY);
    arch = Tcl_GetVar2(interp, "tnm", "arch", TCL_GLOBAL_ONLY);

    /* 
     * Check if we can write a frozen file. Construct the path to the
     * directory where we keep frozen files. Create a machine specific
     * sub-directory if needed. The result of the following block is
     * the native frozen file name or NULL if we can't write a frozen
     * file.
     */

    if (cache != NULL && arch != NULL) {

	char *path;
	Tcl_DString dst;
	
	Tcl_DStringInit(&dst);
	Tcl_DStringAppend(&dst, cache, -1);
	Tcl_DStringAppend(&dst, "/", 1);
	path = Tcl_DStringAppend(&dst, arch, -1);
	(void) TnmMkDir(interp, path);
	Tcl_DStringAppend(&dst, "/", 1);
	Tcl_DStringAppend(&dst, fileArgv[fileArgc-1], -1);
	path = Tcl_DStringAppend(&dst, ".idy", 4);
	frozenFileName = Tcl_TranslateFileName(interp, path,
					       &frozenFileBuffer);
	Tcl_DStringFree(&dst);
    }

    /* 
     * Search for the MIB file we are trying to load. First try the
     * file argument translated into the platform specific format. If
     * not found, check $tnm(library)/site and $tnm(library)/mibs.
     */

    fileName = Tcl_TranslateFileName(interp, file, &fileBuffer);
    if (! fileName) {
	code = TCL_ERROR;
	goto exit;
    }

    if (access(fileName, R_OK) != 0) {

	char *path;
	Tcl_DString dst;

	/*
	 * Copy the whole library path, append "site" or "mibs" and
	 * finally append the file name and path. This makes sure
	 * that relative paths like "sun/SUN-MIB" will be found in
	 * $tnm(library)/site/sun/SUN-MIB.
	 */

	Tcl_DStringInit(&dst);
	Tcl_DStringAppend(&dst, library, -1);
	Tcl_DStringAppend(&dst, "/site/", 6);
	path = Tcl_DStringAppend(&dst, file, -1);
	fileName = Tcl_TranslateFileName(interp, path, &fileBuffer);
	if (fileName && access(fileName, R_OK) != 0) {
	    Tcl_DStringFree(&fileBuffer);
	    Tcl_DStringFree(&dst);
	    Tcl_DStringAppend(&dst, library, -1);
	    Tcl_DStringAppend(&dst, "/mibs/", 6);
	    path = Tcl_DStringAppend(&dst, file, -1);
	    fileName = Tcl_TranslateFileName(interp, path, &fileBuffer);
	}
	if (fileName && access(fileName, R_OK) != 0) {
	    Tcl_DStringFree(&fileBuffer);
	    fileName = NULL;
	}
	Tcl_DStringFree(&dst);
    }

    /*
     * If we have the file name now, call the parser to do its job.
     */

    if (fileName) {
	tnm_MibTree = Tnm_MibParse(fileName, frozenFileName, tnm_MibTree);
	if (tnm_MibTree == NULL) {
	    Tcl_AppendResult(interp, "couldn't parse MIB file \"",
			     fileName,"\"", (char *) NULL);
	    code = TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "couldn't open MIB file \"", file,
			 "\": ", Tcl_PosixError(interp), (char *) NULL);
	code = TCL_ERROR;
    }

 exit:
    /* 
     * Free up all the memory that we have allocated.
     */

    Tcl_DStringFree(&fileBuffer);
    Tcl_DStringFree(&frozenFileBuffer);
    if (fileArgv) {
	ckfree((char *) fileArgv);
    }
    return code;
}

/*
 *----------------------------------------------------------------------
 *
 * WalkTree --
 *
 *	This procedure implements a recursive MIB walk. The varName 
 *	argument defines the Tcl variable used to identify the current
 *	MIB node and label identifies the root node of the sub-tree.
 *	The current position in the MIB tree is given by nodePtr.
 *
 *	The oid and len arguments are optional and only used when the
 *	label is an object identifier and not a name. In this case, we
 *	assemble the current path in the tree in the buffer pointed to
 *	by oid. Note, that the caller of WalkTree() must ensure that the
 *	buffer is large enough and not shared by others.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
WalkTree(interp, varName, label, body, exact, nodePtr, oid, len)
    Tcl_Interp *interp;
    char *varName;
    char *label;
    char *body;
    int exact;
    Tnm_MibNode* nodePtr;
    Tnm_Oid *oid;
    int len;
{
    int doall = 1;
    int result = TCL_OK;

    if (! nodePtr) {
	nodePtr = Tnm_MibFindNode(label, NULL, exact);
	if (!nodePtr) {
	    Tcl_AppendResult(interp, "no object \"", label, "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	doall = 0;
    }

    while (nodePtr) {

	char *val, *label;

	if (doall && oid) {
	    oid[len-1] = nodePtr->subid;
	}

	label = nodePtr->label;

	if (oid) {
	    label = Tnm_OidToStr(oid, len);
	}
	
	val = Tcl_SetVar(interp, varName, label, TCL_LEAVE_ERR_MSG);	
	if (! val) {
	    result = TCL_ERROR;
	    goto loopDone;
	}
	
	result = Tcl_Eval(interp, body);

        if ((result == TCL_OK || result == TCL_CONTINUE) 
	    && nodePtr->childPtr) {
	    result = WalkTree(interp, varName, label, body, exact, 
			      nodePtr->childPtr, oid, len+1);
	}

	if (result != TCL_OK) {
	    if (result == TCL_CONTINUE) {
		result = TCL_OK;
	    } else if (result == TCL_BREAK) {
		goto loopDone;
	    } else if (result == TCL_ERROR) {
		char msg[100];
		sprintf(msg, "\n    (\"mib walk\" body line %d)",
			interp->errorLine);
		Tcl_AddErrorInfo(interp, msg);
		goto loopDone;
	    } else {
		goto loopDone;
	    }
	}

	nodePtr = doall ? nodePtr->nextPtr : NULL;
    }

  loopDone:
    if (result == TCL_OK) {
	Tcl_ResetResult(interp);
    }
    return result;
}
