// SciTE - Scintilla based Text Editor
/** @file IFaceTable.cxx
 ** SciTE iface function and constant descriptors.
 **/
// Copyright 1998-2004 by Neil Hodgson <neilh@scintilla.org>
// The License.txt file describes the conditions under which this software may be distributed.

#include "IFaceTable.h"
#include <string.h>
#include <ctype.h>

int IFaceTable::FindConstant(const char *name) {
	int lo = 0;
	int hi = IFaceTable::constantCount - 1;
	do {
		int idx = (lo+hi)/2;
		int cmp = strcmp(name, constants[idx].name);

		if (cmp > 0) {
			lo = idx + 1;
		} else if (cmp < 0) {
			hi = idx - 1;
		} else {
			return idx;
		}
	} while (lo <= hi);

	return -1;
}

int IFaceTable::FindFunction(const char *name) {
	int lo = 0;
	int hi = IFaceTable::functionCount - 1;
	do {
		int idx = (lo+hi)/2;
		int cmp = strcmp(name, functions[idx].name);
		if (cmp > 0) {
			lo = idx + 1;
		} else if (cmp < 0) {
			hi = idx - 1;
		} else {
			return idx;
		}
	} while (lo <= hi);

	return -1;
}

int IFaceTable::FindFunctionByConstantName(const char *name) {
	if (strncmp(name, "SCI_", 4)==0) {
		// This looks like a constant for an iface function.  This requires
		// a sequential search.  Take special care since the function names
		// are mixed case, whereas the constants are all-caps.

		for (int idx = 0; idx < functionCount; ++idx) {
			const char *nm = name+4;
			const char *fn = IFaceTable::functions[idx].name;
			while (*nm && *fn && (*nm == toupper(*fn))) {
				++nm;
				++fn;
			}
			if (!*nm && !*fn) {
				return idx;
			}
		}
	}
	return -1;
}

int IFaceTable::FindProperty(const char *name) {
	int lo = 0;
	int hi = IFaceTable::propertyCount - 1;
	do {
		int idx = (lo+hi)/2;
		int cmp = strcmp(name, properties[idx].name);

		if (cmp > 0) {
			lo = idx + 1;
		} else if (cmp < 0) {
			hi = idx - 1;
		} else {
			return idx;
		}
	} while (lo <= hi);

	return -1;
}

int IFaceTable::GetConstantName(int value, char *nameOut, unsigned nameBufferLen) {
	if (nameOut && nameBufferLen > 0) {
		*nameOut = '\0';
	}

	// Look in both the constants table and the functions table.  Start with functions.
	for (int funcIdx = 0; funcIdx < functionCount; ++funcIdx) {
		if (functions[funcIdx].value == value) {
			int len = static_cast<int>(strlen(functions[funcIdx].name)) + 4;
			if (nameOut && (static_cast<int>(nameBufferLen) > len)) {
				strcpy(nameOut, "SCI_");
				strcat(nameOut, functions[funcIdx].name);
				// fix case
				for (char *nm = nameOut + 4; *nm; ++nm) {
					if (*nm >= 'a' && *nm <= 'z') {
						*nm = static_cast<char>(*nm - 'a' + 'A');
					}
				}
				return len;
			} else {
				return -1 - len;
			}
		}
	}

	for (int constIdx = 0; constIdx < constantCount; ++constIdx) {
		if (constants[constIdx].value == value) {
			int len = static_cast<int>(strlen(constants[constIdx].name));
			if (nameOut && (static_cast<int>(nameBufferLen) > len)) {
				strcpy(nameOut, constants[constIdx].name);
				return len;
			} else {
				return -1 - len;
			}
		}
	}

	return 0;
}

#if defined(__GNUC__)
#if (__GNUC__ > 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 7))
#pragma GCC diagnostic ignored "-Wnarrowing"
#endif
#endif

//++Autogenerated -- start of section automatically generated from Scintilla.iface

static IFaceConstant ifaceConstants[] = {
	{"ANNOTATION_BOXED",2},
	{"ANNOTATION_HIDDEN",0},
	{"ANNOTATION_STANDARD",1},
	{"CARETSTYLE_BLOCK",2},
	{"CARETSTYLE_INVISIBLE",0},
	{"CARETSTYLE_LINE",1},
	{"CARET_EVEN",0x08},
	{"CARET_JUMPS",0x10},
	{"CARET_SLOP",0x01},
	{"CARET_STRICT",0x04},
	{"EDGE_BACKGROUND",2},
	{"EDGE_LINE",1},
	{"EDGE_NONE",0},
	{"IDM_ABBREV",242},
	{"IDM_ABOUT",902},
	{"IDM_ACTIVATE",320},
	{"IDM_ALLOWACCESS",119},
	{"IDM_BLOCK_COMMENT",243},
	{"IDM_BOOKMARK_CLEARALL",224},
	{"IDM_BOOKMARK_NEXT",221},
	{"IDM_BOOKMARK_NEXT_SELECT",225},
	{"IDM_BOOKMARK_PREV",223},
	{"IDM_BOOKMARK_PREV_SELECT",226},
	{"IDM_BOOKMARK_TOGGLE",222},
	{"IDM_BOX_COMMENT",246},
	{"IDM_BUFFER",1200},
	{"IDM_BUFFERSEP",505},
	{"IDM_BUILD",302},
	{"IDM_CLEAR",206},
	{"IDM_CLEAROUTPUT",420},
	{"IDM_CLOSE",105},
	{"IDM_CLOSEALL",503},
	{"IDM_COMPILE",301},
	{"IDM_COMPLETE",233},
	{"IDM_COMPLETEWORD",234},
	{"IDM_COPY",204},
	{"IDM_COPYASRTF",245},
	{"IDM_COPYPATH",118},
	{"IDM_CUT",203},
	{"IDM_DIRECTIONDOWN",806},
	{"IDM_DIRECTIONUP",805},
	{"IDM_DUPLICATE",250},
	{"IDM_ENCODING_DEFAULT",150},
	{"IDM_ENCODING_UCOOKIE",154},
	{"IDM_ENCODING_UCS2BE",151},
	{"IDM_ENCODING_UCS2LE",152},
	{"IDM_ENCODING_UTF8",153},
	{"IDM_ENTERSELECTION",256},
	{"IDM_EOL_CONVERT",433},
	{"IDM_EOL_CR",431},
	{"IDM_EOL_CRLF",430},
	{"IDM_EOL_LF",432},
	{"IDM_EXPAND",235},
	{"IDM_EXPAND_ENSURECHILDRENVISIBLE",238},
	{"IDM_FILER",114},
	{"IDM_FIND",210},
	{"IDM_FINDINFILES",215},
	{"IDM_FINDNEXT",211},
	{"IDM_FINDNEXTBACK",212},
	{"IDM_FINDNEXTBACKSEL",214},
	{"IDM_FINDNEXTSEL",213},
	{"IDM_FINISHEDEXECUTE",305},
	{"IDM_FOLDMARGIN",406},
	{"IDM_FULLSCREEN",961},
	{"IDM_GO",303},
	{"IDM_GOTO",220},
	{"IDM_HELP",901},
	{"IDM_HELP_SCITE",903},
	{"IDM_IMPORT",1300},
	{"IDM_INCSEARCH",252},
	{"IDM_INS_ABBREV",247},
	{"IDM_JOIN",248},
	{"IDM_LANGUAGE",1400},
	{"IDM_LINENUMBERMARGIN",407},
	{"IDM_LOADSESSION",132},
	{"IDM_LWRCASE",241},
	{"IDM_MACROLIST",314},
	{"IDM_MACROPLAY",313},
	{"IDM_MACRORECORD",311},
	{"IDM_MACROSTOPRECORD",312},
	{"IDM_MACRO_SEP",310},
	{"IDM_MATCHBRACE",230},
	{"IDM_MATCHCASE",801},
	{"IDM_MONOFONT",450},
	{"IDM_MOVETABLEFT",509},
	{"IDM_MOVETABRIGHT",508},
	{"IDM_MRUFILE",1000},
	{"IDM_MRU_SEP",120},
	{"IDM_MRU_SUB",121},
	{"IDM_NEW",101},
	{"IDM_NEXTFILE",502},
	{"IDM_NEXTFILESTACK",507},
	{"IDM_NEXTMATCHPPC",262},
	{"IDM_NEXTMSG",306},
	{"IDM_ONTOP",960},
	{"IDM_OPEN",102},
	{"IDM_OPENABBREVPROPERTIES",463},
	{"IDM_OPENDIRECTORYPROPERTIES",465},
	{"IDM_OPENFILESHERE",413},
	{"IDM_OPENGLOBALPROPERTIES",462},
	{"IDM_OPENLOCALPROPERTIES",460},
	{"IDM_OPENLUAEXTERNALFILE",464},
	{"IDM_OPENSELECTED",103},
	{"IDM_OPENUSERPROPERTIES",461},
	{"IDM_PASTE",205},
	{"IDM_PASTEANDDOWN",208},
	{"IDM_PREVFILE",501},
	{"IDM_PREVFILESTACK",506},
	{"IDM_PREVMATCHPPC",260},
	{"IDM_PREVMSG",307},
	{"IDM_PRINT",131},
	{"IDM_PRINTSETUP",130},
	{"IDM_QUIT",140},
	{"IDM_READONLY",416},
	{"IDM_REDO",202},
	{"IDM_REGEXP",802},
	{"IDM_REPLACE",216},
	{"IDM_REVERT",104},
	{"IDM_RUNWIN",351},
	{"IDM_SAVE",106},
	{"IDM_SAVEACOPY",116},
	{"IDM_SAVEALL",504},
	{"IDM_SAVEAS",110},
	{"IDM_SAVEASHTML",111},
	{"IDM_SAVEASPDF",113},
	{"IDM_SAVEASRTF",112},
	{"IDM_SAVEASTEX",115},
	{"IDM_SAVEASXML",117},
	{"IDM_SAVESESSION",133},
	{"IDM_SELECTALL",207},
	{"IDM_SELECTION_FOR_FIND",217},
	{"IDM_SELECTTOBRACE",231},
	{"IDM_SELECTTONEXTMATCHPPC",263},
	{"IDM_SELECTTOPREVMATCHPPC",261},
	{"IDM_SELMARGIN",405},
	{"IDM_SHOWCALLTIP",232},
	{"IDM_SPLIT",249},
	{"IDM_SPLITVERTICAL",401},
	{"IDM_SRCWIN",350},
	{"IDM_STATUSWIN",353},
	{"IDM_STOPEXECUTE",304},
	{"IDM_STREAM_COMMENT",244},
	{"IDM_SWITCHPANE",421},
	{"IDM_TABSIZE",440},
	{"IDM_TABWIN",354},
	{"IDM_TOGGLEOUTPUT",409},
	{"IDM_TOGGLEPARAMETERS",412},
	{"IDM_TOGGLE_FOLDALL",236},
	{"IDM_TOGGLE_FOLDRECURSIVE",237},
	{"IDM_TOOLS",1100},
	{"IDM_TOOLWIN",352},
	{"IDM_UNDO",201},
	{"IDM_UNSLASH",804},
	{"IDM_UPRCASE",240},
	{"IDM_VIEWEOL",403},
	{"IDM_VIEWGUIDES",404},
	{"IDM_VIEWSPACE",402},
	{"IDM_VIEWSTATUSBAR",411},
	{"IDM_VIEWTABBAR",410},
	{"IDM_VIEWTOOLBAR",408},
	{"IDM_WHOLEWORD",800},
	{"IDM_WRAP",414},
	{"IDM_WRAPAROUND",803},
	{"IDM_WRAPOUTPUT",415},
	{"INDIC0_MASK",0x20},
	{"INDIC1_MASK",0x40},
	{"INDIC2_MASK",0x80},
	{"INDICS_MASK",0xE0},
	{"INDIC_BOX",6},
	{"INDIC_COMPOSITIONTHICK",14},
	{"INDIC_CONTAINER",8},
	{"INDIC_DASH",9},
	{"INDIC_DIAGONAL",3},
	{"INDIC_DOTBOX",12},
	{"INDIC_DOTS",10},
	{"INDIC_HIDDEN",5},
	{"INDIC_MAX",31},
	{"INDIC_PLAIN",0},
	{"INDIC_ROUNDBOX",7},
	{"INDIC_SQUIGGLE",1},
	{"INDIC_SQUIGGLELOW",11},
	{"INDIC_SQUIGGLEPIXMAP",13},
	{"INDIC_STRAIGHTBOX",8},
	{"INDIC_STRIKE",4},
	{"INDIC_TT",2},
	{"INVALID_POSITION",-1},
	{"KEYWORDSET_MAX",8},
	{"MARKER_MAX",31},
	{"SCEN_CHANGE",768},
	{"SCEN_KILLFOCUS",256},
	{"SCEN_SETFOCUS",512},
	{"SCE_4GL_BLOCK",8},
	{"SCE_4GL_BLOCK_",24},
	{"SCE_4GL_CHARACTER",4},
	{"SCE_4GL_CHARACTER_",20},
	{"SCE_4GL_COMMENT1",10},
	{"SCE_4GL_COMMENT1_",26},
	{"SCE_4GL_COMMENT2",11},
	{"SCE_4GL_COMMENT2_",27},
	{"SCE_4GL_COMMENT3",12},
	{"SCE_4GL_COMMENT3_",28},
	{"SCE_4GL_COMMENT4",13},
	{"SCE_4GL_COMMENT4_",29},
	{"SCE_4GL_COMMENT5",14},
	{"SCE_4GL_COMMENT5_",30},
	{"SCE_4GL_COMMENT6",15},
	{"SCE_4GL_COMMENT6_",31},
	{"SCE_4GL_DEFAULT",0},
	{"SCE_4GL_DEFAULT_",16},
	{"SCE_4GL_END",9},
	{"SCE_4GL_END_",25},
	{"SCE_4GL_IDENTIFIER",7},
	{"SCE_4GL_IDENTIFIER_",23},
	{"SCE_4GL_NUMBER",1},
	{"SCE_4GL_NUMBER_",17},
	{"SCE_4GL_OPERATOR",6},
	{"SCE_4GL_OPERATOR_",22},
	{"SCE_4GL_PREPROCESSOR",5},
	{"SCE_4GL_PREPROCESSOR_",21},
	{"SCE_4GL_STRING",3},
	{"SCE_4GL_STRING_",19},
	{"SCE_4GL_WORD",2},
	{"SCE_4GL_WORD_",18},
	{"SCE_A68K_COMMENT",1},
	{"SCE_A68K_COMMENT_DOXYGEN",18},
	{"SCE_A68K_COMMENT_SPECIAL",17},
	{"SCE_A68K_COMMENT_WORD",16},
	{"SCE_A68K_CPUINSTRUCTION",7},
	{"SCE_A68K_DEFAULT",0},
	{"SCE_A68K_DIRECTIVE",10},
	{"SCE_A68K_EXTINSTRUCTION",8},
	{"SCE_A68K_IDENTIFIER",14},
	{"SCE_A68K_LABEL",12},
	{"SCE_A68K_MACRO_ARG",11},
	{"SCE_A68K_MACRO_DECLARATION",15},
	{"SCE_A68K_NUMBER_BIN",3},
	{"SCE_A68K_NUMBER_DEC",2},
	{"SCE_A68K_NUMBER_HEX",4},
	{"SCE_A68K_OPERATOR",6},
	{"SCE_A68K_REGISTER",9},
	{"SCE_A68K_STRING1",5},
	{"SCE_A68K_STRING2",13},
	{"SCE_ABAQUS_ARGUMENT",11},
	{"SCE_ABAQUS_COMMAND",8},
	{"SCE_ABAQUS_COMMENT",1},
	{"SCE_ABAQUS_COMMENTBLOCK",2},
	{"SCE_ABAQUS_DEFAULT",0},
	{"SCE_ABAQUS_FUNCTION",12},
	{"SCE_ABAQUS_NUMBER",3},
	{"SCE_ABAQUS_OPERATOR",5},
	{"SCE_ABAQUS_PROCESSOR",7},
	{"SCE_ABAQUS_SLASHCOMMAND",9},
	{"SCE_ABAQUS_STARCOMMAND",10},
	{"SCE_ABAQUS_STRING",4},
	{"SCE_ABAQUS_WORD",6},
	{"SCE_ADA_CHARACTER",5},
	{"SCE_ADA_CHARACTEREOL",6},
	{"SCE_ADA_COMMENTLINE",10},
	{"SCE_ADA_DEFAULT",0},
	{"SCE_ADA_DELIMITER",4},
	{"SCE_ADA_IDENTIFIER",2},
	{"SCE_ADA_ILLEGAL",11},
	{"SCE_ADA_LABEL",9},
	{"SCE_ADA_NUMBER",3},
	{"SCE_ADA_STRING",7},
	{"SCE_ADA_STRINGEOL",8},
	{"SCE_ADA_WORD",1},
	{"SCE_APDL_ARGUMENT",11},
	{"SCE_APDL_COMMAND",8},
	{"SCE_APDL_COMMENT",1},
	{"SCE_APDL_COMMENTBLOCK",2},
	{"SCE_APDL_DEFAULT",0},
	{"SCE_APDL_FUNCTION",12},
	{"SCE_APDL_NUMBER",3},
	{"SCE_APDL_OPERATOR",5},
	{"SCE_APDL_PROCESSOR",7},
	{"SCE_APDL_SLASHCOMMAND",9},
	{"SCE_APDL_STARCOMMAND",10},
	{"SCE_APDL_STRING",4},
	{"SCE_APDL_WORD",6},
	{"SCE_ASM_CHARACTER",12},
	{"SCE_ASM_COMMENT",1},
	{"SCE_ASM_COMMENTBLOCK",11},
	{"SCE_ASM_COMMENTDIRECTIVE",15},
	{"SCE_ASM_CPUINSTRUCTION",6},
	{"SCE_ASM_DEFAULT",0},
	{"SCE_ASM_DIRECTIVE",9},
	{"SCE_ASM_DIRECTIVEOPERAND",10},
	{"SCE_ASM_EXTINSTRUCTION",14},
	{"SCE_ASM_IDENTIFIER",5},
	{"SCE_ASM_MATHINSTRUCTION",7},
	{"SCE_ASM_NUMBER",2},
	{"SCE_ASM_OPERATOR",4},
	{"SCE_ASM_REGISTER",8},
	{"SCE_ASM_STRING",3},
	{"SCE_ASM_STRINGEOL",13},
	{"SCE_ASN1_ATTRIBUTE",7},
	{"SCE_ASN1_COMMENT",1},
	{"SCE_ASN1_DEFAULT",0},
	{"SCE_ASN1_DESCRIPTOR",8},
	{"SCE_ASN1_IDENTIFIER",2},
	{"SCE_ASN1_KEYWORD",6},
	{"SCE_ASN1_OID",4},
	{"SCE_ASN1_OPERATOR",10},
	{"SCE_ASN1_SCALAR",5},
	{"SCE_ASN1_STRING",3},
	{"SCE_ASN1_TYPE",9},
	{"SCE_ASY_CHARACTER",6},
	{"SCE_ASY_COMMENT",1},
	{"SCE_ASY_COMMENTLINE",2},
	{"SCE_ASY_COMMENTLINEDOC",10},
	{"SCE_ASY_DEFAULT",0},
	{"SCE_ASY_IDENTIFIER",8},
	{"SCE_ASY_NUMBER",3},
	{"SCE_ASY_OPERATOR",7},
	{"SCE_ASY_STRING",5},
	{"SCE_ASY_STRINGEOL",9},
	{"SCE_ASY_WORD",4},
	{"SCE_ASY_WORD2",11},
	{"SCE_AU3_COMMENT",1},
	{"SCE_AU3_COMMENTBLOCK",2},
	{"SCE_AU3_COMOBJ",14},
	{"SCE_AU3_DEFAULT",0},
	{"SCE_AU3_EXPAND",13},
	{"SCE_AU3_FUNCTION",4},
	{"SCE_AU3_KEYWORD",5},
	{"SCE_AU3_MACRO",6},
	{"SCE_AU3_NUMBER",3},
	{"SCE_AU3_OPERATOR",8},
	{"SCE_AU3_PREPROCESSOR",11},
	{"SCE_AU3_SENT",10},
	{"SCE_AU3_SPECIAL",12},
	{"SCE_AU3_STRING",7},
	{"SCE_AU3_UDF",15},
	{"SCE_AU3_VARIABLE",9},
	{"SCE_AVE_COMMENT",1},
	{"SCE_AVE_DEFAULT",0},
	{"SCE_AVE_ENUM",7},
	{"SCE_AVE_IDENTIFIER",9},
	{"SCE_AVE_NUMBER",2},
	{"SCE_AVE_OPERATOR",10},
	{"SCE_AVE_STRING",6},
	{"SCE_AVE_STRINGEOL",8},
	{"SCE_AVE_WORD",3},
	{"SCE_AVE_WORD1",11},
	{"SCE_AVE_WORD2",12},
	{"SCE_AVE_WORD3",13},
	{"SCE_AVE_WORD4",14},
	{"SCE_AVE_WORD5",15},
	{"SCE_AVE_WORD6",16},
	{"SCE_AVS_CLIPPROP",13},
	{"SCE_AVS_COMMENTBLOCK",1},
	{"SCE_AVS_COMMENTBLOCKN",2},
	{"SCE_AVS_COMMENTLINE",3},
	{"SCE_AVS_DEFAULT",0},
	{"SCE_AVS_FILTER",10},
	{"SCE_AVS_FUNCTION",12},
	{"SCE_AVS_IDENTIFIER",6},
	{"SCE_AVS_KEYWORD",9},
	{"SCE_AVS_NUMBER",4},
	{"SCE_AVS_OPERATOR",5},
	{"SCE_AVS_PLUGIN",11},
	{"SCE_AVS_STRING",7},
	{"SCE_AVS_TRIPLESTRING",8},
	{"SCE_AVS_USERDFN",14},
	{"SCE_BAAN_COMMENT",1},
	{"SCE_BAAN_COMMENTDOC",2},
	{"SCE_BAAN_DEFAULT",0},
	{"SCE_BAAN_IDENTIFIER",8},
	{"SCE_BAAN_NUMBER",3},
	{"SCE_BAAN_OPERATOR",7},
	{"SCE_BAAN_PREPROCESSOR",6},
	{"SCE_BAAN_STRING",5},
	{"SCE_BAAN_STRINGEOL",9},
	{"SCE_BAAN_WORD",4},
	{"SCE_BAAN_WORD2",10},
	{"SCE_BAT_COMMAND",5},
	{"SCE_BAT_COMMENT",1},
	{"SCE_BAT_DEFAULT",0},
	{"SCE_BAT_HIDE",4},
	{"SCE_BAT_IDENTIFIER",6},
	{"SCE_BAT_LABEL",3},
	{"SCE_BAT_OPERATOR",7},
	{"SCE_BAT_WORD",2},
	{"SCE_B_ASM",14},
	{"SCE_B_BINNUMBER",18},
	{"SCE_B_COMMENT",1},
	{"SCE_B_COMMENTBLOCK",19},
	{"SCE_B_CONSTANT",13},
	{"SCE_B_DATE",8},
	{"SCE_B_DEFAULT",0},
	{"SCE_B_DOCBLOCK",21},
	{"SCE_B_DOCKEYWORD",22},
	{"SCE_B_DOCLINE",20},
	{"SCE_B_ERROR",16},
	{"SCE_B_HEXNUMBER",17},
	{"SCE_B_IDENTIFIER",7},
	{"SCE_B_KEYWORD",3},
	{"SCE_B_KEYWORD2",10},
	{"SCE_B_KEYWORD3",11},
	{"SCE_B_KEYWORD4",12},
	{"SCE_B_LABEL",15},
	{"SCE_B_NUMBER",2},
	{"SCE_B_OPERATOR",6},
	{"SCE_B_PREPROCESSOR",5},
	{"SCE_B_STRING",4},
	{"SCE_B_STRINGEOL",9},
	{"SCE_CAML_CHAR",9},
	{"SCE_CAML_COMMENT",12},
	{"SCE_CAML_COMMENT1",13},
	{"SCE_CAML_COMMENT2",14},
	{"SCE_CAML_COMMENT3",15},
	{"SCE_CAML_DEFAULT",0},
	{"SCE_CAML_IDENTIFIER",1},
	{"SCE_CAML_KEYWORD",3},
	{"SCE_CAML_KEYWORD2",4},
	{"SCE_CAML_KEYWORD3",5},
	{"SCE_CAML_LINENUM",6},
	{"SCE_CAML_NUMBER",8},
	{"SCE_CAML_OPERATOR",7},
	{"SCE_CAML_STRING",11},
	{"SCE_CAML_TAGNAME",2},
	{"SCE_CAML_WHITE",10},
	{"SCE_CLW_ATTRIBUTE",13},
	{"SCE_CLW_BUILTIN_PROCEDURES_FUNCTION",11},
	{"SCE_CLW_COMMENT",2},
	{"SCE_CLW_COMPILER_DIRECTIVE",9},
	{"SCE_CLW_DEFAULT",0},
	{"SCE_CLW_DEPRECATED",16},
	{"SCE_CLW_ERROR",15},
	{"SCE_CLW_INTEGER_CONSTANT",5},
	{"SCE_CLW_KEYWORD",8},
	{"SCE_CLW_LABEL",1},
	{"SCE_CLW_PICTURE_STRING",7},
	{"SCE_CLW_REAL_CONSTANT",6},
	{"SCE_CLW_RUNTIME_EXPRESSIONS",10},
	{"SCE_CLW_STANDARD_EQUATE",14},
	{"SCE_CLW_STRING",3},
	{"SCE_CLW_STRUCTURE_DATA_TYPE",12},
	{"SCE_CLW_USER_IDENTIFIER",4},
	{"SCE_CMAKE_COMMANDS",5},
	{"SCE_CMAKE_COMMENT",1},
	{"SCE_CMAKE_DEFAULT",0},
	{"SCE_CMAKE_FOREACHDEF",10},
	{"SCE_CMAKE_IFDEFINEDEF",11},
	{"SCE_CMAKE_MACRODEF",12},
	{"SCE_CMAKE_NUMBER",14},
	{"SCE_CMAKE_PARAMETERS",6},
	{"SCE_CMAKE_STRINGDQ",2},
	{"SCE_CMAKE_STRINGLQ",3},
	{"SCE_CMAKE_STRINGRQ",4},
	{"SCE_CMAKE_STRINGVAR",13},
	{"SCE_CMAKE_USERDEFINED",8},
	{"SCE_CMAKE_VARIABLE",7},
	{"SCE_CMAKE_WHILEDEF",9},
	{"SCE_COFFEESCRIPT_CHARACTER",7},
	{"SCE_COFFEESCRIPT_COMMENT",1},
	{"SCE_COFFEESCRIPT_COMMENTBLOCK",22},
	{"SCE_COFFEESCRIPT_COMMENTDOC",3},
	{"SCE_COFFEESCRIPT_COMMENTDOCKEYWORD",17},
	{"SCE_COFFEESCRIPT_COMMENTDOCKEYWORDERROR",18},
	{"SCE_COFFEESCRIPT_COMMENTLINE",2},
	{"SCE_COFFEESCRIPT_COMMENTLINEDOC",15},
	{"SCE_COFFEESCRIPT_DEFAULT",0},
	{"SCE_COFFEESCRIPT_GLOBALCLASS",19},
	{"SCE_COFFEESCRIPT_IDENTIFIER",11},
	{"SCE_COFFEESCRIPT_NUMBER",4},
	{"SCE_COFFEESCRIPT_OPERATOR",10},
	{"SCE_COFFEESCRIPT_PREPROCESSOR",9},
	{"SCE_COFFEESCRIPT_REGEX",14},
	{"SCE_COFFEESCRIPT_STRING",6},
	{"SCE_COFFEESCRIPT_STRINGEOL",12},
	{"SCE_COFFEESCRIPT_STRINGRAW",20},
	{"SCE_COFFEESCRIPT_TRIPLEVERBATIM",21},
	{"SCE_COFFEESCRIPT_UUID",8},
	{"SCE_COFFEESCRIPT_VERBATIM",13},
	{"SCE_COFFEESCRIPT_VERBOSE_REGEX",23},
	{"SCE_COFFEESCRIPT_VERBOSE_REGEX_COMMENT",24},
	{"SCE_COFFEESCRIPT_WORD",5},
	{"SCE_COFFEESCRIPT_WORD2",16},
	{"SCE_CONF_COMMENT",1},
	{"SCE_CONF_DEFAULT",0},
	{"SCE_CONF_DIRECTIVE",9},
	{"SCE_CONF_EXTENSION",4},
	{"SCE_CONF_IDENTIFIER",3},
	{"SCE_CONF_IP",8},
	{"SCE_CONF_NUMBER",2},
	{"SCE_CONF_OPERATOR",7},
	{"SCE_CONF_PARAMETER",5},
	{"SCE_CONF_STRING",6},
	{"SCE_CSOUND_ARATE_VAR",11},
	{"SCE_CSOUND_COMMENT",1},
	{"SCE_CSOUND_COMMENTBLOCK",9},
	{"SCE_CSOUND_DEFAULT",0},
	{"SCE_CSOUND_GLOBAL_VAR",14},
	{"SCE_CSOUND_HEADERSTMT",7},
	{"SCE_CSOUND_IDENTIFIER",5},
	{"SCE_CSOUND_INSTR",4},
	{"SCE_CSOUND_IRATE_VAR",13},
	{"SCE_CSOUND_KRATE_VAR",12},
	{"SCE_CSOUND_NUMBER",2},
	{"SCE_CSOUND_OPCODE",6},
	{"SCE_CSOUND_OPERATOR",3},
	{"SCE_CSOUND_PARAM",10},
	{"SCE_CSOUND_STRINGEOL",15},
	{"SCE_CSOUND_USERKEYWORD",8},
	{"SCE_CSS_ATTRIBUTE",16},
	{"SCE_CSS_CLASS",2},
	{"SCE_CSS_COMMENT",9},
	{"SCE_CSS_DEFAULT",0},
	{"SCE_CSS_DIRECTIVE",12},
	{"SCE_CSS_DOUBLESTRING",13},
	{"SCE_CSS_EXTENDED_IDENTIFIER",19},
	{"SCE_CSS_EXTENDED_PSEUDOCLASS",20},
	{"SCE_CSS_EXTENDED_PSEUDOELEMENT",21},
	{"SCE_CSS_ID",10},
	{"SCE_CSS_IDENTIFIER",6},
	{"SCE_CSS_IDENTIFIER2",15},
	{"SCE_CSS_IDENTIFIER3",17},
	{"SCE_CSS_IMPORTANT",11},
	{"SCE_CSS_MEDIA",22},
	{"SCE_CSS_OPERATOR",5},
	{"SCE_CSS_PSEUDOCLASS",3},
	{"SCE_CSS_PSEUDOELEMENT",18},
	{"SCE_CSS_SINGLESTRING",14},
	{"SCE_CSS_TAG",1},
	{"SCE_CSS_UNKNOWN_IDENTIFIER",7},
	{"SCE_CSS_UNKNOWN_PSEUDOCLASS",4},
	{"SCE_CSS_VALUE",8},
	{"SCE_CSS_VARIABLE",23},
	{"SCE_C_CHARACTER",7},
	{"SCE_C_COMMENT",1},
	{"SCE_C_COMMENTDOC",3},
	{"SCE_C_COMMENTDOCKEYWORD",17},
	{"SCE_C_COMMENTDOCKEYWORDERROR",18},
	{"SCE_C_COMMENTLINE",2},
	{"SCE_C_COMMENTLINEDOC",15},
	{"SCE_C_DEFAULT",0},
	{"SCE_C_GLOBALCLASS",19},
	{"SCE_C_HASHQUOTEDSTRING",22},
	{"SCE_C_IDENTIFIER",11},
	{"SCE_C_NUMBER",4},
	{"SCE_C_OPERATOR",10},
	{"SCE_C_PREPROCESSOR",9},
	{"SCE_C_PREPROCESSORCOMMENT",23},
	{"SCE_C_PREPROCESSORCOMMENTDOC",24},
	{"SCE_C_REGEX",14},
	{"SCE_C_STRING",6},
	{"SCE_C_STRINGEOL",12},
	{"SCE_C_STRINGRAW",20},
	{"SCE_C_TRIPLEVERBATIM",21},
	{"SCE_C_USERLITERAL",25},
	{"SCE_C_UUID",8},
	{"SCE_C_VERBATIM",13},
	{"SCE_C_WORD",5},
	{"SCE_C_WORD2",16},
	{"SCE_DIFF_ADDED",6},
	{"SCE_DIFF_CHANGED",7},
	{"SCE_DIFF_COMMAND",2},
	{"SCE_DIFF_COMMENT",1},
	{"SCE_DIFF_DEFAULT",0},
	{"SCE_DIFF_DELETED",5},
	{"SCE_DIFF_HEADER",3},
	{"SCE_DIFF_POSITION",4},
	{"SCE_DMAP_COMMENT",1},
	{"SCE_DMAP_DEFAULT",0},
	{"SCE_DMAP_IDENTIFIER",7},
	{"SCE_DMAP_NUMBER",2},
	{"SCE_DMAP_OPERATOR",6},
	{"SCE_DMAP_STRING1",3},
	{"SCE_DMAP_STRING2",4},
	{"SCE_DMAP_STRINGEOL",5},
	{"SCE_DMAP_WORD",8},
	{"SCE_DMAP_WORD2",9},
	{"SCE_DMAP_WORD3",10},
	{"SCE_D_CHARACTER",12},
	{"SCE_D_COMMENT",1},
	{"SCE_D_COMMENTDOC",3},
	{"SCE_D_COMMENTDOCKEYWORD",16},
	{"SCE_D_COMMENTDOCKEYWORDERROR",17},
	{"SCE_D_COMMENTLINE",2},
	{"SCE_D_COMMENTLINEDOC",15},
	{"SCE_D_COMMENTNESTED",4},
	{"SCE_D_DEFAULT",0},
	{"SCE_D_IDENTIFIER",14},
	{"SCE_D_NUMBER",5},
	{"SCE_D_OPERATOR",13},
	{"SCE_D_STRING",10},
	{"SCE_D_STRINGB",18},
	{"SCE_D_STRINGEOL",11},
	{"SCE_D_STRINGR",19},
	{"SCE_D_TYPEDEF",9},
	{"SCE_D_WORD",6},
	{"SCE_D_WORD2",7},
	{"SCE_D_WORD3",8},
	{"SCE_D_WORD5",20},
	{"SCE_D_WORD6",21},
	{"SCE_D_WORD7",22},
	{"SCE_ECL_ADDED",24},
	{"SCE_ECL_CHANGED",26},
	{"SCE_ECL_CHARACTER",7},
	{"SCE_ECL_COMMENT",1},
	{"SCE_ECL_COMMENTDOC",23},
	{"SCE_ECL_COMMENTDOCKEYWORD",17},
	{"SCE_ECL_COMMENTDOCKEYWORDERROR",18},
	{"SCE_ECL_COMMENTLINE",2},
	{"SCE_ECL_COMMENTLINEDOC",15},
	{"SCE_ECL_DEFAULT",0},
	{"SCE_ECL_DELETED",25},
	{"SCE_ECL_IDENTIFIER",11},
	{"SCE_ECL_MOVED",27},
	{"SCE_ECL_NUMBER",3},
	{"SCE_ECL_OPERATOR",6},
	{"SCE_ECL_PREPROCESSOR",9},
	{"SCE_ECL_REGEX",14},
	{"SCE_ECL_STRING",4},
	{"SCE_ECL_STRINGEOL",12},
	{"SCE_ECL_UNKNOWN",10},
	{"SCE_ECL_UUID",8},
	{"SCE_ECL_VERBATIM",13},
	{"SCE_ECL_WORD0",5},
	{"SCE_ECL_WORD1",16},
	{"SCE_ECL_WORD2",19},
	{"SCE_ECL_WORD3",20},
	{"SCE_ECL_WORD4",21},
	{"SCE_ECL_WORD5",22},
	{"SCE_EIFFEL_CHARACTER",5},
	{"SCE_EIFFEL_COMMENTLINE",1},
	{"SCE_EIFFEL_DEFAULT",0},
	{"SCE_EIFFEL_IDENTIFIER",7},
	{"SCE_EIFFEL_NUMBER",2},
	{"SCE_EIFFEL_OPERATOR",6},
	{"SCE_EIFFEL_STRING",4},
	{"SCE_EIFFEL_STRINGEOL",8},
	{"SCE_EIFFEL_WORD",3},
	{"SCE_ERLANG_ATOM",7},
	{"SCE_ERLANG_ATOM_QUOTED",18},
	{"SCE_ERLANG_BIFS",22},
	{"SCE_ERLANG_CHARACTER",9},
	{"SCE_ERLANG_COMMENT",1},
	{"SCE_ERLANG_COMMENT_DOC",16},
	{"SCE_ERLANG_COMMENT_DOC_MACRO",17},
	{"SCE_ERLANG_COMMENT_FUNCTION",14},
	{"SCE_ERLANG_COMMENT_MODULE",15},
	{"SCE_ERLANG_DEFAULT",0},
	{"SCE_ERLANG_FUNCTION_NAME",8},
	{"SCE_ERLANG_KEYWORD",4},
	{"SCE_ERLANG_MACRO",10},
	{"SCE_ERLANG_MACRO_QUOTED",19},
	{"SCE_ERLANG_MODULES",23},
	{"SCE_ERLANG_MODULES_ATT",24},
	{"SCE_ERLANG_NODE_NAME",13},
	{"SCE_ERLANG_NODE_NAME_QUOTED",21},
	{"SCE_ERLANG_NUMBER",3},
	{"SCE_ERLANG_OPERATOR",6},
	{"SCE_ERLANG_PREPROC",12},
	{"SCE_ERLANG_RECORD",11},
	{"SCE_ERLANG_RECORD_QUOTED",20},
	{"SCE_ERLANG_STRING",5},
	{"SCE_ERLANG_UNKNOWN",31},
	{"SCE_ERLANG_VARIABLE",2},
	{"SCE_ERR_ABSF",18},
	{"SCE_ERR_BORLAND",5},
	{"SCE_ERR_CMD",4},
	{"SCE_ERR_CTAG",9},
	{"SCE_ERR_DEFAULT",0},
	{"SCE_ERR_DIFF_ADDITION",11},
	{"SCE_ERR_DIFF_CHANGED",10},
	{"SCE_ERR_DIFF_DELETION",12},
	{"SCE_ERR_DIFF_MESSAGE",13},
	{"SCE_ERR_ELF",15},
	{"SCE_ERR_GCC",2},
	{"SCE_ERR_GCC_INCLUDED_FROM",22},
	{"SCE_ERR_IFC",16},
	{"SCE_ERR_IFORT",17},
	{"SCE_ERR_JAVA_STACK",20},
	{"SCE_ERR_LUA",8},
	{"SCE_ERR_MS",3},
	{"SCE_ERR_NET",7},
	{"SCE_ERR_PERL",6},
	{"SCE_ERR_PHP",14},
	{"SCE_ERR_PYTHON",1},
	{"SCE_ERR_TIDY",19},
	{"SCE_ERR_VALUE",21},
	{"SCE_ESCRIPT_BRACE",9},
	{"SCE_ESCRIPT_COMMENT",1},
	{"SCE_ESCRIPT_COMMENTDOC",3},
	{"SCE_ESCRIPT_COMMENTLINE",2},
	{"SCE_ESCRIPT_DEFAULT",0},
	{"SCE_ESCRIPT_IDENTIFIER",8},
	{"SCE_ESCRIPT_NUMBER",4},
	{"SCE_ESCRIPT_OPERATOR",7},
	{"SCE_ESCRIPT_STRING",6},
	{"SCE_ESCRIPT_WORD",5},
	{"SCE_ESCRIPT_WORD2",10},
	{"SCE_ESCRIPT_WORD3",11},
	{"SCE_FORTH_COMMENT",1},
	{"SCE_FORTH_COMMENT_ML",2},
	{"SCE_FORTH_CONTROL",4},
	{"SCE_FORTH_DEFAULT",0},
	{"SCE_FORTH_DEFWORD",6},
	{"SCE_FORTH_IDENTIFIER",3},
	{"SCE_FORTH_KEYWORD",5},
	{"SCE_FORTH_LOCALE",11},
	{"SCE_FORTH_NUMBER",9},
	{"SCE_FORTH_PREWORD1",7},
	{"SCE_FORTH_PREWORD2",8},
	{"SCE_FORTH_STRING",10},
	{"SCE_FS_COMMENT",1},
	{"SCE_FS_COMMENTDOC",3},
	{"SCE_FS_COMMENTDOCKEYWORD",5},
	{"SCE_FS_COMMENTDOCKEYWORDERROR",6},
	{"SCE_FS_COMMENTDOC_C",22},
	{"SCE_FS_COMMENTLINE",2},
	{"SCE_FS_COMMENTLINEDOC",4},
	{"SCE_FS_COMMENTLINEDOC_C",23},
	{"SCE_FS_CONSTANT",18},
	{"SCE_FS_DATE",16},
	{"SCE_FS_DEFAULT",0},
	{"SCE_FS_DEFAULT_C",21},
	{"SCE_FS_DISABLEDCODE",20},
	{"SCE_FS_IDENTIFIER",15},
	{"SCE_FS_IDENTIFIER_C",30},
	{"SCE_FS_KEYWORD",7},
	{"SCE_FS_KEYWORD2",8},
	{"SCE_FS_KEYWORD2_C",25},
	{"SCE_FS_KEYWORD3",9},
	{"SCE_FS_KEYWORD4",10},
	{"SCE_FS_KEYWORD_C",24},
	{"SCE_FS_NUMBER",11},
	{"SCE_FS_NUMBER_C",26},
	{"SCE_FS_OPERATOR",14},
	{"SCE_FS_OPERATOR_C",29},
	{"SCE_FS_PREPROCESSOR",13},
	{"SCE_FS_PREPROCESSOR_C",28},
	{"SCE_FS_STRING",12},
	{"SCE_FS_STRINGEOL",17},
	{"SCE_FS_STRINGEOL_C",31},
	{"SCE_FS_STRING_C",27},
	{"SCE_FS_WORDOPERATOR",19},
	{"SCE_F_COMMENT",1},
	{"SCE_F_CONTINUATION",14},
	{"SCE_F_DEFAULT",0},
	{"SCE_F_IDENTIFIER",7},
	{"SCE_F_LABEL",13},
	{"SCE_F_NUMBER",2},
	{"SCE_F_OPERATOR",6},
	{"SCE_F_OPERATOR2",12},
	{"SCE_F_PREPROCESSOR",11},
	{"SCE_F_STRING1",3},
	{"SCE_F_STRING2",4},
	{"SCE_F_STRINGEOL",5},
	{"SCE_F_WORD",8},
	{"SCE_F_WORD2",9},
	{"SCE_F_WORD3",10},
	{"SCE_GAP_CHAR",7},
	{"SCE_GAP_COMMENT",9},
	{"SCE_GAP_DEFAULT",0},
	{"SCE_GAP_IDENTIFIER",1},
	{"SCE_GAP_KEYWORD",2},
	{"SCE_GAP_KEYWORD2",3},
	{"SCE_GAP_KEYWORD3",4},
	{"SCE_GAP_KEYWORD4",5},
	{"SCE_GAP_NUMBER",10},
	{"SCE_GAP_OPERATOR",8},
	{"SCE_GAP_STRING",6},
	{"SCE_GAP_STRINGEOL",11},
	{"SCE_GC_ATTRIBUTE",5},
	{"SCE_GC_COMMAND",7},
	{"SCE_GC_COMMENTBLOCK",2},
	{"SCE_GC_COMMENTLINE",1},
	{"SCE_GC_CONTROL",6},
	{"SCE_GC_DEFAULT",0},
	{"SCE_GC_EVENT",4},
	{"SCE_GC_GLOBAL",3},
	{"SCE_GC_OPERATOR",9},
	{"SCE_GC_STRING",8},
	{"SCE_HA_CAPITAL",8},
	{"SCE_HA_CHARACTER",5},
	{"SCE_HA_CLASS",6},
	{"SCE_HA_COMMENTBLOCK",14},
	{"SCE_HA_COMMENTBLOCK2",15},
	{"SCE_HA_COMMENTBLOCK3",16},
	{"SCE_HA_COMMENTLINE",13},
	{"SCE_HA_DATA",9},
	{"SCE_HA_DEFAULT",0},
	{"SCE_HA_IDENTIFIER",1},
	{"SCE_HA_IMPORT",10},
	{"SCE_HA_INSTANCE",12},
	{"SCE_HA_KEYWORD",2},
	{"SCE_HA_LITERATE_CODEDELIM",22},
	{"SCE_HA_LITERATE_COMMENT",21},
	{"SCE_HA_MODULE",7},
	{"SCE_HA_NUMBER",3},
	{"SCE_HA_OPERATOR",11},
	{"SCE_HA_PRAGMA",17},
	{"SCE_HA_PREPROCESSOR",18},
	{"SCE_HA_RESERVED_OPERATOR",20},
	{"SCE_HA_STRING",4},
	{"SCE_HA_STRINGEOL",19},
	{"SCE_HBA_COMMENTLINE",82},
	{"SCE_HBA_DEFAULT",81},
	{"SCE_HBA_IDENTIFIER",86},
	{"SCE_HBA_NUMBER",83},
	{"SCE_HBA_START",80},
	{"SCE_HBA_STRING",85},
	{"SCE_HBA_STRINGEOL",87},
	{"SCE_HBA_WORD",84},
	{"SCE_HB_COMMENTLINE",72},
	{"SCE_HB_DEFAULT",71},
	{"SCE_HB_IDENTIFIER",76},
	{"SCE_HB_NUMBER",73},
	{"SCE_HB_START",70},
	{"SCE_HB_STRING",75},
	{"SCE_HB_STRINGEOL",77},
	{"SCE_HB_WORD",74},
	{"SCE_HJA_COMMENT",57},
	{"SCE_HJA_COMMENTDOC",59},
	{"SCE_HJA_COMMENTLINE",58},
	{"SCE_HJA_DEFAULT",56},
	{"SCE_HJA_DOUBLESTRING",63},
	{"SCE_HJA_KEYWORD",62},
	{"SCE_HJA_NUMBER",60},
	{"SCE_HJA_REGEX",67},
	{"SCE_HJA_SINGLESTRING",64},
	{"SCE_HJA_START",55},
	{"SCE_HJA_STRINGEOL",66},
	{"SCE_HJA_SYMBOLS",65},
	{"SCE_HJA_WORD",61},
	{"SCE_HJ_COMMENT",42},
	{"SCE_HJ_COMMENTDOC",44},
	{"SCE_HJ_COMMENTLINE",43},
	{"SCE_HJ_DEFAULT",41},
	{"SCE_HJ_DOUBLESTRING",48},
	{"SCE_HJ_KEYWORD",47},
	{"SCE_HJ_NUMBER",45},
	{"SCE_HJ_REGEX",52},
	{"SCE_HJ_SINGLESTRING",49},
	{"SCE_HJ_START",40},
	{"SCE_HJ_STRINGEOL",51},
	{"SCE_HJ_SYMBOLS",50},
	{"SCE_HJ_WORD",46},
	{"SCE_HPA_CHARACTER",110},
	{"SCE_HPA_CLASSNAME",114},
	{"SCE_HPA_COMMENTLINE",107},
	{"SCE_HPA_DEFAULT",106},
	{"SCE_HPA_DEFNAME",115},
	{"SCE_HPA_IDENTIFIER",117},
	{"SCE_HPA_NUMBER",108},
	{"SCE_HPA_OPERATOR",116},
	{"SCE_HPA_START",105},
	{"SCE_HPA_STRING",109},
	{"SCE_HPA_TRIPLE",112},
	{"SCE_HPA_TRIPLEDOUBLE",113},
	{"SCE_HPA_WORD",111},
	{"SCE_HPHP_COMMENT",124},
	{"SCE_HPHP_COMMENTLINE",125},
	{"SCE_HPHP_COMPLEX_VARIABLE",104},
	{"SCE_HPHP_DEFAULT",118},
	{"SCE_HPHP_HSTRING",119},
	{"SCE_HPHP_HSTRING_VARIABLE",126},
	{"SCE_HPHP_NUMBER",122},
	{"SCE_HPHP_OPERATOR",127},
	{"SCE_HPHP_SIMPLESTRING",120},
	{"SCE_HPHP_VARIABLE",123},
	{"SCE_HPHP_WORD",121},
	{"SCE_HP_CHARACTER",95},
	{"SCE_HP_CLASSNAME",99},
	{"SCE_HP_COMMENTLINE",92},
	{"SCE_HP_DEFAULT",91},
	{"SCE_HP_DEFNAME",100},
	{"SCE_HP_IDENTIFIER",102},
	{"SCE_HP_NUMBER",93},
	{"SCE_HP_OPERATOR",101},
	{"SCE_HP_START",90},
	{"SCE_HP_STRING",94},
	{"SCE_HP_TRIPLE",97},
	{"SCE_HP_TRIPLEDOUBLE",98},
	{"SCE_HP_WORD",96},
	{"SCE_H_ASP",15},
	{"SCE_H_ASPAT",16},
	{"SCE_H_ATTRIBUTE",3},
	{"SCE_H_ATTRIBUTEUNKNOWN",4},
	{"SCE_H_CDATA",17},
	{"SCE_H_COMMENT",9},
	{"SCE_H_DEFAULT",0},
	{"SCE_H_DOUBLESTRING",6},
	{"SCE_H_ENTITY",10},
	{"SCE_H_NUMBER",5},
	{"SCE_H_OTHER",8},
	{"SCE_H_QUESTION",18},
	{"SCE_H_SCRIPT",14},
	{"SCE_H_SGML_1ST_PARAM",23},
	{"SCE_H_SGML_1ST_PARAM_COMMENT",30},
	{"SCE_H_SGML_BLOCK_DEFAULT",31},
	{"SCE_H_SGML_COMMAND",22},
	{"SCE_H_SGML_COMMENT",29},
	{"SCE_H_SGML_DEFAULT",21},
	{"SCE_H_SGML_DOUBLESTRING",24},
	{"SCE_H_SGML_ENTITY",28},
	{"SCE_H_SGML_ERROR",26},
	{"SCE_H_SGML_SIMPLESTRING",25},
	{"SCE_H_SGML_SPECIAL",27},
	{"SCE_H_SINGLESTRING",7},
	{"SCE_H_TAG",1},
	{"SCE_H_TAGEND",11},
	{"SCE_H_TAGUNKNOWN",2},
	{"SCE_H_VALUE",19},
	{"SCE_H_XCCOMMENT",20},
	{"SCE_H_XMLEND",13},
	{"SCE_H_XMLSTART",12},
	{"SCE_INNO_COMMENT",1},
	{"SCE_INNO_COMMENT_PASCAL",7},
	{"SCE_INNO_DEFAULT",0},
	{"SCE_INNO_IDENTIFIER",12},
	{"SCE_INNO_INLINE_EXPANSION",6},
	{"SCE_INNO_KEYWORD",2},
	{"SCE_INNO_KEYWORD_PASCAL",8},
	{"SCE_INNO_KEYWORD_USER",9},
	{"SCE_INNO_PARAMETER",3},
	{"SCE_INNO_PREPROC",5},
	{"SCE_INNO_SECTION",4},
	{"SCE_INNO_STRING_DOUBLE",10},
	{"SCE_INNO_STRING_SINGLE",11},
	{"SCE_KIX_COMMENT",1},
	{"SCE_KIX_DEFAULT",0},
	{"SCE_KIX_FUNCTIONS",8},
	{"SCE_KIX_IDENTIFIER",31},
	{"SCE_KIX_KEYWORD",7},
	{"SCE_KIX_MACRO",6},
	{"SCE_KIX_NUMBER",4},
	{"SCE_KIX_OPERATOR",9},
	{"SCE_KIX_STRING1",2},
	{"SCE_KIX_STRING2",3},
	{"SCE_KIX_VAR",5},
	{"SCE_KVIRC_COMMENT",1},
	{"SCE_KVIRC_COMMENTBLOCK",2},
	{"SCE_KVIRC_DEFAULT",0},
	{"SCE_KVIRC_FUNCTION",7},
	{"SCE_KVIRC_FUNCTION_KEYWORD",6},
	{"SCE_KVIRC_KEYWORD",5},
	{"SCE_KVIRC_NUMBER",9},
	{"SCE_KVIRC_OPERATOR",10},
	{"SCE_KVIRC_STRING",3},
	{"SCE_KVIRC_STRING_FUNCTION",11},
	{"SCE_KVIRC_STRING_VARIABLE",12},
	{"SCE_KVIRC_VARIABLE",8},
	{"SCE_KVIRC_WORD",4},
	{"SCE_LISP_COMMENT",1},
	{"SCE_LISP_DEFAULT",0},
	{"SCE_LISP_IDENTIFIER",9},
	{"SCE_LISP_KEYWORD",3},
	{"SCE_LISP_KEYWORD_KW",4},
	{"SCE_LISP_MULTI_COMMENT",12},
	{"SCE_LISP_NUMBER",2},
	{"SCE_LISP_OPERATOR",10},
	{"SCE_LISP_SPECIAL",11},
	{"SCE_LISP_STRING",6},
	{"SCE_LISP_STRINGEOL",8},
	{"SCE_LISP_SYMBOL",5},
	{"SCE_LOT_ABORT",6},
	{"SCE_LOT_BREAK",2},
	{"SCE_LOT_DEFAULT",0},
	{"SCE_LOT_FAIL",5},
	{"SCE_LOT_HEADER",1},
	{"SCE_LOT_PASS",4},
	{"SCE_LOT_SET",3},
	{"SCE_LOUT_COMMENT",1},
	{"SCE_LOUT_DEFAULT",0},
	{"SCE_LOUT_IDENTIFIER",9},
	{"SCE_LOUT_NUMBER",2},
	{"SCE_LOUT_OPERATOR",8},
	{"SCE_LOUT_STRING",7},
	{"SCE_LOUT_STRINGEOL",10},
	{"SCE_LOUT_WORD",3},
	{"SCE_LOUT_WORD2",4},
	{"SCE_LOUT_WORD3",5},
	{"SCE_LOUT_WORD4",6},
	{"SCE_LUA_CHARACTER",7},
	{"SCE_LUA_COMMENT",1},
	{"SCE_LUA_COMMENTDOC",3},
	{"SCE_LUA_COMMENTLINE",2},
	{"SCE_LUA_DEFAULT",0},
	{"SCE_LUA_IDENTIFIER",11},
	{"SCE_LUA_LABEL",20},
	{"SCE_LUA_LITERALSTRING",8},
	{"SCE_LUA_NUMBER",4},
	{"SCE_LUA_OPERATOR",10},
	{"SCE_LUA_PREPROCESSOR",9},
	{"SCE_LUA_STRING",6},
	{"SCE_LUA_STRINGEOL",12},
	{"SCE_LUA_WORD",5},
	{"SCE_LUA_WORD2",13},
	{"SCE_LUA_WORD3",14},
	{"SCE_LUA_WORD4",15},
	{"SCE_LUA_WORD5",16},
	{"SCE_LUA_WORD6",17},
	{"SCE_LUA_WORD7",18},
	{"SCE_LUA_WORD8",19},
	{"SCE_L_CMDOPT",11},
	{"SCE_L_COMMAND",1},
	{"SCE_L_COMMENT",4},
	{"SCE_L_COMMENT2",7},
	{"SCE_L_DEFAULT",0},
	{"SCE_L_ERROR",12},
	{"SCE_L_MATH",3},
	{"SCE_L_MATH2",6},
	{"SCE_L_SHORTCMD",9},
	{"SCE_L_SPECIAL",10},
	{"SCE_L_TAG",2},
	{"SCE_L_TAG2",5},
	{"SCE_L_VERBATIM",8},
	{"SCE_MAGIK_BRACE_BLOCK",10},
	{"SCE_MAGIK_BRACKET_BLOCK",9},
	{"SCE_MAGIK_CHARACTER",3},
	{"SCE_MAGIK_COMMENT",1},
	{"SCE_MAGIK_CONTAINER",8},
	{"SCE_MAGIK_DEFAULT",0},
	{"SCE_MAGIK_FLOW",7},
	{"SCE_MAGIK_HYPER_COMMENT",16},
	{"SCE_MAGIK_IDENTIFIER",5},
	{"SCE_MAGIK_KEYWORD",13},
	{"SCE_MAGIK_NUMBER",4},
	{"SCE_MAGIK_OPERATOR",6},
	{"SCE_MAGIK_PRAGMA",14},
	{"SCE_MAGIK_SQBRACKET_BLOCK",11},
	{"SCE_MAGIK_STRING",2},
	{"SCE_MAGIK_SYMBOL",15},
	{"SCE_MAGIK_UNKNOWN_KEYWORD",12},
	{"SCE_MAKE_COMMENT",1},
	{"SCE_MAKE_DEFAULT",0},
	{"SCE_MAKE_IDENTIFIER",3},
	{"SCE_MAKE_IDEOL",9},
	{"SCE_MAKE_OPERATOR",4},
	{"SCE_MAKE_PREPROCESSOR",2},
	{"SCE_MAKE_TARGET",5},
	{"SCE_MARKDOWN_BLOCKQUOTE",15},
	{"SCE_MARKDOWN_CODE",19},
	{"SCE_MARKDOWN_CODE2",20},
	{"SCE_MARKDOWN_CODEBK",21},
	{"SCE_MARKDOWN_DEFAULT",0},
	{"SCE_MARKDOWN_EM1",4},
	{"SCE_MARKDOWN_EM2",5},
	{"SCE_MARKDOWN_HEADER1",6},
	{"SCE_MARKDOWN_HEADER2",7},
	{"SCE_MARKDOWN_HEADER3",8},
	{"SCE_MARKDOWN_HEADER4",9},
	{"SCE_MARKDOWN_HEADER5",10},
	{"SCE_MARKDOWN_HEADER6",11},
	{"SCE_MARKDOWN_HRULE",17},
	{"SCE_MARKDOWN_LINE_BEGIN",1},
	{"SCE_MARKDOWN_LINK",18},
	{"SCE_MARKDOWN_OLIST_ITEM",14},
	{"SCE_MARKDOWN_PRECHAR",12},
	{"SCE_MARKDOWN_STRIKEOUT",16},
	{"SCE_MARKDOWN_STRONG1",2},
	{"SCE_MARKDOWN_STRONG2",3},
	{"SCE_MARKDOWN_ULIST_ITEM",13},
	{"SCE_MATLAB_COMMAND",2},
	{"SCE_MATLAB_COMMENT",1},
	{"SCE_MATLAB_DEFAULT",0},
	{"SCE_MATLAB_DOUBLEQUOTESTRING",8},
	{"SCE_MATLAB_IDENTIFIER",7},
	{"SCE_MATLAB_KEYWORD",4},
	{"SCE_MATLAB_NUMBER",3},
	{"SCE_MATLAB_OPERATOR",6},
	{"SCE_MATLAB_STRING",5},
	{"SCE_METAPOST_COMMAND",4},
	{"SCE_METAPOST_DEFAULT",0},
	{"SCE_METAPOST_EXTRA",6},
	{"SCE_METAPOST_GROUP",2},
	{"SCE_METAPOST_SPECIAL",1},
	{"SCE_METAPOST_SYMBOL",3},
	{"SCE_METAPOST_TEXT",5},
	{"SCE_MMIXAL_CHAR",11},
	{"SCE_MMIXAL_COMMENT",1},
	{"SCE_MMIXAL_HEX",14},
	{"SCE_MMIXAL_INCLUDE",17},
	{"SCE_MMIXAL_LABEL",2},
	{"SCE_MMIXAL_LEADWS",0},
	{"SCE_MMIXAL_NUMBER",9},
	{"SCE_MMIXAL_OPCODE",3},
	{"SCE_MMIXAL_OPCODE_POST",7},
	{"SCE_MMIXAL_OPCODE_PRE",4},
	{"SCE_MMIXAL_OPCODE_UNKNOWN",6},
	{"SCE_MMIXAL_OPCODE_VALID",5},
	{"SCE_MMIXAL_OPERANDS",8},
	{"SCE_MMIXAL_OPERATOR",15},
	{"SCE_MMIXAL_REF",10},
	{"SCE_MMIXAL_REGISTER",13},
	{"SCE_MMIXAL_STRING",12},
	{"SCE_MMIXAL_SYMBOL",16},
	{"SCE_MODULA_BADSTR",17},
	{"SCE_MODULA_BASENUM",7},
	{"SCE_MODULA_CHAR",11},
	{"SCE_MODULA_CHARSPEC",12},
	{"SCE_MODULA_COMMENT",1},
	{"SCE_MODULA_DEFAULT",0},
	{"SCE_MODULA_DOXYCOMM",2},
	{"SCE_MODULA_DOXYKEY",3},
	{"SCE_MODULA_FLOAT",8},
	{"SCE_MODULA_KEYWORD",4},
	{"SCE_MODULA_NUMBER",6},
	{"SCE_MODULA_OPERATOR",16},
	{"SCE_MODULA_PRAGMA",14},
	{"SCE_MODULA_PRGKEY",15},
	{"SCE_MODULA_PROC",13},
	{"SCE_MODULA_RESERVED",5},
	{"SCE_MODULA_STRING",9},
	{"SCE_MODULA_STRSPEC",10},
	{"SCE_MSSQL_COLUMN_NAME",8},
	{"SCE_MSSQL_COLUMN_NAME_2",16},
	{"SCE_MSSQL_COMMENT",1},
	{"SCE_MSSQL_DATATYPE",10},
	{"SCE_MSSQL_DEFAULT",0},
	{"SCE_MSSQL_DEFAULT_PREF_DATATYPE",15},
	{"SCE_MSSQL_FUNCTION",13},
	{"SCE_MSSQL_GLOBAL_VARIABLE",12},
	{"SCE_MSSQL_IDENTIFIER",6},
	{"SCE_MSSQL_LINE_COMMENT",2},
	{"SCE_MSSQL_NUMBER",3},
	{"SCE_MSSQL_OPERATOR",5},
	{"SCE_MSSQL_STATEMENT",9},
	{"SCE_MSSQL_STORED_PROCEDURE",14},
	{"SCE_MSSQL_STRING",4},
	{"SCE_MSSQL_SYSTABLE",11},
	{"SCE_MSSQL_VARIABLE",7},
	{"SCE_MYSQL_COMMENT",1},
	{"SCE_MYSQL_COMMENTLINE",2},
	{"SCE_MYSQL_DATABASEOBJECT",9},
	{"SCE_MYSQL_DEFAULT",0},
	{"SCE_MYSQL_DQSTRING",13},
	{"SCE_MYSQL_FUNCTION",15},
	{"SCE_MYSQL_HIDDENCOMMAND",21},
	{"SCE_MYSQL_IDENTIFIER",16},
	{"SCE_MYSQL_KEYWORD",8},
	{"SCE_MYSQL_KNOWNSYSTEMVARIABLE",5},
	{"SCE_MYSQL_MAJORKEYWORD",7},
	{"SCE_MYSQL_NUMBER",6},
	{"SCE_MYSQL_OPERATOR",14},
	{"SCE_MYSQL_PLACEHOLDER",22},
	{"SCE_MYSQL_PROCEDUREKEYWORD",10},
	{"SCE_MYSQL_QUOTEDIDENTIFIER",17},
	{"SCE_MYSQL_SQSTRING",12},
	{"SCE_MYSQL_STRING",11},
	{"SCE_MYSQL_SYSTEMVARIABLE",4},
	{"SCE_MYSQL_USER1",18},
	{"SCE_MYSQL_USER2",19},
	{"SCE_MYSQL_USER3",20},
	{"SCE_MYSQL_VARIABLE",3},
	{"SCE_NNCRONTAB_ASTERISK",6},
	{"SCE_NNCRONTAB_COMMENT",1},
	{"SCE_NNCRONTAB_DEFAULT",0},
	{"SCE_NNCRONTAB_ENVIRONMENT",9},
	{"SCE_NNCRONTAB_IDENTIFIER",10},
	{"SCE_NNCRONTAB_KEYWORD",4},
	{"SCE_NNCRONTAB_MODIFIER",5},
	{"SCE_NNCRONTAB_NUMBER",7},
	{"SCE_NNCRONTAB_SECTION",3},
	{"SCE_NNCRONTAB_STRING",8},
	{"SCE_NNCRONTAB_TASK",2},
	{"SCE_NSIS_COMMENT",1},
	{"SCE_NSIS_COMMENTBOX",18},
	{"SCE_NSIS_DEFAULT",0},
	{"SCE_NSIS_FUNCTION",5},
	{"SCE_NSIS_FUNCTIONDEF",17},
	{"SCE_NSIS_IFDEFINEDEF",11},
	{"SCE_NSIS_LABEL",7},
	{"SCE_NSIS_MACRODEF",12},
	{"SCE_NSIS_NUMBER",14},
	{"SCE_NSIS_PAGEEX",16},
	{"SCE_NSIS_SECTIONDEF",9},
	{"SCE_NSIS_SECTIONGROUP",15},
	{"SCE_NSIS_STRINGDQ",2},
	{"SCE_NSIS_STRINGLQ",3},
	{"SCE_NSIS_STRINGRQ",4},
	{"SCE_NSIS_STRINGVAR",13},
	{"SCE_NSIS_SUBSECTIONDEF",10},
	{"SCE_NSIS_USERDEFINED",8},
	{"SCE_NSIS_VARIABLE",6},
	{"SCE_OPAL_BOOL_CONST",8},
	{"SCE_OPAL_COMMENT_BLOCK",1},
	{"SCE_OPAL_COMMENT_LINE",2},
	{"SCE_OPAL_DEFAULT",32},
	{"SCE_OPAL_INTEGER",3},
	{"SCE_OPAL_KEYWORD",4},
	{"SCE_OPAL_PAR",7},
	{"SCE_OPAL_SORT",5},
	{"SCE_OPAL_SPACE",0},
	{"SCE_OPAL_STRING",6},
	{"SCE_OSCRIPT_BLOCK_COMMENT",2},
	{"SCE_OSCRIPT_CONSTANT",8},
	{"SCE_OSCRIPT_DEFAULT",0},
	{"SCE_OSCRIPT_DOC_COMMENT",3},
	{"SCE_OSCRIPT_DOUBLEQUOTE_STRING",7},
	{"SCE_OSCRIPT_FUNCTION",15},
	{"SCE_OSCRIPT_GLOBAL",10},
	{"SCE_OSCRIPT_IDENTIFIER",9},
	{"SCE_OSCRIPT_KEYWORD",11},
	{"SCE_OSCRIPT_LABEL",13},
	{"SCE_OSCRIPT_LINE_COMMENT",1},
	{"SCE_OSCRIPT_METHOD",18},
	{"SCE_OSCRIPT_NUMBER",5},
	{"SCE_OSCRIPT_OBJECT",16},
	{"SCE_OSCRIPT_OPERATOR",12},
	{"SCE_OSCRIPT_PREPROCESSOR",4},
	{"SCE_OSCRIPT_PROPERTY",17},
	{"SCE_OSCRIPT_SINGLEQUOTE_STRING",6},
	{"SCE_OSCRIPT_TYPE",14},
	{"SCE_PAS_ASM",14},
	{"SCE_PAS_CHARACTER",12},
	{"SCE_PAS_COMMENT",2},
	{"SCE_PAS_COMMENT2",3},
	{"SCE_PAS_COMMENTLINE",4},
	{"SCE_PAS_DEFAULT",0},
	{"SCE_PAS_HEXNUMBER",8},
	{"SCE_PAS_IDENTIFIER",1},
	{"SCE_PAS_NUMBER",7},
	{"SCE_PAS_OPERATOR",13},
	{"SCE_PAS_PREPROCESSOR",5},
	{"SCE_PAS_PREPROCESSOR2",6},
	{"SCE_PAS_STRING",10},
	{"SCE_PAS_STRINGEOL",11},
	{"SCE_PAS_WORD",9},
	{"SCE_PLM_COMMENT",1},
	{"SCE_PLM_CONTROL",6},
	{"SCE_PLM_DEFAULT",0},
	{"SCE_PLM_IDENTIFIER",4},
	{"SCE_PLM_KEYWORD",7},
	{"SCE_PLM_NUMBER",3},
	{"SCE_PLM_OPERATOR",5},
	{"SCE_PLM_STRING",2},
	{"SCE_PL_ARRAY",13},
	{"SCE_PL_BACKTICKS",20},
	{"SCE_PL_BACKTICKS_VAR",57},
	{"SCE_PL_CHARACTER",7},
	{"SCE_PL_COMMENTLINE",2},
	{"SCE_PL_DATASECTION",21},
	{"SCE_PL_DEFAULT",0},
	{"SCE_PL_ERROR",1},
	{"SCE_PL_FORMAT",42},
	{"SCE_PL_FORMAT_IDENT",41},
	{"SCE_PL_HASH",14},
	{"SCE_PL_HERE_DELIM",22},
	{"SCE_PL_HERE_Q",23},
	{"SCE_PL_HERE_QQ",24},
	{"SCE_PL_HERE_QQ_VAR",61},
	{"SCE_PL_HERE_QX",25},
	{"SCE_PL_HERE_QX_VAR",62},
	{"SCE_PL_IDENTIFIER",11},
	{"SCE_PL_LONGQUOTE",19},
	{"SCE_PL_NUMBER",4},
	{"SCE_PL_OPERATOR",10},
	{"SCE_PL_POD",3},
	{"SCE_PL_POD_VERB",31},
	{"SCE_PL_PREPROCESSOR",9},
	{"SCE_PL_PUNCTUATION",8},
	{"SCE_PL_REGEX",17},
	{"SCE_PL_REGEX_VAR",54},
	{"SCE_PL_REGSUBST",18},
	{"SCE_PL_REGSUBST_VAR",55},
	{"SCE_PL_SCALAR",12},
	{"SCE_PL_STRING",6},
	{"SCE_PL_STRING_Q",26},
	{"SCE_PL_STRING_QQ",27},
	{"SCE_PL_STRING_QQ_VAR",64},
	{"SCE_PL_STRING_QR",29},
	{"SCE_PL_STRING_QR_VAR",66},
	{"SCE_PL_STRING_QW",30},
	{"SCE_PL_STRING_QX",28},
	{"SCE_PL_STRING_QX_VAR",65},
	{"SCE_PL_STRING_VAR",43},
	{"SCE_PL_SUB_PROTOTYPE",40},
	{"SCE_PL_SYMBOLTABLE",15},
	{"SCE_PL_VARIABLE_INDEXER",16},
	{"SCE_PL_WORD",5},
	{"SCE_PL_XLAT",44},
	{"SCE_POV_BADDIRECTIVE",9},
	{"SCE_POV_COMMENT",1},
	{"SCE_POV_COMMENTLINE",2},
	{"SCE_POV_DEFAULT",0},
	{"SCE_POV_DIRECTIVE",8},
	{"SCE_POV_IDENTIFIER",5},
	{"SCE_POV_NUMBER",3},
	{"SCE_POV_OPERATOR",4},
	{"SCE_POV_STRING",6},
	{"SCE_POV_STRINGEOL",7},
	{"SCE_POV_WORD2",10},
	{"SCE_POV_WORD3",11},
	{"SCE_POV_WORD4",12},
	{"SCE_POV_WORD5",13},
	{"SCE_POV_WORD6",14},
	{"SCE_POV_WORD7",15},
	{"SCE_POV_WORD8",16},
	{"SCE_POWERPRO_ALTQUOTE",15},
	{"SCE_POWERPRO_COMMENTBLOCK",1},
	{"SCE_POWERPRO_COMMENTLINE",2},
	{"SCE_POWERPRO_DEFAULT",0},
	{"SCE_POWERPRO_DOUBLEQUOTEDSTRING",8},
	{"SCE_POWERPRO_FUNCTION",16},
	{"SCE_POWERPRO_IDENTIFIER",12},
	{"SCE_POWERPRO_LINECONTINUE",10},
	{"SCE_POWERPRO_NUMBER",3},
	{"SCE_POWERPRO_OPERATOR",11},
	{"SCE_POWERPRO_SINGLEQUOTEDSTRING",9},
	{"SCE_POWERPRO_STRINGEOL",13},
	{"SCE_POWERPRO_VERBATIM",14},
	{"SCE_POWERPRO_WORD",4},
	{"SCE_POWERPRO_WORD2",5},
	{"SCE_POWERPRO_WORD3",6},
	{"SCE_POWERPRO_WORD4",7},
	{"SCE_POWERSHELL_ALIAS",10},
	{"SCE_POWERSHELL_CHARACTER",3},
	{"SCE_POWERSHELL_CMDLET",9},
	{"SCE_POWERSHELL_COMMENT",1},
	{"SCE_POWERSHELL_COMMENTDOCKEYWORD",16},
	{"SCE_POWERSHELL_COMMENTSTREAM",13},
	{"SCE_POWERSHELL_DEFAULT",0},
	{"SCE_POWERSHELL_FUNCTION",11},
	{"SCE_POWERSHELL_HERE_CHARACTER",15},
	{"SCE_POWERSHELL_HERE_STRING",14},
	{"SCE_POWERSHELL_IDENTIFIER",7},
	{"SCE_POWERSHELL_KEYWORD",8},
	{"SCE_POWERSHELL_NUMBER",4},
	{"SCE_POWERSHELL_OPERATOR",6},
	{"SCE_POWERSHELL_STRING",2},
	{"SCE_POWERSHELL_USER1",12},
	{"SCE_POWERSHELL_VARIABLE",5},
	{"SCE_PO_COMMENT",1},
	{"SCE_PO_DEFAULT",0},
	{"SCE_PO_ERROR",15},
	{"SCE_PO_FLAGS",11},
	{"SCE_PO_FUZZY",8},
	{"SCE_PO_MSGCTXT",6},
	{"SCE_PO_MSGCTXT_TEXT",7},
	{"SCE_PO_MSGCTXT_TEXT_EOL",14},
	{"SCE_PO_MSGID",2},
	{"SCE_PO_MSGID_TEXT",3},
	{"SCE_PO_MSGID_TEXT_EOL",12},
	{"SCE_PO_MSGSTR",4},
	{"SCE_PO_MSGSTR_TEXT",5},
	{"SCE_PO_MSGSTR_TEXT_EOL",13},
	{"SCE_PO_PROGRAMMER_COMMENT",9},
	{"SCE_PO_REFERENCE",10},
	{"SCE_PROPS_ASSIGNMENT",3},
	{"SCE_PROPS_COMMENT",1},
	{"SCE_PROPS_DEFAULT",0},
	{"SCE_PROPS_DEFVAL",4},
	{"SCE_PROPS_KEY",5},
	{"SCE_PROPS_SECTION",2},
	{"SCE_PS_BADSTRINGCHAR",15},
	{"SCE_PS_BASE85STRING",14},
	{"SCE_PS_COMMENT",1},
	{"SCE_PS_DEFAULT",0},
	{"SCE_PS_DSC_COMMENT",2},
	{"SCE_PS_DSC_VALUE",3},
	{"SCE_PS_HEXSTRING",13},
	{"SCE_PS_IMMEVAL",8},
	{"SCE_PS_KEYWORD",6},
	{"SCE_PS_LITERAL",7},
	{"SCE_PS_NAME",5},
	{"SCE_PS_NUMBER",4},
	{"SCE_PS_PAREN_ARRAY",9},
	{"SCE_PS_PAREN_DICT",10},
	{"SCE_PS_PAREN_PROC",11},
	{"SCE_PS_TEXT",12},
	{"SCE_P_CHARACTER",4},
	{"SCE_P_CLASSNAME",8},
	{"SCE_P_COMMENTBLOCK",12},
	{"SCE_P_COMMENTLINE",1},
	{"SCE_P_DECORATOR",15},
	{"SCE_P_DEFAULT",0},
	{"SCE_P_DEFNAME",9},
	{"SCE_P_IDENTIFIER",11},
	{"SCE_P_NUMBER",2},
	{"SCE_P_OPERATOR",10},
	{"SCE_P_STRING",3},
	{"SCE_P_STRINGEOL",13},
	{"SCE_P_TRIPLE",6},
	{"SCE_P_TRIPLEDOUBLE",7},
	{"SCE_P_WORD",5},
	{"SCE_P_WORD2",14},
	{"SCE_RB_BACKTICKS",18},
	{"SCE_RB_CHARACTER",7},
	{"SCE_RB_CLASSNAME",8},
	{"SCE_RB_CLASS_VAR",17},
	{"SCE_RB_COMMENTLINE",2},
	{"SCE_RB_DATASECTION",19},
	{"SCE_RB_DEFAULT",0},
	{"SCE_RB_DEFNAME",9},
	{"SCE_RB_ERROR",1},
	{"SCE_RB_GLOBAL",13},
	{"SCE_RB_HERE_DELIM",20},
	{"SCE_RB_HERE_Q",21},
	{"SCE_RB_HERE_QQ",22},
	{"SCE_RB_HERE_QX",23},
	{"SCE_RB_IDENTIFIER",11},
	{"SCE_RB_INSTANCE_VAR",16},
	{"SCE_RB_MODULE_NAME",15},
	{"SCE_RB_NUMBER",4},
	{"SCE_RB_OPERATOR",10},
	{"SCE_RB_POD",3},
	{"SCE_RB_REGEX",12},
	{"SCE_RB_STDERR",40},
	{"SCE_RB_STDIN",30},
	{"SCE_RB_STDOUT",31},
	{"SCE_RB_STRING",6},
	{"SCE_RB_STRING_Q",24},
	{"SCE_RB_STRING_QQ",25},
	{"SCE_RB_STRING_QR",27},
	{"SCE_RB_STRING_QW",28},
	{"SCE_RB_STRING_QX",26},
	{"SCE_RB_SYMBOL",14},
	{"SCE_RB_UPPER_BOUND",41},
	{"SCE_RB_WORD",5},
	{"SCE_RB_WORD_DEMOTED",29},
	{"SCE_REBOL_BINARY",11},
	{"SCE_REBOL_BRACEDSTRING",7},
	{"SCE_REBOL_CHARACTER",5},
	{"SCE_REBOL_COMMENTBLOCK",2},
	{"SCE_REBOL_COMMENTLINE",1},
	{"SCE_REBOL_DATE",18},
	{"SCE_REBOL_DEFAULT",0},
	{"SCE_REBOL_EMAIL",16},
	{"SCE_REBOL_FILE",15},
	{"SCE_REBOL_IDENTIFIER",20},
	{"SCE_REBOL_ISSUE",13},
	{"SCE_REBOL_MONEY",12},
	{"SCE_REBOL_NUMBER",8},
	{"SCE_REBOL_OPERATOR",4},
	{"SCE_REBOL_PAIR",9},
	{"SCE_REBOL_PREFACE",3},
	{"SCE_REBOL_QUOTEDSTRING",6},
	{"SCE_REBOL_TAG",14},
	{"SCE_REBOL_TIME",19},
	{"SCE_REBOL_TUPLE",10},
	{"SCE_REBOL_URL",17},
	{"SCE_REBOL_WORD",21},
	{"SCE_REBOL_WORD2",22},
	{"SCE_REBOL_WORD3",23},
	{"SCE_REBOL_WORD4",24},
	{"SCE_REBOL_WORD5",25},
	{"SCE_REBOL_WORD6",26},
	{"SCE_REBOL_WORD7",27},
	{"SCE_REBOL_WORD8",28},
	{"SCE_RUST_CHARACTER",15},
	{"SCE_RUST_COMMENTBLOCK",1},
	{"SCE_RUST_COMMENTBLOCKDOC",3},
	{"SCE_RUST_COMMENTLINE",2},
	{"SCE_RUST_COMMENTLINEDOC",4},
	{"SCE_RUST_DEFAULT",0},
	{"SCE_RUST_IDENTIFIER",17},
	{"SCE_RUST_LEXERROR",20},
	{"SCE_RUST_LIFETIME",18},
	{"SCE_RUST_MACRO",19},
	{"SCE_RUST_NUMBER",5},
	{"SCE_RUST_OPERATOR",16},
	{"SCE_RUST_STRING",13},
	{"SCE_RUST_STRINGR",14},
	{"SCE_RUST_WORD",6},
	{"SCE_RUST_WORD2",7},
	{"SCE_RUST_WORD3",8},
	{"SCE_RUST_WORD4",9},
	{"SCE_RUST_WORD5",10},
	{"SCE_RUST_WORD6",11},
	{"SCE_RUST_WORD7",12},
	{"SCE_R_BASEKWORD",3},
	{"SCE_R_COMMENT",1},
	{"SCE_R_DEFAULT",0},
	{"SCE_R_IDENTIFIER",9},
	{"SCE_R_INFIX",10},
	{"SCE_R_INFIXEOL",11},
	{"SCE_R_KWORD",2},
	{"SCE_R_NUMBER",5},
	{"SCE_R_OPERATOR",8},
	{"SCE_R_OTHERKWORD",4},
	{"SCE_R_STRING",6},
	{"SCE_R_STRING2",7},
	{"SCE_SCRIPTOL_CHARACTER",8},
	{"SCE_SCRIPTOL_CLASSNAME",14},
	{"SCE_SCRIPTOL_COMMENTBLOCK",5},
	{"SCE_SCRIPTOL_COMMENTLINE",2},
	{"SCE_SCRIPTOL_CSTYLE",4},
	{"SCE_SCRIPTOL_DEFAULT",0},
	{"SCE_SCRIPTOL_IDENTIFIER",12},
	{"SCE_SCRIPTOL_KEYWORD",10},
	{"SCE_SCRIPTOL_NUMBER",6},
	{"SCE_SCRIPTOL_OPERATOR",11},
	{"SCE_SCRIPTOL_PERSISTENT",3},
	{"SCE_SCRIPTOL_PREPROCESSOR",15},
	{"SCE_SCRIPTOL_STRING",7},
	{"SCE_SCRIPTOL_STRINGEOL",9},
	{"SCE_SCRIPTOL_TRIPLE",13},
	{"SCE_SCRIPTOL_WHITE",1},
	{"SCE_SH_BACKTICKS",11},
	{"SCE_SH_CHARACTER",6},
	{"SCE_SH_COMMENTLINE",2},
	{"SCE_SH_DEFAULT",0},
	{"SCE_SH_ERROR",1},
	{"SCE_SH_HERE_DELIM",12},
	{"SCE_SH_HERE_Q",13},
	{"SCE_SH_IDENTIFIER",8},
	{"SCE_SH_NUMBER",3},
	{"SCE_SH_OPERATOR",7},
	{"SCE_SH_PARAM",10},
	{"SCE_SH_SCALAR",9},
	{"SCE_SH_STRING",5},
	{"SCE_SH_WORD",4},
	{"SCE_SML_CHAR",9},
	{"SCE_SML_COMMENT",12},
	{"SCE_SML_COMMENT1",13},
	{"SCE_SML_COMMENT2",14},
	{"SCE_SML_COMMENT3",15},
	{"SCE_SML_DEFAULT",0},
	{"SCE_SML_IDENTIFIER",1},
	{"SCE_SML_KEYWORD",3},
	{"SCE_SML_KEYWORD2",4},
	{"SCE_SML_KEYWORD3",5},
	{"SCE_SML_LINENUM",6},
	{"SCE_SML_NUMBER",8},
	{"SCE_SML_OPERATOR",7},
	{"SCE_SML_STRING",11},
	{"SCE_SML_TAGNAME",2},
	{"SCE_SN_CODE",1},
	{"SCE_SN_COMMENTLINE",2},
	{"SCE_SN_COMMENTLINEBANG",3},
	{"SCE_SN_DEFAULT",0},
	{"SCE_SN_IDENTIFIER",11},
	{"SCE_SN_NUMBER",4},
	{"SCE_SN_OPERATOR",10},
	{"SCE_SN_PREPROCESSOR",9},
	{"SCE_SN_REGEXTAG",13},
	{"SCE_SN_SIGNAL",14},
	{"SCE_SN_STRING",6},
	{"SCE_SN_STRINGEOL",12},
	{"SCE_SN_USER",19},
	{"SCE_SN_WORD",5},
	{"SCE_SN_WORD2",7},
	{"SCE_SN_WORD3",8},
	{"SCE_SORCUS_COMMAND",1},
	{"SCE_SORCUS_COMMENTLINE",3},
	{"SCE_SORCUS_CONSTANT",9},
	{"SCE_SORCUS_DEFAULT",0},
	{"SCE_SORCUS_IDENTIFIER",6},
	{"SCE_SORCUS_NUMBER",8},
	{"SCE_SORCUS_OPERATOR",7},
	{"SCE_SORCUS_PARAMETER",2},
	{"SCE_SORCUS_STRING",4},
	{"SCE_SORCUS_STRINGEOL",5},
	{"SCE_SPICE_COMMENTLINE",8},
	{"SCE_SPICE_DEFAULT",0},
	{"SCE_SPICE_DELIMITER",6},
	{"SCE_SPICE_IDENTIFIER",1},
	{"SCE_SPICE_KEYWORD",2},
	{"SCE_SPICE_KEYWORD2",3},
	{"SCE_SPICE_KEYWORD3",4},
	{"SCE_SPICE_NUMBER",5},
	{"SCE_SPICE_VALUE",7},
	{"SCE_SQL_CHARACTER",7},
	{"SCE_SQL_COMMENT",1},
	{"SCE_SQL_COMMENTDOC",3},
	{"SCE_SQL_COMMENTDOCKEYWORD",17},
	{"SCE_SQL_COMMENTDOCKEYWORDERROR",18},
	{"SCE_SQL_COMMENTLINE",2},
	{"SCE_SQL_COMMENTLINEDOC",15},
	{"SCE_SQL_DEFAULT",0},
	{"SCE_SQL_IDENTIFIER",11},
	{"SCE_SQL_NUMBER",4},
	{"SCE_SQL_OPERATOR",10},
	{"SCE_SQL_QUOTEDIDENTIFIER",23},
	{"SCE_SQL_SQLPLUS",8},
	{"SCE_SQL_SQLPLUS_COMMENT",13},
	{"SCE_SQL_SQLPLUS_PROMPT",9},
	{"SCE_SQL_STRING",6},
	{"SCE_SQL_USER1",19},
	{"SCE_SQL_USER2",20},
	{"SCE_SQL_USER3",21},
	{"SCE_SQL_USER4",22},
	{"SCE_SQL_WORD",5},
	{"SCE_SQL_WORD2",16},
	{"SCE_STTXT_CHARACTER",11},
	{"SCE_STTXT_COMMENT",1},
	{"SCE_STTXT_COMMENTLINE",2},
	{"SCE_STTXT_DATETIME",16},
	{"SCE_STTXT_DEFAULT",0},
	{"SCE_STTXT_FB",6},
	{"SCE_STTXT_FUNCTION",5},
	{"SCE_STTXT_HEXNUMBER",8},
	{"SCE_STTXT_IDENTIFIER",15},
	{"SCE_STTXT_KEYWORD",3},
	{"SCE_STTXT_NUMBER",7},
	{"SCE_STTXT_OPERATOR",10},
	{"SCE_STTXT_PRAGMA",9},
	{"SCE_STTXT_PRAGMAS",18},
	{"SCE_STTXT_STRING1",12},
	{"SCE_STTXT_STRING2",13},
	{"SCE_STTXT_STRINGEOL",14},
	{"SCE_STTXT_TYPE",4},
	{"SCE_STTXT_VARS",17},
	{"SCE_ST_ASSIGN",14},
	{"SCE_ST_BINARY",5},
	{"SCE_ST_BOOL",6},
	{"SCE_ST_CHARACTER",15},
	{"SCE_ST_COMMENT",3},
	{"SCE_ST_DEFAULT",0},
	{"SCE_ST_GLOBAL",10},
	{"SCE_ST_KWSEND",13},
	{"SCE_ST_NIL",9},
	{"SCE_ST_NUMBER",2},
	{"SCE_ST_RETURN",11},
	{"SCE_ST_SELF",7},
	{"SCE_ST_SPECIAL",12},
	{"SCE_ST_SPEC_SEL",16},
	{"SCE_ST_STRING",1},
	{"SCE_ST_SUPER",8},
	{"SCE_ST_SYMBOL",4},
	{"SCE_T3_BLOCK_COMMENT",3},
	{"SCE_T3_BRACE",20},
	{"SCE_T3_DEFAULT",0},
	{"SCE_T3_D_STRING",10},
	{"SCE_T3_HTML_DEFAULT",15},
	{"SCE_T3_HTML_STRING",16},
	{"SCE_T3_HTML_TAG",14},
	{"SCE_T3_IDENTIFIER",8},
	{"SCE_T3_KEYWORD",6},
	{"SCE_T3_LIB_DIRECTIVE",12},
	{"SCE_T3_LINE_COMMENT",4},
	{"SCE_T3_MSG_PARAM",13},
	{"SCE_T3_NUMBER",7},
	{"SCE_T3_OPERATOR",5},
	{"SCE_T3_PREPROCESSOR",2},
	{"SCE_T3_S_STRING",9},
	{"SCE_T3_USER1",17},
	{"SCE_T3_USER2",18},
	{"SCE_T3_USER3",19},
	{"SCE_T3_X_DEFAULT",1},
	{"SCE_T3_X_STRING",11},
	{"SCE_TCL_BLOCK_COMMENT",21},
	{"SCE_TCL_COMMENT",1},
	{"SCE_TCL_COMMENTLINE",2},
	{"SCE_TCL_COMMENT_BOX",20},
	{"SCE_TCL_DEFAULT",0},
	{"SCE_TCL_EXPAND",11},
	{"SCE_TCL_IDENTIFIER",7},
	{"SCE_TCL_IN_QUOTE",5},
	{"SCE_TCL_MODIFIER",10},
	{"SCE_TCL_NUMBER",3},
	{"SCE_TCL_OPERATOR",6},
	{"SCE_TCL_SUBSTITUTION",8},
	{"SCE_TCL_SUB_BRACE",9},
	{"SCE_TCL_WORD",12},
	{"SCE_TCL_WORD2",13},
	{"SCE_TCL_WORD3",14},
	{"SCE_TCL_WORD4",15},
	{"SCE_TCL_WORD5",16},
	{"SCE_TCL_WORD6",17},
	{"SCE_TCL_WORD7",18},
	{"SCE_TCL_WORD8",19},
	{"SCE_TCL_WORD_IN_QUOTE",4},
	{"SCE_TCMD_CLABEL",10},
	{"SCE_TCMD_COMMAND",5},
	{"SCE_TCMD_COMMENT",1},
	{"SCE_TCMD_DEFAULT",0},
	{"SCE_TCMD_ENVIRONMENT",8},
	{"SCE_TCMD_EXPANSION",9},
	{"SCE_TCMD_HIDE",4},
	{"SCE_TCMD_IDENTIFIER",6},
	{"SCE_TCMD_LABEL",3},
	{"SCE_TCMD_OPERATOR",7},
	{"SCE_TCMD_WORD",2},
	{"SCE_TEX_COMMAND",4},
	{"SCE_TEX_DEFAULT",0},
	{"SCE_TEX_GROUP",2},
	{"SCE_TEX_SPECIAL",1},
	{"SCE_TEX_SYMBOL",3},
	{"SCE_TEX_TEXT",5},
	{"SCE_TXT2TAGS_BLOCKQUOTE",15},
	{"SCE_TXT2TAGS_CODE",19},
	{"SCE_TXT2TAGS_CODE2",20},
	{"SCE_TXT2TAGS_CODEBK",21},
	{"SCE_TXT2TAGS_COMMENT",22},
	{"SCE_TXT2TAGS_DEFAULT",0},
	{"SCE_TXT2TAGS_EM1",4},
	{"SCE_TXT2TAGS_EM2",5},
	{"SCE_TXT2TAGS_HEADER1",6},
	{"SCE_TXT2TAGS_HEADER2",7},
	{"SCE_TXT2TAGS_HEADER3",8},
	{"SCE_TXT2TAGS_HEADER4",9},
	{"SCE_TXT2TAGS_HEADER5",10},
	{"SCE_TXT2TAGS_HEADER6",11},
	{"SCE_TXT2TAGS_HRULE",17},
	{"SCE_TXT2TAGS_LINE_BEGIN",1},
	{"SCE_TXT2TAGS_LINK",18},
	{"SCE_TXT2TAGS_OLIST_ITEM",14},
	{"SCE_TXT2TAGS_OPTION",23},
	{"SCE_TXT2TAGS_POSTPROC",25},
	{"SCE_TXT2TAGS_PRECHAR",12},
	{"SCE_TXT2TAGS_PREPROC",24},
	{"SCE_TXT2TAGS_STRIKEOUT",16},
	{"SCE_TXT2TAGS_STRONG1",2},
	{"SCE_TXT2TAGS_STRONG2",3},
	{"SCE_TXT2TAGS_ULIST_ITEM",13},
	{"SCE_VHDL_ATTRIBUTE",10},
	{"SCE_VHDL_COMMENT",1},
	{"SCE_VHDL_COMMENTLINEBANG",2},
	{"SCE_VHDL_DEFAULT",0},
	{"SCE_VHDL_IDENTIFIER",6},
	{"SCE_VHDL_KEYWORD",8},
	{"SCE_VHDL_NUMBER",3},
	{"SCE_VHDL_OPERATOR",5},
	{"SCE_VHDL_STDFUNCTION",11},
	{"SCE_VHDL_STDOPERATOR",9},
	{"SCE_VHDL_STDPACKAGE",12},
	{"SCE_VHDL_STDTYPE",13},
	{"SCE_VHDL_STRING",4},
	{"SCE_VHDL_STRINGEOL",7},
	{"SCE_VHDL_USERWORD",14},
	{"SCE_VISUALPROLOG_ANONYMOUS",10},
	{"SCE_VISUALPROLOG_CHARACTER",13},
	{"SCE_VISUALPROLOG_CHARACTER_ESCAPE_ERROR",15},
	{"SCE_VISUALPROLOG_CHARACTER_TOO_MANY",14},
	{"SCE_VISUALPROLOG_COMMENT_BLOCK",4},
	{"SCE_VISUALPROLOG_COMMENT_KEY",6},
	{"SCE_VISUALPROLOG_COMMENT_KEY_ERROR",7},
	{"SCE_VISUALPROLOG_COMMENT_LINE",5},
	{"SCE_VISUALPROLOG_DEFAULT",0},
	{"SCE_VISUALPROLOG_IDENTIFIER",8},
	{"SCE_VISUALPROLOG_KEY_DIRECTIVE",3},
	{"SCE_VISUALPROLOG_KEY_MAJOR",1},
	{"SCE_VISUALPROLOG_KEY_MINOR",2},
	{"SCE_VISUALPROLOG_NUMBER",11},
	{"SCE_VISUALPROLOG_OPERATOR",12},
	{"SCE_VISUALPROLOG_STRING",16},
	{"SCE_VISUALPROLOG_STRING_EOL_OPEN",19},
	{"SCE_VISUALPROLOG_STRING_ESCAPE",17},
	{"SCE_VISUALPROLOG_STRING_ESCAPE_ERROR",18},
	{"SCE_VISUALPROLOG_STRING_VERBATIM",20},
	{"SCE_VISUALPROLOG_STRING_VERBATIM_EOL",22},
	{"SCE_VISUALPROLOG_STRING_VERBATIM_SPECIAL",21},
	{"SCE_VISUALPROLOG_VARIABLE",9},
	{"SCE_V_COMMENT",1},
	{"SCE_V_COMMENTLINE",2},
	{"SCE_V_COMMENTLINEBANG",3},
	{"SCE_V_DEFAULT",0},
	{"SCE_V_IDENTIFIER",11},
	{"SCE_V_NUMBER",4},
	{"SCE_V_OPERATOR",10},
	{"SCE_V_PREPROCESSOR",9},
	{"SCE_V_STRING",6},
	{"SCE_V_STRINGEOL",12},
	{"SCE_V_USER",19},
	{"SCE_V_WORD",5},
	{"SCE_V_WORD2",7},
	{"SCE_V_WORD3",8},
	{"SCE_YAML_COMMENT",1},
	{"SCE_YAML_DEFAULT",0},
	{"SCE_YAML_DOCUMENT",6},
	{"SCE_YAML_ERROR",8},
	{"SCE_YAML_IDENTIFIER",2},
	{"SCE_YAML_KEYWORD",3},
	{"SCE_YAML_NUMBER",4},
	{"SCE_YAML_OPERATOR",9},
	{"SCE_YAML_REFERENCE",5},
	{"SCE_YAML_TEXT",7},
	{"SCFIND_MATCHCASE",0x4},
	{"SCFIND_POSIX",0x00400000},
	{"SCFIND_REGEXP",0x00200000},
	{"SCFIND_WHOLEWORD",0x2},
	{"SCFIND_WORDSTART",0x00100000},
	{"SCI_ANNOTATIONGETLINES",2546},
	{"SCI_ANNOTATIONGETSTYLE",2543},
	{"SCI_ANNOTATIONGETSTYLEOFFSET",2551},
	{"SCI_ANNOTATIONGETSTYLES",2545},
	{"SCI_ANNOTATIONGETTEXT",2541},
	{"SCI_ANNOTATIONGETVISIBLE",2549},
	{"SCI_ANNOTATIONSETSTYLE",2542},
	{"SCI_ANNOTATIONSETSTYLEOFFSET",2550},
	{"SCI_ANNOTATIONSETSTYLES",2544},
	{"SCI_ANNOTATIONSETTEXT",2540},
	{"SCI_ANNOTATIONSETVISIBLE",2548},
	{"SCI_AUTOCGETAUTOHIDE",2119},
	{"SCI_AUTOCGETCANCELATSTART",2111},
	{"SCI_AUTOCGETCASEINSENSITIVEBEHAVIOUR",2635},
	{"SCI_AUTOCGETCHOOSESINGLE",2114},
	{"SCI_AUTOCGETCURRENT",2445},
	{"SCI_AUTOCGETCURRENTTEXT",2610},
	{"SCI_AUTOCGETDROPRESTOFWORD",2271},
	{"SCI_AUTOCGETIGNORECASE",2116},
	{"SCI_AUTOCGETMAXHEIGHT",2211},
	{"SCI_AUTOCGETMAXWIDTH",2209},
	{"SCI_AUTOCGETORDER",2661},
	{"SCI_AUTOCGETSEPARATOR",2107},
	{"SCI_AUTOCGETTYPESEPARATOR",2285},
	{"SCI_AUTOCSETAUTOHIDE",2118},
	{"SCI_AUTOCSETCANCELATSTART",2110},
	{"SCI_AUTOCSETCASEINSENSITIVEBEHAVIOUR",2634},
	{"SCI_AUTOCSETCHOOSESINGLE",2113},
	{"SCI_AUTOCSETDROPRESTOFWORD",2270},
	{"SCI_AUTOCSETFILLUPS",2112},
	{"SCI_AUTOCSETIGNORECASE",2115},
	{"SCI_AUTOCSETMAXHEIGHT",2210},
	{"SCI_AUTOCSETMAXWIDTH",2208},
	{"SCI_AUTOCSETORDER",2660},
	{"SCI_AUTOCSETSEPARATOR",2106},
	{"SCI_AUTOCSETTYPESEPARATOR",2286},
	{"SCI_CALLTIPSETBACK",2205},
	{"SCI_CALLTIPSETFORE",2206},
	{"SCI_CALLTIPSETFOREHLT",2207},
	{"SCI_CALLTIPSETPOSITION",2213},
	{"SCI_CALLTIPSETPOSSTART",2214},
	{"SCI_CALLTIPUSESTYLE",2212},
	{"SCI_DISTANCETOSECONDARYSTYLES",4025},
	{"SCI_GETADDITIONALCARETFORE",2605},
	{"SCI_GETADDITIONALCARETSBLINK",2568},
	{"SCI_GETADDITIONALCARETSVISIBLE",2609},
	{"SCI_GETADDITIONALSELALPHA",2603},
	{"SCI_GETADDITIONALSELECTIONTYPING",2566},
	{"SCI_GETALLLINESVISIBLE",2236},
	{"SCI_GETANCHOR",2009},
	{"SCI_GETAUTOMATICFOLD",2664},
	{"SCI_GETBACKSPACEUNINDENTS",2263},
	{"SCI_GETBUFFEREDDRAW",2034},
	{"SCI_GETCARETFORE",2138},
	{"SCI_GETCARETLINEBACK",2097},
	{"SCI_GETCARETLINEBACKALPHA",2471},
	{"SCI_GETCARETLINEVISIBLE",2095},
	{"SCI_GETCARETLINEVISIBLEALWAYS",2654},
	{"SCI_GETCARETPERIOD",2075},
	{"SCI_GETCARETSTICKY",2457},
	{"SCI_GETCARETSTYLE",2513},
	{"SCI_GETCARETWIDTH",2189},
	{"SCI_GETCHARACTERPOINTER",2520},
	{"SCI_GETCHARAT",2007},
	{"SCI_GETCODEPAGE",2137},
	{"SCI_GETCOLUMN",2129},
	{"SCI_GETCONTROLCHARSYMBOL",2389},
	{"SCI_GETCURRENTPOS",2008},
	{"SCI_GETCURSOR",2387},
	{"SCI_GETDIRECTFUNCTION",2184},
	{"SCI_GETDIRECTPOINTER",2185},
	{"SCI_GETDOCPOINTER",2357},
	{"SCI_GETEDGECOLOUR",2364},
	{"SCI_GETEDGECOLUMN",2360},
	{"SCI_GETEDGEMODE",2362},
	{"SCI_GETENDATLASTLINE",2278},
	{"SCI_GETENDSTYLED",2028},
	{"SCI_GETEOLMODE",2030},
	{"SCI_GETEXTRAASCENT",2526},
	{"SCI_GETEXTRADESCENT",2528},
	{"SCI_GETFIRSTVISIBLELINE",2152},
	{"SCI_GETFOCUS",2381},
	{"SCI_GETFOLDEXPANDED",2230},
	{"SCI_GETFOLDLEVEL",2223},
	{"SCI_GETFOLDPARENT",2225},
	{"SCI_GETFONTQUALITY",2612},
	{"SCI_GETGAPPOSITION",2644},
	{"SCI_GETHIGHLIGHTGUIDE",2135},
	{"SCI_GETHOTSPOTACTIVEUNDERLINE",2496},
	{"SCI_GETHOTSPOTSINGLELINE",2497},
	{"SCI_GETHSCROLLBAR",2131},
	{"SCI_GETIDENTIFIER",2623},
	{"SCI_GETINDENT",2123},
	{"SCI_GETINDENTATIONGUIDES",2133},
	{"SCI_GETINDICATORCURRENT",2501},
	{"SCI_GETINDICATORVALUE",2503},
	{"SCI_GETKEYSUNICODE",2522},
	{"SCI_GETLAYOUTCACHE",2273},
	{"SCI_GETLENGTH",2006},
	{"SCI_GETLEXER",4002},
	{"SCI_GETLEXERLANGUAGE",4012},
	{"SCI_GETLINECOUNT",2154},
	{"SCI_GETLINEENDPOSITION",2136},
	{"SCI_GETLINEENDTYPESACTIVE",2658},
	{"SCI_GETLINEENDTYPESALLOWED",2657},
	{"SCI_GETLINEENDTYPESSUPPORTED",4018},
	{"SCI_GETLINEINDENTATION",2127},
	{"SCI_GETLINEINDENTPOSITION",2128},
	{"SCI_GETLINESTATE",2093},
	{"SCI_GETLINEVISIBLE",2228},
	{"SCI_GETMAINSELECTION",2575},
	{"SCI_GETMARGINCURSORN",2249},
	{"SCI_GETMARGINLEFT",2156},
	{"SCI_GETMARGINMASKN",2245},
	{"SCI_GETMARGINOPTIONS",2557},
	{"SCI_GETMARGINRIGHT",2158},
	{"SCI_GETMARGINSENSITIVEN",2247},
	{"SCI_GETMARGINTYPEN",2241},
	{"SCI_GETMARGINWIDTHN",2243},
	{"SCI_GETMAXLINESTATE",2094},
	{"SCI_GETMODEVENTMASK",2378},
	{"SCI_GETMODIFY",2159},
	{"SCI_GETMOUSEDOWNCAPTURES",2385},
	{"SCI_GETMOUSEDWELLTIME",2265},
	{"SCI_GETMOUSESELECTIONRECTANGULARSWITCH",2669},
	{"SCI_GETMULTIPASTE",2615},
	{"SCI_GETMULTIPLESELECTION",2564},
	{"SCI_GETOVERTYPE",2187},
	{"SCI_GETPASTECONVERTENDINGS",2468},
	{"SCI_GETPOSITIONCACHE",2515},
	{"SCI_GETPRIMARYSTYLEFROMSTYLE",4028},
	{"SCI_GETPRINTCOLOURMODE",2149},
	{"SCI_GETPRINTMAGNIFICATION",2147},
	{"SCI_GETPRINTWRAPMODE",2407},
	{"SCI_GETPROPERTY",4008},
	{"SCI_GETPROPERTYEXPANDED",4009},
	{"SCI_GETPROPERTYINT",4010},
	{"SCI_GETPUNCTUATIONCHARS",2649},
	{"SCI_GETREADONLY",2140},
	{"SCI_GETRECTANGULARSELECTIONANCHOR",2591},
	{"SCI_GETRECTANGULARSELECTIONANCHORVIRTUALSPACE",2595},
	{"SCI_GETRECTANGULARSELECTIONCARET",2589},
	{"SCI_GETRECTANGULARSELECTIONCARETVIRTUALSPACE",2593},
	{"SCI_GETRECTANGULARSELECTIONMODIFIER",2599},
	{"SCI_GETREPRESENTATION",2666},
	{"SCI_GETSCROLLWIDTH",2275},
	{"SCI_GETSCROLLWIDTHTRACKING",2517},
	{"SCI_GETSEARCHFLAGS",2199},
	{"SCI_GETSELALPHA",2477},
	{"SCI_GETSELECTIONEMPTY",2650},
	{"SCI_GETSELECTIONEND",2145},
	{"SCI_GETSELECTIONMODE",2423},
	{"SCI_GETSELECTIONNANCHOR",2579},
	{"SCI_GETSELECTIONNANCHORVIRTUALSPACE",2583},
	{"SCI_GETSELECTIONNCARET",2577},
	{"SCI_GETSELECTIONNCARETVIRTUALSPACE",2581},
	{"SCI_GETSELECTIONNEND",2587},
	{"SCI_GETSELECTIONNSTART",2585},
	{"SCI_GETSELECTIONS",2570},
	{"SCI_GETSELECTIONSTART",2143},
	{"SCI_GETSELEOLFILLED",2479},
	{"SCI_GETSTATUS",2383},
	{"SCI_GETSTYLEAT",2010},
	{"SCI_GETSTYLEBITS",2091},
	{"SCI_GETSTYLEBITSNEEDED",4011},
	{"SCI_GETSTYLEFROMSUBSTYLE",4027},
	{"SCI_GETSUBSTYLEBASES",4026},
	{"SCI_GETSUBSTYLESLENGTH",4022},
	{"SCI_GETSUBSTYLESSTART",4021},
	{"SCI_GETTABINDENTS",2261},
	{"SCI_GETTABWIDTH",2121},
	{"SCI_GETTAG",2616},
	{"SCI_GETTARGETEND",2193},
	{"SCI_GETTARGETSTART",2191},
	{"SCI_GETTECHNOLOGY",2631},
	{"SCI_GETTEXTLENGTH",2183},
	{"SCI_GETTWOPHASEDRAW",2283},
	{"SCI_GETUNDOCOLLECTION",2019},
	{"SCI_GETUSETABS",2125},
	{"SCI_GETVIEWEOL",2355},
	{"SCI_GETVIEWWS",2020},
	{"SCI_GETVIRTUALSPACEOPTIONS",2597},
	{"SCI_GETVSCROLLBAR",2281},
	{"SCI_GETWHITESPACECHARS",2647},
	{"SCI_GETWHITESPACESIZE",2087},
	{"SCI_GETWORDCHARS",2646},
	{"SCI_GETWRAPINDENTMODE",2473},
	{"SCI_GETWRAPMODE",2269},
	{"SCI_GETWRAPSTARTINDENT",2465},
	{"SCI_GETWRAPVISUALFLAGS",2461},
	{"SCI_GETWRAPVISUALFLAGSLOCATION",2463},
	{"SCI_GETXOFFSET",2398},
	{"SCI_GETZOOM",2374},
	{"SCI_INDICGETALPHA",2524},
	{"SCI_INDICGETFORE",2083},
	{"SCI_INDICGETOUTLINEALPHA",2559},
	{"SCI_INDICGETSTYLE",2081},
	{"SCI_INDICGETUNDER",2511},
	{"SCI_INDICSETALPHA",2523},
	{"SCI_INDICSETFORE",2082},
	{"SCI_INDICSETOUTLINEALPHA",2558},
	{"SCI_INDICSETSTYLE",2080},
	{"SCI_INDICSETUNDER",2510},
	{"SCI_LEXER_START",4000},
	{"SCI_LINESONSCREEN",2370},
	{"SCI_MARGINGETSTYLE",2533},
	{"SCI_MARGINGETSTYLEOFFSET",2538},
	{"SCI_MARGINGETSTYLES",2535},
	{"SCI_MARGINGETTEXT",2531},
	{"SCI_MARGINSETSTYLE",2532},
	{"SCI_MARGINSETSTYLEOFFSET",2537},
	{"SCI_MARGINSETSTYLES",2534},
	{"SCI_MARGINSETTEXT",2530},
	{"SCI_MARKERSETALPHA",2476},
	{"SCI_MARKERSETBACK",2042},
	{"SCI_MARKERSETBACKSELECTED",2292},
	{"SCI_MARKERSETFORE",2041},
	{"SCI_OPTIONAL_START",3000},
	{"SCI_RGBAIMAGESETHEIGHT",2625},
	{"SCI_RGBAIMAGESETSCALE",2651},
	{"SCI_RGBAIMAGESETWIDTH",2624},
	{"SCI_SELECTIONISRECTANGLE",2372},
	{"SCI_SETADDITIONALCARETFORE",2604},
	{"SCI_SETADDITIONALCARETSBLINK",2567},
	{"SCI_SETADDITIONALCARETSVISIBLE",2608},
	{"SCI_SETADDITIONALSELALPHA",2602},
	{"SCI_SETADDITIONALSELBACK",2601},
	{"SCI_SETADDITIONALSELECTIONTYPING",2565},
	{"SCI_SETADDITIONALSELFORE",2600},
	{"SCI_SETANCHOR",2026},
	{"SCI_SETAUTOMATICFOLD",2663},
	{"SCI_SETBACKSPACEUNINDENTS",2262},
	{"SCI_SETBUFFEREDDRAW",2035},
	{"SCI_SETCARETFORE",2069},
	{"SCI_SETCARETLINEBACK",2098},
	{"SCI_SETCARETLINEBACKALPHA",2470},
	{"SCI_SETCARETLINEVISIBLE",2096},
	{"SCI_SETCARETLINEVISIBLEALWAYS",2655},
	{"SCI_SETCARETPERIOD",2076},
	{"SCI_SETCARETSTICKY",2458},
	{"SCI_SETCARETSTYLE",2512},
	{"SCI_SETCARETWIDTH",2188},
	{"SCI_SETCODEPAGE",2037},
	{"SCI_SETCONTROLCHARSYMBOL",2388},
	{"SCI_SETCURRENTPOS",2141},
	{"SCI_SETCURSOR",2386},
	{"SCI_SETDOCPOINTER",2358},
	{"SCI_SETEDGECOLOUR",2365},
	{"SCI_SETEDGECOLUMN",2361},
	{"SCI_SETEDGEMODE",2363},
	{"SCI_SETENDATLASTLINE",2277},
	{"SCI_SETEOLMODE",2031},
	{"SCI_SETEXTRAASCENT",2525},
	{"SCI_SETEXTRADESCENT",2527},
	{"SCI_SETFIRSTVISIBLELINE",2613},
	{"SCI_SETFOCUS",2380},
	{"SCI_SETFOLDEXPANDED",2229},
	{"SCI_SETFOLDFLAGS",2233},
	{"SCI_SETFOLDLEVEL",2222},
	{"SCI_SETFONTQUALITY",2611},
	{"SCI_SETHIGHLIGHTGUIDE",2134},
	{"SCI_SETHOTSPOTACTIVEUNDERLINE",2412},
	{"SCI_SETHOTSPOTSINGLELINE",2421},
	{"SCI_SETHSCROLLBAR",2130},
	{"SCI_SETIDENTIFIER",2622},
	{"SCI_SETIDENTIFIERS",4024},
	{"SCI_SETINDENT",2122},
	{"SCI_SETINDENTATIONGUIDES",2132},
	{"SCI_SETINDICATORCURRENT",2500},
	{"SCI_SETINDICATORVALUE",2502},
	{"SCI_SETKEYSUNICODE",2521},
	{"SCI_SETKEYWORDS",4005},
	{"SCI_SETLAYOUTCACHE",2272},
	{"SCI_SETLEXER",4001},
	{"SCI_SETLEXERLANGUAGE",4006},
	{"SCI_SETLINEENDTYPESALLOWED",2656},
	{"SCI_SETLINEINDENTATION",2126},
	{"SCI_SETLINESTATE",2092},
	{"SCI_SETMAINSELECTION",2574},
	{"SCI_SETMARGINCURSORN",2248},
	{"SCI_SETMARGINLEFT",2155},
	{"SCI_SETMARGINMASKN",2244},
	{"SCI_SETMARGINOPTIONS",2539},
	{"SCI_SETMARGINRIGHT",2157},
	{"SCI_SETMARGINSENSITIVEN",2246},
	{"SCI_SETMARGINTYPEN",2240},
	{"SCI_SETMARGINWIDTHN",2242},
	{"SCI_SETMODEVENTMASK",2359},
	{"SCI_SETMOUSEDOWNCAPTURES",2384},
	{"SCI_SETMOUSEDWELLTIME",2264},
	{"SCI_SETMOUSESELECTIONRECTANGULARSWITCH",2668},
	{"SCI_SETMULTIPASTE",2614},
	{"SCI_SETMULTIPLESELECTION",2563},
	{"SCI_SETOVERTYPE",2186},
	{"SCI_SETPASTECONVERTENDINGS",2467},
	{"SCI_SETPOSITIONCACHE",2514},
	{"SCI_SETPRINTCOLOURMODE",2148},
	{"SCI_SETPRINTMAGNIFICATION",2146},
	{"SCI_SETPRINTWRAPMODE",2406},
	{"SCI_SETPROPERTY",4004},
	{"SCI_SETPUNCTUATIONCHARS",2648},
	{"SCI_SETREADONLY",2171},
	{"SCI_SETRECTANGULARSELECTIONANCHOR",2590},
	{"SCI_SETRECTANGULARSELECTIONANCHORVIRTUALSPACE",2594},
	{"SCI_SETRECTANGULARSELECTIONCARET",2588},
	{"SCI_SETRECTANGULARSELECTIONCARETVIRTUALSPACE",2592},
	{"SCI_SETRECTANGULARSELECTIONMODIFIER",2598},
	{"SCI_SETREPRESENTATION",2665},
	{"SCI_SETSCROLLWIDTH",2274},
	{"SCI_SETSCROLLWIDTHTRACKING",2516},
	{"SCI_SETSEARCHFLAGS",2198},
	{"SCI_SETSELALPHA",2478},
	{"SCI_SETSELECTIONEND",2144},
	{"SCI_SETSELECTIONMODE",2422},
	{"SCI_SETSELECTIONNANCHOR",2578},
	{"SCI_SETSELECTIONNANCHORVIRTUALSPACE",2582},
	{"SCI_SETSELECTIONNCARET",2576},
	{"SCI_SETSELECTIONNCARETVIRTUALSPACE",2580},
	{"SCI_SETSELECTIONNEND",2586},
	{"SCI_SETSELECTIONNSTART",2584},
	{"SCI_SETSELECTIONSTART",2142},
	{"SCI_SETSELEOLFILLED",2480},
	{"SCI_SETSTATUS",2382},
	{"SCI_SETSTYLEBITS",2090},
	{"SCI_SETTABINDENTS",2260},
	{"SCI_SETTABWIDTH",2036},
	{"SCI_SETTARGETEND",2192},
	{"SCI_SETTARGETSTART",2190},
	{"SCI_SETTECHNOLOGY",2630},
	{"SCI_SETTWOPHASEDRAW",2284},
	{"SCI_SETUNDOCOLLECTION",2012},
	{"SCI_SETUSETABS",2124},
	{"SCI_SETVIEWEOL",2356},
	{"SCI_SETVIEWWS",2021},
	{"SCI_SETVIRTUALSPACEOPTIONS",2596},
	{"SCI_SETVSCROLLBAR",2280},
	{"SCI_SETWHITESPACECHARS",2443},
	{"SCI_SETWHITESPACESIZE",2086},
	{"SCI_SETWORDCHARS",2077},
	{"SCI_SETWRAPINDENTMODE",2472},
	{"SCI_SETWRAPMODE",2268},
	{"SCI_SETWRAPSTARTINDENT",2464},
	{"SCI_SETWRAPVISUALFLAGS",2460},
	{"SCI_SETWRAPVISUALFLAGSLOCATION",2462},
	{"SCI_SETXOFFSET",2397},
	{"SCI_SETZOOM",2373},
	{"SCI_START",2000},
	{"SCI_STYLEGETBACK",2482},
	{"SCI_STYLEGETBOLD",2483},
	{"SCI_STYLEGETCASE",2489},
	{"SCI_STYLEGETCHANGEABLE",2492},
	{"SCI_STYLEGETCHARACTERSET",2490},
	{"SCI_STYLEGETEOLFILLED",2487},
	{"SCI_STYLEGETFONT",2486},
	{"SCI_STYLEGETFORE",2481},
	{"SCI_STYLEGETHOTSPOT",2493},
	{"SCI_STYLEGETITALIC",2484},
	{"SCI_STYLEGETSIZE",2485},
	{"SCI_STYLEGETSIZEFRACTIONAL",2062},
	{"SCI_STYLEGETUNDERLINE",2488},
	{"SCI_STYLEGETVISIBLE",2491},
	{"SCI_STYLEGETWEIGHT",2064},
	{"SCI_STYLESETBACK",2052},
	{"SCI_STYLESETBOLD",2053},
	{"SCI_STYLESETCASE",2060},
	{"SCI_STYLESETCHANGEABLE",2099},
	{"SCI_STYLESETCHARACTERSET",2066},
	{"SCI_STYLESETEOLFILLED",2057},
	{"SCI_STYLESETFONT",2056},
	{"SCI_STYLESETFORE",2051},
	{"SCI_STYLESETHOTSPOT",2409},
	{"SCI_STYLESETITALIC",2054},
	{"SCI_STYLESETSIZE",2055},
	{"SCI_STYLESETSIZEFRACTIONAL",2061},
	{"SCI_STYLESETUNDERLINE",2059},
	{"SCI_STYLESETVISIBLE",2074},
	{"SCI_STYLESETWEIGHT",2063},
	{"SCK_ADD",310},
	{"SCK_BACK",8},
	{"SCK_DELETE",308},
	{"SCK_DIVIDE",312},
	{"SCK_DOWN",300},
	{"SCK_END",305},
	{"SCK_ESCAPE",7},
	{"SCK_HOME",304},
	{"SCK_INSERT",309},
	{"SCK_LEFT",302},
	{"SCK_MENU",315},
	{"SCK_NEXT",307},
	{"SCK_PRIOR",306},
	{"SCK_RETURN",13},
	{"SCK_RIGHT",303},
	{"SCK_RWIN",314},
	{"SCK_SUBTRACT",311},
	{"SCK_TAB",9},
	{"SCK_UP",301},
	{"SCK_WIN",313},
	{"SCLEX_A68K",100},
	{"SCLEX_ABAQUS",84},
	{"SCLEX_ADA",20},
	{"SCLEX_APDL",61},
	{"SCLEX_ASM",34},
	{"SCLEX_ASN1",63},
	{"SCLEX_ASYMPTOTE",85},
	{"SCLEX_AU3",60},
	{"SCLEX_AUTOMATIC",1000},
	{"SCLEX_AVE",19},
	{"SCLEX_AVS",104},
	{"SCLEX_BAAN",31},
	{"SCLEX_BASH",62},
	{"SCLEX_BATCH",12},
	{"SCLEX_BLITZBASIC",66},
	{"SCLEX_BULLANT",27},
	{"SCLEX_CAML",65},
	{"SCLEX_CLW",45},
	{"SCLEX_CLWNOCASE",46},
	{"SCLEX_CMAKE",80},
	{"SCLEX_COBOL",92},
	{"SCLEX_COFFEESCRIPT",102},
	{"SCLEX_CONF",17},
	{"SCLEX_CONTAINER",0},
	{"SCLEX_CPP",3},
	{"SCLEX_CPPNOCASE",35},
	{"SCLEX_CSOUND",74},
	{"SCLEX_CSS",38},
	{"SCLEX_D",79},
	{"SCLEX_DIFF",16},
	{"SCLEX_DMAP",112},
	{"SCLEX_ECL",105},
	{"SCLEX_EIFFEL",23},
	{"SCLEX_EIFFELKW",24},
	{"SCLEX_ERLANG",53},
	{"SCLEX_ERRORLIST",10},
	{"SCLEX_ESCRIPT",41},
	{"SCLEX_F77",37},
	{"SCLEX_FLAGSHIP",73},
	{"SCLEX_FORTH",52},
	{"SCLEX_FORTRAN",36},
	{"SCLEX_FREEBASIC",75},
	{"SCLEX_GAP",81},
	{"SCLEX_GUI4CLI",58},
	{"SCLEX_HASKELL",68},
	{"SCLEX_HTML",4},
	{"SCLEX_INNOSETUP",76},
	{"SCLEX_KIX",57},
	{"SCLEX_KVIRC",110},
	{"SCLEX_LATEX",14},
	{"SCLEX_LISP",21},
	{"SCLEX_LITERATEHASKELL",108},
	{"SCLEX_LOT",47},
	{"SCLEX_LOUT",40},
	{"SCLEX_LUA",15},
	{"SCLEX_MAGIK",87},
	{"SCLEX_MAKEFILE",11},
	{"SCLEX_MARKDOWN",98},
	{"SCLEX_MATLAB",32},
	{"SCLEX_METAPOST",50},
	{"SCLEX_MMIXAL",44},
	{"SCLEX_MODULA",101},
	{"SCLEX_MSSQL",55},
	{"SCLEX_MYSQL",89},
	{"SCLEX_NIMROD",96},
	{"SCLEX_NNCRONTAB",26},
	{"SCLEX_NSIS",43},
	{"SCLEX_NULL",1},
	{"SCLEX_OCTAVE",54},
	{"SCLEX_OPAL",77},
	{"SCLEX_OSCRIPT",106},
	{"SCLEX_PASCAL",18},
	{"SCLEX_PERL",6},
	{"SCLEX_PHPSCRIPT",69},
	{"SCLEX_PLM",82},
	{"SCLEX_PO",90},
	{"SCLEX_POV",39},
	{"SCLEX_POWERBASIC",51},
	{"SCLEX_POWERPRO",95},
	{"SCLEX_POWERSHELL",88},
	{"SCLEX_PROGRESS",83},
	{"SCLEX_PROPERTIES",9},
	{"SCLEX_PS",42},
	{"SCLEX_PUREBASIC",67},
	{"SCLEX_PYTHON",2},
	{"SCLEX_R",86},
	{"SCLEX_REBOL",71},
	{"SCLEX_RUBY",22},
	{"SCLEX_RUST",111},
	{"SCLEX_SCRIPTOL",33},
	{"SCLEX_SMALLTALK",72},
	{"SCLEX_SML",97},
	{"SCLEX_SORCUS",94},
	{"SCLEX_SPECMAN",59},
	{"SCLEX_SPICE",78},
	{"SCLEX_SQL",7},
	{"SCLEX_STTXT",109},
	{"SCLEX_TACL",93},
	{"SCLEX_TADS3",70},
	{"SCLEX_TAL",91},
	{"SCLEX_TCL",25},
	{"SCLEX_TCMD",103},
	{"SCLEX_TEX",49},
	{"SCLEX_TXT2TAGS",99},
	{"SCLEX_VB",8},
	{"SCLEX_VBSCRIPT",28},
	{"SCLEX_VERILOG",56},
	{"SCLEX_VHDL",64},
	{"SCLEX_VISUALPROLOG",107},
	{"SCLEX_XCODE",13},
	{"SCLEX_XML",5},
	{"SCLEX_YAML",48},
	{"SCMOD_ALT",4},
	{"SCMOD_CTRL",2},
	{"SCMOD_META",16},
	{"SCMOD_NORM",0},
	{"SCMOD_SHIFT",1},
	{"SCMOD_SUPER",8},
	{"SCVS_NONE",0},
	{"SCVS_RECTANGULARSELECTION",1},
	{"SCVS_USERACCESSIBLE",2},
	{"SCWS_INVISIBLE",0},
	{"SCWS_VISIBLEAFTERINDENT",2},
	{"SCWS_VISIBLEALWAYS",1},
	{"SC_ALPHA_NOALPHA",256},
	{"SC_ALPHA_OPAQUE",255},
	{"SC_ALPHA_TRANSPARENT",0},
	{"SC_AUTOMATICFOLD_CHANGE",0x0004},
	{"SC_AUTOMATICFOLD_CLICK",0x0002},
	{"SC_AUTOMATICFOLD_SHOW",0x0001},
	{"SC_CACHE_CARET",1},
	{"SC_CACHE_DOCUMENT",3},
	{"SC_CACHE_NONE",0},
	{"SC_CACHE_PAGE",2},
	{"SC_CARETSTICKY_OFF",0},
	{"SC_CARETSTICKY_ON",1},
	{"SC_CARETSTICKY_WHITESPACE",2},
	{"SC_CASEINSENSITIVEBEHAVIOUR_IGNORECASE",1},
	{"SC_CASEINSENSITIVEBEHAVIOUR_RESPECTCASE",0},
	{"SC_CASE_LOWER",2},
	{"SC_CASE_MIXED",0},
	{"SC_CASE_UPPER",1},
	{"SC_CHARSET_8859_15",1000},
	{"SC_CHARSET_ANSI",0},
	{"SC_CHARSET_ARABIC",178},
	{"SC_CHARSET_BALTIC",186},
	{"SC_CHARSET_CHINESEBIG5",136},
	{"SC_CHARSET_CYRILLIC",1251},
	{"SC_CHARSET_DEFAULT",1},
	{"SC_CHARSET_EASTEUROPE",238},
	{"SC_CHARSET_GB2312",134},
	{"SC_CHARSET_GREEK",161},
	{"SC_CHARSET_HANGUL",129},
	{"SC_CHARSET_HEBREW",177},
	{"SC_CHARSET_JOHAB",130},
	{"SC_CHARSET_MAC",77},
	{"SC_CHARSET_OEM",255},
	{"SC_CHARSET_RUSSIAN",204},
	{"SC_CHARSET_SHIFTJIS",128},
	{"SC_CHARSET_SYMBOL",2},
	{"SC_CHARSET_THAI",222},
	{"SC_CHARSET_TURKISH",162},
	{"SC_CHARSET_VIETNAMESE",163},
	{"SC_CP_UTF8",65001},
	{"SC_CURSORARROW",2},
	{"SC_CURSORNORMAL",-1},
	{"SC_CURSORREVERSEARROW",7},
	{"SC_CURSORWAIT",4},
	{"SC_EFF_QUALITY_ANTIALIASED",2},
	{"SC_EFF_QUALITY_DEFAULT",0},
	{"SC_EFF_QUALITY_LCD_OPTIMIZED",3},
	{"SC_EFF_QUALITY_MASK",0xF},
	{"SC_EFF_QUALITY_NON_ANTIALIASED",1},
	{"SC_EOL_CR",1},
	{"SC_EOL_CRLF",0},
	{"SC_EOL_LF",2},
	{"SC_FOLDACTION_CONTRACT",0},
	{"SC_FOLDACTION_EXPAND",1},
	{"SC_FOLDACTION_TOGGLE",2},
	{"SC_FOLDFLAG_LEVELNUMBERS",0x0040},
	{"SC_FOLDFLAG_LINEAFTER_CONTRACTED",0x0010},
	{"SC_FOLDFLAG_LINEAFTER_EXPANDED",0x0008},
	{"SC_FOLDFLAG_LINEBEFORE_CONTRACTED",0x0004},
	{"SC_FOLDFLAG_LINEBEFORE_EXPANDED",0x0002},
	{"SC_FOLDLEVELBASE",0x400},
	{"SC_FOLDLEVELHEADERFLAG",0x2000},
	{"SC_FOLDLEVELNUMBERMASK",0x0FFF},
	{"SC_FOLDLEVELWHITEFLAG",0x1000},
	{"SC_FONT_SIZE_MULTIPLIER",100},
	{"SC_IV_LOOKBOTH",3},
	{"SC_IV_LOOKFORWARD",2},
	{"SC_IV_NONE",0},
	{"SC_IV_REAL",1},
	{"SC_LASTSTEPINUNDOREDO",0x100},
	{"SC_LINE_END_TYPE_DEFAULT",0},
	{"SC_LINE_END_TYPE_UNICODE",1},
	{"SC_MARGINOPTION_NONE",0},
	{"SC_MARGINOPTION_SUBLINESELECT",1},
	{"SC_MARGIN_BACK",2},
	{"SC_MARGIN_FORE",3},
	{"SC_MARGIN_NUMBER",1},
	{"SC_MARGIN_RTEXT",5},
	{"SC_MARGIN_SYMBOL",0},
	{"SC_MARGIN_TEXT",4},
	{"SC_MARKNUM_FOLDER",30},
	{"SC_MARKNUM_FOLDEREND",25},
	{"SC_MARKNUM_FOLDERMIDTAIL",27},
	{"SC_MARKNUM_FOLDEROPEN",31},
	{"SC_MARKNUM_FOLDEROPENMID",26},
	{"SC_MARKNUM_FOLDERSUB",29},
	{"SC_MARKNUM_FOLDERTAIL",28},
	{"SC_MARK_ARROW",2},
	{"SC_MARK_ARROWDOWN",6},
	{"SC_MARK_ARROWS",24},
	{"SC_MARK_AVAILABLE",28},
	{"SC_MARK_BACKGROUND",22},
	{"SC_MARK_BOOKMARK",31},
	{"SC_MARK_BOXMINUS",14},
	{"SC_MARK_BOXMINUSCONNECTED",15},
	{"SC_MARK_BOXPLUS",12},
	{"SC_MARK_BOXPLUSCONNECTED",13},
	{"SC_MARK_CHARACTER",10000},
	{"SC_MARK_CIRCLE",0},
	{"SC_MARK_CIRCLEMINUS",20},
	{"SC_MARK_CIRCLEMINUSCONNECTED",21},
	{"SC_MARK_CIRCLEPLUS",18},
	{"SC_MARK_CIRCLEPLUSCONNECTED",19},
	{"SC_MARK_DOTDOTDOT",23},
	{"SC_MARK_EMPTY",5},
	{"SC_MARK_FULLRECT",26},
	{"SC_MARK_LCORNER",10},
	{"SC_MARK_LCORNERCURVE",16},
	{"SC_MARK_LEFTRECT",27},
	{"SC_MARK_MINUS",7},
	{"SC_MARK_PIXMAP",25},
	{"SC_MARK_PLUS",8},
	{"SC_MARK_RGBAIMAGE",30},
	{"SC_MARK_ROUNDRECT",1},
	{"SC_MARK_SHORTARROW",4},
	{"SC_MARK_SMALLRECT",3},
	{"SC_MARK_TCORNER",11},
	{"SC_MARK_TCORNERCURVE",17},
	{"SC_MARK_UNDERLINE",29},
	{"SC_MARK_VLINE",9},
	{"SC_MASK_FOLDERS",static_cast<int>(0xFE000000)},
	{"SC_MAX_MARGIN",4},
	{"SC_MODEVENTMASKALL",0xFFFFF},
	{"SC_MOD_BEFOREDELETE",0x800},
	{"SC_MOD_BEFOREINSERT",0x400},
	{"SC_MOD_CHANGEANNOTATION",0x20000},
	{"SC_MOD_CHANGEFOLD",0x8},
	{"SC_MOD_CHANGEINDICATOR",0x4000},
	{"SC_MOD_CHANGELINESTATE",0x8000},
	{"SC_MOD_CHANGEMARGIN",0x10000},
	{"SC_MOD_CHANGEMARKER",0x200},
	{"SC_MOD_CHANGESTYLE",0x4},
	{"SC_MOD_CONTAINER",0x40000},
	{"SC_MOD_DELETETEXT",0x2},
	{"SC_MOD_INSERTTEXT",0x1},
	{"SC_MOD_LEXERSTATE",0x80000},
	{"SC_MULTILINEUNDOREDO",0x1000},
	{"SC_MULTIPASTE_EACH",1},
	{"SC_MULTIPASTE_ONCE",0},
	{"SC_MULTISTEPUNDOREDO",0x80},
	{"SC_ORDER_CUSTOM",2},
	{"SC_ORDER_PERFORMSORT",1},
	{"SC_ORDER_PRESORTED",0},
	{"SC_PERFORMED_REDO",0x40},
	{"SC_PERFORMED_UNDO",0x20},
	{"SC_PERFORMED_USER",0x10},
	{"SC_PRINT_BLACKONWHITE",2},
	{"SC_PRINT_COLOURONWHITE",3},
	{"SC_PRINT_COLOURONWHITEDEFAULTBG",4},
	{"SC_PRINT_INVERTLIGHT",1},
	{"SC_PRINT_NORMAL",0},
	{"SC_SEL_LINES",2},
	{"SC_SEL_RECTANGLE",1},
	{"SC_SEL_STREAM",0},
	{"SC_SEL_THIN",3},
	{"SC_STARTACTION",0x2000},
	{"SC_STATUS_BADALLOC",2},
	{"SC_STATUS_FAILURE",1},
	{"SC_STATUS_OK",0},
	{"SC_TECHNOLOGY_DEFAULT",0},
	{"SC_TECHNOLOGY_DIRECTWRITE",1},
	{"SC_TIME_FOREVER",10000000},
	{"SC_TYPE_BOOLEAN",0},
	{"SC_TYPE_INTEGER",1},
	{"SC_TYPE_STRING",2},
	{"SC_UPDATE_CONTENT",0x1},
	{"SC_UPDATE_H_SCROLL",0x8},
	{"SC_UPDATE_SELECTION",0x2},
	{"SC_UPDATE_V_SCROLL",0x4},
	{"SC_WEIGHT_BOLD",700},
	{"SC_WEIGHT_NORMAL",400},
	{"SC_WEIGHT_SEMIBOLD",600},
	{"SC_WRAPINDENT_FIXED",0},
	{"SC_WRAPINDENT_INDENT",2},
	{"SC_WRAPINDENT_SAME",1},
	{"SC_WRAPVISUALFLAGLOC_DEFAULT",0x0000},
	{"SC_WRAPVISUALFLAGLOC_END_BY_TEXT",0x0001},
	{"SC_WRAPVISUALFLAGLOC_START_BY_TEXT",0x0002},
	{"SC_WRAPVISUALFLAG_END",0x0001},
	{"SC_WRAPVISUALFLAG_MARGIN",0x0004},
	{"SC_WRAPVISUALFLAG_NONE",0x0000},
	{"SC_WRAPVISUALFLAG_START",0x0002},
	{"SC_WRAP_CHAR",2},
	{"SC_WRAP_NONE",0},
	{"SC_WRAP_WORD",1},
	{"STYLE_BRACEBAD",35},
	{"STYLE_BRACELIGHT",34},
	{"STYLE_CALLTIP",38},
	{"STYLE_CONTROLCHAR",36},
	{"STYLE_DEFAULT",32},
	{"STYLE_INDENTGUIDE",37},
	{"STYLE_LASTPREDEFINED",39},
	{"STYLE_LINENUMBER",33},
	{"STYLE_MAX",255},
	{"UNDO_MAY_COALESCE",1},
	{"VISIBLE_SLOP",0x01},
	{"VISIBLE_STRICT",0x04}
};

static IFaceFunction ifaceFunctions[] = {
	{"AddRefDocument", 2376, iface_void, {iface_void, iface_int}},
	{"AddSelection", 2573, iface_int, {iface_int, iface_int}},
	{"AddStyledText", 2002, iface_void, {iface_length, iface_cells}},
	{"AddText", 2001, iface_void, {iface_length, iface_string}},
	{"AddUndoAction", 2560, iface_void, {iface_int, iface_int}},
	{"Allocate", 2446, iface_void, {iface_int, iface_void}},
	{"AllocateExtendedStyles", 2553, iface_int, {iface_int, iface_void}},
	{"AllocateSubStyles", 4020, iface_int, {iface_int, iface_int}},
	{"AnnotationClearAll", 2547, iface_void, {iface_void, iface_void}},
	{"AppendText", 2282, iface_void, {iface_length, iface_string}},
	{"AssignCmdKey", 2070, iface_void, {iface_keymod, iface_int}},
	{"AutoCActive", 2102, iface_bool, {iface_void, iface_void}},
	{"AutoCCancel", 2101, iface_void, {iface_void, iface_void}},
	{"AutoCComplete", 2104, iface_void, {iface_void, iface_void}},
	{"AutoCPosStart", 2103, iface_position, {iface_void, iface_void}},
	{"AutoCSelect", 2108, iface_void, {iface_void, iface_string}},
	{"AutoCShow", 2100, iface_void, {iface_int, iface_string}},
	{"AutoCStops", 2105, iface_void, {iface_void, iface_string}},
	{"BackTab", 2328, iface_void, {iface_void, iface_void}},
	{"BeginUndoAction", 2078, iface_void, {iface_void, iface_void}},
	{"BraceBadLight", 2352, iface_void, {iface_position, iface_void}},
	{"BraceBadLightIndicator", 2499, iface_void, {iface_bool, iface_int}},
	{"BraceHighlight", 2351, iface_void, {iface_position, iface_position}},
	{"BraceHighlightIndicator", 2498, iface_void, {iface_bool, iface_int}},
	{"BraceMatch", 2353, iface_position, {iface_position, iface_void}},
	{"CallTipActive", 2202, iface_bool, {iface_void, iface_void}},
	{"CallTipCancel", 2201, iface_void, {iface_void, iface_void}},
	{"CallTipPosStart", 2203, iface_position, {iface_void, iface_void}},
	{"CallTipSetHlt", 2204, iface_void, {iface_int, iface_int}},
	{"CallTipShow", 2200, iface_void, {iface_position, iface_string}},
	{"CanPaste", 2173, iface_bool, {iface_void, iface_void}},
	{"CanRedo", 2016, iface_bool, {iface_void, iface_void}},
	{"CanUndo", 2174, iface_bool, {iface_void, iface_void}},
	{"Cancel", 2325, iface_void, {iface_void, iface_void}},
	{"ChangeLexerState", 2617, iface_int, {iface_position, iface_position}},
	{"CharLeft", 2304, iface_void, {iface_void, iface_void}},
	{"CharLeftExtend", 2305, iface_void, {iface_void, iface_void}},
	{"CharLeftRectExtend", 2428, iface_void, {iface_void, iface_void}},
	{"CharPositionFromPoint", 2561, iface_position, {iface_int, iface_int}},
	{"CharPositionFromPointClose", 2562, iface_position, {iface_int, iface_int}},
	{"CharRight", 2306, iface_void, {iface_void, iface_void}},
	{"CharRightExtend", 2307, iface_void, {iface_void, iface_void}},
	{"CharRightRectExtend", 2429, iface_void, {iface_void, iface_void}},
	{"ChooseCaretX", 2399, iface_void, {iface_void, iface_void}},
	{"Clear", 2180, iface_void, {iface_void, iface_void}},
	{"ClearAll", 2004, iface_void, {iface_void, iface_void}},
	{"ClearAllCmdKeys", 2072, iface_void, {iface_void, iface_void}},
	{"ClearCmdKey", 2071, iface_void, {iface_keymod, iface_void}},
	{"ClearDocumentStyle", 2005, iface_void, {iface_void, iface_void}},
	{"ClearRegisteredImages", 2408, iface_void, {iface_void, iface_void}},
	{"ClearRepresentation", 2667, iface_void, {iface_string, iface_void}},
	{"ClearSelections", 2571, iface_void, {iface_void, iface_void}},
	{"Colourise", 4003, iface_void, {iface_position, iface_position}},
	{"ContractedFoldNext", 2618, iface_int, {iface_int, iface_void}},
	{"ConvertEOLs", 2029, iface_void, {iface_int, iface_void}},
	{"Copy", 2178, iface_void, {iface_void, iface_void}},
	{"CopyAllowLine", 2519, iface_void, {iface_void, iface_void}},
	{"CopyRange", 2419, iface_void, {iface_position, iface_position}},
	{"CopyText", 2420, iface_void, {iface_length, iface_string}},
	{"CountCharacters", 2633, iface_int, {iface_int, iface_int}},
	{"CreateDocument", 2375, iface_int, {iface_void, iface_void}},
	{"CreateLoader", 2632, iface_int, {iface_int, iface_void}},
	{"Cut", 2177, iface_void, {iface_void, iface_void}},
	{"DelLineLeft", 2395, iface_void, {iface_void, iface_void}},
	{"DelLineRight", 2396, iface_void, {iface_void, iface_void}},
	{"DelWordLeft", 2335, iface_void, {iface_void, iface_void}},
	{"DelWordRight", 2336, iface_void, {iface_void, iface_void}},
	{"DelWordRightEnd", 2518, iface_void, {iface_void, iface_void}},
	{"DeleteBack", 2326, iface_void, {iface_void, iface_void}},
	{"DeleteBackNotLine", 2344, iface_void, {iface_void, iface_void}},
	{"DeleteRange", 2645, iface_void, {iface_position, iface_int}},
	{"DescribeKeyWordSets", 4017, iface_int, {iface_void, iface_stringresult}},
	{"DescribeProperty", 4016, iface_int, {iface_string, iface_stringresult}},
	{"DocLineFromVisible", 2221, iface_int, {iface_int, iface_void}},
	{"DocumentEnd", 2318, iface_void, {iface_void, iface_void}},
	{"DocumentEndExtend", 2319, iface_void, {iface_void, iface_void}},
	{"DocumentStart", 2316, iface_void, {iface_void, iface_void}},
	{"DocumentStartExtend", 2317, iface_void, {iface_void, iface_void}},
	{"DropSelectionN", 2671, iface_void, {iface_int, iface_void}},
	{"EditToggleOvertype", 2324, iface_void, {iface_void, iface_void}},
	{"EmptyUndoBuffer", 2175, iface_void, {iface_void, iface_void}},
	{"EncodedFromUTF8", 2449, iface_int, {iface_string, iface_stringresult}},
	{"EndUndoAction", 2079, iface_void, {iface_void, iface_void}},
	{"EnsureVisible", 2232, iface_void, {iface_int, iface_void}},
	{"EnsureVisibleEnforcePolicy", 2234, iface_void, {iface_int, iface_void}},
	{"ExpandChildren", 2239, iface_void, {iface_int, iface_int}},
	{"FindColumn", 2456, iface_int, {iface_int, iface_int}},
	{"FindIndicatorFlash", 2641, iface_void, {iface_position, iface_position}},
	{"FindIndicatorHide", 2642, iface_void, {iface_void, iface_void}},
	{"FindIndicatorShow", 2640, iface_void, {iface_position, iface_position}},
	{"FindText", 2150, iface_position, {iface_int, iface_findtext}},
	{"FoldAll", 2662, iface_void, {iface_int, iface_void}},
	{"FoldChildren", 2238, iface_void, {iface_int, iface_int}},
	{"FoldLine", 2237, iface_void, {iface_int, iface_int}},
	{"FormFeed", 2330, iface_void, {iface_void, iface_void}},
	{"FormatRange", 2151, iface_position, {iface_bool, iface_formatrange}},
	{"FreeSubStyles", 4023, iface_void, {iface_void, iface_void}},
	{"GetCurLine", 2027, iface_int, {iface_length, iface_stringresult}},
	{"GetHotspotActiveBack", 2495, iface_colour, {iface_void, iface_void}},
	{"GetHotspotActiveFore", 2494, iface_colour, {iface_void, iface_void}},
	{"GetLastChild", 2224, iface_int, {iface_int, iface_int}},
	{"GetLine", 2153, iface_int, {iface_int, iface_stringresult}},
	{"GetLineSelEndPosition", 2425, iface_position, {iface_int, iface_void}},
	{"GetLineSelStartPosition", 2424, iface_position, {iface_int, iface_void}},
	{"GetRangePointer", 2643, iface_int, {iface_int, iface_int}},
	{"GetSelText", 2161, iface_int, {iface_void, iface_stringresult}},
	{"GetStyledText", 2015, iface_int, {iface_void, iface_textrange}},
	{"GetText", 2182, iface_int, {iface_length, iface_stringresult}},
	{"GetTextRange", 2162, iface_int, {iface_void, iface_textrange}},
	{"GotoLine", 2024, iface_void, {iface_int, iface_void}},
	{"GotoPos", 2025, iface_void, {iface_position, iface_void}},
	{"GrabFocus", 2400, iface_void, {iface_void, iface_void}},
	{"HideLines", 2227, iface_void, {iface_int, iface_int}},
	{"HideSelection", 2163, iface_void, {iface_bool, iface_void}},
	{"Home", 2312, iface_void, {iface_void, iface_void}},
	{"HomeDisplay", 2345, iface_void, {iface_void, iface_void}},
	{"HomeDisplayExtend", 2346, iface_void, {iface_void, iface_void}},
	{"HomeExtend", 2313, iface_void, {iface_void, iface_void}},
	{"HomeRectExtend", 2430, iface_void, {iface_void, iface_void}},
	{"HomeWrap", 2349, iface_void, {iface_void, iface_void}},
	{"HomeWrapExtend", 2450, iface_void, {iface_void, iface_void}},
	{"IndicatorAllOnFor", 2506, iface_int, {iface_int, iface_void}},
	{"IndicatorClearRange", 2505, iface_void, {iface_int, iface_int}},
	{"IndicatorEnd", 2509, iface_int, {iface_int, iface_int}},
	{"IndicatorFillRange", 2504, iface_void, {iface_int, iface_int}},
	{"IndicatorStart", 2508, iface_int, {iface_int, iface_int}},
	{"IndicatorValueAt", 2507, iface_int, {iface_int, iface_int}},
	{"InsertText", 2003, iface_void, {iface_position, iface_string}},
	{"LineCopy", 2455, iface_void, {iface_void, iface_void}},
	{"LineCut", 2337, iface_void, {iface_void, iface_void}},
	{"LineDelete", 2338, iface_void, {iface_void, iface_void}},
	{"LineDown", 2300, iface_void, {iface_void, iface_void}},
	{"LineDownExtend", 2301, iface_void, {iface_void, iface_void}},
	{"LineDownRectExtend", 2426, iface_void, {iface_void, iface_void}},
	{"LineDuplicate", 2404, iface_void, {iface_void, iface_void}},
	{"LineEnd", 2314, iface_void, {iface_void, iface_void}},
	{"LineEndDisplay", 2347, iface_void, {iface_void, iface_void}},
	{"LineEndDisplayExtend", 2348, iface_void, {iface_void, iface_void}},
	{"LineEndExtend", 2315, iface_void, {iface_void, iface_void}},
	{"LineEndRectExtend", 2432, iface_void, {iface_void, iface_void}},
	{"LineEndWrap", 2451, iface_void, {iface_void, iface_void}},
	{"LineEndWrapExtend", 2452, iface_void, {iface_void, iface_void}},
	{"LineFromPosition", 2166, iface_int, {iface_position, iface_void}},
	{"LineLength", 2350, iface_int, {iface_int, iface_void}},
	{"LineScroll", 2168, iface_void, {iface_int, iface_int}},
	{"LineScrollDown", 2342, iface_void, {iface_void, iface_void}},
	{"LineScrollUp", 2343, iface_void, {iface_void, iface_void}},
	{"LineTranspose", 2339, iface_void, {iface_void, iface_void}},
	{"LineUp", 2302, iface_void, {iface_void, iface_void}},
	{"LineUpExtend", 2303, iface_void, {iface_void, iface_void}},
	{"LineUpRectExtend", 2427, iface_void, {iface_void, iface_void}},
	{"LinesJoin", 2288, iface_void, {iface_void, iface_void}},
	{"LinesSplit", 2289, iface_void, {iface_int, iface_void}},
	{"LoadLexerLibrary", 4007, iface_void, {iface_void, iface_string}},
	{"LowerCase", 2340, iface_void, {iface_void, iface_void}},
	{"MarginTextClearAll", 2536, iface_void, {iface_void, iface_void}},
	{"MarkerAdd", 2043, iface_int, {iface_int, iface_int}},
	{"MarkerAddSet", 2466, iface_void, {iface_int, iface_int}},
	{"MarkerDefine", 2040, iface_void, {iface_int, iface_int}},
	{"MarkerDefinePixmap", 2049, iface_void, {iface_int, iface_string}},
	{"MarkerDefineRGBAImage", 2626, iface_void, {iface_int, iface_string}},
	{"MarkerDelete", 2044, iface_void, {iface_int, iface_int}},
	{"MarkerDeleteAll", 2045, iface_void, {iface_int, iface_void}},
	{"MarkerDeleteHandle", 2018, iface_void, {iface_int, iface_void}},
	{"MarkerEnableHighlight", 2293, iface_void, {iface_bool, iface_void}},
	{"MarkerGet", 2046, iface_int, {iface_int, iface_void}},
	{"MarkerLineFromHandle", 2017, iface_int, {iface_int, iface_void}},
	{"MarkerNext", 2047, iface_int, {iface_int, iface_int}},
	{"MarkerPrevious", 2048, iface_int, {iface_int, iface_int}},
	{"MarkerSymbolDefined", 2529, iface_int, {iface_int, iface_void}},
	{"MoveCaretInsideView", 2401, iface_void, {iface_void, iface_void}},
	{"MoveSelectedLinesDown", 2621, iface_void, {iface_void, iface_void}},
	{"MoveSelectedLinesUp", 2620, iface_void, {iface_void, iface_void}},
	{"NewLine", 2329, iface_void, {iface_void, iface_void}},
	{"Null", 2172, iface_void, {iface_void, iface_void}},
	{"PageDown", 2322, iface_void, {iface_void, iface_void}},
	{"PageDownExtend", 2323, iface_void, {iface_void, iface_void}},
	{"PageDownRectExtend", 2434, iface_void, {iface_void, iface_void}},
	{"PageUp", 2320, iface_void, {iface_void, iface_void}},
	{"PageUpExtend", 2321, iface_void, {iface_void, iface_void}},
	{"PageUpRectExtend", 2433, iface_void, {iface_void, iface_void}},
	{"ParaDown", 2413, iface_void, {iface_void, iface_void}},
	{"ParaDownExtend", 2414, iface_void, {iface_void, iface_void}},
	{"ParaUp", 2415, iface_void, {iface_void, iface_void}},
	{"ParaUpExtend", 2416, iface_void, {iface_void, iface_void}},
	{"Paste", 2179, iface_void, {iface_void, iface_void}},
	{"PointXFromPosition", 2164, iface_int, {iface_void, iface_position}},
	{"PointYFromPosition", 2165, iface_int, {iface_void, iface_position}},
	{"PositionAfter", 2418, iface_position, {iface_position, iface_void}},
	{"PositionBefore", 2417, iface_position, {iface_position, iface_void}},
	{"PositionFromLine", 2167, iface_position, {iface_int, iface_void}},
	{"PositionFromPoint", 2022, iface_position, {iface_int, iface_int}},
	{"PositionFromPointClose", 2023, iface_position, {iface_int, iface_int}},
	{"PositionRelative", 2670, iface_position, {iface_position, iface_int}},
	{"PrivateLexerCall", 4013, iface_int, {iface_int, iface_int}},
	{"PropertyNames", 4014, iface_int, {iface_void, iface_stringresult}},
	{"PropertyType", 4015, iface_int, {iface_string, iface_void}},
	{"Redo", 2011, iface_void, {iface_void, iface_void}},
	{"RegisterImage", 2405, iface_void, {iface_int, iface_string}},
	{"RegisterRGBAImage", 2627, iface_void, {iface_int, iface_string}},
	{"ReleaseAllExtendedStyles", 2552, iface_void, {iface_void, iface_void}},
	{"ReleaseDocument", 2377, iface_void, {iface_void, iface_int}},
	{"ReplaceSel", 2170, iface_void, {iface_void, iface_string}},
	{"ReplaceTarget", 2194, iface_int, {iface_length, iface_string}},
	{"ReplaceTargetRE", 2195, iface_int, {iface_length, iface_string}},
	{"RotateSelection", 2606, iface_void, {iface_void, iface_void}},
	{"ScrollCaret", 2169, iface_void, {iface_void, iface_void}},
	{"ScrollRange", 2569, iface_void, {iface_position, iface_position}},
	{"ScrollToEnd", 2629, iface_void, {iface_void, iface_void}},
	{"ScrollToStart", 2628, iface_void, {iface_void, iface_void}},
	{"SearchAnchor", 2366, iface_void, {iface_void, iface_void}},
	{"SearchInTarget", 2197, iface_int, {iface_length, iface_string}},
	{"SearchNext", 2367, iface_int, {iface_int, iface_string}},
	{"SearchPrev", 2368, iface_int, {iface_int, iface_string}},
	{"SelectAll", 2013, iface_void, {iface_void, iface_void}},
	{"SelectionDuplicate", 2469, iface_void, {iface_void, iface_void}},
	{"SetCharsDefault", 2444, iface_void, {iface_void, iface_void}},
	{"SetEmptySelection", 2556, iface_void, {iface_position, iface_void}},
	{"SetFoldMarginColour", 2290, iface_void, {iface_bool, iface_colour}},
	{"SetFoldMarginHiColour", 2291, iface_void, {iface_bool, iface_colour}},
	{"SetHotspotActiveBack", 2411, iface_void, {iface_bool, iface_colour}},
	{"SetHotspotActiveFore", 2410, iface_void, {iface_bool, iface_colour}},
	{"SetLengthForEncode", 2448, iface_void, {iface_int, iface_void}},
	{"SetSavePoint", 2014, iface_void, {iface_void, iface_void}},
	{"SetSel", 2160, iface_void, {iface_position, iface_position}},
	{"SetSelBack", 2068, iface_void, {iface_bool, iface_colour}},
	{"SetSelFore", 2067, iface_void, {iface_bool, iface_colour}},
	{"SetSelection", 2572, iface_int, {iface_int, iface_int}},
	{"SetStyling", 2033, iface_void, {iface_length, iface_int}},
	{"SetStylingEx", 2073, iface_void, {iface_length, iface_string}},
	{"SetText", 2181, iface_void, {iface_void, iface_string}},
	{"SetVisiblePolicy", 2394, iface_void, {iface_int, iface_int}},
	{"SetWhitespaceBack", 2085, iface_void, {iface_bool, iface_colour}},
	{"SetWhitespaceFore", 2084, iface_void, {iface_bool, iface_colour}},
	{"SetXCaretPolicy", 2402, iface_void, {iface_int, iface_int}},
	{"SetYCaretPolicy", 2403, iface_void, {iface_int, iface_int}},
	{"ShowLines", 2226, iface_void, {iface_int, iface_int}},
	{"StartRecord", 3001, iface_void, {iface_void, iface_void}},
	{"StartStyling", 2032, iface_void, {iface_position, iface_int}},
	{"StopRecord", 3002, iface_void, {iface_void, iface_void}},
	{"StutteredPageDown", 2437, iface_void, {iface_void, iface_void}},
	{"StutteredPageDownExtend", 2438, iface_void, {iface_void, iface_void}},
	{"StutteredPageUp", 2435, iface_void, {iface_void, iface_void}},
	{"StutteredPageUpExtend", 2436, iface_void, {iface_void, iface_void}},
	{"StyleClearAll", 2050, iface_void, {iface_void, iface_void}},
	{"StyleResetDefault", 2058, iface_void, {iface_void, iface_void}},
	{"SwapMainAnchorCaret", 2607, iface_void, {iface_void, iface_void}},
	{"Tab", 2327, iface_void, {iface_void, iface_void}},
	{"TargetAsUTF8", 2447, iface_int, {iface_void, iface_stringresult}},
	{"TargetFromSelection", 2287, iface_void, {iface_void, iface_void}},
	{"TextHeight", 2279, iface_int, {iface_int, iface_void}},
	{"TextWidth", 2276, iface_int, {iface_int, iface_string}},
	{"ToggleCaretSticky", 2459, iface_void, {iface_void, iface_void}},
	{"ToggleFold", 2231, iface_void, {iface_int, iface_void}},
	{"Undo", 2176, iface_void, {iface_void, iface_void}},
	{"UpperCase", 2341, iface_void, {iface_void, iface_void}},
	{"UsePopUp", 2371, iface_void, {iface_bool, iface_void}},
	{"UserListShow", 2117, iface_void, {iface_int, iface_string}},
	{"VCHome", 2331, iface_void, {iface_void, iface_void}},
	{"VCHomeDisplay", 2652, iface_void, {iface_void, iface_void}},
	{"VCHomeDisplayExtend", 2653, iface_void, {iface_void, iface_void}},
	{"VCHomeExtend", 2332, iface_void, {iface_void, iface_void}},
	{"VCHomeRectExtend", 2431, iface_void, {iface_void, iface_void}},
	{"VCHomeWrap", 2453, iface_void, {iface_void, iface_void}},
	{"VCHomeWrapExtend", 2454, iface_void, {iface_void, iface_void}},
	{"VerticalCentreCaret", 2619, iface_void, {iface_void, iface_void}},
	{"VisibleFromDocLine", 2220, iface_int, {iface_int, iface_void}},
	{"WordEndPosition", 2267, iface_int, {iface_position, iface_bool}},
	{"WordLeft", 2308, iface_void, {iface_void, iface_void}},
	{"WordLeftEnd", 2439, iface_void, {iface_void, iface_void}},
	{"WordLeftEndExtend", 2440, iface_void, {iface_void, iface_void}},
	{"WordLeftExtend", 2309, iface_void, {iface_void, iface_void}},
	{"WordPartLeft", 2390, iface_void, {iface_void, iface_void}},
	{"WordPartLeftExtend", 2391, iface_void, {iface_void, iface_void}},
	{"WordPartRight", 2392, iface_void, {iface_void, iface_void}},
	{"WordPartRightExtend", 2393, iface_void, {iface_void, iface_void}},
	{"WordRight", 2310, iface_void, {iface_void, iface_void}},
	{"WordRightEnd", 2441, iface_void, {iface_void, iface_void}},
	{"WordRightEndExtend", 2442, iface_void, {iface_void, iface_void}},
	{"WordRightExtend", 2311, iface_void, {iface_void, iface_void}},
	{"WordStartPosition", 2266, iface_int, {iface_position, iface_bool}},
	{"WrapCount", 2235, iface_int, {iface_int, iface_void}},
	{"ZoomIn", 2333, iface_void, {iface_void, iface_void}},
	{"ZoomOut", 2334, iface_void, {iface_void, iface_void}}
};

static IFaceProperty ifaceProperties[] = {
	{"AdditionalCaretFore", 2605, 2604, iface_colour, iface_void},
	{"AdditionalCaretsBlink", 2568, 2567, iface_bool, iface_void},
	{"AdditionalCaretsVisible", 2609, 2608, iface_bool, iface_void},
	{"AdditionalSelAlpha", 2603, 2602, iface_int, iface_void},
	{"AdditionalSelBack", 0, 2601, iface_colour, iface_void},
	{"AdditionalSelFore", 0, 2600, iface_colour, iface_void},
	{"AdditionalSelectionTyping", 2566, 2565, iface_bool, iface_void},
	{"AllLinesVisible", 2236, 0, iface_bool, iface_void},
	{"Anchor", 2009, 2026, iface_position, iface_void},
	{"AnnotationLines", 2546, 0, iface_int, iface_int},
	{"AnnotationStyle", 2543, 2542, iface_int, iface_int},
	{"AnnotationStyleOffset", 2551, 2550, iface_int, iface_void},
	{"AnnotationStyles", 2545, 2544, iface_stringresult, iface_int},
	{"AnnotationText", 2541, 2540, iface_stringresult, iface_int},
	{"AnnotationVisible", 2549, 2548, iface_int, iface_void},
	{"AutoCAutoHide", 2119, 2118, iface_bool, iface_void},
	{"AutoCCancelAtStart", 2111, 2110, iface_bool, iface_void},
	{"AutoCCaseInsensitiveBehaviour", 2635, 2634, iface_int, iface_void},
	{"AutoCChooseSingle", 2114, 2113, iface_bool, iface_void},
	{"AutoCCurrent", 2445, 0, iface_int, iface_void},
	{"AutoCCurrentText", 2610, 0, iface_stringresult, iface_void},
	{"AutoCDropRestOfWord", 2271, 2270, iface_bool, iface_void},
	{"AutoCFillUps", 0, 2112, iface_string, iface_void},
	{"AutoCIgnoreCase", 2116, 2115, iface_bool, iface_void},
	{"AutoCMaxHeight", 2211, 2210, iface_int, iface_void},
	{"AutoCMaxWidth", 2209, 2208, iface_int, iface_void},
	{"AutoCOrder", 2661, 2660, iface_int, iface_void},
	{"AutoCSeparator", 2107, 2106, iface_int, iface_void},
	{"AutoCTypeSeparator", 2285, 2286, iface_int, iface_void},
	{"AutomaticFold", 2664, 2663, iface_int, iface_void},
	{"BackSpaceUnIndents", 2263, 2262, iface_bool, iface_void},
	{"BufferedDraw", 2034, 2035, iface_bool, iface_void},
	{"CallTipBack", 0, 2205, iface_colour, iface_void},
	{"CallTipFore", 0, 2206, iface_colour, iface_void},
	{"CallTipForeHlt", 0, 2207, iface_colour, iface_void},
	{"CallTipPosStart", 0, 2214, iface_int, iface_void},
	{"CallTipPosition", 0, 2213, iface_bool, iface_void},
	{"CallTipUseStyle", 0, 2212, iface_int, iface_void},
	{"CaretFore", 2138, 2069, iface_colour, iface_void},
	{"CaretLineBack", 2097, 2098, iface_colour, iface_void},
	{"CaretLineBackAlpha", 2471, 2470, iface_int, iface_void},
	{"CaretLineVisible", 2095, 2096, iface_bool, iface_void},
	{"CaretLineVisibleAlways", 2654, 2655, iface_bool, iface_void},
	{"CaretPeriod", 2075, 2076, iface_int, iface_void},
	{"CaretSticky", 2457, 2458, iface_int, iface_void},
	{"CaretStyle", 2513, 2512, iface_int, iface_void},
	{"CaretWidth", 2189, 2188, iface_int, iface_void},
	{"CharAt", 2007, 0, iface_int, iface_position},
	{"CharacterPointer", 2520, 0, iface_int, iface_void},
	{"CodePage", 2137, 2037, iface_int, iface_void},
	{"Column", 2129, 0, iface_int, iface_position},
	{"ControlCharSymbol", 2389, 2388, iface_int, iface_void},
	{"CurrentPos", 2008, 2141, iface_position, iface_void},
	{"Cursor", 2387, 2386, iface_int, iface_void},
	{"DirectFunction", 2184, 0, iface_int, iface_void},
	{"DirectPointer", 2185, 0, iface_int, iface_void},
	{"DistanceToSecondaryStyles", 4025, 0, iface_int, iface_void},
	{"DocPointer", 2357, 2358, iface_int, iface_void},
	{"EOLMode", 2030, 2031, iface_int, iface_void},
	{"EdgeColour", 2364, 2365, iface_colour, iface_void},
	{"EdgeColumn", 2360, 2361, iface_int, iface_void},
	{"EdgeMode", 2362, 2363, iface_int, iface_void},
	{"EndAtLastLine", 2278, 2277, iface_bool, iface_void},
	{"EndStyled", 2028, 0, iface_position, iface_void},
	{"ExtraAscent", 2526, 2525, iface_int, iface_void},
	{"ExtraDescent", 2528, 2527, iface_int, iface_void},
	{"FirstVisibleLine", 2152, 2613, iface_int, iface_void},
	{"Focus", 2381, 2380, iface_bool, iface_void},
	{"FoldExpanded", 2230, 2229, iface_bool, iface_int},
	{"FoldFlags", 0, 2233, iface_int, iface_void},
	{"FoldLevel", 2223, 2222, iface_int, iface_int},
	{"FoldParent", 2225, 0, iface_int, iface_int},
	{"FontQuality", 2612, 2611, iface_int, iface_void},
	{"GapPosition", 2644, 0, iface_position, iface_void},
	{"HScrollBar", 2131, 2130, iface_bool, iface_void},
	{"HighlightGuide", 2135, 2134, iface_int, iface_void},
	{"HotspotActiveUnderline", 2496, 2412, iface_bool, iface_void},
	{"HotspotSingleLine", 2497, 2421, iface_bool, iface_void},
	{"Identifier", 2623, 2622, iface_int, iface_void},
	{"Identifiers", 0, 4024, iface_string, iface_int},
	{"Indent", 2123, 2122, iface_int, iface_void},
	{"IndentationGuides", 2133, 2132, iface_int, iface_void},
	{"IndicAlpha", 2524, 2523, iface_int, iface_int},
	{"IndicFore", 2083, 2082, iface_colour, iface_int},
	{"IndicOutlineAlpha", 2559, 2558, iface_int, iface_int},
	{"IndicStyle", 2081, 2080, iface_int, iface_int},
	{"IndicUnder", 2511, 2510, iface_bool, iface_int},
	{"IndicatorCurrent", 2501, 2500, iface_int, iface_void},
	{"IndicatorValue", 2503, 2502, iface_int, iface_void},
	{"KeyWords", 0, 4005, iface_string, iface_int},
	{"KeysUnicode", 2522, 2521, iface_bool, iface_void},
	{"LayoutCache", 2273, 2272, iface_int, iface_void},
	{"Length", 2006, 0, iface_int, iface_void},
	{"Lexer", 4002, 4001, iface_int, iface_void},
	{"LexerLanguage", 4012, 4006, iface_stringresult, iface_void},
	{"LineCount", 2154, 0, iface_int, iface_void},
	{"LineEndPosition", 2136, 0, iface_position, iface_int},
	{"LineEndTypesActive", 2658, 0, iface_int, iface_void},
	{"LineEndTypesAllowed", 2657, 2656, iface_int, iface_void},
	{"LineEndTypesSupported", 4018, 0, iface_int, iface_void},
	{"LineIndentPosition", 2128, 0, iface_position, iface_int},
	{"LineIndentation", 2127, 2126, iface_int, iface_int},
	{"LineState", 2093, 2092, iface_int, iface_int},
	{"LineVisible", 2228, 0, iface_bool, iface_int},
	{"LinesOnScreen", 2370, 0, iface_int, iface_void},
	{"MainSelection", 2575, 2574, iface_int, iface_void},
	{"MarginCursorN", 2249, 2248, iface_int, iface_int},
	{"MarginLeft", 2156, 2155, iface_int, iface_void},
	{"MarginMaskN", 2245, 2244, iface_int, iface_int},
	{"MarginOptions", 2557, 2539, iface_int, iface_void},
	{"MarginRight", 2158, 2157, iface_int, iface_void},
	{"MarginSensitiveN", 2247, 2246, iface_bool, iface_int},
	{"MarginStyle", 2533, 2532, iface_int, iface_int},
	{"MarginStyleOffset", 2538, 2537, iface_int, iface_void},
	{"MarginStyles", 2535, 2534, iface_stringresult, iface_int},
	{"MarginText", 2531, 2530, iface_stringresult, iface_int},
	{"MarginTypeN", 2241, 2240, iface_int, iface_int},
	{"MarginWidthN", 2243, 2242, iface_int, iface_int},
	{"MarkerAlpha", 0, 2476, iface_int, iface_int},
	{"MarkerBack", 0, 2042, iface_colour, iface_int},
	{"MarkerBackSelected", 0, 2292, iface_colour, iface_int},
	{"MarkerFore", 0, 2041, iface_colour, iface_int},
	{"MaxLineState", 2094, 0, iface_int, iface_void},
	{"ModEventMask", 2378, 2359, iface_int, iface_void},
	{"Modify", 2159, 0, iface_bool, iface_void},
	{"MouseDownCaptures", 2385, 2384, iface_bool, iface_void},
	{"MouseDwellTime", 2265, 2264, iface_int, iface_void},
	{"MouseSelectionRectangularSwitch", 2669, 2668, iface_bool, iface_void},
	{"MultiPaste", 2615, 2614, iface_int, iface_void},
	{"MultipleSelection", 2564, 2563, iface_bool, iface_void},
	{"Overtype", 2187, 2186, iface_bool, iface_void},
	{"PasteConvertEndings", 2468, 2467, iface_bool, iface_void},
	{"PositionCache", 2515, 2514, iface_int, iface_void},
	{"PrimaryStyleFromStyle", 4028, 0, iface_int, iface_int},
	{"PrintColourMode", 2149, 2148, iface_int, iface_void},
	{"PrintMagnification", 2147, 2146, iface_int, iface_void},
	{"PrintWrapMode", 2407, 2406, iface_int, iface_void},
	{"Property", 4008, 4004, iface_stringresult, iface_string},
	{"PropertyExpanded", 4009, 0, iface_stringresult, iface_string},
	{"PropertyInt", 4010, 0, iface_int, iface_string},
	{"PunctuationChars", 2649, 2648, iface_stringresult, iface_void},
	{"RGBAImageHeight", 0, 2625, iface_int, iface_void},
	{"RGBAImageScale", 0, 2651, iface_int, iface_void},
	{"RGBAImageWidth", 0, 2624, iface_int, iface_void},
	{"ReadOnly", 2140, 2171, iface_bool, iface_void},
	{"RectangularSelectionAnchor", 2591, 2590, iface_position, iface_void},
	{"RectangularSelectionAnchorVirtualSpace", 2595, 2594, iface_int, iface_void},
	{"RectangularSelectionCaret", 2589, 2588, iface_position, iface_void},
	{"RectangularSelectionCaretVirtualSpace", 2593, 2592, iface_int, iface_void},
	{"RectangularSelectionModifier", 2599, 2598, iface_int, iface_void},
	{"Representation", 2666, 2665, iface_stringresult, iface_string},
	{"ScrollWidth", 2275, 2274, iface_int, iface_void},
	{"ScrollWidthTracking", 2517, 2516, iface_bool, iface_void},
	{"SearchFlags", 2199, 2198, iface_int, iface_void},
	{"SelAlpha", 2477, 2478, iface_int, iface_void},
	{"SelEOLFilled", 2479, 2480, iface_bool, iface_void},
	{"SelectionEmpty", 2650, 0, iface_bool, iface_void},
	{"SelectionEnd", 2145, 2144, iface_position, iface_void},
	{"SelectionIsRectangle", 2372, 0, iface_bool, iface_void},
	{"SelectionMode", 2423, 2422, iface_int, iface_void},
	{"SelectionNAnchor", 2579, 2578, iface_position, iface_int},
	{"SelectionNAnchorVirtualSpace", 2583, 2582, iface_int, iface_int},
	{"SelectionNCaret", 2577, 2576, iface_position, iface_int},
	{"SelectionNCaretVirtualSpace", 2581, 2580, iface_int, iface_int},
	{"SelectionNEnd", 2587, 2586, iface_position, iface_int},
	{"SelectionNStart", 2585, 2584, iface_position, iface_int},
	{"SelectionStart", 2143, 2142, iface_position, iface_void},
	{"Selections", 2570, 0, iface_int, iface_void},
	{"Status", 2383, 2382, iface_int, iface_void},
	{"StyleAt", 2010, 0, iface_int, iface_position},
	{"StyleBack", 2482, 2052, iface_colour, iface_int},
	{"StyleBits", 2091, 2090, iface_int, iface_void},
	{"StyleBitsNeeded", 4011, 0, iface_int, iface_void},
	{"StyleBold", 2483, 2053, iface_bool, iface_int},
	{"StyleCase", 2489, 2060, iface_int, iface_int},
	{"StyleChangeable", 2492, 2099, iface_bool, iface_int},
	{"StyleCharacterSet", 2490, 2066, iface_int, iface_int},
	{"StyleEOLFilled", 2487, 2057, iface_bool, iface_int},
	{"StyleFont", 2486, 2056, iface_stringresult, iface_int},
	{"StyleFore", 2481, 2051, iface_colour, iface_int},
	{"StyleFromSubStyle", 4027, 0, iface_int, iface_int},
	{"StyleHotSpot", 2493, 2409, iface_bool, iface_int},
	{"StyleItalic", 2484, 2054, iface_bool, iface_int},
	{"StyleSize", 2485, 2055, iface_int, iface_int},
	{"StyleSizeFractional", 2062, 2061, iface_int, iface_int},
	{"StyleUnderline", 2488, 2059, iface_bool, iface_int},
	{"StyleVisible", 2491, 2074, iface_bool, iface_int},
	{"StyleWeight", 2064, 2063, iface_int, iface_int},
	{"SubStyleBases", 4026, 0, iface_stringresult, iface_void},
	{"SubStylesLength", 4022, 0, iface_int, iface_int},
	{"SubStylesStart", 4021, 0, iface_int, iface_int},
	{"TabIndents", 2261, 2260, iface_bool, iface_void},
	{"TabWidth", 2121, 2036, iface_int, iface_void},
	{"Tag", 2616, 0, iface_stringresult, iface_int},
	{"TargetEnd", 2193, 2192, iface_position, iface_void},
	{"TargetStart", 2191, 2190, iface_position, iface_void},
	{"Technology", 2631, 2630, iface_int, iface_void},
	{"TextLength", 2183, 0, iface_int, iface_void},
	{"TwoPhaseDraw", 2283, 2284, iface_bool, iface_void},
	{"UndoCollection", 2019, 2012, iface_bool, iface_void},
	{"UseTabs", 2125, 2124, iface_bool, iface_void},
	{"VScrollBar", 2281, 2280, iface_bool, iface_void},
	{"ViewEOL", 2355, 2356, iface_bool, iface_void},
	{"ViewWS", 2020, 2021, iface_int, iface_void},
	{"VirtualSpaceOptions", 2597, 2596, iface_int, iface_void},
	{"WhitespaceChars", 2647, 2443, iface_stringresult, iface_void},
	{"WhitespaceSize", 2087, 2086, iface_int, iface_void},
	{"WordChars", 2646, 2077, iface_stringresult, iface_void},
	{"WrapIndentMode", 2473, 2472, iface_int, iface_void},
	{"WrapMode", 2269, 2268, iface_int, iface_void},
	{"WrapStartIndent", 2465, 2464, iface_int, iface_void},
	{"WrapVisualFlags", 2461, 2460, iface_int, iface_void},
	{"WrapVisualFlagsLocation", 2463, 2462, iface_int, iface_void},
	{"XOffset", 2398, 2397, iface_int, iface_void},
	{"Zoom", 2374, 2373, iface_int, iface_void}
};

enum {
	ifaceFunctionCount = 284,
	ifaceConstantCount = 2487,
	ifacePropertyCount = 215
};

//--Autogenerated

const IFaceFunction * const IFaceTable::functions = ifaceFunctions;
const int IFaceTable::functionCount = ifaceFunctionCount;

const IFaceConstant * const IFaceTable::constants = ifaceConstants;
const int IFaceTable::constantCount = ifaceConstantCount;

const IFaceProperty * const IFaceTable::properties = ifaceProperties;
const int IFaceTable::propertyCount = ifacePropertyCount;

#ifdef _MSC_VER
// Unreferenced inline functions are OK
#pragma warning(disable: 4514)
#endif
