/*
 * Copyright (C) 1983-1989  Masatoshi Kurihara <kurihara@sra.co.jp>
 * Copyright (C) 1999, 2000 and 2001
 * Jun-ichiro itojun Hagino <itojun@iijlab.net>
 * All rights reserved.
 *
 * Note well: this is not a normal 3-clause BSD copyright;
 * commercial use of the software is restricted.
 *
 * Redistribution and non-commercial use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Copyright (C) 1985 - 1989 by Software Research Associates, Inc. */

#include <curses.h>
#include <common.h>
#ifdef MULTIBYTE
# include <wchar.h>
#endif

#include <def.h>
#include <entry.h>
#include <var.h>
#include <common.h>
#include <sub.h>
#include <sch.h>
#include <opts.h>

static int doshell __P((const char *, int));
static int dowq __P((const char *, int));
static int doquit __P((const char *, int));
static int dowrite __P((const char *, int));
static int docolon __P((void));

#ifdef MULTIBYTE
#include <machine/limits.h>

/*
 * screen width -> the offset of character cursor is sitting on
 */
int
scrwidth2strlen(str, scrpos)
	const wchar_t *str;
	int scrpos;
{
	const wchar_t *p;
	int pos;

	pos = 0;
	p = str;
	while (*p && pos + scrwidth(*p) - 1 < scrpos) {
		pos += scrwidth(*p);
		p++;
	}
	return p - str;
}

/*
 * the offset of character -> screen width
 */
int
strlen2scrwidth(str, len)
	const wchar_t *str;
	int len;
{
	int i;
	int width;

	width = 0;
	for (i = 0; str[i] && i < len; i++) {
		width += scrwidth(str[i]);
	}
	return width;
}

/*
 * Column(screen column!) cut for widechar
 */
wchar_t *
wcolcut(line, tail)
	const wchar_t *line;
	int tail;
{
	int	cnt;
	static wchar_t msgbuf[LEN_BUFF];

	wcscpy(msgbuf, line);
	cnt = scrwidth2strlen(msgbuf, tail);
	if (cnt < wcslen(msgbuf))
		msgbuf[cnt] = 0;
	return msgbuf;
}
#endif /*MULTIBYTE*/

/*
 * Change message
 */
void
chnfld(field)
	int field;
{
	int cnt, incnt, maxlen, maxdisp, curx, cury;
	int insert = FALSE;
#ifndef MULTIBYTE
	int inchar;
	char *cpyback;
	static char savebuf[LEN_BUFF], editbuf[LEN_BUFF], edittmp[LEN_BUFF];
#else
	wchar_t inchar;
	void *cpyback;
	static wchar_t savebuf[LEN_BUFF], editbuf[LEN_BUFF], edittmp[LEN_BUFF];
	int wbedit = 0;
	int curx0;
	int i;
#endif

	/*
	 * Save cursor and current message
	 */
	getyx(schwin, cury, curx);
#ifdef MULTIBYTE
	curx0 = curx;
#endif
	if (prjsw) {
		memset(savebuf, 0, sizeof(savebuf));
		switch (field) {
		case FLD_WORK:
			incnt  = curx - SCR_WORK;
			maxlen = SCR_PRJ - SCR_WORK - 1;
			maxdisp= maxlen;
#ifndef MULTIBYTE
			strcpy(savebuf, cpyback = sch[cursch]->work);
#else
			cpyback = sch[cursch]->work;
			mbstowcs(savebuf, sch[cursch]->work,
				sizeof(savebuf)/sizeof(savebuf[0]));
#endif
			break;
		case FLD_PRJ:
			incnt  = curx - SCR_PRJ;
			maxlen = scrmsg - SCR_PRJ - 1;
			maxdisp= maxlen;
#ifndef MULTIBYTE
			strcpy(savebuf, cpyback = sch[cursch]->prj);
#else
			cpyback = sch[cursch]->work;
			mbstowcs(savebuf, sch[cursch]->prj,
				sizeof(savebuf)/sizeof(savebuf[0]));
#endif
			break;
		case FLD_MSG:
#ifndef MULTIBYTE
			incnt  = curx - scrmsg;
#else
			incnt = scrwidth2strlen(sch[cursch]->msg, curx - scrmsg);
#endif
			maxlen = LEN_MESG - 2;
			maxdisp= COLS - scrmsg - 1;
#ifndef MULTIBYTE
			strcpy(savebuf, cpyback = sch[cursch]->msg);
#else
			wcscpy(savebuf, cpyback = sch[cursch]->msg);
			wbedit++;
#endif
			break;
		}
	}
	else {
#ifndef MULTIBYTE
		incnt  = curx - scrmsg;
#else
		incnt = scrwidth2strlen(sch[cursch]->msg, curx - scrmsg);
#endif
		maxlen = LEN_MESG - 2;
		maxdisp= COLS - scrmsg - 1;
#ifndef MULTIBYTE
		strcpy(savebuf, cpyback = sch[cursch]->msg);
#else
		wcscpy(savebuf, cpyback = sch[cursch]->msg);
		wbedit++;
#endif
	}
	clrline(editbuf, LEN_BUFF);
#ifndef MULTIBYTE
	strcpy(editbuf, savebuf);
#else
	wcscpy(editbuf, savebuf);
#endif

	/*
	 * Command and string get
	 */
	while ((inchar = getch()) != UPD_EXIT) {
		/*
		 * Convert erase character
		 */
#ifdef HAVE_TERMIOS_H
		if (inchar == tio.c_cc[VERASE])
			inchar = ERS_CHAR;
		else if (inchar == tio.c_cc[VKILL])
			inchar = ERS_LINE;
#else
		if (inchar == ttyb.sg_erase)
			inchar = ERS_CHAR;
		else if (inchar == ttyb.sg_kill)
			inchar = ERS_LINE;
#endif

		switch (inchar) {
		case UPD_RTCHAR:
			/*
			 * Next character
			 */
			if (incnt < maxdisp && editbuf[incnt]) {
#ifndef MULTIBYTE
				wmove(schwin, cury, ++curx);
#else
				curx += scrwidth(editbuf[incnt]);
				wmove(schwin, cury, curx);
#endif
				incnt++;
				wrefresh(schwin);
			}
			break;

		case UPD_LFCHAR:
			/*
			 * Previous character
			 */
			if (incnt) {
#ifndef MULTIBYTE
				wmove(schwin, cury, --curx);
#else
				curx -= scrwidth(editbuf[incnt - 1]);
				wmove(schwin, cury, curx);
#endif
				incnt--;
				wrefresh(schwin);
			}
			break;

		case TAB:
		case UPD_RTWORD:
			/*
			 * Next word
			 */
			cnt = incnt;
			while (incnt < maxdisp && editbuf[incnt] && 
			       ! isspace(editbuf[incnt])) {
#ifndef MULTIBYTE
				curx++;
#else
				curx += scrwidth(editbuf[incnt]);
#endif
				incnt++;
			}
			while (incnt < maxdisp && editbuf[incnt] && 
			       isspace(editbuf[incnt])) {
#ifndef MULTIBYTE
				curx++;
#else
				curx += scrwidth(editbuf[incnt]);
#endif
				incnt++;
			}
			if (cnt != incnt) {
				wmove(schwin, cury, curx);
				wrefresh(schwin);
			}
			break;

		case UPD_LFWORD:
			/*
			 * Previous word
			 */
			if (incnt) {
				cnt = incnt--;
				curx--;
				while (incnt && isspace(editbuf[incnt])) {
					incnt--;
					curx--;
				}
				while (incnt && !isspace(editbuf[incnt])) {
					incnt--;
					curx--;
				}
				if (incnt) {
					incnt++;
					curx++;
				}
				if (cnt != incnt) {
					wmove(schwin, cury, curx);
					wrefresh(schwin);
				}
			}
			break;

		case UPD_EOLDEL:
			/*
			 * Erase to EOL
			 */
#if 0
			for (cnt = incnt; cnt < maxdisp; cnt++) {
				waddch(schwin, SPACE);
			}
#else
			wclrtoeol(schwin);
#endif
			clrline(editbuf + incnt, LEN_BUFF - incnt);
			wmove(schwin, cury, curx);
			wrefresh(schwin);
			break;

		case UPD_WRDDEL:
			/*
			 * Erase word
			 */
			cnt = incnt;
			while (editbuf[cnt] && !isspace(editbuf[cnt]))
				cnt++;
			while (editbuf[cnt] && isspace(editbuf[cnt]))
				cnt++;
			if (cnt != incnt) {
				if (incnt && !isspace(editbuf[incnt-1]))
					cnt--;
#ifndef MULTIBYTE
				strcpy(editbuf + incnt, editbuf + cnt);
				cnt = strlen(editbuf);
#else
				wcscpy(editbuf + incnt, editbuf + cnt);
				cnt = wcslen(editbuf);
#endif
				clrline(editbuf + cnt, LEN_BUFF - cnt);
				for (cnt = incnt; cnt < maxdisp; cnt++)
					waddch(schwin, SPACE);
				wmove(schwin, cury, curx);
#ifndef MULTIBYTE
				waddstr(schwin,
					 jisdsp(colcut(editbuf + incnt,
							 maxdisp - incnt)));
#else
				wmbdsp(schwin, wcolcut(editbuf + incnt,
				    maxdisp - strlen2scrwidth(editbuf, incnt)));
#endif
				wmove(schwin, cury, curx);
				wrefresh(schwin);
			}
			break;

		case UPD_1DELCHR:
		case UPD_2DELCHR:
			/*
			 * Erase character
			 */
			if (editbuf[incnt]) {
				cnt = incnt + 1;
#ifndef MULTIBYTE
				strcpy(editbuf + incnt, editbuf + cnt);
				cnt = strlen(editbuf);
				clrline(editbuf + cnt, LEN_BUFF - cnt);
				for (cnt = incnt; cnt < maxdisp; cnt++)
					waddch(schwin, SPACE);
				wmove(schwin, cury, curx);
				waddstr(schwin,
					 jisdsp(colcut(editbuf + incnt,
							 maxdisp - incnt)));
#else
				memmove(editbuf + incnt, editbuf + cnt,
					sizeof(editbuf[0]) * (LEN_BUFF - cnt));
				wclrtoeol(schwin);
				wmbdsp(schwin, wcolcut(editbuf + incnt,
				    maxdisp - strlen2scrwidth(editbuf, incnt)));
#endif
				wmove(schwin, cury, curx);
				wrefresh(schwin);
			}
			break;

		case UPD_UNDO:
			/*
			 * Undo all update
			 */
			clrline(editbuf, LEN_BUFF);
#ifndef MULTIBYTE
			strcpy(editbuf, savebuf);
			curx -= incnt;
#else
			wcscpy(editbuf, savebuf);
			curx = curx0;
#endif
			incnt = 0;
			wmove(schwin, cury, curx);
#if 0
			for (cnt = 0; cnt < maxdisp; cnt++)
				waddch(schwin, SPACE);
#else
			wclrtoeol(schwin);
#endif
			wmove(schwin, cury, curx);
#ifndef MULTIBYTE
			waddstr(schwin, jisdsp(colcut(editbuf, maxdisp)));
#else
			wmbdsp(schwin, wcolcut(editbuf, maxdisp));
#endif
			wmove(schwin, cury, curx);
			wrefresh(schwin);
			break;

		case UPD_INSERT:
			/*
			 * Insert mode change
			 */
			if (insert) {
				insert = FALSE;
				wstandout(errwin);
				mesg(COLS - 12, " [replace] ");
				wstandend(errwin);
			}
			else {
				insert = TRUE;
				wstandout(errwin);
				mesg(COLS - 12, " [insert] ");
				wstandend(errwin);
			}
			break;

		case UPD_NXEXIT:
			/*
			 * Exit and goto next line
			 */
			if (cursch < maxsch - 1) {
				if (cury < getmaxy(schwin) - 1) {
					wmove(schwin, cury + 1, SCR_EDAY);
					cursch++;
				} else {
					roll(1);
					cursch++;
					wmove(schwin, cury, SCR_EDAY);
				}
			} else
				wmove(schwin, cury, SCR_EDAY);
			wrefresh(schwin);
			goto UPD_END;

		case EOF:
			/*
			 * High value on "cbreak" mode,  should be ignore.
			 */
			break;

		default:
			/*
			 * New character set
			 */
#ifdef MULTIBYTE
			ungetc(inchar, stdin);
			inchar = fgetrune(stdin);
#endif
			if (!isprint(inchar) && !isspace(inchar)) {
				errbell(0);
				break;
			}
#ifndef MULTIBYTE
			if ((incnt >= maxdisp ||
			    (insert && strlen(editbuf) >= maxlen)))
#else
			if ((strlen2scrwidth(editbuf, incnt) >= maxdisp ||
			    (insert && wcslen(editbuf) >= maxlen)))
#endif
			{
				if (inchar != SPACE)
					errbell(0);
				if (++curx < COLS) {
					wmove(schwin, cury, curx);
					wrefresh(schwin);
				}
				goto	UPD_END;
			}
			if (!editbuf[incnt]) {
				/*
				 * Append to tail
				 */
				editbuf[incnt++] = inchar;
				editbuf[incnt] = '\0';
#ifndef MULTIBYTE
				waddch(schwin, inchar);
#else
				wmbdsp(schwin, editbuf + incnt - 1);
#endif
			}
			else if (insert) {
				/*
				 * Insert mode
				 */
#ifndef MULTIBYTE
				strcpy(edittmp, editbuf + incnt);
				strcpy(editbuf + incnt + 1, edittmp);
				editbuf[incnt++] = inchar;
				waddch(schwin, inchar);
				waddstr(schwin, jisdsp(
					colcut(editbuf + incnt,
						  maxdisp - incnt)));
#else
				wcscpy(edittmp, editbuf + incnt);
				wcscpy(editbuf + incnt + 1, edittmp);
				editbuf[incnt++] = inchar;
				wclrtoeol(schwin);
				wmbdsp(schwin, wcolcut(editbuf + incnt - 1,
				  maxdisp - strlen2scrwidth(editbuf, incnt - 1)));
#endif
			}
			else {
				/*
				 * Replace mode
				 */
#ifndef MULTIBYTE
				editbuf[incnt++] = inchar;
				waddch(schwin, inchar);
#else
			    {
				char tmp[16];	/*XXX*/
				if (scrwidth(editbuf[incnt])
						== scrwidth(inchar)) {
					int i;

					editbuf[incnt++] = inchar;
					tmp[0] = '\0';
					i = wctomb(tmp, inchar);
					if (i < 0) {
						tmp[i] = '\0';
						waddstr(schwin, tmp);
					}
				} else {
					editbuf[incnt++] = inchar;
					tmp[0] = '\0';
					wclrtoeol(schwin);
					wmbdsp(schwin,
					  wcolcut(editbuf + incnt - 1,
					    maxdisp - strlen2scrwidth(editbuf, incnt - 1)));
				}
			    }
#endif
			}
#ifndef MULTIBYTE
			curx++;
#else
			curx += scrwidth(inchar);
#endif
			wmove(schwin, cury, curx);
			wrefresh(schwin);
		}
	}

	UPD_END:
	/*
	 * Copy back
	 */
#ifndef MULTIBYTE
	strcpy(cpyback, editbuf);
#else
	if (wbedit)
		wcscpy((wchar_t *)cpyback, editbuf);
	else {
		*(char *)cpyback = '\0';
		wcstombs((char *)cpyback, editbuf, LEN_BUFF);	/*XXX*/
	}
#endif /*MULTIBYTE*/
	updated = TRUE;
}

static int
doshell(cmd, bang)
	const char *cmd;
	int bang;
{
	const char *p;

	p = cmd;
	while (p && *p && !isspace(*p))
		p++;
	while (p && *p && isspace(*p))
		p++;
	if (*p == '\0') {
		error("no command specified");
		return -1;
	}

	execom(p);
	return 0;
}

static int
dowq(cmd, bang)
	const char *cmd;
	int bang;
{
	if (dowrite("w", bang) < 0)
		return -1;
	return doquit(cmd, bang);
}

static int
doquit(cmd, bang)
	const char *cmd;
	int bang;
{
	int inchar;
	char buf[LEN_BUFF];

	/*
	 * Quit without write back
	 */
	if (strcmp(cmd, "Q") == 0) {
		/* interactive */
		if (verbose || updated) {
			snprintf(buf, sizeof(buf),
				"Quit; tap again to confirm%s",
				updated ? " (file modified since the last write)" : "");
			mesg(0, buf);
		}
		inchar = getch();
		if (inchar == 'Q')
			fin();
		else {
			mesg(-1, "");
			return 0;
		}
	}

	/* colon mode */
	if (!bang && updated) {
		mesg(0, "File modified since last complete write; "
			"write or use ! to override");
	} else
		fin();
	return 0;
}

static int
dowrite(cmd, bang)
	const char *cmd;
	int bang;
{
	/*
	 * Write back
	 */
	if (!bang && ! writesw) {
		error("File is read only");
		return -1;
	}
#if 0
	else if (!bang && ! updated) {
		error("Data is not updated");
		return -1;
	}
#endif
	wmove(errwin, 0, 0);
	if (newsw) {
		wprintw(errwin, "\"%s\" [New file]", schsname);
		newsw = FALSE;
	} else
		wprintw(errwin, "\"%s\"", schsname);
	wclrtoeol(errwin);
	wrefresh(errwin);
	writesch(schfname);
	wprintw(errwin, " %d lines ", maxsch);
	wrefresh(errwin);
	wrefresh(schwin);
	updated = FALSE;
	msgsw = TRUE;
	return 0;
}

static int
docolon()
{
	const char *cmd, *p;
	char *q;
	int bang;
	char buf[LEN_BUFF];
	int longcmd;
	int ret;
	int repaint;

	longcmd = 0;
	repaint = 0;

again:
	mesg(0, ":");
	cmd = wgets(errwin, 0, 1);
	if (cmd == NULL) {
		error("Invalid command");
		return -1;
	}
	if (longcmd) {
		wmove(errwin, 0, 0);
		wrefresh(errwin);
		fputc('\n', stdout);
		fflush(stdout);
	}

	for (p = cmd; p && *p && isspace(*p); p++)
		;
	if (p == NULL || *p == '\0') {
		ret =  -1;
		goto end;
	}

	strcpy(buf, p);
	for (q = buf; q && *q && !isspace(*q); q++)
		;
	if (q > buf && q[-1] == '!') {
		bang = 1;
		q--;
	} else
		bang = 0;

	if (bang && q == buf) {
		ret = doshell(buf, bang);
		if (ret >= 0)
			longcmd++;
	} else if (strncmp(buf, "w", q - buf) == 0)
		ret = dowrite(buf, bang);
	else if (strncmp(buf, "wq", q - buf) == 0)
		ret = dowq(buf, bang);
	else if (strncmp(buf, "q", q - buf) == 0)
		ret = doquit(buf, bang);
	else if (strncmp(buf, "set", q - buf) == 0) {
		int r;
		ret = doset(buf, bang, &r);
		if (ret == 1)
			longcmd++;
		if (r)
			repaint++;
	} else {
		error("Invalid command");
		ret = -1;
	}

	if (longcmd) {
		wmove(errwin, 0, 0);
		wclrtoeol(errwin);
		waddstr(errwin, "[Hit any key to continue]");
		wrefresh(errwin);

		ret = getch();
		if (ret == ':')
			goto again;
		ret = 0;
	}

end:
	if (repaint) {
		if (prjsw)
			scrmsg	= SCR_MSG;
		else
			scrmsg	= SCR_WORK;
		puthead();
		schput(cursch);
		refwin();
	}
	return ret;
}

/*
 * Command get and execute
 */
void
command()
{
	int inchar, incnt, cnt, curx, cury;
	char *keyin;
	char msgbuf[LEN_BUFF];
#ifdef MULTIBYTE
	int woff;
#endif
	int yankbuf;
	int count;

	yankbuf = 0;
	count = 0;
LOOP:
	/*
	 * Command get
	 */
	
#ifdef HAVE_TERMIOS_H
	if ((inchar = getch()) == tio.c_cc[VERASE])
		inchar = CMD_SHCHAR;
#else
	if ((inchar = getch()) == ttyb.sg_erase)
		inchar = CMD_SHCHAR;
#endif
	if (isdigit(inchar)) {
		if (count == 0 && inchar == '0')
			; /* '0' is command by itself */
		else {
			count = count * 10 + inchar - '0';
			goto LOOP;
		}
	}
	switch (inchar) {
	case 'F' & 0x1f:
		/*
		 * Next screen
		 */
		if (count == 0)
			count = 1;
		count *= getmaxy(schwin);
		count -= 2;
		if (count <= 0)
			count = 1;
		roll(count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case 'D' & 0x1f:
		/*
		 * Next half screen
		 */
		if (count == 0)
			count = getmaxy(schwin) / 2;
		if (count <= 0)
			count = 1;
		roll(count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case 'B' & 0x1f:
		/*
		 * Previous screen
		 */
		if (count == 0)
			count = 1;
		count *= getmaxy(schwin);
		count -= 2;
		if (count <= 0)
			count = 1;
		roll(-1 * count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case 'U' & 0x1f:
		/*
		 * Previous half screen
		 */
		if (count == 0)
			count = getmaxy(schwin) / 2;
		if (count <= 0)
			count = 1;
		roll(- 1 * count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case '+':
	case '\n':
		/*
		 * Next line
		 */
		if (cursch < maxsch - 1) {
			getyx(schwin, cury, curx);
			if (cury < getmaxy(schwin) - 1) {
				wmove(schwin, cury + 1, SCR_EDAY);
				cursch++;
			} else
				schput(cursch + 1);
			wrefresh(schwin);
			mesg(-1, "");
		}
		break;

	case '-':
		/*
		 * Previous line
		 */
		if (cursch > 0) {
			getyx(schwin, cury, curx);
			if (cury > 0) {
				wmove(schwin, cury - 1, SCR_EDAY);
				cursch--;
			} else
				schput(cursch - 1);
			wrefresh(schwin);
			mesg(-1, "");
		}
		break;

	case 'Y' & 0x1f:
		/*
		 * Scroll 1 line
		 */
		if (count == 0)
			count = 1;
		getyx(schwin, cury, curx);
		if (cursch - cury - count < 0)
			count = cursch - cury;
		roll(-1 * count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case 'E' & 0x1f:
		/*
		 * Scroll 1 line
		 */
		if (count == 0)
			count = 1;
		incnt = 0;
		getyx(schwin, cury, curx);
		if (cursch - cury + getmaxy(schwin) >= maxsch)
			break;
		if (cursch - cury + count + getmaxy(schwin) >= maxsch)
			count = maxsch - getmaxy(schwin) - (cursch - cury);
		roll(count);
		mesg(-1, "");
		wrefresh(schwin);
		break;

	case 'j':
		/*
		 * Next line same column
		 */
		if (count == 0)
			count = 1;
		while (count-- > 0 && cursch < maxsch - 1) {
			getyx(schwin, cury, curx);
			if (curx > scrmsg) {
#ifndef MULTIBYTE
				if (!sch[cursch+1]->msg[curx-scrmsg]) {
					curx = scrmsg
						+ strlen(sch[cursch + 1]->msg);
				}
#else
				woff = scrwidth2strlen(sch[cursch + 1]->msg,
					curx - scrmsg);
				curx = scrmsg
					+ strlen2scrwidth(sch[cursch + 1]->msg,
						woff);
#endif
			}
			if (cury < getmaxy(schwin) - 1)
				wmove(schwin, cury + 1, curx);
			else {
				screen_scroll(1);
				wmove(schwin, cury, curx);
			}
			cursch++;
			mesg(-1, "");
		}
		wrefresh(schwin);
		break;

	case 'k':
		/*
		 * Previous line same column
		 */
		if (count == 0)
			count = 1;
		while (count-- > 0 && cursch > 0) {
			getyx(schwin, cury, curx);
			if (curx > scrmsg) {
#ifndef MULTIBYTE
				if (!(sch[cursch - 1]->msg[curx - scrmsg])) {
					curx = scrmsg
						+ strlen(sch[cursch - 1]->msg);
				}
#else
				woff = scrwidth2strlen(sch[cursch - 1]->msg,
					curx - scrmsg);
				curx = scrmsg
					+ strlen2scrwidth(sch[cursch - 1]->msg,
						woff);
#endif
			}

			if (cury > 0)
				wmove(schwin, cury - 1, curx);
			else {
				screen_scroll(-1);
				wmove(schwin, cury, curx);
			}
			cursch--;
			mesg(-1, "");
		}
		wrefresh(schwin);
		break;

	case 'l':
	case ' ':
		/*
		 * Next character
		 */
		if (count == 0)
			count = 1;
		getyx(schwin, cury, curx);
		while (count-- > 0 && curx < COLS - 1) {
			incnt = 0;
			if (curx < scrmsg)
				incnt = 1;
			else {
#ifndef MULTIBYTE
				if (sch[cursch]->msg[curx - scrmsg])
					incnt = 1;
#else
				woff = scrwidth2strlen(sch[cursch]->msg,
					curx - scrmsg);
				if (sch[cursch]->msg[woff])
					incnt = scrwidth(sch[cursch]->msg[woff]);
#endif
				if (!(curx + incnt < COLS - 1))
					incnt = 0;
			}
			if (incnt)
				wmove(schwin, cury, curx + incnt);
			getyx(schwin, cury, curx);
		}
		wrefresh(schwin);
		break;

	case 'h':
	case '\010':
		/*
		 * Previous character
		 */
		if (count == 0)
			count = 1;
		getyx(schwin, cury, curx);
		while (count-- > 0 && curx > 0) {
#ifndef MULTIBYTE
			incnt = 1;
#else
			if (curx <= scrmsg)
				incnt = 1;
			else {
				woff = scrwidth2strlen(sch[cursch]->msg,
					curx - scrmsg);
				incnt = scrwidth(sch[cursch]->msg[woff - 1]);
			}
#endif
			wmove(schwin, cury, curx - incnt);
			getyx(schwin, cury, curx);
		}
		wrefresh(schwin);
		break;

	case '0':
		/*
		 * Head of line
		 */
		getyx(schwin, cury, curx);
		wmove(schwin, cury, SCR_EDAY);
		wrefresh(schwin);
		break;

	case '$':
		/*
		 * End of line
		 */
#ifndef MULTIBYTE
		cnt = scrmsg + strlen(sch[cursch]->msg) < COLS - 1 ?
		      scrmsg + strlen(sch[cursch]->msg) : COLS - 1;
#else
		for (incnt = 0; sch[cursch]->msg[incnt]; incnt++)
			;
		cnt = scrmsg + strlen2scrwidth(sch[cursch]->msg, incnt);
		if (COLS <= cnt)
			cnt = COLS - 1;
#endif
		getyx(schwin, cury, curx);
		wmove(schwin, cury, cnt);
		wrefresh(schwin);
		break;

	case 'H':
		/*
		 * Head of window
		 */
		getyx(schwin, cury, curx);
		wmove(schwin, 0, SCR_EDAY);
		cursch -= cury;
		wrefresh(schwin);
		mesg(-1, "");
		break;

	case 'M':
		/*
		 * Middle of window
		 */
		getyx(schwin, cury, curx);
		cnt = cury + (maxsch - cursch - 1) < getmaxy(schwin) - 1 ?
		      cury + (maxsch - cursch - 1) : getmaxy(schwin) - 1;
		cnt = cnt / 2 - cury;
		wmove(schwin, cury + cnt, SCR_EDAY);
		cursch = cursch + cnt;
		wrefresh(schwin);
		mesg(-1, "");
		break;

	case 'L':
		/*
		 * Bottom of window
		 */
		getyx(schwin, cury, curx);
		cnt = cury + (maxsch - cursch - 1) < getmaxy(schwin) - 1 ?
		      cury + (maxsch - cursch - 1) : getmaxy(schwin) - 1;
		wmove(schwin, cnt, SCR_EDAY);
		cursch = cursch + cnt - cury;
		wrefresh(schwin);
		mesg(-1, "");
		break;

	case 'w':
		/*
		 * Next field, or 3-hour jump
		 */
		getyx(schwin, cury, curx);
		if (prjsw && curx < SCR_WORK)
			incnt = SCR_WORK;
		else if (!prjsw && curx < scrmsg)
			incnt = scrmsg;
		else
			incnt = 0;
		if (incnt) {
			for (curx++; curx < incnt; curx++) {
				cnt = (curx - SCR_BAR) / hourwidth + daystart;
				if ((curx - SCR_BAR) % hourwidth == 0
				 && cnt % 3 == 0)
					break;
			}
			wmove(schwin, cury, curx);
			wrefresh(schwin);
			break;
		}
		/*FALLTHROUGH*/
	case '\t':
	case 'W':
		/*
		 * Next field
		 */
		getyx(schwin, cury, curx);
		if (curx < SCR_BAR)
			wmove(schwin, cury, SCR_BAR);
		else if (prjsw && curx < SCR_WORK)
			wmove(schwin, cury, SCR_WORK);
		else if (prjsw && curx < SCR_PRJ)
			wmove(schwin, cury, SCR_PRJ);
		else if (curx < scrmsg)
			wmove(schwin, cury, scrmsg);
		else {
			/*
			 * Next word
			 */
#ifndef MULTIBYTE
			incnt = curx - scrmsg;
#else
			incnt = scrwidth2strlen(sch[cursch]->msg, curx - scrmsg);
#endif
			cnt = incnt;
			while (curx < COLS - 1 && sch[cursch]->msg[incnt] && 
			       ! isspace(sch[cursch]->msg[incnt])) {
#ifndef MULTIBYTE
				curx++;
#else
				curx += scrwidth(sch[cursch]->msg[incnt]);
#endif
				incnt++;
			}
			while (curx < COLS - 1 && sch[cursch]->msg[incnt] && 
			       isspace(sch[cursch]->msg[incnt])) {
#ifndef MULTIBYTE
				curx++;
#else
				curx += scrwidth(sch[cursch]->msg[incnt]);
#endif
				incnt++;
			}
			if (cnt != incnt)
				wmove(schwin, cury, curx);
			else {
				/*
				 * Next line
				 */
				if (cursch < maxsch - 1) {
					getyx(schwin, cury, curx);
					if (cury < getmaxy(schwin) - 1) {
						wmove(schwin, 
							 cury + 1, SCR_EDAY);
						cursch++;
					} else
						schput(cursch + 1);
					mesg(-1, "");
				}
			}
		}
		wrefresh(schwin);
		break;

	case 'b':
		/*
		 * Previous field, or 3-hour jump
		 */
		getyx(schwin, cury, curx);
		if (prjsw && curx <= SCR_WORK)
			incnt = SCR_WORK;
		else if (!prjsw && curx <= scrmsg)
			incnt = scrmsg;
		else
			incnt = 0;
		if (SCR_BAR < curx && incnt) {
			for (curx--; SCR_BAR < curx; curx--) {
				cnt = (curx - SCR_BAR) / hourwidth + daystart;
				if ((curx - SCR_BAR) % hourwidth == 0
				 && cnt % 3 == 0)
					break;
			}
			wmove(schwin, cury, curx);
			wrefresh(schwin);
			break;
		}
		/*FALLTHROUGH*/
	case 'B':
		/*
		 * Previous field
		 */
		getyx(schwin, cury, curx);
		if (curx <= SCR_EDAY) {
			/*
			 * Previous line
			 */
			if (cursch > 0) {
				getyx(schwin, cury, curx);
				if (cury > 0) {
					wmove(schwin, cury - 1, scrmsg);
					cursch--;
				} else
					schput(cursch - 1);
				mesg(-1, "");
			}
		}
		else if (curx <= SCR_BAR)
			wmove(schwin, cury, SCR_EDAY);
		else if (prjsw && curx <= SCR_WORK)
			wmove(schwin, cury, SCR_BAR);
		else if (prjsw && curx <= SCR_PRJ)
			wmove(schwin, cury, SCR_WORK);
		else if (curx <= scrmsg) {
			if (prjsw)
				wmove(schwin, cury, SCR_PRJ);
			else
				wmove(schwin, cury, SCR_BAR);
		}
		else {
			/*
			 * Previous word
			 */
#ifndef MULTIBYTE
			incnt = curx - scrmsg;
#else
			incnt = scrwidth2strlen(sch[cursch]->msg, curx - scrmsg);
#endif

			incnt--;
			while (incnt && isspace(sch[cursch]->msg[incnt]))
				incnt--;
			while (incnt && !isspace(sch[cursch]->msg[incnt]))
				incnt--;
			if (incnt)
				incnt++;
#ifndef MULTIBYTE
			curx = scrmsg + incnt;
#else
			curx = scrmsg + strlen2scrwidth(sch[cursch]->msg, incnt);
#endif
			wmove(schwin, cury, curx);
		}
		wrefresh(schwin);
		break;

	case 'W' & 0x1f:
		/*
		 * Project code mode change
		 */
		getyx(schwin, cury, curx);
		if (prjsw) {
			prjsw = FALSE;
			scrmsg = SCR_WORK;
			puthead();
		} else {
			prjsw = TRUE;
			scrmsg = SCR_MSG;
			puthead();
		}
		schput(cursch - cury);
		/* cursch modified */
		goline(cursch + cury);
		refwin();
		break;

	case 'A' & 0x1f:
	case 'C' & 0x1f:
		/*
		 * Calander mode change
		 */
		getyx(schwin, cury, curx);
		if (calsw) {
			calsw = FALSE;
			schwin = schfull;
		} else {
			if (LEN_HFWSCH <= 0) {
				error("Screen too small");
				break;
			}
			calsw = TRUE;
			schwin = schhalf;
		}
		if (cury < getmaxy(schwin)) {
			schput(cursch - cury);
			/* cursch modified */
			goline(cursch + cury);
			wmove(schwin, cury, curx);
		} else {
			schput(cursch - (getmaxy(schwin) - 1));
			/* cursch modified */
			goline(cursch + getmaxy(schwin) - 1);
		}
		refwin();
		break;

#if 0
	case CMD_AUTOWB:
		/*
		 * Auto write mode change
		 */
		if (autowrite) {
			autowrite = FALSE;
			mesg(0, "Auto Write Disable");
		}
		else {
			autowrite = TRUE;
			mesg(0, "Auto Write Enable");
		}
		break;
#endif

	case CMD_VERSION:
		/*
		 * Version display
		 */
		sprintf(msgbuf, "\"sch\" version %s, %s", VERSION, VERDATE);
		mesg(0, msgbuf);
		break;

	case CMD_STATUS:
		/*
		 * Status display
		 */
		sprintf(msgbuf,
			 "\"%s\"%s%s%s date %d/%d, line %d of %d --%d%%--",
			 schsname, writesw ? "" : " [ReadOnly]",
			 updated ? " [Modified]" : "",
			 (writesw && autowrite) ? " [AutoWrite]" : "",
			 sch[cursch]->month, sch[cursch]->day,
			 cursch + 1, maxsch, (100 * (cursch + 1)) / maxsch);
		msgbuf[COLS - 1] = NULL;
		mesg(0, msgbuf);
		break;

	case CMD_HARDCOPY:
		/*
		 * Hardcopy print
		 */
		mesg(0, "Hardcopy File ? ");
		keyin = wgets(errwin, 0, 16);
		if (*keyin)
			hardcopy(keyin);
		else
			mesg(-1, "");
		break;

	case 'L' & 0x1f:
	case 'R' & 0x1f:
		/*
		 * Redraw
		 */
		touchwin(curscr);
		clearok(curscr, TRUE);
		wrefresh(curscr);
		wrefresh(schwin);
		break;

	case 's':
		/*
		 * Data sort
		 */
		if (! writesw)
			error("File is read only");
		else {
			getyx(schwin, cury, curx);
			qsort(sch, maxsch, sizeof(sch[0]), schcmp);
			if (cury > 3) {
				cnt = cursch;
				schput(cursch - cury);
				goline(cnt);
			} else
				schput(cursch);
			updated = TRUE;
			mesg(-1, "");
			wrefresh(schwin);
		}
		break;

	case CMD_PUT:
		/*
		 * Put field
		 */
		if (! writesw)
			error("File is read only");
		else if ((cnt = ismsg()) >= FLD_WORK) {
			if ((cnt == FLD_WORK && isdisp(yank[yankbuf].work)) ||
			    (cnt == FLD_PRJ && isdisp(yank[yankbuf].prj)) ||
			    (cnt == FLD_MSG && isdisp(yank[yankbuf].msg)))
				fldrest(yankbuf, cnt);
			else {
				sprintf(msgbuf, 
					 "Register %d is empty", yankbuf);
				mesg(0, msgbuf);
			}
		} else
			error("Not message field");
		break;

	case CMD_LNPUT:
		/*
		 * Put line
		 */
		if (! writesw)
			error("File is read only");
		else if (maxsch < MAXDATA) {
			if (isdisp(yank[yankbuf].work) ||
			    isdisp(yank[yankbuf].prj) ||
			    isdisp(yank[yankbuf].msg)) {
				openline(cursch + 1, -1);
				elmrest(yankbuf);
				mesg(-1, "");
			} else {
				sprintf(msgbuf, 
					 "Register %d is empty", yankbuf);
				mesg(0, msgbuf);
			}
		} else
			error("No free space in buffer");
		break;

	case '"':
		incnt = getch();
		if (isdigit(incnt))
			yankbuf = incnt - '0';
		else if (incnt != ESCAPE && incnt != SPACE)
			error("Register must be `0' to `9'");
		break;

	case CMD_YANK:
		/*
		 * Yanc line
		 */
		if (! writesw) {
			error("File is read only");
			break;
		}
		if (verbose)
			mesg(0, "Yank; speicify register");
		incnt = getch();
		if (incnt == CMD_YANK)
			incnt = yankbuf + '0';
		if (isdigit(incnt)) {
			incnt -= '0';
			elmsave(incnt);
			sprintf(msgbuf, "Saved in register %d", incnt);
			mesg(0, msgbuf);
		} else if (incnt != ESCAPE && incnt != SPACE)
			error("Register must be `0' to `9'");
		break;

	case CMD_PRYANK:
		/*
		 * Print yanked line
		 */
		yanklist();
		break;

	case CMD_MARK:
		/*
		 * Mark line
		 */
		incnt = getch();
		if (incnt == ESCAPE || incnt == SPACE)
			break;
		else if (incnt >= '1' && incnt <= '9') {
			incnt -= '0';
			mark[incnt] = cursch;
			sprintf(msgbuf, "Marked in register %d", incnt);
			mesg(0, msgbuf);
		} else
			error("Register must be `1' to `9'");
		break;

	case 'G':
		/*
		 * Jump
		 */
		if (count == 0)
			count = maxsch;
		if (count > maxsch) {
			if (verbose)
				error("Already at end-of-file");
			else
				errbell(0);
			break;
		}
		count--;
		goline(count);
		wrefresh(schwin);
		break;

	case 'g':
		/*
		 * Genarate blank line
		 */
		if (! writesw) {
			error("File is read only");
			break;
		}
		genlines();
		break;

	case '\'':
		/*
		 * Goto marked line
		 */
		incnt = getch();
		if (incnt == '0') {
			/*
			 * Search today
			 */
			sprintf(msgbuf, "Search Today, %d/%d",
				 inttime->tm_mon + 1, inttime->tm_mday);
			mesg(0, msgbuf);
			godate(msgbuf + 14);
		} else if (incnt >= '1' && incnt <= '9') {
			incnt -= '0';
			if (!validmark(mark[incnt])) {
				sprintf(msgbuf, 
					 "No mark in register %d", incnt);
				mesg(0, msgbuf);
			} else {
				goline(mark[incnt]);
				mesg(-1, "");
			}
		} else if (incnt != ESCAPE && incnt != SPACE)
			error("Register must be `1' to `9'");
		break;

#if 0
	case CMD_WRITE:
		(void)dowrite("w", 0);
		break;
#endif

	case 'Q':
		(void)doquit("Q", 0);
		break;

	case 'Z':
		/*
		 * Write back and exit
		 */
		if (getch() == inchar)
			(void)dowq("wq", 0);
		break;

#if 0
	case CMD_SHELL:
		/*
		 * SHELL command invoke
		 */
		mesg(0, "Shell Command ? ");
		keyin = wgets(errwin, 0, 16);
		if (*keyin)
			execom(keyin);
		else
			mesg(-1, "");
		break;
#endif

	case '/':
	case '?':
		/*
		 * Search forward/backward
		 */
		sprintf(msgbuf, "%c", inchar);
		mesg(0, msgbuf);
		keyin = wgets(errwin, 0, 1);
		if (*keyin) {
#ifdef HAVE_REGEX_H
			if (psearch) {
				free(psearch);
				psearch = NULL;
			}
			psearch = (regex_t *)malloc(sizeof(regex_t));
			if (!psearch)
				error("malloc");
			if (regcomp(psearch, keyin, 0) < 0)
#else
			if (!(psearch = regcomp(keyin) < 0))
#endif
				error("regcomp");
			else
				search(inchar == '/' ? SRCH_FOR : SRCH_BACK);
		} else
			mesg(-1, "");
		break;

	case 'n':
	case 'N':
		/*
		 * Search forward/backward same string
		 */
		if (psearch) {
			mesg(0, inchar == 'n' ? "/" : "?");
			search(inchar == 'n' ? SRCH_FOR : SRCH_BACK);
		} else
			error("No previous regular expression");
		break;

	case '#':
		/*
		 * Search today
		 */
		sprintf(msgbuf, "Search Today, %d/%d",
			 inttime->tm_mon + 1, inttime->tm_mday);
		mesg(0, msgbuf);
		godate(msgbuf + 14);
		break;

	case ':':
		(void)docolon();
		if (!msgsw)
			mesg(-1, "");
		break;

	case 'D':
		/*
		 * Field erase
		 */
		if (! writesw)
			error("File is read only");
		else if ((cnt = ismsg()) >= FLD_WORK)
			delfld(cnt);
		else
			error("Not message field");
		break;

	case 'd':
		/*
		 * Delete line
		 */
		if (!writesw) {
			error("File is read only");
			break;
		} else if (maxsch <= 1) { 
			error("No data in buffer");
			break;
		}
		if (verbose)
			mesg(0, "Delete; tap again to confirm");
		if (getch() == inchar)
			delline(yankbuf);
		else
			mesg(-1, "");
		break;

	case 'O':
	case 'o':
		/*
		 * Open above/below
		 */
		if (!writesw)
			error("File is read only");
		else if (maxsch < MAXDATA) {
			if (inchar == 'O')
				openline(cursch, 1);
			else
				openline(cursch + 1, -1);
			mesg(-1, "");
		} else
			error("No free space in buffer");
		break;

	case CMD_REPLACE:
		/*
		 * Change message 
		 */
		if (! writesw)
			error("File is read only");
		else if ((cnt = ismsg()) == FLD_DAY) {
			/*
			 * Change date
			 */
			mesg(0, "New Date [DD, MM/DD or MM/DD/YYYY] ? ");
			keyin = wgets(errwin, 0, 37);
			if (*keyin)
				chndate(keyin);
			else
				mesg(-1, "");
		} else if (cnt > FLD_DAY) {
			/*
			 * Change message 
			 */
			mesg(0, UPD_HELP);
			wstandout(errwin);
			mesg(COLS - 12, " [replace] ");
			wstandend(errwin);
			chnfld(cnt);
			mesg(-1, "");
		} else {
			mesg(0, BAR_HELP);
			getbar();
			mesg(-1, "");
		}
		break;

	case CMD_INSERT:
	case CMD_APPEND:
		/*
		 * Insert/Append message, almost same to Change message
		 */
		if (! writesw)
			error("File is read only");
		else if ((cnt = ismsg()) == FLD_DAY) {
			/*
			 * Change date
			 */
			mesg(0, "New Date [DD, MM/DD or MM/DD/YYYY] ? ");
			keyin = wgets(errwin, 0, 37);
			if (*keyin)
				chndate(keyin);
			else
				mesg(-1, "");
		}
		else if (cnt > FLD_DAY) {
			/*
			 * Change message 
			 */
			mesg(0, UPD_HELP);
			wstandout(errwin);
			mesg(COLS - 12, " [insert] ");
			wstandend(errwin);
			ungetc(UPD_INSERT, stdin);
			chnfld(cnt);
			mesg(-1, "");
		} else {
			mesg(0, BAR_HELP);
			getbar();
			mesg(-1, "");
		}
		break;

	case CMD_ALLREP:
		/*
		 * Reset all data in line
		 */
		if (! writesw)
			error("File is read only");
		else {
			if (sch[cursch]->sthour || sch[cursch]->stmin ||
			    sch[cursch]->enhour || sch[cursch]->enmin ||
			    isalnum(sch[cursch]->work[0])           ||
			    isalnum(sch[cursch]->prj[0])            ||
			    isalnum(sch[cursch]->msg[0])) {
				mesg(0, "Clear Really ? ");
				keyin = wgets(errwin, 0, 15);
				if (*keyin == 'Y' || *keyin == 'y')
					chnline();
				else
					mesg(-1, "");
			} else
				chnline();
		}
		break;

	case '@':
		/*
		 * Help
		 */
		if (!calsw) {
			error("Help is supported in Calendar Mode");
			break;
		}
		getyx(schwin, cury, curx);
		help(HLP_PNEXT);
		wmove(schwin, cury, curx);
		wrefresh(calwin);
		wrefresh(schwin);
		mesg(-1, "");
		break;

	case 'S':
		/*
		 * Send bug to author
		 */
		sendbug();
		break;

	case EOF:
		/*
		 * High value on "cbreak" mode,  should be ignore.
		 */
		break;

	case 'Z' & 0x1f:
		restart(0);
		break;

	case '[' & 0x1f:
		break;

	default:
		if (verbose) {
			if (isprint(inchar) && !isspace(inchar))
				sprintf(msgbuf, "%c", inchar);
			else if (isalpha(inchar + 'a' - 1))
				sprintf(msgbuf, "^%c", inchar + 'a' - 1);
			else
				sprintf(msgbuf, "\\%03o", inchar);
			strcat(msgbuf, " isn't a sch command");
			error(msgbuf);
		}
		errbell(0);
	}
	count = 0;
	goto LOOP;
}
