/*
 * Copyright (C) 1983-1989  Masatoshi Kurihara <kurihara@sra.co.jp>
 * Copyright (C) 1999, 2000 and 2001
 * Jun-ichiro itojun Hagino <itojun@iijlab.net>
 * All rights reserved.
 *
 * Note well: this is not a normal 3-clause BSD copyright;
 * commercial use of the software is restricted.
 *
 * Redistribution and non-commercial use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* Copyright (C) 1985 - 1989 by Software Research Associates, Inc. */

#include <curses.h>
#include <common.h>
#ifdef MULTIBYTE
# include <wchar.h>
#endif

#include <common.h>
#include <opts.h>
#include <entry.h>
#include <def.h>
#include <var.h>
#include <sch.h>

enum vartype {
	v_bool, v_int, v_str
};

struct vartab {
	char *name;
	enum vartype type;
	void *var;
	int (*action) __P((struct vartab *, int, const char *));
	int update;
};

static int action_prohibit __P((struct vartab *, int, const char *));
static int action_thru __P((struct vartab *, int, const char *));
static int action_repaint __P((struct vartab *, int, const char *));
static int action_repaint_day __P((struct vartab *, int, const char *));
static int action_curzone __P((struct vartab *, int, const char *));
#ifdef MULTIBYTE
static int action_localelang __P((struct vartab *, int, const char *));
#endif

static struct vartab vartab[] = {
	{ "autowrite",	v_bool,	&autowrite, action_thru, 0 },
	{ "calendar",	v_bool,	&calsw, action_repaint, 0 },
	{ "curzone",	v_int,	&curzone, action_curzone, 0 },
	{ "dayend",	v_int,	&dayend, action_repaint_day, 0 },
	{ "daystart",	v_int,	&daystart, action_repaint_day, 0 },
	{ "homezone",	v_int,	&homezone, action_prohibit, 0 },
	{ "hourwidth",	v_int,	&hourwidth, action_repaint, 0 },
#ifdef MULTIBYTE
	{ "localelang",	v_str,	localelang, action_localelang, 0 },
#endif
	{ "oldformat",	v_bool,	&oldformat, action_thru, 0 },
	{ "tabhour",	v_int,	&tabhour, action_thru, 0 },
	{ "verbose",	v_bool,	&verbose, action_thru, 0 },
	{ NULL, NULL, NULL },
};

static int repaint;

static int
action_prohibit(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
	error("Not modifyable");
	return -1;
}

static int
action_thru(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
	*(int *)v->var = x;
	return 0;
}

static int
action_repaint(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
	*(int *)v->var = x;
	repaint++;
	return 0;
}

static int
action_repaint_day(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
	int ret;

	if (x < 0) {
		error("Invalid value specified");
		return -1;
	}

	ret = 0;
	*(int *)v->var = x;
	if (daystart > dayend) {
		int t;
		t = daystart;
		daystart = dayend;
		dayend = t;
	} else if (daystart == dayend) {
		daystart = 9;
		dayend = 21;
		ret = -1;
	}

	repaint++;
	if (ret < 0)
		error("Invalid value specified; setting default");
	return ret;
}

static int
action_curzone(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
#if defined(HAVE_OFFTIME) && defined(HAVE_TIMEOFF)
	int i;

	for (i = 0; i < maxsch; i++)
		sch_tznorm(sch[i], x, curzone);
	*(int *)v->var = x;
	repaint++;
	return 0;
#else
	error("Timezone offset not supported");
	return -1;
#endif
}

#ifdef MULTIBYTE
static int
action_localelang(v, x, s)
	struct vartab *v;
	int x;
	const char *s;
{
	if (localeset) {
		error("locale already set");
		return -1;
	}
	strcpy(localelang, s);
	setrunelocale(localelang);
	localeset++;
}
#endif

int
saveopts(fp, all)
	FILE *fp;
	int all;
{
	struct vartab *v;

	for (v = vartab; v->name; v++) {
		if (!all && !v->update)
			continue;
		fprintf(fp, ":set ");
		switch (v->type) {
		case v_bool:
			fprintf(fp, "%s%s", *(int *)v->var ? "" : "no",
				v->name);
			break;
		case v_int:
			fprintf(fp, "%s=%d", v->name, *(int *)v->var);
			break;
		case v_str:
			fprintf(fp, "%s=\"%s\"", v->name, (char *)v->var);
			break;
		}
		fprintf(fp, "\n");
	}
	return 0;
}

int
dosetit(cmd, bang)
	const char *cmd;
	int bang;
{
	const char *q;
	struct vartab *v;
	int value;
	const char *svalue = NULL;
	char *sbuf = NULL;

	for (q = cmd; *q && *q != '='; q++)
		;
	for (v = vartab; v->name; v++) {
		switch (v->type) {
		case v_bool:
			svalue = NULL;
			if (strcmp(v->name, cmd) == 0) {
				value = 1;
				break;
			}
			if (strncmp("no", cmd, 2) == 0
			 && strcmp(v->name, cmd + 2) == 0) {
				value = 0;
				break;
			}
			continue;
		case v_int:
			if (strncmp(v->name, cmd, q - cmd) != 0) 
				continue;
			if (*q != '=') {
				error("Invalid option string");
				return -1;
			}
			svalue = q + 1;
			value = atoi(svalue);
			break;
		case v_str:
			if (strncmp(v->name, cmd, q - cmd) != 0) 
				continue;
			if (*q != '=' || strlen(q + 1) > OPT_STRLEN) {
				error("Invalid option string");
				return -1;
			}
			svalue = q + 1;
			if (*svalue == '"' && svalue[strlen(svalue) - 1] == '"') {
				sbuf = strdup(svalue + 1);
				sbuf[strlen(sbuf) - 1] = '\0';
				svalue = sbuf;
			}
			value = atoi(svalue);
			break;
		}
		if (v->action == NULL) {
			error("Attempt to update constant");
			if (sbuf)
				free(sbuf);
			return -1;
		}
		if ((*v->action)(v, value, svalue) < 0) {
			if (sbuf)
				free(sbuf);
			return -2;
		}
		v->update++;
		if (sbuf)
			free(sbuf);
		return 0;
	}
	return -1;
}

int
doset(cmd, bang, repaintp)
	const char *cmd;
	int bang;
	int *repaintp;
{
	const char *p;
	struct vartab *v;
	int w;
	int ret;

	for (p = cmd; *p && !isspace(*p); p++)
		;
	for (/*nothing*/; *p && isspace(*p); p++)
		;

	if (*p == '\0' || strcmp(p, "all") == 0) {
		/* show all variables */
#ifdef SIGTSTP
		signal(SIGTSTP, SIG_IGN);
#endif
		signal(SIGINT,  SIG_IGN);
		signal(SIGQUIT, SIG_IGN);

		wmove(errwin, 0, 0);
		wrefresh(errwin);
#if 0
		fputc('\n', stdout);
		fflush(stdout);
#endif

		for (v = vartab; v->name; v++) {
			if (v->var == NULL)
				continue;
			switch (v->type) {
			case v_bool:
				w = printf("%s%s", *(int *)v->var ? "" : "no",
					v->name);
				break;
			case v_int:
				w = printf("%s=%d", v->name, *(int *)v->var);
				break;
			case v_str:
				w = printf("%s=\"%s\"", v->name, (char *)v->var);
				break;
			}
			w %= 16;
			printf("%*s", 16 - w, "");
		}

		fputs("\n", stdout);
#if 0
		standout();
		fputs("[Hit return to continue]", stdout);
		standend();
#endif
		fflush(stdout);
#if 0
		getch();
		refwin();
#endif
#ifdef SIGTSTP
		signal(SIGTSTP, restart);
#endif
#ifdef SIGWINCH
		signal(SIGWINCH, resize);
#endif
		signal(SIGINT,  errbell);
		signal(SIGQUIT, errbell);

		return 1;
	}

	repaint = 0;

	ret = dosetit(p, bang);

	if (repaintp)
		*repaintp = repaint;

	switch (ret) {
	case -1:
		mesg(0, "Undefined variable");
		return -1;
	case -2:
		return -1;
	default:
		return 0;
	}
}
