/*
 * Copyright (C) 1983-1989  Masatoshi Kurihara <kurihara@sra.co.jp>
 * Copyright (C) 1999, 2000 and 2001
 * Jun-ichiro itojun Hagino <itojun@iijlab.net>
 * All rights reserved.
 *
 * Note well: this is not a normal 3-clause BSD copyright;
 * commercial use of the software is restricted.
 *
 * Redistribution and non-commercial use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <curses.h>
#include <common.h>
#ifdef MULTIBYTE
# include <wchar.h>
#endif
#include <err.h>

#include <entry.h>
#include <var.h>

/* create new schedule entry */
struct sch *
sch_new()
{
	struct sch *s;
	time_t ltime;
	struct tm *inttime;

	(void)time(&ltime);
	inttime = localtime(&ltime);

	s = calloc(sizeof(*s), 1);
	if (!s) {
		err(1, "sch_new");
		/*NOTREACHED*/
	}

	s->year = inttime->tm_year + 1900;
	s->month = inttime->tm_mon + 1;
	s->day = inttime->tm_mday;
	s->sthour = 0;
	s->stmin = 0;
	s->enhour = 0;
	s->enmin = 0;
	strcpy(s->work, "");
	strcpy(s->prj , "");
	strcpy(s->msg, "");

	return s;
}

void
sch_free(s)
	struct sch *s;
{
	free(s);
}

int
sch_wday(s, zone)
	struct sch *s;
	int zone;
{
	struct tm tm1, *tm2;
	time_t t;

	memset(&tm1, 0, sizeof(tm1));
	tm1.tm_year = s->year - 1900;
	tm1.tm_mon = s->month - 1;
	tm1.tm_mday = s->day;
#if defined(HAVE_OFFTIME) && defined(HAVE_TIMEOFF)
	t = timeoff(&tm1, zone * -3600);

	tm2 = offtime(&t, zone * -3600);
#else
	t = mktime(&tm1);

	tm2 = localtime(&t);
#endif
	return tm2->tm_wday;
}

/* parse schedule entry in old format */
struct sch *
sch_oparse(line)
	char *line;
{
	struct sch *s;
	char msg[LEN_MESG];
	int n;
	char *p;
	time_t ltime;
	struct tm *inttime;

	(void)time(&ltime);
	inttime = localtime(&ltime);

	s = calloc(sizeof(*s), 1);
	if (!s) {
		err(1, "sch_oparse");
		/*NOTREACHED*/
	}

	n = sscanf(line, "%2d/%2d: %2d:%2d-%2d:%2d",
		&s->month, &s->day, &s->sthour, &s->stmin,
		&s->enhour, &s->enmin);
	if (n != 6) {
		free(s);
		return NULL;
	}
	p = line;
	while (!isspace(*p))
		p++;
	p++;
	while (!isspace(*p))
		p++;
	p++;
	strncpy(s->work, p, 11);
	p += 12;
	strncpy(s->prj, p, 3);
	p += 4;
	strcpy(msg, p);

	s->year = inttime->tm_year + 1900;

#ifndef MULTIBYTE
	strcpy(s->msg , msg);
#else
	memset(s->msg, 0, sizeof(s->msg));
	mbstowcs(s->msg, msg, sizeof(s->msg) / sizeof(s->msg[0]));
#endif

	return s;
}

/* print schedule entry in old format */
void
sch_oprint(fp, s)
	FILE *fp;
	struct sch *s;
{
#ifdef MULTIBYTE
	char msg[LEN_MESG * 5];	/*XXX*/
	mbstate_t state;
#endif

#ifdef MULTIBYTE
	wcsrtombs(NULL, "", 0, &state);
	msg[0] = '\0';
	wcsrtombs(msg, s->msg, sizeof(msg), &state);
	wcsrtombs(msg + strlen(msg), "", sizeof(msg) - strlen(msg), &state);
#endif
	fprintf(fp, OPUT_FORM, 
		 s->month, s->day, s->sthour, s->stmin, 
		 s->enhour, s->enmin, s->work, s->prj,
#ifndef MULTIBYTE
		 s->msg
#else
		 msg
#endif
		 );
}

/* parse schedule entry in new format */
struct sch *
sch_parse(line)
	char *line;
{
	struct sch *s;
	char msg[LEN_MESG];
	int n;
	char *p;

	s = calloc(sizeof(*s), 1);
	if (!s) {
		err(1, "sch_oparse");
		/*NOTREACHED*/
	}

	n = sscanf(line, "%4d/%2d/%2d: %2d:%2d-%2d:%2d",
		&s->year, &s->month, &s->day, &s->sthour, &s->stmin,
		&s->enhour, &s->enmin);
	if (n != 7) {
		free(s);
		return NULL;
	}
	p = line;
	while (!isspace(*p))
		p++;
	p++;
	while (!isspace(*p))
		p++;
	p++;
	strncpy(s->work, p, 11);
	p += 12;
	strncpy(s->prj, p, 3);
	p += 4;
	strcpy(msg, p);

#ifndef MULTIBYTE
	strcpy(s->msg , msg);
#else
	memset(s->msg, 0, sizeof(s->msg));
	mbstowcs(s->msg, msg, sizeof(s->msg) / sizeof(s->msg[0]));
#endif

	return s;
}

/* print schedule entry in new format */
void
sch_print(fp, s)
	FILE *fp;
	struct sch *s;
{
#ifdef MULTIBYTE
	char msg[LEN_MESG * 5];	/*XXX*/
	mbstate_t state;
#endif

#ifdef MULTIBYTE
	wcsrtombs(NULL, "", 0, &state);
	msg[0] = '\0';
	wcsrtombs(msg, s->msg, sizeof(msg), &state);
	wcsrtombs(msg + strlen(msg), "", sizeof(msg) - strlen(msg), &state);
#endif
	fprintf(fp, PUT_FORM, s->year,
		 s->month, s->day, s->sthour, s->stmin, 
		 s->enhour, s->enmin, s->work, s->prj,
#ifndef MULTIBYTE
		 s->msg
#else
		 msg
#endif
		 );
}

#if defined(HAVE_OFFTIME) && defined(HAVE_TIMEOFF)
void
sch_tznorm(s, to, from)
	struct sch *s;
	int to;
	int from;
{
	struct tm tm1, *tm2;
	time_t t;

	if (s->sthour == 0 && s->enhour == 0)
		return;

	memset(&tm1, 0, sizeof(tm1));
	tm1.tm_year = s->year - 1900;
	tm1.tm_mon = s->month - 1;
	tm1.tm_mday = s->day;
	tm1.tm_hour = s->sthour;
	tm1.tm_gmtoff = from * -3600;
	t = timeoff(&tm1, from * -3600);

	tm2 = offtime(&t, to * -3600);
	s->year = tm2->tm_year + 1900;
	s->month = tm2->tm_mon + 1;
	s->day = tm2->tm_mday;
	s->enhour = tm2->tm_hour + (s->enhour - s->sthour);
	s->sthour = tm2->tm_hour;
}
#endif

void
sch_dayoffset(s, off)
	struct sch *s;
	int off;
{
	struct tm tm1, *tm2;
	time_t t;

	memset(&tm1, 0, sizeof(tm1));
	tm1.tm_year = s->year - 1900;
	tm1.tm_mon = s->month - 1;
	tm1.tm_mday = s->day;
	tm1.tm_hour = 12;	/* avoid possible leap second at 0AM */
	t = mktime(&tm1);

	t += ((long)off) * 60 * 60 * 24; 

	tm2 = gmtime(&t);
	s->year = tm2->tm_year + 1900;
	s->month = tm2->tm_mon + 1;
	s->day = tm2->tm_mday;
}
