/* bin2.c */
/* binary file to C
   Copyright (C) 1999 Darren Salt
   All Rights Reserved

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/* Assumptions:
     ANSI C compiler, or GCC
     ASCII or a superset of ASCII
     Type sizes (octets):
       char = 1, short = 2, int = long = 4, double = 8, long long = 8
     Absolutely *NOT* EBCDIC.  */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <setjmp.h>
#include <ctype.h>
#include <locale.h>

#ifndef __riscos
#include <sys/stat.h>
#endif

#include "mdwopt.h"
#include "error.h"

#include "bin2c.h"

/*************
 * Variables *
 *************/

/* Output workspace buffer size.  */
#ifndef BIN2C_BUFFER_SIZE
#define BIN2C_BUFFER_SIZE 1024
#endif

#ifndef OUTPUT_WIDTH
#define OUTPUT_WIDTH 64
#endif

static const char version[] = "bin2c version 0.1";

#ifndef xmalloc
extern void *xmalloc (size_t);
extern void *xrealloc (void *, size_t);
#endif

/* Get type information.  */
static void read_type_info (const char *);

/* Validate a variable name; 0 if valid. */
static int validate_name (const char *name, const char *desc);

/* Binary to C... */
static void process (void);


/* Type processing procedures.  */
static int do_uchar (void);
static int do_schar (void);
static int do_ushort (void);
static int do_sshort (void);
static int do_ulong (void);
static int do_slong (void);
static int do_ulonglong (void);
static int do_slonglong (void);
static int do_double (void);
static int do_string (void);


typedef int (*process_fn) (void);

typedef struct
  {
    const char *name;		/* main part of type name (spaces allowed) */
    const char *extra;		/* optional part of type name (no spaces) */
    process_fn fn[4];		/* type: plain, unsigned, signed, pointer */
    const char suffix[4][4];	/* type suffix */
  }
typename_t;


/* Type names.  */
static const typename_t typenames[] =
{
  {"long long", "int", {do_longlong, do_ulonglong, do_slonglong, 0}, {"ll", "ull", "ll", ""}},
  {"long", "int", {do_long, do_ulong, do_slong, 0}, {"l", "ul", "l", ""}},
  {"short", "int", {do_short, do_ushort, do_sshort, 0}, {"", "u", "", ""}},
  {"char", 0, {do_char, do_uchar, do_schar, do_string}, {"", "u", "", ""}},
  {"double", 0, {do_double, 0, 0, 0}, {"", "", "", ""}},
  {0, "int", {do_long, do_ulong, do_slong, 0}, {"", "", "", ""}},
  {0}
};


/* Output control information.  Default is int.  */
static struct
  {
    const char *name;		/* type name */
    char suffix[4];		/* type suffix, eg. "ul" */
    process_fn fn;
    char is_const;
    char is_static;
    char sign;			/* 0 none, 1 unsigned, 2 signed, 3 pointer */
    char ptr_is_const;
  }
c_type =
{
  "int", "", do_long, 0, 0, 0, 0
};

/* The output array name.  */
static const char *array_name = "foo";

/* The output array length name, and the array length.  */
static const char *array_length_name;
static int array_length;

/* Array specifiers.
   Note that the output always has one leading unspecified dimension.
 */
static int dimensions;
static unsigned int *dimension_list;

/* Output base.  */
static enum
  {
    DECIMAL,
    OCTAL,
    HEXADECIMAL
  }
output_base;

/* Input endianness.  */
#ifdef BIG_ENDIAN
static int big_endian = 1;
#else
static int big_endian;
#endif

/* For char output, use character constants for printable characters.  */
static int use_char_consts;

/* String input separator: '\0' (0) or '\n' (1).  */
static int separator_is_newline;

/* Input & output filenames, and FILE blocks.  */
static const char *infilename;
static const char *outfilename;
static FILE *infile = stdin;
static FILE *outfile = stdout;
static long infilesize;

/* The name used to run this program.  */
char *program_name;


static int show_help;
static int show_version;

/***********
 * Options *
 ***********/

static const char short_opts[] = "t:n:doheEZNpcl::";
static struct option long_opts[] =
{
  {/*t*/"type", required_argument, NULL, 't'},
  {/*n*/"name", required_argument, NULL, 'n'},
  {/*d*/"decimal", no_argument, (int *) &output_base, DECIMAL},
  {/*o*/"octal", no_argument, (int *) &output_base, OCTAL},
  {/*h*/"hex", no_argument, (int *) &output_base, HEXADECIMAL},
  {/*h*/"hexadecimal", no_argument, (int *) &output_base, HEXADECIMAL},
  {/*e*/"little-endian", no_argument, &big_endian, 0},
  {/*E*/"big-endian", no_argument, &big_endian, 1},
  {/*Z*/"nul-ending", no_argument, &separator_is_newline, 0},
  {/*N*/"nl-ending", no_argument, &separator_is_newline, 1},
  {/*c*/"char-consts", no_argument, &use_char_consts, 1},
  {/*l*/"length-name", optional_argument, NULL, 'l'},
  {"help", no_argument, &show_help, 1},
  {"version", no_argument, &show_version, 1},
  {0, 0, 0, 0}
};

/********************
 * Usage, help text *
 *******************/


static void
usage (int status)
{
  fprintf (stderr, "Try `%s --help' for more information.\n", program_name);
  exit (status);
}


static void
help (void)
{
  printf ("\
Usage: %s [OPTION]... [INFILE [OUTFILE]]\n\
\n\
  -t, --type=TYPE           output array type (see documentation)\n\
  -n, --name=NAME           output array name\n\
  -l, --length-name=NAME    output array length name\n\
\n\
Integral type options:\n\
  -d, --decimal             print numbers in decimal (default)\n\
  -o, --octal               print numbers in octal\n\
  -h, --hex, --hexadecimal  print numbers in hexadecimal\n"
#ifdef BIG_ENDIAN
	  "\
  -e, --little-endian       interpret input as little endian\n\
  -E, --big-endian          interpret input as big endian (default)\n"
#else
	  "\
  -e, --little-endian       interpret input as little endian (default)\n\
  -E, --big-endian          interpret input as big endian\n"
#endif
"\
  -c, --char-consts         use character constants for type char\n\
\n\
String type options:\n\
  -Z, --nul-ending          use NUL \\0 as string terminator (default)\n\
  -N, --nl-ending           use NL \\n as string terminator\n\
\n\
      --help                display this help\n\
      --version             output version information\n\
\n\
Input cannot be read from standard input (unless it's a file).\n\
With no output file, write to standard output.\n\
\n\
Report bugs to ds@youmustbejoking.demon.co.uk\n",
	  program_name);
  exit (0);
}

/****************
 * Main program *
 ****************/

int
main (int argc, char *argv[])
{
  int c;
  struct stat st;

  {
#ifdef __riscos
    char *dot = strrchr (argv[0], '.');
    char *colon = strrchr (argv[0], ':');
    if (dot < colon)
      dot = colon;
    program_name = dot ? dot+1 : argv[0];
#else
    char *slash = strrchr (argv[0], '/');
    program_name = slash ? slash+1 : argv[0];
#endif
  }
  /* Choose an appropriate character set, such as ISO8859-1.  */
  setlocale (LC_CTYPE, "");

  /* Read the options.  */
/*while ((c = getopt_long (argc, argv, short_opts, long_opts, NULL)) != EOF)*/
  while ((c = mdwopt (argc, argv, short_opts, long_opts, NULL, 0, 16)) != EOF)
    switch (c)
      {
      default:
	usage (1);
      case 0:
	break;
      case 'c':
	use_char_consts = 1;
	break;
      case 'd':
	output_base = DECIMAL;
	break;
      case 'e':
	big_endian = 0;
	break;
      case 'E':
	big_endian = 1;
	break;
      case 'h':
	output_base = HEXADECIMAL;
	break;
      case 'l':
	if (optarg && *optarg)
	  {
	    if (validate_name (optarg, "array length"))
	      error (1, 0, "invalid array length variable name '%s'", optarg);
	    array_length_name = optarg;
	  }
	else
	  array_length_name = "";
	break;
      case 'n':
	if (validate_name (optarg, "array"))
	  error (1, 0, "invalid array variable name '%s'", optarg);
	array_name = optarg;
	break;
      case 'N':
	separator_is_newline = 1;
	break;
      case 'o':
	output_base = OCTAL;
	break;
      case 't':
	read_type_info (optarg);
	break;
      case 'Z':
	separator_is_newline = 0;
	break;
      }
  if (show_version)
    {
      printf ("%s\n", version);
      exit (0);
    }
  if (show_help)
    help ();

  if (array_length_name && !*array_length_name)
    {
      array_length_name = xmalloc (strlen (array_name) + 9);
      sprintf ((char *)array_length_name, "%s_length", array_name);
    }

  argv += optind;
  argc -= optind;
  if (argc > 2)
    {
      error (0, 0, "too many filenames");
      usage (1);
    }

  if (array_length_name && !strcmp (array_name, array_length_name))
    error (1, 0,
	   "the array and array length variable names must be different");

  /* Open the input file if given, else use stdin if it's a file.  */
  if (!argc || !strcmp (argv[0], "-"))
    {
      /* stdin - if it's a file, we're OK :-)  */
      if (fstat (0, &st) < 0)
	error (2, errno, "couldn't stat standard input", infilename);
      if (!S_ISREG (st.st_mode) /* or pipe? I don't know */ )
	{
	  error (0, 0, "standard input is not a file");
	  usage (1);
	}
    }
  else if (argc)
    {
      /* File - find its size and try to open it.  */
      infilename = argv[0];
      if (stat (infilename, &st) < 0)
	error (2, errno, "couldn't stat '%s'", infilename);
      if (!S_ISREG (st.st_mode))
	error (2, errno, "'%s' is not a regular file", infilename);
      infile = fopen (infilename, "rb");
      if (!infile)
	error (2, errno, "couldn't open input file '%s'", infilename);
    }
  /* We need the input file size...  */
  infilesize = st.st_size;

  /* Open the output file, or use stdout.  */
  if (argc == 2 && strcmp (argv[1], "-"))
    {
      outfilename = argv[1];
      outfile = fopen (outfilename, "w");
      if (!outfile)
	error (2, errno, "couldn't open output file '%s'", outfilename);
    }

  process ();

  if (outfile != stdout && fclose (outfile))
    error (2, errno, "output file '%s' close error", outfilename);
  /* Allow the input file to be closed automatically.  */

  return 0;
}

/**********************************
 * Command line parsing functions *
 **********************************/


/* Validate a variable name; cause an error on failure.
   ## NOTE: ASCII dependent.  */
static int validate_name (const char *name, const char *desc)
{
  const char *p = name;
  char c;
  c = toupper (*p++);
  if (c < 'A' || c > 'Z')
    return 1;
  while ((c = toupper(*p++)) != 0)
    if (c < '0' || (c > '9' && c < 'A') || (c > 'Z' && c != '_'))
      return 1;
  return 0;
}


/* Skip white space.  */
#define SKIP(p) { while (isspace (*(p))) (p)++; }
void (SKIP)(const char *);

/* This is set if we've already parsed a keyword.  */
static int require_space;


/* Compare a keyword string to the input. */
static int
keyword (const char **const ptr,
	 const char *const word, const char *const extra)
{
  int len;
  /* Local copies...  */
  const char *p = *ptr;
  const char *w = word;
  int rq = require_space;
  /* 'word' is required, eg. "long" in "long int".  */
  if (w)
    for (;;)
      {
	char *spc = strchr (w, ' ');
	len = spc ? spc - w : strlen (w);
	if (rq)
	  {
	    if (!isspace (*p))
	      return 0;
	    SKIP (p);
	    rq = 0;
	  }
	if (memcmp (p, w, len))
	  /* Unmatched.  */
	  return 0;
	p += len;
	rq = 1;
	if (!spc)
	  {
	    /* We have a match; export our local copies then check extra.  */
	    *ptr = p;
	    require_space = rq;
	    break;
	  }
	w = spc + 1;
      }
  /* 'extra' is optional, eg. "int" in "long int".  */
  if (extra)
    keyword (ptr, extra, 0);
  return 1;
}


/* Parse the type string.  */
static void
read_type_info (const char *type)
{
  const char *ptr = type;
  const typename_t *cmp = typenames;
  int started_dims_list = 0;

  /* Set some defaults.  */
  c_type.is_const = 0;
  c_type.ptr_is_const = 0;
  dimensions = 0;
  free (dimension_list);
  dimension_list = 0;

  SKIP (ptr);

  /* static?  const?  */
  if (keyword (&ptr, "static", 0))
    c_type.is_static = 1;
  if (keyword (&ptr, "const", 0))
    c_type.is_const = 1;

  /* unsigned?  signed?  */
  if (keyword (&ptr, "unsigned", 0))
    c_type.sign = 1;
  else if (keyword (&ptr, "signed", 0))
    c_type.sign = 2;
  else
    {
      char *p = strchr (ptr, '*');
      char *b = strchr (ptr, '[');
      if (p && (!b || p < b))
	c_type.sign = 3;	/* possibly a pointer */
    }

  /* Identify the type.  */
  for (;;)
    {
      if (cmp->fn[c_type.sign] && keyword (&ptr, cmp->name, cmp->extra))
	break;
      cmp++;
      if (!cmp->name && !cmp->extra)
	goto unrecognised;
    }

  if (c_type.sign == 3)
    {
      SKIP (ptr);
      if (*ptr != '*')
	goto unrecognised;
      ptr++;
    }

  c_type.name = cmp->name ? cmp->name : cmp->extra;
  memcpy (c_type.suffix, cmp->suffix[c_type.sign], 4);
  c_type.fn = cmp->fn[c_type.sign];

  SKIP (ptr);

  /* Const pointer?  */
  if (c_type.sign == 3 && !memcmp (ptr, "const", 5))
    {
      ptr += 5;
      c_type.ptr_is_const = 1;
    }

  /* Get dimensions.  */
  for (;;)
    {
      unsigned int index;

      SKIP (ptr);
      switch (*ptr)
	{
	case '\0':
	  return;
	case '[':
	  /* We have a dimension...  */
	  started_dims_list = 1;
	  if (c_type.sign == 3)
	    error (1, 0,
		   "sorry, can't have array dimensions list with a pointer");
	  ptr++;
	  SKIP (ptr);
	  /* Get the number of elements.  */
	  index = (unsigned int) strtoul (ptr, (char **) &ptr, 10);
	  if (errno)
	    goto syntax;
	  dimensions++;
	  switch (index)
	    {
	    case 0:
	      error (1, 0, "illegal dimension [0] in '%s'", type);
	    case 1:
	      error (0, 0, "warning: odd dimension [1] (depth %i)",
		     dimensions);
	    }
	  dimension_list = xrealloc (dimension_list, dimensions * sizeof (int));
	  dimension_list[dimensions - 1] = index;
	  SKIP (ptr);
	  /* Check for correct termination.  */
	  if (*ptr++ == ']')
	    break;
	  goto syntax;
	default:
	  if (started_dims_list)
	    goto syntax;
	  else
	    goto unrecognised;
	}
    }

  /* The following are only reached on error.  */

syntax:
  error (1, 0, "syntax error in dimensions list in '%s'", type);

unrecognised:
  error (1, 0, "unrecognised type name '%s'", type);
}

/********************
 * Output functions *
 ********************/


/* Output workspace buffer; four extra characters for overflow.  */
static char buffer[BIN2C_BUFFER_SIZE + 4];
static jmp_buf jmp;
static int width;
static int file_is_short;


/* Indent appropriately for the current nesting.
   Use tabs if possible; assumed to be 8 characters wide.  */
static void
indent (int d)
{
  int i = d + 2;
  width = i;
  while (i >= 8)
    {
      if (fputc ('\t', outfile) == EOF)
	longjmp (jmp, 1);
      i -= 8;
    }
  while (i)
    {
      if (fputc (' ', outfile) == EOF)
	longjmp (jmp, 1);
      i--;
    }
}


/* Output lowest-level elements.  */
static void
process_file (unsigned int elements)
{
  unsigned int e = elements;

  do
    {
      buffer[0] = 0;
      /* Get the string to output.
	 The function is expected to return the number of characters which it
	 has written to outfile; any unwritten characters must be in a
	 NUL-terminated string in |buffer|.  */
      width += c_type.fn ();
      if (!dimensions)
        array_length ++;
      /* Write out the buffer contents.  */
      if (ferror (outfile) || fputs (buffer, outfile) == EOF)
	longjmp (jmp, 1);
      /* Append a suffix, if there is one.  */
      if (c_type.suffix)
	strcat (buffer, c_type.suffix);
      /* No more to do?  */
      if (!infilesize)
	break;
      width += strlen (buffer);
      if (e > 1)
	{
	  /* Print a comma, and a newline if necessary.  */
	  if (width < OUTPUT_WIDTH)
	    {
	      fprintf (outfile, ", ");
	      width += 2;
	    }
	  else
	    {
	      fprintf (outfile, ",\n");
	      if (!ferror (outfile))
		indent (dimensions);
	    }
	}
      if (ferror (outfile))
	longjmp (jmp, 1);
    }
  while (--e);
}


/* Do an array element (which will have subdimensions).  */
static void
process_nested (int up, int depth)
{
  unsigned int i = dimension_list[depth];

  /* Open brace.  */
  if (!width)
    indent (depth);
  if (fputc ('{', outfile) == EOF)
    longjmp (jmp, 1);
  width++;

  if (depth == dimensions - 1)
    {
      /* One dimension left; output its content.  */
      process_file (i);
    }
  else
    {
      /* More dimensions to go; recurse.  */
      do
	{
	  process_nested (i, depth + 1);
	  if (ferror (outfile))
	    longjmp (jmp, 1);
	}
      while (infilesize && --i);
      if (!infilesize && (up > 1 || i > 1))
	/* Flag an incompletely filled element.  */
	file_is_short = 1;
    }

  /* Close brace.  */
  if (infilesize)
    {
      if (up != 1)
	width = 0;
      fprintf (outfile, "}%s", (up == 1) ? "" : ",\n");
    }
  else
    {
      if (fputc ('}', outfile) == EOF)
	longjmp (jmp, 1);
    }
  if (ferror (outfile))
    longjmp (jmp, 1);
}


/* Process the input file.  */
static void
process (void)
{
  int i;

  /* Print the type information and name.  */
  fprintf (outfile, "%s%s%s%s%s%s%s%s[]",
	   c_type.is_static ? "static " : "",
	   c_type.is_const ? "const " : "",
	   c_type.sign == 1 ? "unsigned "
	   : c_type.sign == 2 ? "unsigned " + 2 : "",
	   c_type.name,
	   c_type.sign == 3 ? " *" : "",
	   c_type.ptr_is_const ? "const" : "",
	   (c_type.sign != 3 || c_type.ptr_is_const) ? " " : "",
	   array_name);
  i = 0;
  while (i < dimensions)
    {
      fprintf (outfile, "[%i]", dimension_list[i]);
      if (ferror (outfile))
	goto output_err;
      i++;
    }
  /* Indicate that this is defining...  */
  fprintf (outfile, " =\n{\n");
  if (ferror (outfile))
    goto output_err;

  if (dimensions == 0)
    {
      /* Only one dimension of unknown size.  */
      fprintf (outfile, "  ");
      if (ferror (outfile))
	goto output_err;
      width = 2;
      process_file (-1);
      if (c_type.sign == 3)
	{
	  fprintf (outfile, ",\n 0");
	  if (ferror (outfile))
	    longjmp (jmp, 1);
	}
    }
  else
    {
      /* One dimension of unknown size and at least one of known size.  */
      if (!setjmp (jmp))
	do
	  {
	    if (fputc (' ', outfile) == EOF)
	      goto output_err;
	    width = 1;
	    process_nested (0, 0);
	    array_length ++;
	  }
	while (infilesize && !ferror (outfile));
    }

  if (!ferror (outfile))
    {
      /* Finish off.  */
      fprintf (outfile, "\n};\n");
      if (!ferror (outfile) && array_length_name)
	fprintf (outfile, "\n%sconst int %s = %i;\n",
		 c_type.is_static ? "static " : "",
		 array_length_name, array_length);
      if (ferror (outfile))
        goto output_err;
      if (file_is_short)
	error (0, 0, "warning: input file does not properly fill the array");      return;
    }

output_err:
  error (2, errno, "output error");
}


/* Read a block of bytes, reversing them if need be. */
static int
endian_read (void *buf, int size)
{
  int i, c;
  char *b = buf;
  memset (b, 0, size);

  infilesize -= size;
  if (infilesize < 0)
    infilesize = 0;

  /* If the endianness matches that of the machine,
     it's simple: just read the data in the normal order.  */
#ifdef BIG_ENDIAN
  if (big_endian)
#else
  if (!big_endian)
#endif
    {
      i = fread (b, 1, size, infile);
      if (i < size)
	file_is_short = 1;
      return i;
    }
  i = size;
  /* Read in the data, in reverse order.  */
  do
    {
      c = fgetc (infile);
    }
  while (c != EOF && ((b[--i] = c), i));
  if (i)
    file_is_short = 1;
  return size - i;
}

/* Input stream parsing procedures, and some specific output code.  */

/* Types.  */
static const char u_type[][5] =
{"%u", "0%o", "0x%X"};
static const char s_type[][5] =
{"%i", "0%o", "0x%X"};


/* For certain characters, write the corresponding escape sequence (2 chars)
   in the supplied buffer, null terminate and return non-zero.  */
static int
write_escape (char *buf, char c)
{
  char o;
  switch (c)
    {
      default:
	return 0;
      case '\0':
	o = '0';
	break;
      case '\a':
	o = 'a';
	break;
      case '\b':
	o = 'b';
	break;
      case '\f':
	o = 'f';
	break;
      case '\n':
	o = 'n';
	break;
      case '\r':
	o = 'r';
	break;
      case '\t':
	o = 't';
	break;
      case '\v':
	o = 'v';
	break;
      case '"':
      case '\\':
	o = c;
    }
  return 1;
}


/* Write out a character as an int or a character constant.  */
static void
do_char_output (const char *fmt, int c)
{
  unsigned char cc = c;
  if (use_char_consts)
    {
      if (write_escape (buffer + 1, cc))
	{
	  buffer[3] = buffer[0] = '\'';
	  buffer[4] = 0;
	}
      else if (isprint (cc))
	sprintf (buffer, "'%c'", cc);
      else
	sprintf (buffer, fmt, cc);
    }
  else
    sprintf (buffer, fmt, cc);
}


/* Process an unsigned char.  */
static int
do_uchar (void)
{
  unsigned int c = fgetc (infile);
  if (c != EOF)
    {
      infilesize--;
      do_char_output (u_type[output_base], c);
    }
  return 0;
}


/* Process a signed char.  */
static int
do_schar (void)
{
  signed int c = fgetc (infile);
  if (c != EOF)
    {
      infilesize--;
      c <<= 24;
      do_char_output (s_type[output_base], c >> 24);
    }
  return 0;
}


/* Process an unsigned short int.  */
static int
do_ushort (void)
{
  unsigned short c;
  if (endian_read (&c, 2) > 0)
    sprintf (buffer, u_type[output_base], c);
  return 0;
}


/* Process a signed short int.  */
static int
do_sshort (void)
{
  signed short c;
  if (endian_read (&c, 2) > 0)
    sprintf (buffer, s_type[output_base], c);
  return 0;
}


/* Process an unsigned long int.  */
static int
do_ulong (void)
{
  static const char type[][6] =
  {"%lu", "0%lo", "0x%lX"};
  unsigned long c;
  if (endian_read (&c, 4) > 0)
    sprintf (buffer, type[output_base], c);
  return 0;
}


/* Process a signed long int.  */
static int
do_slong (void)
{
  static const char type[][6] =
  {"%li", "0%lo", "0x%lX"};
  signed long c;
  if (endian_read (&c, 4) > 0)
    sprintf (buffer, type[output_base], c);
  return 0;
}


#if !defined(__GNUC__) || defined(ACORN_SHARED_C_LIBRARY)

#ifdef BIG_ENDIAN
/* I *hope* that this is right...!  */
#define LOW_WORD  1
#define HIGH_WORD 0
#else
#define LOW_WORD  0
#define HIGH_WORD 1
#endif


/* This routine outputs a long long, signed or unsigned, where the
   compiler and/or the library does not support the type.  */
static void
do_longlong_output (unsigned long c[], int sign)
{
  char *p = buffer;

  if (c[0] == 0 && c[1] == 0)
    /* Zero - easy.  */
    switch (output_base)
      {
      default:
	sprintf (p, "0");
	return;
      case OCTAL:
	sprintf (p, "00");
	return;
      case HEXADECIMAL:
	sprintf (p, "0x0");
	return;
      }

  /* The sign.  Only of relevance to decimal numbers.  */
  if (sign)
    *p++ = '-';

  switch (output_base)
    {
    default:
      /* Decimal.  If you can do better, feel free to replace this code.  */
      {
#ifdef HAVE_LONG_LONG
	unsigned long long div = 10000000000000000000ull;
	unsigned long long value = c[LOW_WORD]
				 | (unsigned long long) (c[HIGH_WORD]) << 32;
	unsigned long l;
	int out = 0;
	do
	  {
	    int digit = (int) (value / div);
	    value %= div;
	    if (digit || out)
	      {
		*p++ = digit + '0';
		out = 1;
	      }
	    div /= 10LL;
	  }
	while (div >= 1000000000ull);
	l = (unsigned long) value;
	/* just sprintf() the rest */
#else
	static const unsigned long div[][2] =
	{
	  {232830643ul, 2808348672ul},
	  {23283064ul, 1569325056ul},
	  {2328306ul, 1874919424ul},
	  {232830ul, 2764472320ul},
	  {23283ul, 276447232ul},
	  {2328ul, 1316134912ul},
	  {232ul, 3567587328ul},
	  {23ul, 1215752192ul},
	  {2ul, 1410065408ul},
	  {0ul, 1000000000ul}
	  /* Beyond this, we can use sprintf().  */
	};
	int i = 0, out = 0;
	unsigned long h = c[HIGH_WORD], l = c[LOW_WORD];
	while (i < sizeof (div) / sizeof (int[2])
	       && (div[i][0] > h || (div[i][0] == h && div[i][1] > l)))
	  i++;
	while (i < sizeof (div) / sizeof (int[2]))
	  {
	    char digit = 0;
	    while (h > div[i][0] || (h == div[i][0] && l >= div[i][1]))
	      {
		h -= div[i][0];
		if (l < div[i][1])
		  h --;
		l -= div[i][1];
		digit++;
	      }
	    if (digit || out)
	      {
		*p++ = digit + '0';
		out = 1;
	      }
	    i++;
	  }
	/* just sprintf() the rest */
#endif
	sprintf (p, out ? "%09i" : "%i", l);
	break;
      }
    case OCTAL:
      /* Fairly trivial; mind that middle digit!  */
      if (c[HIGH_WORD] > 1)
	sprintf (p, "0%lo%01lo%010lo",
		 c[HIGH_WORD] >> 1,
		 (c[HIGH_WORD] & 1) * 4 + (c[LOW_WORD] >> 30),
		 c[LOW_WORD] & 0x3FFFFFFF);
      else if (c[HIGH_WORD] == 1)
	sprintf (p, "0%lo%010lo",
		 4 + (c[LOW_WORD] >> 30),
		 c[LOW_WORD] & 0x3FFFFFFF);
      else
	sprintf (p, "0%lo", c[LOW_WORD]);
      break;
    case HEXADECIMAL:
      /* Trivial.  */
      if (c[HIGH_WORD])
	sprintf (p, "0x%lX%08lX", c[HIGH_WORD], c[LOW_WORD]);
      else
	sprintf (p, "0x%lX", c[LOW_WORD]);
      break;
    }
}


/* Process an unsigned long long (where we can't use long long).  */
static int
do_ulonglong (void)
{
  unsigned long c[2];
  if (endian_read (c, 8) > 0)
    do_longlong_output (c, 0);
  return 0;
}


/* Process a signed long long (where we can't use long long).  */
static int
do_slonglong (void)
{
  signed long c[2];
  if (endian_read (&c, 8) > 0)
    {
      int sign = output_base == DECIMAL && c[HIGH_WORD] < 0;
      if (sign)
	{
	  c[LOW_WORD] = -c[LOW_WORD];
	  c[HIGH_WORD] = c[LOW_WORD] ? ~c[HIGH_WORD] : -c[HIGH_WORD];
	}
      do_longlong_output ((unsigned long *) c, sign);
    }
  return 0;
}

#else /* __GNUC__ && !ACORN_SHARED_C_LIBRARY */

/* Process an unsigned long long (where we can use long long).  */
static int
do_ulonglong (void)
{
  static const char type[][7] =
  {"%llu", "0%llo", "0x%llX"};
  unsigned long long c;
  if (endian_read (&c, 8) > 0)
    sprintf (buffer, type[output_base], c);
  return 0;
}


/* Process a signed long long (where we can use long long).  */
static int
do_slonglong (void)
{
  static const char type[][7] =
  {"%lli", "0%llo", "0x%llX"};
  signed long long c;
  if (endian_read (&c, 8) > 0)
    sprintf (buffer, type[output_base], c);
  return 0;
}
#endif /* __GNUC__ && !ACORN_SHARED_C_LIBRARY */


/* Process a double.  */
static int
do_double (void)
{
  double c;
  if (endian_read (&c, 8) > 0)
    sprintf (buffer, "%0.16g", c);
  return 0;
}


/* Process a string (we're building a char* array).
   Note that this function performs its own output.  */
static int
do_string (void)
{
  int pos = 1;
  int eos = separator_is_newline ? '\n' : '\0';
  int c;
  int nul = 0;

  /* Open quotes.  */
  buffer[0] = '"';
  /* Fill the buffer until EOF or the end-of-string character is read.  */
  do
    {
      do
	{
	  c = fgetc (infile);
	  if (c != EOF)
	    infilesize--;
	  if (c == EOF || c == eos)
	    break;
	  if (iscntrl (c) || c == '\\' || c == '"')
	    {
	      nul = (c == '\0');
	      if (write_escape (buffer + pos, c))
	        pos += 2;
	      else
		{
		  sprintf (buffer + pos, "\\x%02X", c);
		  pos += 4;
		}
	    }
	  else
	    {
	      if (nul && c >= '0' && c < '8')
		{
		  buffer[pos++] = '0';
		  buffer[pos++] = '0';
		  buffer[pos++] = '0';
		  nul = 0;
		}
	      buffer[pos] = c;
	    }
	}
      while (++pos < BIN2C_BUFFER_SIZE && infilesize);
      if (fwrite (buffer, 1, pos, outfile) < pos)
	break;
      pos = 0;
    }
  while (c != EOF && c != eos && infilesize);
  /* Close quotes.  */
  if (!ferror (outfile))
    fputc ('"', outfile);

  buffer[0] = 0;
  /* Force one string per line.  */
  return 1 << 30;
}
