/****h* ROBODoc/LaTeX_Generator
 * FUNCTION
 *   Generator for LaTeX output.  Supports singledoc mode.
 *
 *******
 */

#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "generator.h"
#include "util.h"
#include "links.h"
#include "latex_generator.h"
#include "globals.h"
#include "robodoc.h"

#ifdef DMALLOC
#include <dmalloc.h>
#endif

static int verbatim = FALSE;


/****f* LaTeX_Generator/RB_LaTeX_Generate_String
 * FUNCTION
 *   Write a string to the destination document, escaping
 *   characters where necessary.
 ******
 */

void
RB_LaTeX_Generate_String( FILE* dest_doc, char *a_string )
{
    int                 i;
    int                 l = strlen( a_string );
    unsigned char       c;

    for ( i = 0; i < l; i++ )
    {
        c = a_string[i];
        RB_LaTeX_Generate_Char( dest_doc, c );
    }
}


void
RB_LaTeX_Generate_False_Link( FILE* dest_doc, char *name )
{
    RB_LaTeX_Generate_String( dest_doc, name );
}

void
RB_LaTeX_Generate_Item_Begin( FILE* dest_doc )
{
    /* Empty */
}

void
RB_LaTeX_Generate_Item_End( FILE* dest_doc )
{
    /* Empty */
}

void
RB_LaTeX_Generate_Item_Name( FILE* dest_doc, char *name )
{
    RB_LaTeX_Generate_String( dest_doc, name );
    fprintf( dest_doc, "\n" );
}


void
RB_LaTeX_Generate_BeginSection( FILE * dest_doc, int depth, char *name )
{
    switch ( depth )
    {
    case 1:
        fprintf( dest_doc, "\\section{" );
        break;
    case 2:
        fprintf( dest_doc, "\\subsection{" );
        break;
    case 3:
        fprintf( dest_doc, "\\subsubsection{" );
        break;
    case 4:
    case 5:
    default:
        /* Too deep so just make it a subsubsection */
        fprintf( dest_doc, "\\subsubsection{" );
    }
    RB_LaTeX_Generate_String( dest_doc, name );
    fprintf( dest_doc, "}\n" );
}

void
RB_LaTeX_Generate_EndSection( FILE * dest_doc, int depth, char *name )
{
    /* Empty */
}

char               *
RB_LaTeX_Get_Default_Extension( void )
{
    return ".tex";
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_Doc_Start
 * NAME
 *   RB_LaTeX_Generate_Doc_Start --
 *****
 */

void
RB_LaTeX_Generate_Doc_Start( FILE * dest_doc, char *src_name, char *name,
                             char toc )
{
    fprintf( dest_doc, "%% Document: %s\n", name );
    fprintf( dest_doc, "%% Source: %s\n", src_name );
    fprintf( dest_doc, "%% " COMMENT_ROBODOC );
    fprintf( dest_doc, "%% " COMMENT_COPYRIGHT );
    if ( course_of_action & DO_HEADLESS ) 
    {
        /* The user wants a headless document, so we skip all
         * initialization.  It is up to the user to put this in.
         */
    }
    else
    {
        fprintf( dest_doc, "\\documentclass{article}\n" );
        fprintf( dest_doc, "\\usepackage{makeidx}\n" );
        fprintf( dest_doc, "\\oddsidemargin  0.15 in\n" );
        fprintf( dest_doc, "\\evensidemargin 0.35 in\n" );
        fprintf( dest_doc, "\\marginparwidth 1 in   \n" );
        fprintf( dest_doc, "\\oddsidemargin 0.25 in \n" );
        fprintf( dest_doc, "\\evensidemargin 0.25 in\n" );
        fprintf( dest_doc, "\\marginparwidth 0.75 in\n" );
        fprintf( dest_doc, "\\textwidth 5.875 in\n" );

        fprintf( dest_doc, "\\setlength{\\parindent}{0in}\n" );
        fprintf( dest_doc, "\\setlength{\\parskip}{.08in}\n\n" );

        /* changed default header to use boldface (vs slant) */
        fprintf( dest_doc, "\\pagestyle{headings}\n" );

        /* TODO   This global is never set */
        if ( document_title )
        {
            fprintf( dest_doc, "\\title{%s}\n", document_title );
        }
        else
        {
            fprintf( dest_doc, "\\title{API Reference}\n" );
        }
        fprintf( dest_doc, "\\author{%s}\n", COMMENT_ROBODOC );
        fprintf( dest_doc, "\\makeindex\n" );
        fprintf( dest_doc, "\\begin{document}\n" );
        fprintf( dest_doc, "\\maketitle\n" );
        /* autogenerate table of contents! */
        fprintf( dest_doc, "\\printindex\n" );
        fprintf( dest_doc, "\\tableofcontents\n" );
        fprintf( dest_doc, "\\newpage\n" );
        /* trick to disable the autogenerated \newpage */
        fprintf( dest_doc, "\n" );
    }
}

/****f* LaTeX_Generator/RB_LaTeX_Generate_Doc_End
 * NAME
 *   RB_LaTeX_Generate_Doc_End --
 *****
 */

void
RB_LaTeX_Generate_Doc_End( FILE * dest_doc, char *name )
{
    if ( course_of_action & DO_FOOTLESS ) 
    {
        /* The user does not want the foot of the
         * document
         */
    }
    else
    {
        fprintf( dest_doc, "\\end{document}\n" );
    }
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_Index_Entry
 * FUNCTION
 *   Creates a entry for the index.
 *
 *******
 */

void
RB_LaTeX_Generate_Index_Entry( FILE * dest_doc, struct RB_header *cur_header )
{
    assert( cur_header->function_name );

    fprintf( dest_doc, "\\index{unsorted!" );
    RB_LaTeX_Generate_String( dest_doc, cur_header->function_name );
    if ( cur_header->is_internal )
    {
        fprintf( dest_doc, "}\\index{internal_%s!", cur_header->htype->indexName );
    }
    else
    {
        fprintf( dest_doc, "}\\index{%s!", cur_header->htype->indexName );
    }
    RB_LaTeX_Generate_String( dest_doc, cur_header->function_name );
    fprintf( dest_doc, "}\n" );
}

/****f* LaTeX_Generator/RB_LaTeX_Generate_Header_Start
 * NAME
 *   RB_LaTeX_Generate_Header_Start --
 *****
 */

void
RB_LaTeX_Generate_Header_Start( FILE * dest_doc,
                                struct RB_header *cur_header )
{
    fprintf( dest_doc, "\\subsection{" );
    RB_LaTeX_Generate_String( dest_doc, cur_header->name );
    fprintf( dest_doc, "}\n" );
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_Header_End
 * NAME
 *   RB_LaTeX_Generate_Header_End --
 *****
 */

void
RB_LaTeX_Generate_Header_End( FILE * dest_doc, struct RB_header *cur_header )
{
    fputc( '\n', dest_doc );
}


/*x**f* LaTeX_Generator/RB_LaTeX_Generate_Index
 * NAME
 *   RB_LaTeX_Generate_Index --
 *****
 */

void
RB_LaTeX_Generate_Index( FILE * dest, char *source )
{
    assert( 0 );
#if 0
    RB_Generate_Doc_Start( dest, source, "Master File", 0 );
    RB_Generate_LaTeX_Includes( dest );
    RB_Generate_Doc_End( dest, source );
#endif
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_Index_Table
 * NAME
 *   RB_LaTeX_Generate_Index_Table --
 *****
 */

void
RB_LaTeX_Generate_Index_Table( FILE * dest, int type, char *title )
{
    /* Empty */
}

/****f* LaTeX_Generator/Generate_LaTeX_Includes
 * NAME
 *   Generate_LaTeX_Includes -- generate include commands
 * SYNOPSIS
 *   void RB_Generate_LaTeX_Includes (FILE *dest)
 * FUNCTION
 *   Generates a series of \include commands to include the
 *   documentation generated for each source file into one
 *   big file.
 ****
 */

void
RB_Generate_LaTeX_Includes( FILE * dest )
{
#if 0
    struct RB_link     *cur_link;

    for ( cur_link = first_link; cur_link; cur_link = cur_link->next_link )
    {
        {
            if ( cur_link->type == NO_HEADER )
                fprintf( dest, "\\include{%s}\n", cur_link->label_name );
        }
    }
#endif
}



/****f* LaTeX_Generator/RB_LaTeX_Generate_Empty_Item
 * NAME
 *   RB_LaTeX_Generate_Empty_Item --
 *****
 */

void
RB_LaTeX_Generate_Empty_Item( FILE * dest_doc )
{
    fprintf( dest_doc, "\\\\\n" );
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_EscapedChar
 * FUNCTION
 *   Generate a single character.  These characters are outside
 *   a begin{verbatim} end{verbatim} block. So we need to escape is
 *   special characters.
 *   These are
 *       _  =>  \_
 *       %  =>  \%
 *       $  =>  \$
 * SYNOPSIS
 *   void RB_LaTeX_Generate_EscapedChar( FILE* dest_doc, int c )
 * SEE ALSO
 *   RB_LaTeX_Generate_Char()
 *****
 */

void
RB_LaTeX_Generate_EscapedChar( FILE * dest_doc, int c )
{
    switch ( c )
    {
        case '\\':
        case '&':
        case '~':
        case '_':
        case '%':
        case '^':
        case '{':
        case '}':
        case '$':
        case '#':
            fputc( '\\', dest_doc );
            fputc( c, dest_doc );
            break;
        default:
            fputc( c, dest_doc );
            break;
    }
}


/****f* LaTeX_Generator/RB_LaTeX_Generate_Char
 * FUNCTION
 *   Generate a single character.  These characters are generated
 *   within a begin{verbatim} end{verbatim} block So no escaping is
 *   necessary.
 * SYNOPSIS
 *   void RB_LaTeX_Generate_Char( FILE* dest_doc, int c )
 *****
 */

void
RB_LaTeX_Generate_Char( FILE * dest_doc, int c )
{
    if ( verbatim ) 
    {
        switch ( c )
        {
            case '\n':
                assert( 0 );
                break;
            case '\t':
                assert( 0 );
                break;
            default:
                fputc( c, dest_doc );
        }
    }
    else
    {
        RB_LaTeX_Generate_EscapedChar( dest_doc, c );
    }
}


void LaTeX_Generate_Begin_Paragraph( FILE* dest_doc )
{
    fprintf( dest_doc, "\n" );
}

void LaTeX_Generate_End_Paragraph( FILE* dest_doc )
{
    fprintf( dest_doc, "\n" );
}


void LaTeX_Generate_Begin_Preformatted( FILE* dest_doc )
{
    fprintf( dest_doc, "\\begin{verbatim}\n" );
    verbatim = TRUE;
}

void LaTeX_Generate_End_Preformatted( FILE* dest_doc )
{
    verbatim = FALSE;
    fprintf( dest_doc, "\\end{verbatim}\n" );
}


void LaTeX_Generate_Begin_List( FILE* dest_doc )
{
    fprintf( dest_doc, "\\begin{itemize}\n" );
}

void LaTeX_Generate_End_List( FILE* dest_doc )
{
    fprintf( dest_doc, "\\end{itemize}\n" );
}

void LaTeX_Generate_Begin_List_Item( FILE* dest_doc )
{
    fprintf( dest_doc, "  \\item " );
}

void LaTeX_Generate_End_List_Item( FILE* dest_doc )
{
    fprintf( dest_doc, "" );
}


