/****h* ROBODoc/Analyser
 * NAME
 *   Analyser -- Functions to scan source and collect headers
 * FUNCTION
 *   This module provides the functions to scan a sourcefile and
 *   collect all the headers. 
 *
 *****
 * $Id: analyser.c,v 1.47 2004/06/16 20:31:48 gumpu Exp $
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#include "robodoc.h"
#include "globals.h"
#include "headers.h"
#include "headertypes.h"
#include "items.h"
#include "util.h"
#include "folds.h"
#include "links.h"
#include "analyser.h"
#include "document.h"
#include "file.h"
#include "part.h"


#ifdef DMALLOC
#include <dmalloc.h>
#endif

static int               ToBeAdded( struct RB_Document *document, struct RB_header *header );
static char             *Find_Header_Name( void );
static struct RB_header* Grab_Header( FILE* sourcehandle, struct RB_Document* arg_document );
static char             *Function_Name( char *header_name );
static char             *Module_Name( char *header_name );
static int               Find_End_Marker( FILE * document, struct RB_header *new_header );
struct RB_HeaderType*    AnalyseHeaderType( char **cur_char, int *is_internal );
static struct RB_HeaderType* RB_Find_Marker( FILE* document, int* is_internal, int reuse_previous_line );
static int               Analyse_Items( struct RB_header* arg_header );
static int               Is_ListItem_Start( char* arg_line, int arg_indent );

/****f* Analyser/Is_Pipe_Marker
 * NAME
 *   Is_Pipe_Marker
 * FUNCTION
 *   Check for "pipe" markers e.g. "|html ". 
 * SYNOPSIS
 *   static char* Is_Pipe_Marker(char *cur_char, int *pipe_mode )
 * RESULT
 *   Pointer to the data to be piped to document or in case no pointers
 *   are found.
 * SEE ALSO
 *   RB_Check_Pipe
 *******
 */

static char*
Is_Pipe_Marker( char *cur_char, int *pipe_mode )
{
    char               *s = cur_char + 1;

    *pipe_mode = -1;
    if ( *cur_char == '|' && *s )
    {
        if ( strncmp( "html ", s, 5 ) == 0 )
        {
            *pipe_mode = HTML;
            return ( s + 5 );
        }
        else if ( strncmp( "latex ", s, 6 ) == 0 )
        {
            *pipe_mode = LATEX;
            return ( s + 6 );
        }
        else if ( strncmp( "rtf ", s, 4 ) == 0 )
        {
            *pipe_mode = RTF;
            return ( s + 4 );
        }
        else if ( strncmp( "dbxml ", s, 6 ) == 0 )
        {
            *pipe_mode = XMLDOCBOOK;
            return ( s + 6 );
        }
        else if ( strncmp( "ascii ", s, 6 ) == 0 )
        {
            *pipe_mode = ASCII;
            return ( s + 6 );
        }
    }

    return ( char * ) NULL;
}


/****f* Analyser/RB_Analyse_Document
 *   foo
 * FUNCTION
 *   Scan all the sourcefiles of all parts of a document for
 *   headers.  Store these headers in each part (RB_Part).
 * SYNOPSIS
 *   void RB_Analyse_Document( struct RB_Document* arg_document )
 * INPUTS
 *   o document -- document to be analysed.
 * RESULT
 *   Each part will contain the headers that were found in the
 *   sourcefile of the part.
 * SOURCE
 */

void
RB_Analyse_Document( struct RB_Document *arg_document )
{
    struct RB_Part     *a_part;
    struct RB_Filename *a_filename;
    FILE               *filehandle;

    for ( a_part = arg_document->parts; a_part; a_part = a_part->next )
    {
        struct RB_header   *new_header = NULL;

        a_filename = a_part->filename;
        RB_Say( "analysing %s\n", Get_Fullname( a_filename ) );
        RB_SetCurrentFile( Get_Fullname( a_filename ) );

        RB_Header_Lock_Reset();
        filehandle = RB_Open_Source( a_part );
        line_number = 0;

        for ( new_header = Grab_Header( filehandle, arg_document );
              new_header; new_header = Grab_Header( filehandle, arg_document ) )
        {
            if ( ToBeAdded( arg_document, new_header ) )
            {
                /* The Add is required before the 
                 * Analyse because Add sets the owner of the header
                 * which is needed for error messages.
                 */
                RB_Part_Add_Header( a_part, new_header );
                Analyse_Items( new_header );
            }
            else
            {
                RB_Free_Header( new_header );
            }
        }
        fclose( filehandle );
    }
}

/*****/



/****f* Analyser/Check_Header_Start
 * FUNCION
 *   Sometimes a user makes a spelling mistake in the name of the first item.
 *   ROBODoc then ignores all the lines leading up to the second item,
 *   (which is the first item ROBODoc recognized).  This function
 *   checks for this condition and gives the user a warning.
 * INPUTS
 *   * arg_header -- the header to be checked.
 *   * first      -- the line on which the first item was found
 * RESULT
 *   A warning is given if the condition occured.
 * SOURCE
 */

static void Check_Header_Start( struct RB_header* arg_header, int first_item )
{
    if ( first_item ) 
    {
        int i = 0;
        for( i = 0; i < first_item; ++i )
        {
            char* c = arg_header->lines[ i ];
            c = RB_Skip_Whitespace( c );
            if ( RB_Has_Remark_Marker( c ) )
            {
                c = RB_Skip_Remark_Marker( c );
                for ( ; *c; ++c ) 
                {
                    if ( !isspace( *c ) ) 
                    {
                        RB_Warning_Full(
                                Get_Fullname( arg_header->owner->filename ),
                                arg_header->line_number,
                                "Header \"%s\" contains text before the fist item, "
                                "this might be caused by a misspelled item name.\n", 
                                arg_header->name );
                        return;
                    }
                }
            }
        }
    }
}

/*******/


/* asumes Copy_Lines_To_Item has been run on the item */

static int Is_Empty_Line( char* line ) 
{
   line = RB_Skip_Whitespace( line );
   return ( *line == '\0' );
}


/****f* Analyser/Get_Indent
 * FUNCION
 *   Determine the indentation of a line by counting
 *   the number of spaces at the begining of the line.
 * INPUTS
 *   * line -- the line
 * RESULT
 *   The indentation.
 * SOURCE
 */

static int Get_Indent( char* line )
{
    int i;
    for ( i = 0; line[ i ] && isspace( line[ i ] ); ++i )
    {
        /* empty */
    }
    return i;
}

/*****/


/****f* Analyser/Is_ListItem_Start
 * FUNCTION
 *   Test if a line starts with something that indicates a list item.
 *   List items start with '*', '-', or 'o'.
 * INPUTS
 *   * line -- the line
 * RESULT
 *   * TRUE  -- it did start with one of those characters
 *   * FALSE -- it did not.
 * SOURCE
 */

static int Is_ListItem_Start( char* arg_line, int arg_indent ) 
{
    char*c = arg_line;
    int cur_indent = Get_Indent( arg_line );

    if ( cur_indent == arg_indent ) 
    {
        /* TODO  Think there is a function for this */
        for ( ; *c && isspace( *c ); ++c ) { /* empty */ };

        if ( *c && ( strlen( c ) >= 3 ) ) 
        {
            if ( strchr( "*-o", *c ) && isspace( *(c+1) ) ) 
            {
                return TRUE;
            }
        }
    }
    else
    {
        /* The line is indented so it must be
         * the start of a pre block  */
    }

    return FALSE;
}

/*****/

/****f* Analyser/Is_List_Item_Continuation
 * FUNCTION
 *   Is it like the second line in something like:
 *     * this is a list item
 *       that continues 
 * INPUTS
 *   * arg_line  -- the current line
 *   * indent    -- the indent of the current item.
 * RESULT
 *   * TRUE  -- it is.
 *   * FALSE -- it is not.
 * SOURCE
 */

static int Is_List_Item_Continuation( char* arg_line, int indent )
{
    int this_indent = Get_Indent( arg_line );
    return ( this_indent > indent );
}

/*****/


static int Is_End_of_List( char* arg_line, int indent )
{
    int this_indent = Get_Indent( arg_line );
    return ( this_indent <= indent );
}

static int Is_Start_List( char* arg_line, int indent )
{
    int this_indent = Get_Indent( arg_line );
    char* c = strrchr( arg_line, ':' );
    if ( ( this_indent == indent ) && c )
    {
        for ( ++c; *c; ++c ) 
        {
            if ( !isspace( *c ) ) 
            {
                return FALSE;
            }
        }
        return TRUE;
    }
    return FALSE;
}



static void Remove_List_Char( struct RB_Item* arg_item, int start_index )
{
    char* c = arg_item->lines[ start_index ]->line;

    for ( ; *c && isspace( *c ); ++c ) { /* empty */ };
    if ( *c && ( strlen( c ) >= 3 ) ) 
    {
        if ( strchr( "*-o", *c ) && isspace( *(c+1) ) ) 
        {
            char* temp = arg_item->lines[ start_index ]->line;
            *c = ' ';
            arg_item->lines[ start_index ]->line = RB_StrDup( temp + 2 );
            free( temp );
        }
    }
}


static int Analyse_ListBody( struct RB_Item* arg_item, int start_index, int arg_indent )
{
    int i = start_index;
    for (; i < arg_item->no_lines; i++ )
    {
        char* line = arg_item->lines[ i ]->line;
        if ( ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN ) ||
             ( arg_item->lines[ i ]->kind == ITEM_LINE_END ) )
        {
            if ( Is_ListItem_Start( line, arg_indent ) ) 
            {
                arg_item->lines[ i ]->format |= RBILA_END_LIST_ITEM;
                arg_item->lines[ i ]->format |= RBILA_BEGIN_LIST_ITEM;
                Remove_List_Char( arg_item, i );
            }
            else if ( Is_List_Item_Continuation( line, arg_indent ) )
            {
                /* Nothing */
            }
            else
            {
                /* Must be the end of the list */
                arg_item->lines[ i ]->format |= RBILA_END_LIST_ITEM;
                arg_item->lines[ i ]->format |= RBILA_END_LIST;
                break;
            }
        }
    }
    return i;
}


/****f* Analyser/Analyse_List
 * FUNCTION 
 *   Parse the item text to see if there are any lists.
 *   A list is either case I:
 *      ITEMNAME
 *         o bla bla
 *         o bla bla
 *   or case II:
 *      some text:     <-- begin of a list
 *      o bla bla      <-- list item
 *        bla bla bla  <-- continuation of list item.
 *      o bla bla      <-- list item
 *                     <-- end of a list 
 *      bla bla        <-- this can also be the end of a list.
 * INPUTS
 *   * arg_item  -- the item to be parsed.
 *   * indent    -- the indent of this item.
 * OUTPUT
 *   * arg_item  -- the itemlines will contain formatting information 
 *                  for all the lists that were found.
 * SOURCE
 */

static void Analyse_List( struct RB_Item* arg_item, int indent )
{
    if ( arg_item->no_lines >= 1 ) 
    {
        int i = 0;
        char* line = arg_item->lines[ i ]->line;

        if ( ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN ) &&
             Is_ListItem_Start( line, indent ) )
        {
            /* Case I, the is a list item right after the item name */
            arg_item->lines[ i ]->format |= RBILA_BEGIN_LIST;
            arg_item->lines[ i ]->format |= RBILA_BEGIN_LIST_ITEM;
            Remove_List_Char( arg_item, i );
            /* Now try to find the end of the list */
            i = Analyse_ListBody( arg_item, 1, indent );
        }
        /* Now search for case II cases */
        for ( ;i < arg_item->no_lines; i++ )
        {
            line = arg_item->lines[ i ]->line;
            if ( ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN ) &&
                    Is_Start_List( line, indent ) ) 
            {
                ++i;
                if ( i < arg_item->no_lines ) 
                {
                    line = arg_item->lines[ i ]->line;
                    if ( ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN ) &&
                            Is_ListItem_Start( line, indent ) )
                    {
                        arg_item->lines[ i ]->format |= RBILA_BEGIN_LIST;
                        arg_item->lines[ i ]->format |= RBILA_BEGIN_LIST_ITEM;
                        Remove_List_Char( arg_item, i );
                        ++i;
                        i = Analyse_ListBody( arg_item, i, indent );
                    }
                }
            }
        }
    }
}

/******/

static void Dump_Item( struct RB_Item* arg_item )
{
    int i;
    /* preformatted blocks */
    for ( i = 0; i < arg_item->no_lines; i++ )
    {
        char* line = arg_item->lines[ i ]->line;
        int format = arg_item->lines[ i ]->format;
        printf( ( format & RBILA_END_PARAGRAPH ) ? "p" : " " );
        printf( ( format & RBILA_BEGIN_PARAGRAPH ) ? "P" : " " );
        printf( ( format & RBILA_END_PRE ) ? "e" : " " );
        printf( ( format & RBILA_BEGIN_PRE ) ? "E" : " " );
        printf( ( format & RBILA_END_LIST ) ? "l" : " " );
        printf( ( format & RBILA_BEGIN_LIST ) ? "L" : " " );
        printf( ( format & RBILA_END_LIST_ITEM ) ? "i" : " " );
        printf( ( format & RBILA_BEGIN_LIST_ITEM ) ? "I" : " " );
        printf( " (%s)\n", line );
    }
}


static void Preformat_All( struct RB_Item* arg_item )
{
    int i;
    int preformatted = FALSE;
    char* line = NULL;
    if ( arg_item->no_lines > 0 ) 
    {
        i = 0;
        /* Skip any pipe stuff */
        for (; ( i < arg_item->no_lines ) && (arg_item->lines[ i ]->kind == ITEM_LINE_PIPE ); ++i)
        {
            /* Empty */
        }

        line = arg_item->lines[ i ]->line;
        if ( ( arg_item->lines[ i ]->kind == ITEM_LINE_RAW ) ||
             ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN ) )
        {
            arg_item->lines[ i ]->format |= RBILA_BEGIN_PRE;
            preformatted = TRUE;

            for ( ++i; i < arg_item->no_lines; i++ )
            {
                if ( arg_item->lines[ i ]->kind == ITEM_LINE_PIPE )
                {
                    /* Temporarily end the preformatting to allow
                     * the piping to happen
                     */
                    arg_item->lines[ i ]->format |= RBILA_END_PRE;
                    /* Find the end of the pipe stuff */
                    for (;( i < arg_item->no_lines ) && 
                            ( arg_item->lines[ i ]->kind == ITEM_LINE_PIPE ); 
                            ++i) 
                    { /* Empty */ };
                    /* Every item ends with an ITEM_LINE_END, so: */
                    assert( i < arg_item->no_lines );
                    /* And re-enable preformatting */
                    arg_item->lines[ i ]->format |= RBILA_BEGIN_PRE;
                }

                if ( arg_item->lines[ i ]->kind == ITEM_LINE_END )
                {
                    arg_item->lines[ i ]->format |= RBILA_END_PRE;
                }
            }
        }
    }
}


/*
 *     aaaaa
 *       aa
 *               </p>
 *       aa      <p>
 *     aaa
 *     aaaa
 *
 *
 */

static void Analyse_Preformatted( struct RB_Item* arg_item, int indent )
{
    int i;
    int in_list = FALSE;
    int new_indent = -1;
    int preformatted = FALSE;
    char* line = NULL;

    /* preformatted blocks */
    if ( arg_item->no_lines > 0 ) 
    {
        i = 0;
        /* Skip any pipe stuff */
        for (; ( i < arg_item->no_lines ) && (arg_item->lines[ i ]->kind == ITEM_LINE_PIPE); ++i)
        {
            /* Empty */
        }

        line = arg_item->lines[ i ]->line;

        if ( ( !in_list ) && ( arg_item->lines[ i ]->format & RBILA_BEGIN_LIST ) )
        {
            in_list = TRUE;
        }
        if ( ( in_list ) && ( arg_item->lines[ i ]->format & RBILA_END_LIST ) )
        {
            in_list = FALSE;
        }

        for ( ++i; i < arg_item->no_lines; i++ )
        {
            if ( arg_item->lines[ i ]->kind == ITEM_LINE_PIPE )
            {
                if ( preformatted ) 
                {
                    arg_item->lines[ i ]->format |= RBILA_END_PRE;
                }
                for (;( i < arg_item->no_lines ) && 
                      ( arg_item->lines[ i ]->kind == ITEM_LINE_PIPE ); 
                      ++i) 
                { /* Empty */ };
                /* Every item ends with an ITEM_LINE_END, so: */
                assert( i < arg_item->no_lines );
                if ( preformatted )
                {
                    arg_item->lines[ i ]->format |= RBILA_BEGIN_PRE;
                }
            }

            line = arg_item->lines[ i ]->line;
            new_indent = Get_Indent( line );

            if ( ( !in_list ) && ( arg_item->lines[ i ]->format & RBILA_BEGIN_LIST ) )
            {
                in_list = TRUE;
            }
            if ( ( in_list ) && ( arg_item->lines[ i ]->format & RBILA_END_LIST ) )
            {
                in_list = FALSE;
            }

            if ( !in_list )
            {
                if ( ( new_indent > indent ) && !preformatted ) 
                {
                    preformatted = TRUE;
                    arg_item->lines[ i ]->format |= RBILA_BEGIN_PRE;
                }
                else if ( ( new_indent <= indent ) && preformatted ) 
                {
                    preformatted = FALSE;
                    arg_item->lines[ i ]->format |= RBILA_END_PRE;
                }
                else
                {
                    /* An empty line */
                }
            }
            else
            {
                /* We are in a list, do nothing */
            }
        }
    }
}


static void Analyse_Paragraphs( struct RB_Item* arg_item )
{
    int i;
    int in_par          = FALSE;
    int in_list         = FALSE;
    int in_pre          = FALSE;
    int is_empty        = FALSE;
    int prev_is_empty   = FALSE;

    for ( i = 0; 
          ( i < arg_item->no_lines ) && (arg_item->lines[ i ]->kind == ITEM_LINE_PIPE); 
          ++i)
    {
        /* Empty */
    }
    assert( i < arg_item->no_lines );

    if ( ( arg_item->lines[ i ]->format == 0 ) )
    {
        arg_item->lines[ i ]->format |= RBILA_BEGIN_PARAGRAPH;
        in_par = TRUE;
    }
    for ( ; i < arg_item->no_lines; i++ )
    {
        char* line = arg_item->lines[ i ]->line;
        prev_is_empty = is_empty;
        is_empty = Is_Empty_Line( line );
        if ( arg_item->lines[ i ]->format & RBILA_BEGIN_LIST ) 
        {
            in_list = TRUE;
        }
        if ( arg_item->lines[ i ]->format & RBILA_BEGIN_PRE )
        {
            in_pre = TRUE;
        }
        if ( arg_item->lines[ i ]->format & RBILA_END_LIST )
        {
            in_list = FALSE;
        }
        if ( arg_item->lines[ i ]->format & RBILA_END_PRE )
        {
            in_pre = FALSE;
        }
        if ( in_par )
        {
            if ( ( arg_item->lines[ i ]->format & RBILA_BEGIN_LIST ) ||
                 ( arg_item->lines[ i ]->format & RBILA_BEGIN_PRE ) ||
                 is_empty )
            {
                in_par = FALSE;
                arg_item->lines[ i ]->format |= RBILA_END_PARAGRAPH;
            }
        }
        else
        {
            if ( ( arg_item->lines[ i ]->format & RBILA_END_LIST ) ||
                 ( arg_item->lines[ i ]->format & RBILA_END_PRE ) ||
                 ( !is_empty && prev_is_empty && !in_list && !in_pre ) )
            {
                in_par = TRUE;
                arg_item->lines[ i ]->format |= RBILA_BEGIN_PARAGRAPH;
            }
        }
    }
    if ( in_par ) 
    {
        arg_item->lines[ arg_item->no_lines - 1 ]->format |= RBILA_END_PARAGRAPH;
    }
}

static int Analyse_Indentation( struct RB_Item* arg_item )
{
    int i;
    int indent = -1;
    assert( arg_item->no_lines > 0 );

    for ( i = 0; i < arg_item->no_lines; ++i )
    {
        if ( arg_item->lines[ i ]->kind == ITEM_LINE_PLAIN )
        {
            char* line = arg_item->lines[ i ]->line;
            if ( Is_Empty_Line( line ) )
            {
                /* Empty */
                indent = 0;
            }
            else
            {
                indent = Get_Indent( line );
                break;
            }
        }
    }
    assert( indent >= 0 );
    return indent;
}

/****f* Analyser/Analyse_Item_Format
 * FUNCTION
 *   Try to determine the formatting of an item.
 *   An empty line generates a new paragraph
 *   Things that are indented more that the rest of the text
 *   are preformatted.
 *   Things that start with a '*' or '-' create list items
 *   unless they are indented more that the rest of the text.
 * INPUTS
 *   * arg_item -- the item to be analysed.
 * SOURCE
 */

static void Analyse_Item_Format( struct RB_Item* arg_item )
{
    if ( ( arg_item->no_lines ) && ( arg_item->type != SOURCE_ITEM ) ) 
    {
        int indent = Analyse_Indentation( arg_item );
        if ( course_of_action & DO_NOPRE )
        {
            Analyse_List( arg_item, indent );
            Analyse_Preformatted( arg_item, indent );
            Analyse_Paragraphs( arg_item );
        }
        else
        {
            Preformat_All( arg_item );
        }
    } 
    else if ( ( arg_item->no_lines ) && ( arg_item->type == SOURCE_ITEM ) )
    {
        Preformat_All( arg_item );
    }
    else
    {
        /* item is empty */
    }
    /* Dump_Item( arg_item ); */
}

/*****/


static void Copy_Lines_To_Item( struct RB_header* arg_header, struct RB_Item* arg_item )
{
#if 0
    printf( "%d\n%d\n%s\n%s\n",
            arg_item->begin_index,
            arg_item->end_index,
            arg_header->lines[ arg_item->begin_index ],
            arg_header->lines[ arg_item->end_index ] );
#endif
    arg_item->no_lines = arg_item->end_index - arg_item->begin_index + 1;
    if ( arg_item->no_lines > 1 ) 
    {
        char* c;
        int i = 0;
        int j = 0;

        /* trim all the empty lines at the begin of an item.  We skip
         * the first line because this is the name if the item 
         */
        i = arg_item->begin_index + 1;
        if ( arg_item->type == SOURCE_ITEM )
        {
            /* We skip the first line after the source item. */
            ++i;
        }
        do {
            c = arg_header->lines[ i ];
            c = RB_Skip_Whitespace( c );
            if ( RB_Has_Remark_Marker( c ) )
            {
                c = RB_Skip_Remark_Marker( c );
            }
            c = RB_Skip_Whitespace( c );
            if ( *c == '\0' ) {
                ++i;
            }
        } while ( ( *c == '\0' ) && ( i < arg_item->end_index ) );
        if ( i < arg_item->end_index ) {
            arg_item->begin_index = i;

            /* trim all the empty lines at the end of an item */
            i = arg_item->end_index;
            do {
                c = arg_header->lines[ i ];
                c = RB_Skip_Whitespace( c );
                if ( RB_Has_Remark_Marker( c ) )
                {
                    c = RB_Skip_Remark_Marker( c );
                }
                c = RB_Skip_Whitespace( c );
                if ( *c == '\0' ) {
                    --i;
                }
            } while ( ( *c == '\0' ) && ( i > arg_item->begin_index ) );
            arg_item->end_index = i;
        }
        else
        {
            arg_item->begin_index = arg_item->end_index;
        }

        assert( arg_item->begin_index <= arg_item->end_index );

        /* Now copy any of the remaining lines */
        arg_item->no_lines = arg_item->end_index - arg_item->begin_index + 1;
        if ( arg_item->no_lines > 0 ) 
        {
            struct RB_Item_Line* itemline = NULL;
            /* Allocate enough memory for all the lines, plus one
             * extra line
             */
            ++arg_item->no_lines;
            arg_item->lines = calloc( arg_item->no_lines, sizeof( struct RB_Item_Line* ) ) ;
            assert( arg_item->lines );

            /* And create an RB_Item_Line for each of them, and add
             * those to the RB_Item
             */
            for ( i = 0; i < arg_item->no_lines - 1; ++i ) 
            {
                char* c = arg_header->lines[ arg_item->begin_index + i ];
                /* TODO should be a Create_ItemLine() */
                itemline = malloc( sizeof ( struct RB_Item_Line ) );
                assert( itemline );

                c = ExpandTab( c );
                c = RB_Skip_Whitespace( c );
                if ( RB_Has_Remark_Marker( c ) && ( arg_item->type != SOURCE_ITEM ) )
                {
                    char* c2 = c;
                    int   pipe_mode;
                    c = RB_Skip_Remark_Marker( c );
                    c2 = RB_Skip_Whitespace( c );
                    if ( *c2 ) 
                    {
                        c2 = Is_Pipe_Marker( c2, &pipe_mode );
                        if ( c2 != NULL ) 
                        {
                            itemline->kind = ITEM_LINE_PIPE;
                            itemline->pipe_mode = pipe_mode;
                            c = c2;
                        }
                        else
                        {
                            itemline->kind = ITEM_LINE_PLAIN;
                        }
                    }
                    else
                    {
                        itemline->kind = ITEM_LINE_PLAIN;
                    }
                }
                else
                {
                    itemline->kind = ITEM_LINE_RAW;
                    /* The is raw code, so we do not want to have the
                     * whitespace stripped of
                     */
                    c = arg_header->lines[ arg_item->begin_index + i ];
                    c = ExpandTab( c );
                }
                if ( ( ( arg_item->type != SOURCE_ITEM ) && 
                            ( ( itemline->kind == ITEM_LINE_PLAIN ) ||
                              ( itemline->kind == ITEM_LINE_PIPE ) ) ) ||
                        ( arg_item->type == SOURCE_ITEM ) ) 
                {
                    itemline->line = RB_StrDup( c );
                    itemline->format = 0;
                    arg_item->lines[ j ] = itemline;
                    ++j;
                }
                else
                {
                    /* We dump the RAW item lines if we are not in a
                     * source item.
                     */
                    free( itemline );
                }
            }
            if ( j > 0 ) 
            {
                /* And one empty line to mark the end of an item and
                 * to be able to store some additional formatting actions
                 */
                itemline = malloc( sizeof ( struct RB_Item_Line ) );
                itemline->kind = ITEM_LINE_END;
                itemline->line = RB_StrDup("");
                itemline->format = 0;
                arg_item->lines[ j ] = itemline;

                /* Store the real number of lines we copied */
                assert( arg_item->no_lines >= ( j + 1 ) );
                arg_item->no_lines = j + 1;
            }
            else
            {
                arg_item->no_lines = 0;
                free( arg_item->lines );
                arg_item->lines = NULL;
            }
        }
        else
        {
            arg_item->no_lines = 0;
            arg_item->lines = NULL;
        }
    }
    else
    {
        arg_item->no_lines = 0;
        arg_item->lines = NULL;
    }
}


/****f* Analyser/RB_Analyse_Items
 * FUNCTION
 *   Locate the items in the header and create RB_Item structures for
 *   them.
 * SOURCE
 */

static int Analyse_Items( struct RB_header* arg_header )
{
    int                 line_nr;
    enum ItemType       item_type = NO_ITEM;
    struct RB_Item     *new_item;
    struct RB_Item     *cur_item;
    RB_Item_Lock_Reset();
//    printf( "--%d\n", arg_header->no_lines );

    /* find the first item */
    for ( line_nr = 0; line_nr < arg_header->no_lines; ++line_nr )
    {
        item_type = RB_Is_ItemName( arg_header->lines[ line_nr ] );
        if ( item_type != NO_ITEM ) { 
            break; 
        }
    }

    /* and all the others */
    while ( ( item_type != NO_ITEM ) && ( line_nr < arg_header->no_lines ) )
    {
        new_item = RB_Create_Item( item_type );
        new_item->begin_index = line_nr;

        /* Add the item to the end of the list of items. */
        if ( arg_header->items )
        {
            for ( cur_item = arg_header->items; cur_item->next; cur_item = cur_item->next )
            {
                /* Empty */
            }
            cur_item->next = new_item;
        }
        else
        {
            arg_header->items = new_item;
        }
        /* Find the next item */
        for ( ++line_nr; line_nr < arg_header->no_lines; ++line_nr ) 
        {
            item_type = RB_Is_ItemName( arg_header->lines[ line_nr ] );
            if ( item_type != NO_ITEM ) { 
                break; 
            }
        }

        /* This points to the last line in the item */
        new_item->end_index = line_nr - 1;

        assert( new_item->end_index >= new_item->begin_index );

        /* Now analyse and copy the lines */
        Copy_Lines_To_Item( arg_header, new_item );
        Analyse_Item_Format( new_item );
    }

    return 0;
}

/******/



/****f* Analyser/ToBeAdded
 * FUNCTION
 *   Test whether or not a header needs to be added to the
 *   list of headers. This implements the options 
 *      --internal 
 *   and
 *      --internalonly
 * INPUTS
 *   o document  -- a document (to determine the options)
 *   o header    -- a header
 * RESULT
 *   TRUE  -- Add header
 *   FALSE -- Don't add header
 * SOURCE
 */

static int ToBeAdded( struct RB_Document *document, struct RB_header *header )
{
    int                 add = FALSE;

    if ( header->is_internal )
    {
        if ( ( document->actions & DO_INCLUDE_INTERNAL ) ||
                ( document->actions & DO_INTERNAL_ONLY ) )
        {
            add = TRUE;
        }
        else
        {
            add = FALSE;
        }
    }
    else
    {
        if ( document->actions & DO_INTERNAL_ONLY )
        {
            add = FALSE;
        }
        else
        {
            add = TRUE;
        }
    }
    return add;
}

/******/



/****f* Analyser/Grab_Header
 * FUNCTION
 *   Grab a header from a source file, that is scan a source file
 *   until the start of a header is found.  Then search for the end
 *   of a header and store all the lines in between.
 * SYNPOPSIS
 *   struct RB_header* Grab_Header( FILE * sourcehandle )
 * INPUTS
 *   o sourcehandle -- an opened source file.
 * OUTPUT
 *   o sourcehandle -- will point to the line following the end marker.
 * RESULT
 *   0 if no header was found, or a pointer to a new header otherwise.
 * SOURCE
 */

static struct RB_header* Grab_Header( FILE* sourcehandle, struct RB_Document* arg_document )
{
    struct RB_header   *new_header = NULL;
    int                 is_internal = 0;
    struct RB_HeaderType* header_type = NULL;
    int                 good_header = FALSE;
    int                 reuse = FALSE;
    do {
        good_header = FALSE;
        header_type = RB_Find_Marker( sourcehandle, &is_internal, reuse );
        reuse = FALSE;
        if ( header_type )
        {
            struct RB_header   *duplicate_header = NULL;
            long                previous_line = 0;
            new_header = RB_Alloc_Header(  );
            new_header->htype = header_type;
            new_header->is_internal = is_internal;

            if ( ( new_header->name = Find_Header_Name(  ) ) != NULL )
            {
                new_header->line_number = line_number;
                RB_Say( "found header [line %5d]: \"%s\"\n", line_number, new_header->name );
                duplicate_header = RB_Document_Check_For_Duplicate( arg_document, new_header->name );
                if ( duplicate_header )
                {
                    RB_Warning( "A header with the name \"%s\" already exist. See %s:%d\n", 
                            new_header->name,
                            Get_Fullname( duplicate_header->owner->filename ),
                            duplicate_header->line_number );
                }
                else
                {
                    /* Duplicate headers do not crash the program so
                     * we accept them.  But we do warn the user.
                     */
                }

                if ( ( new_header->function_name =
                            Function_Name( new_header->name ) ) == NULL )
                {
                    RB_Warning( "Can't determine the \"function\" name.\n" );
                    RB_Free_Header( new_header );
                    new_header = NULL;
                }
                else
                {
                    if ( ( new_header->module_name =
                                Module_Name( new_header->name ) ) == NULL )
                    {
                        RB_Warning( "Can't determine the \"module\" name.\n" );
                        RB_Free_Header( new_header );
                        new_header = NULL;
                    }
                    else
                    {
                        previous_line = line_number;
                        if ( Find_End_Marker( sourcehandle, new_header ) == 0 )
                        {
                            RB_Warning( "found header on line %d with name \"%s\""
                                    "but I can't find the end marker\n",
                                    previous_line, new_header->name );
                            /* Reuse the current line while finding the next
                             * Marking using RB_Find_Marker()
                             */
                            reuse = TRUE;
                            RB_Free_Header( new_header );
                            new_header = NULL;
                        }
                        else
                        {
                            RB_Say( "found end header [line %5d]:\n", line_number );
                            /* Good header found, we can stop */
                            good_header = TRUE;
                        }
                    }
                }
            }
            else
            {
                RB_Warning( "found header marker but no name\n" );
                RB_Free_Header( new_header );
                new_header = NULL;
            }
        }
        else
        {
            /* end of the file */
            good_header = TRUE;
        }
    } while ( ! good_header );
    return new_header;
}

/*******/



/****f* Analyser/Module_Name
 * FUNCTION
 *   Get the module name from the header name.  The header name will be
 *   something like
 *
 *     module/functionname.
 *
 * SYNPOPSIS
 *   char* Module_Name( char *header_name )
 * INPUTS
 *   o header_name -- a pointer to a nul terminated string.
 * RESULT
 *   Pointer to the modulename.  You're responsible for freeing it.
 * SEE ALSO
 *   Function_Name()
 * SOURCE
 */

static char        *
Module_Name( char *header_name )
{
    char               *cur_char;
    char                c;
    char               *name = NULL;

    assert( header_name );

    for ( cur_char = header_name; *cur_char && *cur_char != '/'; ++cur_char );
    if ( *cur_char )
    {
        c = *cur_char;
        *cur_char = '\0';
        name = RB_StrDup( header_name );
        *cur_char = c;
    }
    return name;
}

/******/



/****f* Analyser/Function_Name
 * FUNCTION
 *   A header name is consists of two parts. The module name and
 *   the function name. This returns a pointer to the function name.
 *   The name "function name" is a bit obsolete. It is really the name
 *   of any of objects that can be documented; classes, methods,
 *   variables, functions, projects, etc.
 * SYNOPSIS
 *   char *RB_NamePart(char *header_name)
 * SOURCE
 */

static char        *
Function_Name( char *header_name )
{
    char               *cur_char;
    char               *name;

    name = NULL;
    if ( ( cur_char = header_name ) != NULL )
    {
        for ( ; *cur_char != '\0'; ++cur_char )
        {
            if ( '/' == *cur_char )
            {
                ++cur_char;
                if ( *cur_char )
                {
                    name = cur_char;
                    break;
                }
            }
        }
    }
    if ( name )
    {
        char               *temp;
        temp = ( char * ) malloc( ( strlen( name ) + 1 ) * sizeof( char ) );
        assert( temp );
        strcpy( temp, name );
        return temp;
    }
    else
    {
        return ( name );
    }
}

/*** RB_Name_Part ***/


/****f* Analyser/RB_Find_Marker
 * NAME
 *   RB_Find_Marker -- Search for header marker in document.
 * SYNOPSIS
 *   header_type = RB_Find_Marker (document)
 * FUNCTION
 *   Read document file line by line, and search each line for 
 *   any of the headers defined in the array  header_markers (OR
 *   if using the -rh switch, robo_head)
 * INPUTS
 *   document - pointer to the file to be searched.
 *   the gobal buffer line_buffer.
 * OUTPUT
 *   o document will point to the line after the line with 
 *     the header marker.
 *   o is_internal will be TRUE if the header is an internal
 *     header.
 * RESULT
 *   o header type
 * BUGS
 *   Bad use of feof(), fgets().
 * SEE ALSO
 *   Find_End_Marker
 * SOURCE
 */

static struct RB_HeaderType*
RB_Find_Marker( FILE* document, int* is_internal, int reuse_previous_line )
{
    int                 found;
    char               *cur_char;
    struct RB_HeaderType* header_type = 0;

    cur_char = NULL;
    found = FALSE;
    while ( !feof( document ) && !found )
    {
        if ( reuse_previous_line )
        {
            /* reuse line in the line_buffer */
            reuse_previous_line = FALSE;
        }
        else {
            *line_buffer = '\0';
            fgets( line_buffer, MAX_LINE_LEN, document );
        }
        if ( !feof( document ) )
        {
            line_number++;
            found = RB_Is_Begin_Marker( line_buffer, &cur_char );
            if ( found )
            {
                header_type = AnalyseHeaderType( &cur_char, is_internal );
                RB_Say( "found header marker of type %s\n", header_type->indexName );
            }
        }
    }

    return header_type;
}

/******** END RB_Find_Marker ******/


/****f* Analyser/AnalyseHeaderType
 * FUNCTION
 *   Determine the type of the header.
 * INPUTS
 *   o cur_char -- pointer to the header type character
 * OUTPUT
 *   o is_internal -- indicates if it is an internal header or not.*
 *   o cur_char -- points to the header type character
 * RESULT
 *   o pointer to a RB_HeaderType
 * SOURCE
 */ 

struct RB_HeaderType* AnalyseHeaderType( char **cur_char, int *is_internal )
{
    struct RB_HeaderType*  headertype = 0;

    *is_internal = RB_IsInternalHeader( **cur_char );

    if ( *is_internal ) 
    {
        /* Skip the character */
        ++( *cur_char );
    }
    headertype = RB_FindHeaderType( **cur_char );
    if ( !headertype )
    {
        RB_Panic( "Undefined headertype (%c)\n", **cur_char );
    }

    return headertype;
}

/*******/



/****f* Analyser/Find_End_Marker
 * FUNCTION
 *   Scan and store all lines from a source file until
 *   an end marker is found.
 * SYNOPSIS
 *   int Find_End_Marker( FILE* document, struct RB_header *new_header )
 * INPUTS
 *   o document -- a pointer to an opened source file.
 * OUTPUT
 *   o new_header -- the lines of source code will be added
 *                   here.
 * RESULT
 *   o TRUE  -- an end marker was found.
 *   o FALSE -- no end marker was found while scanning the
 *              source file.
 * SOURCE
 */

static int
Find_End_Marker( FILE * document, struct RB_header *new_header )
{
    int                 found = FALSE;
    unsigned int        no_lines = 0;
    unsigned int        max_no_lines = 10;
    char              **lines = NULL;
    char              **new_lines = NULL;
    char               *dummy;

    lines = ( char ** ) calloc( max_no_lines, sizeof( char * ) );
    while ( fgets( line_buffer, MAX_LINE_LEN, document ) != NULL )
    {
        ++line_number;          /* global linecounter, koessi */
        if ( RB_Is_Begin_Marker( line_buffer, &dummy ) )
        {
            /* Bad... found a begin marker but was expecting to
               find an end marker.  Panic... */
            found = FALSE;
            return found;
        }
        else if ( RB_Is_End_Marker( line_buffer ) )
        {
            RB_Say( "Found end marker \"%s\"", line_buffer );
            found = TRUE;
            break;
        }
        else
        {
            unsigned int        n;
            char               *line;

            line = RB_StrDup( line_buffer );
            n = strlen( line );
            assert( n > 0 );
            assert( line[n - 1] == '\n' );
            /* Strip CR */
            line[n - 1] = '\0';
            lines[no_lines] = line;
            ++no_lines;
            if ( no_lines == max_no_lines )
            {
                max_no_lines *= 2;
                new_lines = realloc( lines, max_no_lines * sizeof( char * ) );
                assert( new_lines );
                lines = new_lines;
            }
        }
    }

    new_header->no_lines = no_lines;
    new_header->lines = lines;

    return found;
}

/******/

/****f* Analyser/Find_Header_Name   [3.0b]
 * FUNCTION
 *   Searches the line buffer for the header name.
 *   It assumes that the header name follows after the
 *   header marker, seperated by one or more spaces, and terminated
 *   by one or more spaces or a '\n'.
 *   It allocates an array of chars and copies the name to this array.
 * SYNOPSIS
 *   result = Find_Header_Name ()
 * INPUTS
 *   the gobal buffer line_buffer.
 * RESULT
 *   pointer to the allocated array of chars that contains the name,
 *   terminated with a '\0'.
 *   NULL if no header name was found.
 * MODIFICATION HISTORY
 *   8. August 1995      --  optimized by koessi
 * SEE ALSO
 *   RB_WordLen(), RB_StrDup()
 * SOURCE
 */

static char        *
Find_Header_Name( void )
{
    char               *cur_char;

    cur_char = line_buffer;
    skip_while( *cur_char != '*' );
    skip_while( !isspace( *cur_char ) );
    skip_while( isspace( *cur_char ) );
    if ( *cur_char )
    {
        char               *end_char, old_char;

        end_char = cur_char + RB_WordLen( cur_char );
        old_char = *end_char;
        *end_char = '\0';
        cur_char = RB_StrDup( cur_char );
        *end_char = old_char;
        return ( cur_char );
    }
    return ( NULL );
}

/*****  Find_Header_Name  *****/

