/* riemann/query.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <riemann/query.h>

#include <errno.h>
#include <stdlib.h>
#include <string.h>

riemann_query_t *
riemann_query_new (const char *string)
{
  riemann_query_t *query;

  query = (riemann_query_t *)
    malloc (sizeof (riemann_query_t));
  query__init (query);

  if (!string)
    return query;

  /* This cannot fail, because if malloc failed, we crashed already,
     and the previous check makes sure we bail out if string is empty. */
  riemann_query_set_string (query, string);

  return query;
}

void
riemann_query_free (riemann_query_t *query)
{
  if (!query)
    {
      errno = EINVAL;
      return;
    }

  query__free_unpacked (query, NULL);
}

int
riemann_query_set_string (riemann_query_t *query, const char *string)
{
  if (!query || !string)
    return -EINVAL;

  if (query->string)
    free (query->string);

  query->string = strdup (string);

  return 0;
}

riemann_query_t *
riemann_query_clone (const riemann_query_t *query)
{
  if (!query)
    {
      errno = EINVAL;
      return NULL;
    }

  return riemann_query_new (query->string);
}
