/* riemann/client/tls-wolfssl.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2021-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "riemann/_private.h"
#include "riemann/platform.h"

#include "riemann/client/socket.h"
#include "riemann/client/tls.h"

#include <wolfssl/options.h>
#include <wolfssl/ssl.h>

typedef struct {
  riemann_client_connection_socket_t s;

  struct
  {
    WOLFSSL_CTX* ctx;
    WOLFSSL* ssl;

    riemann_client_tls_options_t options;
  } tls;
} riemann_client_connection_wolfssl_t;


int
_riemann_client_disconnect_tls (riemann_client_t *client)
{
  riemann_client_connection_wolfssl_t *conn =
    (riemann_client_connection_wolfssl_t *) client->connection;

  if (!conn || client->type != RIEMANN_CLIENT_TLS)
    return -ENOTCONN;

  if (conn->tls.ssl) {
    wolfSSL_shutdown (conn->tls.ssl);
    wolfSSL_free (conn->tls.ssl);
  }
  if (conn->tls.ctx) {
    wolfSSL_CTX_free (conn->tls.ctx);
  }

  wolfSSL_Cleanup ();

  _riemann_client_tls_options_free (&conn->tls.options);
  return _riemann_client_disconnect_socket (client);
}

int
_riemann_client_connect_setup_tls (riemann_client_t *client,
                                   va_list aq)
{
  va_list ap;
  riemann_client_option_t option;
  riemann_client_connection_wolfssl_t *conn;
  int err_ret = EINVAL;

  conn = (riemann_client_connection_wolfssl_t *)
    calloc (1, sizeof (riemann_client_connection_wolfssl_t));

  conn->s.sock = -1;
  conn->tls.options.handshake_timeout = 40000;

  client->type = RIEMANN_CLIENT_TLS;
  client->send = _riemann_client_send_message_tls;
  client->recv = _riemann_client_recv_message_tls;
  client->connect = _riemann_client_connect_tls;
  client->disconnect = _riemann_client_disconnect_tls;
  client->get_fd = _riemann_client_get_fd_socket;
  client->set_timeout = _riemann_client_set_timeout_socket;
  client->connection = (void *) conn;

  va_copy (ap, aq);

  option = (riemann_client_option_t) va_arg (ap, int);
  do
    {
      switch (option)
        {
        case RIEMANN_CLIENT_OPTION_NONE:
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CA_FILE:
          _riemann_set_string (&(conn->tls.options.cafn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CERT_FILE:
          _riemann_set_string (&(conn->tls.options.certfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_KEY_FILE:
          _riemann_set_string (&(conn->tls.options.keyfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_HANDSHAKE_TIMEOUT:
          conn->tls.options.handshake_timeout = va_arg (ap, unsigned int);
          break;

        case RIEMANN_CLIENT_OPTION_TLS_PRIORITIES:
          _riemann_set_string (&(conn->tls.options.priorities), va_arg (ap, char *));
          break;

        default:
          va_end (ap);
          goto err;
        }

      if (option != RIEMANN_CLIENT_OPTION_NONE)
        option = (riemann_client_option_t) va_arg (ap, int);
    }
  while (option != RIEMANN_CLIENT_OPTION_NONE);
  va_end (ap);

  if (!conn->tls.options.cafn || !conn->tls.options.certfn || !conn->tls.options.keyfn)
    {
      goto err;
    }

  return 0;

 err:
  _riemann_client_tls_options_free (&conn->tls.options);
  free (conn);
  client->connection = NULL;
  return -err_ret;
}

int
_riemann_client_connect_tls (riemann_client_t *client,
                             const char *hostname,
                             int port)
{
  riemann_client_connection_wolfssl_t *conn =
    (riemann_client_connection_wolfssl_t *) client->connection;
  struct timeval timeout;
  int e = 0;

  wolfSSL_Init();

  conn->tls.ctx = wolfSSL_CTX_new(wolfTLSv1_2_client_method());
  if (conn->tls.ctx == NULL) {
    e = WOLFSSL_FAILURE;
    goto end;
  }

  e = wolfSSL_CTX_load_verify_locations(conn->tls.ctx,
                                        conn->tls.options.cafn, 0);
  if (e != WOLFSSL_SUCCESS) {
    goto end;
  }

  e = wolfSSL_CTX_use_certificate_file(conn->tls.ctx,
                                       conn->tls.options.certfn,
                                       WOLFSSL_FILETYPE_PEM);
  if (e != WOLFSSL_SUCCESS) {
    goto end;
  }

  e = wolfSSL_CTX_use_PrivateKey_file(conn->tls.ctx,
                                      conn->tls.options.keyfn,
                                      WOLFSSL_FILETYPE_PEM);
  if (e != WOLFSSL_SUCCESS) {
    goto end;
  }

  if (conn->tls.options.priorities)
    {
      if (wolfSSL_CTX_set_cipher_list(conn->tls.ctx,
                                      conn->tls.options.priorities) != WOLFSSL_SUCCESS) {
        e = WOLFSSL_FAILURE;
        goto end;
      }
    }

  conn->tls.ssl = wolfSSL_new(conn->tls.ctx);
  if (conn->tls.ssl == NULL) {
    e = WOLFSSL_FAILURE;
    goto end;
  }

  e = _riemann_client_connect_socket (client, hostname, port);
  if (e != 0) {
    client->disconnect (client);
    return e;
  }

  timeout.tv_sec = conn->tls.options.handshake_timeout / 1000;
  timeout.tv_usec = conn->tls.options.handshake_timeout % 1000;

  e = _riemann_client_set_timeout_socket (client, &timeout);
  if (e != 0) {
    e = WOLFSSL_FAILURE;
    goto end;
  }

  wolfSSL_set_fd(conn->tls.ssl, conn->s.sock);

  e = wolfSSL_connect(conn->tls.ssl);

 end:

  if (e != WOLFSSL_SUCCESS) {
    client->disconnect (client);
    return -EPROTO;
  }
  return 0;
}

int
_riemann_client_send_message_tls (riemann_client_t *client,
                                  riemann_message_t *message)
{
  uint8_t *buffer;
  size_t len;
  ssize_t sent;
  riemann_client_connection_wolfssl_t *conn =
    (riemann_client_connection_wolfssl_t *) client->connection;

  buffer = riemann_message_to_buffer (message, &len);
  if (!buffer)
    return -errno;

  sent = wolfSSL_write (conn->tls.ssl, buffer, len);
  if (sent < 0 || (size_t)sent != len)
    {
      free (buffer);
      return -EPROTO;
    }
  free (buffer);
  return 0;
}

riemann_message_t *
_riemann_client_recv_message_tls (riemann_client_t *client)
{
  uint32_t header, len;
  uint8_t *buffer;
  ssize_t received;
  riemann_message_t *message;
  riemann_client_connection_wolfssl_t *conn =
    (riemann_client_connection_wolfssl_t *) client->connection;

  received = wolfSSL_read(conn->tls.ssl, &header, sizeof (header));
  if (received != sizeof (header))
    {
      errno = EPROTO;
      return NULL;
    }
  len = ntohl (header);

  buffer = (uint8_t *) calloc (len, sizeof (uint8_t));

  received = wolfSSL_read (conn->tls.ssl, buffer, len);
  if (received != len)
    {
      free (buffer);
      errno = EPROTO;
      return NULL;
    }

  message = riemann_message_from_buffer (buffer, len);
  if (message == NULL)
    {
      int e = errno;

      free (buffer);
      errno = e;
      return NULL;
    }
  free (buffer);

  return message;
}

riemann_client_tls_library_t
riemann_client_get_tls_library (void) {
  return RIEMANN_CLIENT_TLS_LIBRARY_WOLFSSL;
}
