/* riemann/client/tls-gnutls.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "riemann/_private.h"
#include "riemann/platform.h"

#include "riemann/client/socket.h"
#include "riemann/client/tls.h"

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

typedef struct {
  riemann_client_connection_socket_t s;

  struct
  {
    gnutls_session_t session;
    gnutls_certificate_credentials_t creds;

    riemann_client_tls_options_t options;
  } tls;
} riemann_client_connection_gnutls_t;

static int
_verify_certificate_callback (gnutls_session_t session)
{
  unsigned int status;
  int ret;
  const char *hostname;
  gnutls_typed_vdata_st data[2];

  hostname = (const char *) gnutls_session_get_ptr (session);

  memset (data, 0, sizeof (data));

  data[0].type = GNUTLS_DT_DNS_HOSTNAME;
  data[0].data = (unsigned char *) hostname;

  data[1].type = GNUTLS_DT_KEY_PURPOSE_OID;
  data[1].data = (unsigned char *) GNUTLS_KP_TLS_WWW_SERVER;

  ret = gnutls_certificate_verify_peers (session, data, 2,
                                         &status);
  if (ret < 0 || status != 0)
    return GNUTLS_E_CERTIFICATE_ERROR;

   return 0;
}

static void
_tls_handshake_setup (riemann_client_t *client,
                      riemann_client_tls_options_t *tls_options)
{
  riemann_client_connection_gnutls_t *conn =
    (riemann_client_connection_gnutls_t *) client->connection;

  gnutls_transport_set_int (conn->tls.session, conn->s.sock);
  gnutls_handshake_set_timeout (conn->tls.session,
                                tls_options->handshake_timeout);
}

int
_riemann_client_disconnect_tls (riemann_client_t *client)
{
  riemann_client_connection_gnutls_t *conn =
    (riemann_client_connection_gnutls_t *) client->connection;

  if (!conn || client->type != RIEMANN_CLIENT_TLS)
    return -ENOTCONN;

  if (conn->tls.session)
    {
      gnutls_deinit (conn->tls.session);
    }

  if (conn->tls.creds)
    {
      gnutls_certificate_free_credentials (conn->tls.creds);
    }

  _riemann_client_tls_options_free (&conn->tls.options);
  return _riemann_client_disconnect_socket (client);
}

int
_riemann_client_connect_setup_tls (riemann_client_t *client,
                                   va_list aq)
{
  va_list ap;
  riemann_client_option_t option;
  riemann_client_connection_gnutls_t *conn;

  conn = (riemann_client_connection_gnutls_t *)
    calloc (1, sizeof (riemann_client_connection_gnutls_t));

  conn->s.sock = -1;
  conn->tls.options.handshake_timeout = GNUTLS_DEFAULT_HANDSHAKE_TIMEOUT;

  client->type = RIEMANN_CLIENT_TLS;
  client->send = _riemann_client_send_message_tls;
  client->recv = _riemann_client_recv_message_tls;
  client->connect = _riemann_client_connect_tls;
  client->disconnect = _riemann_client_disconnect_tls;
  client->get_fd = _riemann_client_get_fd_socket;
  client->set_timeout = _riemann_client_set_timeout_socket;
  client->connection = (void *) conn;

  va_copy (ap, aq);

  option = (riemann_client_option_t) va_arg (ap, int);
  do
    {
      switch (option)
        {
        case RIEMANN_CLIENT_OPTION_NONE:
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CA_FILE:
          _riemann_set_string (&(conn->tls.options.cafn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_CERT_FILE:
          _riemann_set_string (&(conn->tls.options.certfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_KEY_FILE:
          _riemann_set_string (&(conn->tls.options.keyfn), va_arg (ap, char *));
          break;

        case RIEMANN_CLIENT_OPTION_TLS_HANDSHAKE_TIMEOUT:
          conn->tls.options.handshake_timeout = va_arg (ap, unsigned int);
          break;

        case RIEMANN_CLIENT_OPTION_TLS_PRIORITIES:
          _riemann_set_string (&(conn->tls.options.priorities), va_arg (ap, char *));
          break;

        default:
          va_end (ap);
          goto err;
        }

      if (option != RIEMANN_CLIENT_OPTION_NONE)
        option = (riemann_client_option_t) va_arg (ap, int);
    }
  while (option != RIEMANN_CLIENT_OPTION_NONE);
  va_end (ap);

  if (!conn->tls.options.cafn || !conn->tls.options.certfn || !conn->tls.options.keyfn)
    {
      goto err;
    }

  return 0;

 err:
  _riemann_client_tls_options_free (&conn->tls.options);
  free (conn);
  client->connection = NULL;
  return -EINVAL;
}

int
_riemann_client_connect_tls (riemann_client_t *client,
                             const char *hostname,
                             int port)
{
  riemann_client_connection_gnutls_t *conn =
    (riemann_client_connection_gnutls_t *) client->connection;

  int e = _riemann_client_connect_socket (client, hostname, port);
  if (e != 0) {
    return e;
  }

  gnutls_certificate_allocate_credentials (&(conn->tls.creds));
  gnutls_certificate_set_x509_trust_file (conn->tls.creds, conn->tls.options.cafn,
                                          GNUTLS_X509_FMT_PEM);
  gnutls_certificate_set_verify_function (conn->tls.creds,
                                          _verify_certificate_callback);

  e = gnutls_certificate_set_x509_key_file (conn->tls.creds,
                                            conn->tls.options.certfn,
                                            conn->tls.options.keyfn,
                                            GNUTLS_X509_FMT_PEM);
  if (e != 0) {
    client->disconnect (client);
    return -EPROTO;
  }

  gnutls_init (&conn->tls.session, GNUTLS_CLIENT);

  if (conn->tls.options.priorities)
    {
      if (gnutls_priority_set_direct (conn->tls.session,
                                      conn->tls.options.priorities,
                                      NULL) != GNUTLS_E_SUCCESS)
        {
          e = -1;
          goto end;
        }
    }
  else
    gnutls_set_default_priority (conn->tls.session);

  gnutls_credentials_set (conn->tls.session, GNUTLS_CRD_CERTIFICATE,
                          conn->tls.creds);

  _tls_handshake_setup (client, &conn->tls.options);

  do {
    e = gnutls_handshake (conn->tls.session);
  } while (e < 0 && e != GNUTLS_E_AGAIN && gnutls_error_is_fatal (e) == 0);

 end:
  if (e != 0)
    {
      client->disconnect(client);

      return -EPROTO;
    }

  return 0;
}

int
_riemann_client_send_message_tls (riemann_client_t *client,
                                  riemann_message_t *message)
{
  uint8_t *buffer;
  size_t len;
  ssize_t sent;
  riemann_client_connection_gnutls_t *conn =
    (riemann_client_connection_gnutls_t *) client->connection;

  buffer = riemann_message_to_buffer (message, &len);
  if (!buffer)
    return -errno;

  size_t left = len;
  while (left > 0)
    {
      do
        {
          sent = gnutls_record_send (conn->tls.session, buffer + len - left, left);
        }
      while (sent == GNUTLS_E_AGAIN || sent == GNUTLS_E_INTERRUPTED);

      if (sent < 0)
        {
          free (buffer);
          return -EPROTO;
        }

      left -= sent;
    }
  free (buffer);
  return 0;
}

riemann_message_t *
_riemann_client_recv_message_tls (riemann_client_t *client)
{
  uint32_t header, len;
  uint8_t *buffer;
  ssize_t received;
  size_t left;
  riemann_message_t *message;
  riemann_client_connection_gnutls_t *conn =
    (riemann_client_connection_gnutls_t *) client->connection;

  len = sizeof (header);
  left = len;
  while (left > 0)
    {
      do
        {
          received = gnutls_record_recv (conn->tls.session, &header + len - left, left);
        }
      while (received == GNUTLS_E_AGAIN || received == GNUTLS_E_INTERRUPTED);

      if (received <= 0)
        {
          errno = EPROTO;
          return NULL;
        }

      left -= received;
    }

  len = ntohl (header);

  buffer = (uint8_t *) calloc (len, sizeof (uint8_t));

  left = len;
  while (left > 0)
    {
      do
        {
          received = gnutls_record_recv (conn->tls.session, buffer + len - left, left);
        }
      while (received == GNUTLS_E_AGAIN || received == GNUTLS_E_INTERRUPTED);

      if (received <= 0)
        {
          free (buffer);
          errno = EPROTO;
          return NULL;
        }

      left -= received;
    }

  message = riemann_message_from_buffer (buffer, len);
  if (message == NULL)
    {
      int e = errno;

      free (buffer);
      errno = e;
      return NULL;
    }
  free (buffer);

  return message;
}

riemann_client_tls_library_t
riemann_client_get_tls_library (void) {
  return RIEMANN_CLIENT_TLS_LIBRARY_GNUTLS;
}
