/* riemann/client/socket.c -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <stdarg.h>

#include "riemann/_private.h"
#include "riemann/platform.h"
#include "riemann/client/socket.h"

void
_riemann_client_connect_setup_socket (riemann_client_t *client)
{
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *)
    calloc (1, sizeof (riemann_client_connection_socket_t));

  conn->sock = -1;

  client->connection = (void *) conn;

  client->connect = _riemann_client_connect_socket;
  client->disconnect = _riemann_client_disconnect_socket;
  client->get_fd = _riemann_client_get_fd_socket;
  client->set_timeout = _riemann_client_set_timeout_socket;
}

static int
_riemann_client_disconnect_socket_only (riemann_client_t *client)
{
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  if (!conn || conn->sock == -1)
    return -ENOTCONN;

  if (conn->srv_addr)
    freeaddrinfo (conn->srv_addr);
  if (close (conn->sock) != 0)
    return -errno;

  return 0;
}

int
_riemann_client_disconnect_socket (riemann_client_t *client)
{
  if (!client || !client->connection)
    return -ENOTCONN;

  int e = _riemann_client_disconnect_socket_only (client);
  free (client->connection);
  client->connection = NULL;

  return e;
}

int
_riemann_client_get_fd_socket (riemann_client_t *client)
{
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  return conn->sock;
}

int
_riemann_client_set_timeout_socket (riemann_client_t *client,
                                    struct timeval *timeout)
{

  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  if (conn->sock < 0)
    return -EINVAL;

  if (setsockopt (conn->sock, SOL_SOCKET, SO_SNDTIMEO, timeout,
                  sizeof (struct timeval)) == -1)
    return -errno;
  if (setsockopt (conn->sock, SOL_SOCKET, SO_RCVTIMEO, timeout,
                  sizeof (struct timeval)) == -1)
    return -errno;

  return 0;
}

int
_riemann_client_connect_socket (riemann_client_t *client,
                                const char *hostname, int port)
{
  struct addrinfo hints, *res;
  int sock;
  riemann_client_connection_socket_t *conn =
    (riemann_client_connection_socket_t *) client->connection;

  memset (&hints, 0, sizeof (hints));
  if (client->type == RIEMANN_CLIENT_UDP)
    {
      hints.ai_socktype = SOCK_DGRAM;
    }
  else
    {
      hints.ai_socktype = SOCK_STREAM;
    }

  if (getaddrinfo (hostname, NULL, &hints, &res) != 0)
    {
      free (conn);
      client->connection = NULL;
      return -EADDRNOTAVAIL;
    }

  sock = socket (res->ai_family, res->ai_socktype, 0);
  if (sock == -1)
    {
      int e = errno;

      freeaddrinfo (res);
      client->disconnect (client);

      return -e;
    }

  ((struct sockaddr_in *)res->ai_addr)->sin_port = htons (port);

  if (connect (sock, res->ai_addr, res->ai_addrlen) != 0)
    {
      int e = errno;

      freeaddrinfo (res);
      client->disconnect (client);

      return -e;
    }

  _riemann_client_disconnect_socket_only (client);

  conn->sock = sock;
  conn->srv_addr = res;

  return 0;
}
