# SPDX-FileCopyrightText: 2023-2024 Gergely Nagy
# SPDX-FileContributor: Gergely Nagy
#
# SPDX-License-Identifier: EUPL-1.2

{
  description = "A C client library for the Riemann monitoring system";

  inputs = {
    nixpkgs.url = "github:NixOS/nixpkgs/nixos-23.11";
    nixpkgs-unstable.url = "github:NixOS/nixpkgs/nixos-unstable";
    systems.url = "github:nix-systems/default";
    flake-parts.url = "github:hercules-ci/flake-parts";

    pre-commit-hooks = {
      url = "github:cachix/pre-commit-hooks.nix";
      inputs.nixpkgs.follows = "nixpkgs";
      inputs.nixpkgs-stable.follows = "nixpkgs";
    };
    treefmt-nix = {
      url = "github:numtide/treefmt-nix";
      inputs.nixpkgs.follows = "nixpkgs";
    };
  };

  outputs =
    inputs@{ flake-parts, ... }:
    flake-parts.lib.mkFlake { inherit inputs; } {
      systems = import inputs.systems;
      imports = [
        inputs.pre-commit-hooks.flakeModule
        inputs.treefmt-nix.flakeModule
      ];
      perSystem =
        {
          system,
          config,
          pkgs,
          pkgs-unstable,
          self',
          ...
        }:
        {
          _module.args.pkgs = import inputs.nixpkgs { inherit system; };
          _module.args.pkgs-unstable = import inputs.nixpkgs-unstable { inherit system; };

          pre-commit = {
            settings.hooks = {
              treefmt = {
                enable = true;
                always_run = true;
              };
              reuse = {
                enable = true;
                name = "reuse";
                description = "Run REUSE compliance tests";
                entry = "${pkgs.reuse}/bin/reuse lint";
                pass_filenames = false;
                always_run = true;
              };
            };
          };
          treefmt.config = {
            projectRootFile = "flake.nix";
            programs = {
              statix.enable = true;
              deadnix.enable = true;
              shellcheck.enable = true;
              nixfmt = {
                enable = true;
                package = pkgs-unstable.nixfmt-rfc-style;
              };
            };
          };

          devShells = {
            default = pkgs.mkShell {
              buildInputs =
                self'.packages.default.buildInputs
                ++ (with pkgs; [
                  gnutls
                  libressl
                  openssl
                  reuse
                  wolfssl
                ]);
              inherit (self'.packages.default) nativeBuildInputs;

              inputsFrom = [
                config.pre-commit.devShell
                config.treefmt.build.devShell
              ];
            };
            gen-certs = pkgs.mkShell {
              buildInputs = with pkgs; [
                openssl.bin
                hostname
              ];
            };
          };

          packages =
            let
              makePackageWithTLS =
                with-tls:
                (
                  with pkgs;
                  stdenv.mkDerivation {
                    pname = "riemann_c_client";
                    version = "git";

                    nativeBuildInputs = [
                      autoreconfHook
                      check
                      pkg-config
                      lcov
                    ];
                    buildInputs = [
                      protobufc
                      json_c
                      (if with-tls == "wolfssl" then wolfssl else "")
                      (if with-tls == "gnutls" then gnutls else "")
                      (if with-tls == "openssl" then openssl else "")
                      (if with-tls == "libressl" then libressl else "")
                    ];

                    configureFlags =
                      (if with-tls == "wolfssl" then [ "--with-tls=wolfssl" ] else [ ])
                      ++ (if with-tls == "gnutls" then [ "--with-tls=gnutls" ] else [ ])
                      ++ (if with-tls == "openssl" then [ "--with-tls=openssl" ] else [ ])
                      ++ (if with-tls == "libressl" then [ "--with-tls=openssl" ] else [ ])
                      ++ (if with-tls == "no" then [ "--without-tls" ] else [ ]);

                    doCheck = true;
                    enableParallelBuilding = true;

                    src = ./.;

                    meta.mainProgram = "riemann-client";
                  }
                );
            in
            {
              default = self'.packages.no-tls;
              no-tls = makePackageWithTLS "no";
              wolfssl = makePackageWithTLS "wolfssl";
              gnutls = makePackageWithTLS "gnutls";
              openssl = makePackageWithTLS "openssl";
              libressl = makePackageWithTLS "libressl";
            };
        };
    };
}
