/*
 * nasd_edrfs_format.c
 *
 * Format a NASD partition for EDRFS.
 *
 * Authors: Jim Zelenka, Nat Lanza
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_freelist.h>
#include <nasd/nasd_itypes.h>
#include <nasd/nasd_mem.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_control.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_edrfs_dir.h>
#include <nasd/nasd_edrfs_types.h>
#include <nasd/nasd_edrfs_types_marshall.h>

char *progname;

void
usage()
{
  fprintf(stderr, "USAGE: %s [options] servername partnum\n", progname);
  fprintf(stderr, "  No options supported.\n");
  fprintf(stderr, "  We have eliminated your need to make decisions.\n");
  fprintf(stderr, "  We have eliminated your need for choice.\n");
  fprintf(stderr, "  Have a nice day. (You have no choice.)\n");
  fflush(stderr);
  exit(1);
}

int
main(
  int     argc,
  char  **argv)
{
  nasd_p_smpl_op_dr_args_t wr_args;
  nasd_p_fastwrite_dr_res_t wr_res;
  nasd_p_getattr_dr_args_t ga_args;
  nasd_p_setattr_dr_args_t sa_args;
  nasd_p_getattr_dr_res_t ga_res;
  nasd_p_setattr_dr_res_t sa_res;
  nasd_edrfs_attributes_t edrfsattr;
  nasd_ctrl_part_info_t ptinfo;
  nasd_edrfs_dirpage_t dirpage;
  char                 raw_page[NASD_EDRFS_DIRPAGE_SIZE];
  nasd_error_string_t err_str;
  nasd_rpc_status_t status;
  nasd_attribute_t *attr;
  nasd_drive_handle_t h;
  nasd_identifier_t nid;
  nasd_cookie_t cookie;
  nasd_security_param_t sp;
  char c, *server_name;
  nasd_len_t out_len;
  nasd_status_t rc;
  int partnum;

  progname = argv[0];
  partnum = (-1);

  while (nasd_getopt(argc, argv, "p:", &c)) {
    switch(c) {
      case 'p':
        if (sscanf(nasd_optarg, "%d", &partnum) != 1)
          usage();
        break;
      default:
        fprintf(stderr, "Unknown option '%c'\n", c);
        usage();
    }
  }
  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;
  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &partnum) != 1) {
    fprintf(stderr, "\"%s\" is not a valid partition number\n", argv[nasd_optind]);
    usage();
  }
  nasd_optind++;
  if (nasd_optind < argc) {
    fprintf(stderr, "Extra arguments detected on command line\n");
    usage();
  }

  if (partnum < 0) {
    fprintf(stderr, "%s: bad partition number %d\n", progname, partnum);
    fflush(stderr);
    exit(1);
  }

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr,
      "ERROR: cannot initialize client library, error 0x%x (%s)\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, NASD_PDRIVE_PORT, NASD_BIND_DEFAULT,
    NULL, 0, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s error 0x%x (%s)\n",
      server_name, rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  /* @@@ init cookie */
  bzero(&cookie, sizeof(nasd_cookie_t));
  bzero(&sp, sizeof(nasd_security_param_t));
  sp.actual_protection = NASD_NO_PROTECTION;
  rc = nasd_cl_p_ctrl_get_part_info(h, NULL, &sp, NULL, partnum, &ptinfo);
  if ((rc == NASD_CTRL_ID_CHECK_FAILED) && (ptinfo.ctrl_id == 0)) {
    /* got back a page of zeroes - no such partition */
    fprintf(stderr, "ERROR: partition %d does not exist\n", partnum);
    fflush(stderr);
    exit(1);
  }
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) getting partition info\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(1);
  }

  if (ptinfo.num_obj != 1) {
    fprintf(stderr, "ERROR: partition contains multiple objects. Reformatting\n");
    fprintf(stderr, "  is not supported.\n");
    fflush(stderr);
    exit(1);
  }

  nid = ptinfo.first_obj;

  /* @@@ init cookie */

  printf("Partition has %" NASD_64u_FMT " blocks, root is 0x%"
          NASD_ID_FMT "\n",
          ptinfo.part_size, nid);

  ga_args.in_partnum = partnum;
  ga_args.in_identifier = nid;

  nasd_cl_p_getattr_dr(h, NULL, &sp, NULL, &ga_args, &ga_res, &status);
  if (ga_res.nasd_status || status) {
    fprintf(stderr,
      "ERROR: got nasd_status 0x%x (%s) status 0x%x (%s) getting root obj status\n",
      ga_res.nasd_status, nasd_error_string(ga_res.nasd_status),
      status, nasd_cl_error_string(h, status, err_str));
    fflush(stderr);
    exit(1);
  }

  attr = &ga_res.out_attribute;
  if (attr->object_len) {
    fprintf(stderr, "Partition is not virgin: root object length %lu\n",
      (unsigned long)attr->object_len);
    fflush(stderr);
    exit(1);
  }

  if (bcmp((char *)&attr->object_create_time,
    (char *)&attr->object_modify_time, sizeof(nasd_timespec_t)))
  {
    fprintf(stderr, "Partition is not virgin: modify time != create time\n");
    fflush(stderr);
    exit(1);
  }
  if (bcmp((char *)&attr->object_create_time,
    (char *)&attr->attr_modify_time, sizeof(nasd_timespec_t)))
  {
    fprintf(stderr,
      "Partition is not virgin: attr modify time != create time\n");
    fflush(stderr);
    exit(1);
  }

  bzero((char *)&sa_args, sizeof(sa_args));
  sa_args.in_identifier = nid;
  sa_args.in_fieldmask = NASD_ATTR_FS_SPECIFIC;
  sa_args.in_partnum = partnum;
  sa_args.in_guard = 0;
  bcopy((char *)attr, (char *)&sa_args.in_attribute, sizeof(nasd_attribute_t));
  attr = &sa_args.in_attribute;
  edrfsattr.type = NASD_EDRFS_TYPE_DIR;
  edrfsattr.mode = NASD_EDRFS_S_IFDIR | NASD_EDRFS_S_IRWXU | NASD_EDRFS_S_IXGRP | NASD_EDRFS_S_IRGRP | NASD_EDRFS_S_IXOTH | NASD_EDRFS_S_IROTH;
  edrfsattr.nlink = 2;
  edrfsattr.uid = 0;
  edrfsattr.gid = 0;
  edrfsattr.clean = NASD_EDRFS_CLEAN;
  nasd_edrfs_attributes_t_marshall(&edrfsattr, (nasd_otw_base_t *)
                                   attr->fs_specific);

  nasd_cl_p_setattr_dr(h, NULL, &sp, NULL, &sa_args, &sa_res, &status);
  if (sa_res.nasd_status || status) {
    fprintf(stderr,
      "ERROR: got nasd_status 0x%x (%s) status 0x%x (%s) setting root obj status\n",
      sa_res.nasd_status, nasd_error_string(sa_res.nasd_status),
      status, nasd_cl_error_string(h, status, err_str));
    fflush(stderr);
    exit(1);
  }

  /* directory contains only "." and ".." entry */
  bzero((char *)&dirpage, sizeof(dirpage));

  nasd_edrfs_dir_init_page(&dirpage);
  nasd_edrfs_dir_add_entry(&dirpage, "..", nid, NASD_EDRFS_DIRD_TYPE_DIR);
  nasd_edrfs_dir_add_entry(&dirpage,  ".", nid, NASD_EDRFS_DIRD_TYPE_DIR);
  nasd_edrfs_dir_marshall_page(&dirpage, (void *) raw_page);

  wr_args.in_partnum = partnum;
  wr_args.in_identifier = nid;
  wr_args.in_offset = 0;
  wr_args.in_len = NASD_EDRFS_DIRPAGE_SIZE;
  sp.type = cookie.capability.type;
  sp.partnum = partnum;
  sp.actual_protection = cookie.capability.min_protection;
  nasd_cl_p_write_simple_dr(h, cookie.key, &sp, &cookie.capability,
                            &wr_args, raw_page, &wr_res, &status);
  out_len = wr_res.out_datalen;
  rc = wr_res.nasd_status;
  if (rc || status) {
    fprintf(stderr,
      "ERROR: got nasd_status 0x%x (%s) status 0x%x (%s) writing root dir\n",
      rc, nasd_error_string(rc), status,
      nasd_cl_error_string(h, status, err_str));
    fflush(stderr);
    exit(1);
  }
  if (out_len != sizeof(dirpage)) {
    fprintf(stderr, "ERROR: write %d/%d bytes for root directory\n",
      (int)out_len, (int)sizeof(dirpage));
    fflush(stderr);
    exit(1);
  }

  nasd_cl_p_sync_dr(h, &rc, &status);
  if (rc || status) {
    fprintf(stderr,
      "ERROR: got nasd_status 0x%x (%s) status 0x%x (%s) forcing drive writeback\n",
      rc, nasd_error_string(rc), status,
      nasd_cl_error_string(h, status, err_str));
    fflush(stderr);
    exit(1);
  }

  printf("Format complete.\n");

  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
