/*
 * _sio_rcobj.c
 *
 * SIO Internal Ref-Counted Objects
 *
 * Authors: Khalil Amiri, CMU SCS/ECE, July 18 1997
 *          Sean Levy, CMU SCS, July 1999
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1996,1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */

#include "_sio_internal.h"
#include <stdio.h>

void *
_sio_rcobj_create(size_t size)
{
	struct _sio_rcobj *nobj;
  int rc;

	if (size < sizeof(*nobj))
		_sio_panic("_sio_rcobj_create: size requested (%d) too small", size);
	nobj = (struct _sio_rcobj *)malloc(size);
	if (nobj == NULL)
		return (NULL);

	rc = nasd_mutex_init(&nobj->rco_mutex);
  if (rc) {
    free(nobj);
    nobj = NULL;
  } else {
    rc = nasd_cond_init(&nobj->rco_refcnt_cond);
    if (rc) {
      (void) nasd_mutex_destroy(&nobj->rco_mutex);
      free(nobj);
      nobj = NULL;
    } else
      nobj->rco_refcnt = 1;
  }

	return nobj;
}

void
_sio_rcobj_destroy(void *objv)
{
	struct _sio_rcobj *obj = (struct _sio_rcobj *)objv;
  int rc;

	if (obj->rco_refcnt != 1)
		_sio_panic("_sio_rcobj_destroy: refcnt (%d) > 1",  obj->rco_refcnt);
  rc = nasd_mutex_destroy(&obj->rco_mutex);
  if (rc)
    _sio_panic("_sio_rcobj_destroy: could not destroy mutex");
  rc = nasd_cond_destroy(&obj->rco_refcnt_cond);
  if (rc)
    _sio_panic("_sio_rcobj_destroy: could not destroy condition");
	free(obj);
}

void
_sio_rcobj_ref(void *objv)
{
	struct _sio_rcobj *obj = objv;

	NASD_LOCK_MUTEX(obj->rco_mutex);
	if (obj->rco_refcnt < 1)
		_sio_panic("_sio_rcobj_ref: refcnt (%d) < 1", obj->rco_refcnt);
	obj->rco_refcnt++;
	NASD_UNLOCK_MUTEX(obj->rco_mutex);
}

void
_sio_rcobj_unref(void *objv)
{
	struct _sio_rcobj *obj = objv;

	NASD_LOCK_MUTEX(obj->rco_mutex);
	obj->rco_refcnt--;
	if (obj->rco_refcnt < 1)
		_sio_panic("_sio_rcobj_unref: refcnt (%d) < 1", obj->rco_refcnt);
	NASD_BROADCAST_COND(obj->rco_refcnt_cond);
	NASD_UNLOCK_MUTEX(obj->rco_mutex);
}

unsigned int
_sio_rcobj_refcnt(void *objv)
{
	struct _sio_rcobj *obj = objv;

	return (obj->rco_refcnt);
}

void
_sio_rcobj_waitrefcnt(void *objv, unsigned int refcnt)
{
	struct _sio_rcobj *obj = objv;

	NASD_LOCK_MUTEX(obj->rco_mutex);
	while (obj->rco_refcnt != refcnt) {
		NASD_WAIT_COND(obj->rco_refcnt_cond, obj->rco_mutex);
	}
	NASD_UNLOCK_MUTEX(obj->rco_mutex);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
