/*  $Id: util.h,v 1.1 2012/07/08 09:35:54 sarrazip Exp $
    util.h - Utility functions.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#ifndef _H_util
#define _H_util

#include <flatzebra/RCouple.h>

#include <SDL/SDL_gfxPrimitives.h>

#include <stdlib.h>
#include <vector>
#include <iostream>  // for ready access to std::cout


inline Uint32
mapColor(Uint8 r, Uint8 g, Uint8 b, Uint8 a = 255)
{
    return ((Uint32) r << 24) | ((Uint32) g << 16) | ((Uint32) b << 8) | (Uint32) a;
}


// CAUTION: this operation is O(1) but changes the order of the vector's elements.
//
template <typename T>
inline void
removeVectorElementQuick(std::vector<T *> &vec, typename std::vector<T *>::iterator it)
{
    if (it == vec.end())  // if vector empty
        return;

    if (it + 1 != vec.end())  // if 'it' is NOT last element
        *it = vec.back();  // copy last element to spot at 'it'

    vec.resize(vec.size() - 1);
}


template <typename T>
inline void
deleteVectorElements(std::vector<T *> &v, bool nullPointersAllowed = false)
{
    for (typename std::vector<T *>::iterator it = v.begin(); it != v.end(); ++it)
    {
        assert(nullPointersAllowed || *it != NULL);
        delete *it;
    }
    v.clear();
}


// Returns a uniformly random integer between 'min' and 'max', inclusively.
// 'min' must be lower or equal to 'max'.
//
inline size_t
randSize(size_t min, size_t max)
{
    size_t num = max - min + 1;
    size_t r = (ulong(rand()) << 30) | (ulong(rand()) << 15) | rand();
    return r % num + min;
}


// Returns a uniformly random real number between 'min' and 'max', inclusively.
// 'min' must be lower or equal to 'max'.
// The number of possible values is at most RAND_MAX + 1.
//
inline double
randReal(double min, double max)
{
    return rand() / double(RAND_MAX) * (max - min) + min;;
}


inline double
square(double x)
{
    return x * x;
}


inline void
clamp(double &value, double min, double max)
{
    if (value < min)
        value = min;
    else if (value > max)
        value = max;
}


inline bool
between(double x, double min, double max)
{
    return x >= min && x <= max;
}


inline std::ostream &
operator << (std::ostream &out, const flatzebra::RCouple &c)
{
    return out << '(' << c.x << ", " << c.y << ')';
}


bool segmentIntersection(double &xi, double &yi,
                    double &coef0, double &coef1,
                    double x1, double y1,
                    double x2, double y2,
                    double x3, double y3,
                    double x4, double y4);


inline bool
segmentIntersection(const flatzebra::RCouple &a1, const flatzebra::RCouple &a2,
                    const flatzebra::RCouple &b1, const flatzebra::RCouple &b2,
                    flatzebra::RCouple &intersection)
{
    //cout << "- " << (a2 - a1).length() << endl;
    double xi, yi, coef0, coef1;
    if (!segmentIntersection(xi, yi, coef0, coef1,
                             a1.x, a1.y, a2.x, a2.y, b1.x, b1.y, b2.x, b2.y))
        return false;

    intersection.x = xi;
    intersection.y = yi;
    return true;
}


bool isPointInPolygon(const flatzebra::RCouple &pt,
                      const std::vector<flatzebra::RCouple> &polygon);


#endif  /* _H_util */
