/*  $Id: CircularPointBuffer.cpp,v 1.1 2012/07/08 09:35:52 sarrazip Exp $
    CircularPointBuffer.cpp - Fixed-size circular buffer containing 2D positions.

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "CircularPointBuffer.h"

#include "util.h"

using namespace std;
using namespace flatzebra;


void
CircularPointBuffer::addPoint(const RCouple &c)
{
    size_t indexOfNewElement = logToPhys(numElements);

    if (numElements == capacity)
    {
        removeIntersection(indexOfFirstElement);
        indexOfFirstElement = logToPhys(1);
    }
    else
        ++numElements;

    points[indexOfNewElement] = c;
}


Intersection &
CircularPointBuffer::addIntersection(const RCouple &inter,
                                     size_t indexOfFirstSegment,
                                     size_t indexOfSecondSegment)
{
    assert(indexOfFirstSegment < indexOfSecondSegment);  // logical indexes

    indexOfFirstSegment  = logToPhys(indexOfFirstSegment);
    indexOfSecondSegment = logToPhys(indexOfSecondSegment);

    intersections.push_back(new Intersection(inter, indexOfFirstSegment, indexOfSecondSegment));

    return *intersections.back();
}


#if 0
static ostream &
operator << (ostream &out, const Intersection &inter)
{
    return out << "Intersection(" << inter.point
               << ", " << inter.indexOfFirstSegment
               << ", " << inter.indexOfSecondSegment << ")";
}
#endif


bool
CircularPointBuffer::findNewestLoop(vector<RCouple> &polygon)
{
    //cout << "findNewestLoop: start: " << intersections.size() << " intersection(s)" << endl;

    if (intersections.size() == 0)
        return false;

    const Intersection &newestInter = *intersections.back();

    //cout << "  First loop point: " << newestInter << endl;
    polygon.push_back(newestInter.point);

    //cout << "  All intersections:\n";
    //for (std::vector<Intersection *>::iterator it = intersections.begin();
    //                                          it != intersections.end(); ++it)
    //    cout << "    " << *it << "\n";

    size_t logIndexFirstSeg  = physToLog(newestInter.indexOfFirstSegment);
    size_t logIndexSecondSeg = physToLog(newestInter.indexOfSecondSegment);

    assert(logIndexFirstSeg < logIndexSecondSeg);

    //cout << "  logIndexFirstSeg=" << logIndexFirstSeg << endl;
    //cout << "  logIndexSecondSeg=" << logIndexSecondSeg << endl;
    //cout << "  numElements=" << numElements << endl;

    for (size_t logIndex = logIndexFirstSeg; ; )
    {
        ++logIndex;
        //cout << "  logIndex now " << logIndex << endl;
        if (logIndex == numElements)
            break;

        polygon.push_back(getPoint(logIndex));
        //cout << "  Next loop point: " << polygon.back() << endl;

        const Intersection *inter = findIntersectionAtSegment(logIndex);
        if (inter != NULL)
        {
            //cout << "  Found intersection " << *inter << endl;

            if (*inter == newestInter)
            {
                //cout << "  This is the initial intersection. Polygon has "
                //     << polygon.size() << " vertices\n";
                return true;
            }
        }
    }

    //cout << "findNewestLoop: end: no polygon found\n";
    return false;
}


const Intersection *
CircularPointBuffer::findIntersectionAtSegment(size_t logicalIndex) const
{
    size_t physicalIndex = logToPhys(logicalIndex);

    for (std::vector<Intersection *>::const_iterator it = intersections.begin();
                                                    it != intersections.end(); ++it)
        if ((*it)->indexOfFirstSegment == physicalIndex
                    || (*it)->indexOfSecondSegment == physicalIndex)
            return *it;

    return NULL;
}


// Removes elements of 'intersections' that refer to 'indexOfRemovedSegment',
// which must be a *physical* index.
//
void
CircularPointBuffer::removeIntersection(size_t indexOfRemovedSegment)
{
    for (std::vector<Intersection *>::iterator it = intersections.begin();
                                            it != intersections.end(); )
        if ((*it)->indexOfFirstSegment == indexOfRemovedSegment
                    || (*it)->indexOfSecondSegment == indexOfRemovedSegment)
        {
            delete *it;
            removeVectorElementQuick(intersections, it);
        }
        else
            ++it;
}
