/* -*- mode: c++; indent-tabs-mode: nil -*- */
/** @file JsonRpcClient.qpp defines the JsonRpcClient class */
/*
    QC_JsonRpcClient.qpp

    Qore Programming Language

    Copyright (C) 2006 - 2019 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "qore-json-module.h"

#include "QC_JsonRpcClient.h"

#include <qore/ReferenceHolder.h>

#include "ql_json.h"

class HTTPInfoRefHelper {
protected:
    const ReferenceNode *ref;
    ExceptionSink *xsink;
    ReferenceHolder<QoreHashNode> info;

public:
    DLLLOCAL HTTPInfoRefHelper(const ReferenceNode *n_ref, QoreStringNode *msg, ExceptionSink *n_xsink) : ref(n_ref), xsink(n_xsink), info(new QoreHashNode(autoTypeInfo), xsink) {
        info->setKeyValue("request", msg, xsink);
    }
    DLLLOCAL ~HTTPInfoRefHelper() {
        // we have to create a temporary ExceptionSink if there is
        // an active exception, otherwise writing back the reference will fail
        ExceptionSink *txsink = *xsink ? new ExceptionSink : xsink;

        // write info hash to reference
        AutoVLock vl(txsink);
        QoreTypeSafeReferenceHelper rh(ref, vl, txsink);
        if (!rh)
            return;

        if (rh.assign(info.release()))
            return;

        if (txsink != xsink)
            xsink->assimilate(txsink);
    }
    DLLLOCAL QoreHashNode *operator*() {
        return *info;
    }
};

QoreValue JsonRpcClient::call(QoreStringNode *msg, QoreHashNode *info, ExceptionSink *xsink) {
    ReferenceHolder<QoreHashNode> response(send("POST", 0, 0, *msg, true, info, xsink), xsink);
    if (!response)
        return QoreValue();

    ValueHolder ans(response->takeKeyValue("body"), xsink);

    if (!ans)
        return QoreValue();

    QoreValue ah = *ans;
    if (info) {
        info->setKeyValue("response", ans.release(), xsink);
        info->setKeyValue("response_headers", response.release(), xsink);
    }

    if (ah.getType() != NT_STRING) {
        xsink->raiseException("JSONRPCCLIENT-RESPONSE-ERROR", "undecoded binary response received from remote server");
        return QoreValue();
    }

    // parse JSON-RPC response
    return parse_json(ah.get<QoreStringNode>(), xsink);
}
/* Qore class Qore::Json::JsonRpcClient */

qore_classid_t CID_JSONRPCCLIENT;
QoreClass* QC_JSONRPCCLIENT;

// hash JsonRpcClient::call(string method, ...){}
static QoreValue JsonRpcClient_call_VsVV(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 220 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
   SimpleRefHolder<QoreStringNode> msg(make_jsonrpc_request(client->getVersionStr(), args, 1, 0, xsink));
   if (!msg)
      return 0;

   return client->call(*msg, 0, xsink);
}

// hash JsonRpcClient::callArgs(string method, auto args){}
static QoreValue JsonRpcClient_callArgs_VsVA(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 191 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
   SimpleRefHolder<QoreStringNode> msg(make_jsonrpc_request_args(client->getVersionStr(), args, 1, 0, xsink));
   if (!msg)
      return 0;

   return client->call(*msg, 0, xsink);
}

// hash JsonRpcClient::callArgsWithInfo(reference info, string method, any args){}
static QoreValue JsonRpcClient_callArgsWithInfo_VrVsVa(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
# 287 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
    QoreStringNode *msg = make_jsonrpc_request_args(client->getVersionStr(), args, 1, 1, xsink);
    if (!msg)
        return QoreValue();

    HTTPInfoRefHelper irh(info, msg, xsink);

    // send the message to the server and get the response as a JSON string
    return client->call(msg, *irh, xsink);
}

// hash JsonRpcClient::callId(any id, string method, ...){}
static QoreValue JsonRpcClient_callId_VaVsVV(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    QoreValue id = get_param_value(args, 0);
# 250 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
   SimpleRefHolder<QoreStringNode> msg(make_jsonrpc_request(client->getVersionStr(), args, id, 1, xsink));
   if (!msg)
      return 0;

   return client->call(*msg, 0, xsink);
}

// hash JsonRpcClient::callIdArgsWithInfo(reference info, any id, string method, any args){}
static QoreValue JsonRpcClient_callIdArgsWithInfo_VrVaVsVa(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
    QoreValue id = get_param_value(args, 1);
# 328 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
    QoreStringNode *msg = make_jsonrpc_request_args(client->getVersionStr(), args, id, 1, xsink);
    if (!msg)
        return QoreValue();

    HTTPInfoRefHelper irh(info, msg, xsink);

    // send the message to the server and get the response as a JSON string
    return client->call(msg, *irh, xsink);
}

// hash JsonRpcClient::callIdWithInfo(reference info, any id, string method, ...){}
static QoreValue JsonRpcClient_callIdWithInfo_VrVaVsVV(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
    QoreValue id = get_param_value(args, 1);
# 407 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
    QoreStringNode *msg = make_jsonrpc_request(client->getVersionStr(), args, id, 1, xsink);
    if (!msg)
        return QoreValue();

    HTTPInfoRefHelper irh(info, msg, xsink);

    // send the message to the server and get the response as a JSON string
    return client->call(msg, *irh, xsink);
}

// hash JsonRpcClient::callWithInfo(reference info, string method, ...){}
static QoreValue JsonRpcClient_callWithInfo_VrVsVV(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const ReferenceNode* info = HARD_QORE_VALUE_REF(args, 0);
# 367 "QC_JsonRpcClient.qpp"
    // create the outgoing message in JSON-RPC call format
    QoreStringNode *msg = make_jsonrpc_request(client->getVersionStr(), args, 1, 1, xsink);
    if (!msg)
        return QoreValue();

    HTTPInfoRefHelper irh(info, msg, xsink);

    // send the message to the server and get the response as a JSON string
    return client->call(msg, *irh, xsink);
}

// JsonRpcClient::constructor() {}
static void JsonRpcClient_constructor(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 132 "QC_JsonRpcClient.qpp"
self->setPrivate(CID_JSONRPCCLIENT, new JsonRpcClient);
}

// JsonRpcClient::constructor(hash opts, softbool no_connect = False) {}
static void JsonRpcClient_constructor_Vhvb(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* opts = HARD_QORE_VALUE_HASH(args, 0);
    bool no_connect = HARD_QORE_VALUE_BOOL(args, 1);
# 160 "QC_JsonRpcClient.qpp"
self->setPrivate(CID_JSONRPCCLIENT, new JsonRpcClient(opts, no_connect, xsink));
}

// JsonRpcClient::copy() {}
static void JsonRpcClient_copy(QoreObject* self, QoreObject* old, JsonRpcClient* client, ExceptionSink* xsink) {
# 167 "QC_JsonRpcClient.qpp"
xsink->raiseException("JSONRPCCLIENT-COPY-ERROR", "copying JsonRpcClient objects is not yet supported.");
}

// string JsonRpcClient::getVersion(){}
static QoreValue JsonRpcClient_getVersion(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 458 "QC_JsonRpcClient.qpp"
    QoreStringNode* str = new QoreStringNode(QCS_UTF8);
    client->getVersion(*str);
    return str;
}

// nothing JsonRpcClient::setEventQueue(){}
static QoreValue JsonRpcClient_setEventQueue(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 425 "QC_JsonRpcClient.qpp"
    client->setEventQueue(0, xsink);
    return QoreValue();
}

// nothing JsonRpcClient::setEventQueue(Queue queue){}
static QoreValue JsonRpcClient_setEventQueue_C5Queue(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    HARD_QORE_VALUE_OBJ_DATA(queue, Queue, args, 0, CID_QUEUE, "JsonRpcClient::setEventQueue()", "Queue", xsink);
    if (*xsink)
        return 0;
# 437 "QC_JsonRpcClient.qpp"
    // pass reference from QoreObject::getReferencedPrivateData() to function
    client->setEventQueue(queue, xsink);
    return QoreValue();
}

// nothing JsonRpcClient::setVersion(string version){}
static QoreValue JsonRpcClient_setVersion_Vs(QoreObject* self, JsonRpcClient* client, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* version = HARD_QORE_VALUE_STRING(args, 0);
# 447 "QC_JsonRpcClient.qpp"
    TempEncodingHelper tstr(version, QCS_UTF8, xsink);
    if (tstr)
        client->setVersion(tstr->c_str());
    return QoreValue();
}

DLLLOCAL void preinitJsonRpcClientClass() {
    QC_JSONRPCCLIENT = new QoreClass("JsonRpcClient", "::Qore::Json::JsonRpcClient", QDOM_DEFAULT);
    CID_JSONRPCCLIENT = QC_JSONRPCCLIENT->getID();
    QC_JSONRPCCLIENT->setSystem();
}

DLLLOCAL QoreClass* initJsonRpcClientClass(QoreNamespace& ns) {
    if (!QC_JSONRPCCLIENT)
        preinitJsonRpcClientClass();

    // set parent class
    assert(QC_HTTPCLIENT);
    QC_JSONRPCCLIENT->addBuiltinVirtualBaseClass(QC_HTTPCLIENT);

    // hash JsonRpcClient::call(string method, ...){}
    QC_JSONRPCCLIENT->addMethod("call", (q_method_n_t)JsonRpcClient_call_VsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // hash JsonRpcClient::callArgs(string method, auto args){}
    QC_JSONRPCCLIENT->addMethod("callArgs", (q_method_n_t)JsonRpcClient_callArgs_VsVA, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 2, stringTypeInfo, QORE_PARAM_NO_ARG, "method", autoTypeInfo, QORE_PARAM_NO_ARG, "args");

    // hash JsonRpcClient::callArgsWithInfo(reference info, string method, any args){}
    QC_JSONRPCCLIENT->addMethod("callArgsWithInfo", (q_method_n_t)JsonRpcClient_callArgsWithInfo_VrVsVa, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 3, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");

    // hash JsonRpcClient::callId(any id, string method, ...){}
    QC_JSONRPCCLIENT->addMethod("callId", (q_method_n_t)JsonRpcClient_callId_VaVsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 2, anyTypeInfo, QORE_PARAM_NO_ARG, "id", stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // hash JsonRpcClient::callIdArgsWithInfo(reference info, any id, string method, any args){}
    QC_JSONRPCCLIENT->addMethod("callIdArgsWithInfo", (q_method_n_t)JsonRpcClient_callIdArgsWithInfo_VrVaVsVa, Public, QCF_NO_FLAGS, QDOM_DEFAULT, hashTypeInfo, 4, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", anyTypeInfo, QORE_PARAM_NO_ARG, "id", stringTypeInfo, QORE_PARAM_NO_ARG, "method", anyTypeInfo, QORE_PARAM_NO_ARG, "args");

    // hash JsonRpcClient::callIdWithInfo(reference info, any id, string method, ...){}
    QC_JSONRPCCLIENT->addMethod("callIdWithInfo", (q_method_n_t)JsonRpcClient_callIdWithInfo_VrVaVsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 3, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", anyTypeInfo, QORE_PARAM_NO_ARG, "id", stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // hash JsonRpcClient::callWithInfo(reference info, string method, ...){}
    QC_JSONRPCCLIENT->addMethod("callWithInfo", (q_method_n_t)JsonRpcClient_callWithInfo_VrVsVV, Public, QCF_USES_EXTRA_ARGS, QDOM_DEFAULT, hashTypeInfo, 2, referenceTypeInfo, QORE_PARAM_NO_ARG, "info", stringTypeInfo, QORE_PARAM_NO_ARG, "method");

    // JsonRpcClient::constructor() {}
    QC_JSONRPCCLIENT->addConstructor(JsonRpcClient_constructor, Public, QCF_NO_FLAGS, QDOM_DEFAULT);

    // JsonRpcClient::constructor(hash opts, softbool no_connect = False) {}
    QC_JSONRPCCLIENT->addConstructor(JsonRpcClient_constructor_Vhvb, Public, QCF_NO_FLAGS, QDOM_DEFAULT, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "opts", softBoolTypeInfo, QoreSimpleValue().assign(false), "no_connect");

    // JsonRpcClient::copy() {}
    QC_JSONRPCCLIENT->setCopy((q_copy_t)JsonRpcClient_copy);

    // string JsonRpcClient::getVersion(){}
    QC_JSONRPCCLIENT->addMethod("getVersion", (q_method_n_t)JsonRpcClient_getVersion, Public, QCF_CONSTANT, QDOM_DEFAULT, stringTypeInfo);

    // nothing JsonRpcClient::setEventQueue(){}
    QC_JSONRPCCLIENT->addMethod("setEventQueue", (q_method_n_t)JsonRpcClient_setEventQueue, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo);

    // nothing JsonRpcClient::setEventQueue(Queue queue){}
    QC_JSONRPCCLIENT->addMethod("setEventQueue", (q_method_n_t)JsonRpcClient_setEventQueue_C5Queue, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, QC_QUEUE->getTypeInfo(), QORE_PARAM_NO_ARG, "queue");

    // nothing JsonRpcClient::setVersion(string version){}
    QC_JSONRPCCLIENT->addMethod("setVersion", (q_method_n_t)JsonRpcClient_setVersion_Vs, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "version");

    return QC_JSONRPCCLIENT;
}
