# Helper functions for fsbuild
# This file is automatically generated by fs-package

import os
import platform
import subprocess
import sys
import time
from os import path
import xml.etree.ElementTree as ET
from typing import Dict, List, Optional


class PackageInformation:
    def __init__(self):
        self.values: Dict[str, str] = {}

    @property
    def bundleId(self) -> str:
        return self.values.get(
            "PACKAGE_BUNDLE_ID", self.values.get("PACKAGE_MACOS_BUNDLE_ID", "")
        )

    @property
    def displayName(self) -> str:
        """E.g. FS-UAE Launcher."""
        return self.values.get("PACKAGE_DISPLAY_NAME", self.prettyName)

    @property
    def name(self) -> str:
        """E.g. fs-uae-launcher."""
        return self.values["PACKAGE_NAME"]

    @property
    def prettyName(self) -> str:
        """E.g. FS-UAE-Launcher."""
        return self.values["PACKAGE_NAME_PRETTY"]

    @property
    def type(self):
        return self.values["PACKAGE_TYPE"]

    @property
    def version(self):
        return self.values["PACKAGE_VERSION"]


_packageInformation: Optional[PackageInformation] = None


def checkNotarizationResult(requestUuid: str):
    for _ in range(60):
        time.sleep(10.0)
        result = run(
            [
                "xcrun",
                "altool",
                "--notarization-info",
                requestUuid,
                "-u",
                getNotarizationUserName(),
                "-p",
                "@env:NOTARIZATION_PASSWORD",
                "-itc_provider",
                getNotarizationItcProvider(),
                "--output-format",
                "xml",
            ]
        )
        if "<string>success</string>" in result:
            break
        elif "<string>in progress</string>" in result:
            print("in progress...")
            continue
        else:
            print(result)
            raise Exception("Unknown notarization result")
    else:
        raise Exception("Gave up notarization")


def getPackageInformation() -> PackageInformation:
    global _packageInformation
    if _packageInformation is None:
        _packageInformation = PackageInformation()
        with open("PACKAGE.FS", "r") as f:
            for line in f:
                try:
                    key, value = line.split("=", 1)
                    key = key.strip()
                    value = value.strip()
                    value = value.strip('"')
                    _packageInformation.values[key] = value
                except ValueError:
                    pass
    return _packageInformation


def getAppName() -> str:
    return getPackageInformation().displayName + ".app"


def getArchitecture() -> str:
    machine = platform.machine()
    if machine == "arm64":
        arch = "ARM64"
    else:
        arch = "x86-64"  # FIXME
    return arch


def getBundleName() -> str:
    packageInformation = getPackageInformation()
    if packageInformation.type == "fs-library-plugin":
        return getFrameworkName()
    else:
        return getAppName()


def getBundlePath(prefix: str = "fsbuild/_build/") -> str:
    prettyName = getPackageInformation().prettyName
    bundleName = getBundleName()
    arch = getArchitecture()
    path = f"{prefix}{prettyName}/macOS/{arch}/{bundleName}"
    return path


def getDmgPath() -> str:
    packageInformation = getPackageInformation()
    prettyName = packageInformation.prettyName
    version = packageInformation.version
    arch = getArchitecture()
    osDist = getOperatingSystemDist()
    path = f"fsbuild/_dist/{prettyName}_{version}_{osDist}_{arch}.dmg"
    return path


def getFrameworkName() -> str:
    return getPackageInformation().prettyName + ".framework"


def getNotarizationItcProvider() -> str:
    return os.environ.get("NOTARIZATION_PROVIDER", "")


def getNotarizationUserName() -> str:
    return os.environ.get("NOTARIZATION_USERNAME", "")


def getOperatingSystemDist() -> str:
    envValue = os.environ.get("SYSTEM_OS_DIST", "")
    if envValue:
        return envValue
    elif sys.platform == "linux":
        return "Linux"
    elif sys.platform == "darwin":
        return "macOS"
    elif sys.platform == "win32":
        return "Windows"
    return "Unknown"


def isMacOS() -> bool:
    return sys.platform == "darwin"


def notarizeApp(pathToNotarize: str, bundleId: str) -> str:
    assert path.exists(pathToNotarize)
    print(f"Notarizing {path.basename(pathToNotarize)}")

    result = shell(
        "xcrun altool --notarize-app -t osx "
        "-f {pathToNotarize} "
        "--primary-bundle-id {bundleId} "
        "-u {appleIdUser} "
        "-p @env:NOTARIZATION_PASSWORD "
        "-itc_provider {itcProvider} "
        "--output-format xml".format(
            appleIdUser=getNotarizationUserName(),
            bundleId=bundleId,
            itcProvider=getNotarizationItcProvider(),
            pathToNotarize=pathToNotarize,
        )
    )
    print(result)
    root = ET.fromstring(result)
    dictNode = root.find("dict")
    assert dictNode is not None
    dictNode2 = dictNode.find("dict")
    assert dictNode2 is not None
    stringNode = dictNode2.find("string")
    assert stringNode is not None
    requestUuid = stringNode.text
    assert requestUuid
    print(requestUuid)
    return requestUuid


def run(args: List[str]) -> str:
    print(quoteArgs(args))
    # p = subprocess.Popen(args, stdout = subprocess.PIPE)
    # assert p.wait() == 0
    # assert p.stdout
    # output = p.stdout.read().decode("UTF-8")
    # assert p.wait() == 0
    # return output
    return subprocess.check_output(args).decode("UTF-8")


def runCodeSign(args: List[str]) -> None:
    # Signing sometimes fails due to Apple errors (timeouts, etc). So we try
    # multiple times before giving up.
    for i in range(20):
        try:
            shell(quoteArgs(args))
        except Exception:
            time.sleep(1.0 * i)
            print("Attempt", i + 2)
        else:
            break
    else:
        raise Exception("Giving up signing")


def quoteArg(arg: str) -> str:
    if " " in arg:
        return f'"{arg}"'
    return arg


def quoteArgs(args: List[str]) -> str:
    return " ".join(f"{quoteArg(a)}" for a in args)


def shell(cmd: str) -> str:
    print(cmd)
    return subprocess.run(
        cmd, shell=True, check=True, stdout=subprocess.PIPE
    ).stdout.decode("UTF-8")
