/* pslib.c 
   This program is a wrapper for pslib in Python.
   Written by Stefan Schroeder (stefan at fkp.uni-hannover.de)
   (C) GPL 4/2004, read license at gnu.org.
   pslib implements almost all functions of pslib 0.1.15-1 as 
   shipped with Debian Sarge 5/2004. You can use the manpages
   that come with the C-library.

   read README for more information.
   
*/

#include <Python.h>
#include <libps/pslib.h>

/*
DRAWING
Done PS_arc (3)           - Draws an arc
Done PS_arcn (3)          - Draws an arc
Done PS_circle (3)        - Draws a circle
Done PS_curveto (3)       - Draws a curve
Done PS_lineto (3)        - Draws line
Done PS_rect (3)          - Draws a rectangle

Done PS_fill (3)          - Fill the current path
Done PS_fill_stroke (3)   - Fills and strokes the current path

Done PS_open_image (3)    - Reads an image for later placement
Done PS_open_image_file (3) - Open image from file
Done PS_place_image (3)   - Places image on the page
Done PS_close_image (3)   - Close image an free memory

Done PS_setcolor (3)      - Sets color
Done PS_setdash (3)       - Sets appearance of a dashed line
Done PS_setflat (3)       - (unbekanntes Thema)
Done PS_setgray (3)       - Sets gray value
Done PS_setlinecap (3)    - Sets appearance of line ends
Done PS_setlinejoin (3)   - Sets how connected lines are joined
Done PS_setlinewidth (3)  - Set width of line
Done PS_setmiterlimit (3) - (unbekanntes Thema)
Done PS_setpolydash (3)   - Sets appearance of a dashed line

MANAGEMENT
PS_open_fp (3)       - Uses an already open file pointer as the output file
Done PS_begin_page (3)    - Start a new page
Done PS_boot (3)          - Initialize library
Done PS_close (3)         - Closes a PostScript document
Done PS_delete (3)        - Deletes all resources of a PostScript document
Done PS_end_page (3)      - End a page
Done PS_get_buffer - (3) [PS_get_buffer] - (unbekanntes Thema)
Done PS_get_majorversion (3) - returns the major version number of the PostScript library
Done PS_get_minorversion (3) - returns the minor version number of the PostScript library
Done PS_get_parameter (3) - Sets certain parameters
Done PS_get_subminorversion (3) - returns the subminor version number of the PostScript library
Done PS_get_value (3)     - Sets certain parameters
Done PS_list_parameters (3) - List all set parameters
Done PS_list_resources (3) - List all resources
Done PS_list_values (3)   - List all set values
Done PS_new (3)           - Creating a new PostScript document object
Done PS_open_file (3)     - Opens a file for output
Done PS_restore (3)       - Restore previously save context
Done PS_save (3)          - Save current context
Done PS_set_info (3)      - Sets information fields of document
Done PS_set_parameter (3) - Sets certain parameters
Done PS_set_value (3)     - Sets certain values
Done PS_shutdown (3)      - Final clean up of library

These more esoteric functions are rather difficult to implement.
??? PS_get_opaque (3)    - Returns pointer which has been passed to PS_new2()
??? PS_new2 (3)          - Creating a new PostScript document object
??? PS_open_mem (3)      - Creates a PostScript document in memory

PATH-HANDLING
Done PS_moveto (3)        - Set current point
Done PS_clip (3)          - Clips drawing to current path
Done PS_closepath (3)     - Closes path
Done PS_closepath_stroke (3) - Closes path
Done PS_stroke (3)        - Draws the current path

FONT&TEXT
Done PS_findfont (3)      - Loads a font
Done PS_continue_text (3) - Continue text in next line
Done PS_continue_text2 (3) - (Output text)
Done PS_show (3)          - Output text at current position
Done PS_show2 (3)         - Output text with given length
Done PS_show_boxed (3)    - Output text in a box
Done PS_show_xy (3)       - Output text at given position
Done PS_show_xy2 (3)      - Output text at given position
Done PS_setfont (3)       - Sets font to use for following output
Done PS_stringwidth (3)   - Get width of a string
Done PS_stringwidth2 (3)  - Get width of a string
Done PS_string_geometry (3) - Get geometry of a string

TRANSFORMATION
Done PS_translate (3)     - Sets translation
Done PS_rotate (3)        - Sets rotation factor
Done PS_scale (3)         - Sets scaling factor

HYPERREFS:
Done PS_set_border_color (3) - Sets color of border for annotations
Done PS_set_border_dash (3) - Sets length of dashes for border of annotations
Done PS_set_border_style (3) - Sets border style of annotations
Done PS_add_bookmark   - (3) [PS_add_bookmark] - (unbekanntes Thema)
Done PS_add_launchlink - (3) [PS_add_launchlink] - (unbekanntes Thema)
Done PS_add_locallink  - (3) [PS_add_locallink] - (unbekanntes Thema)
Done PS_add_note       - (3) [PS_add_note] - (unbekanntes Thema)
Done PS_add_pdflink    - (3) [PS_add_pdflink] - (unbekanntes Thema)
Done PS_add_weblink    - (3) [PS_add_weblink] - (unbekanntes Thema)

*/


/******************************
 FUNCTIONS with no arguments 
 ******************************/

/********************
PY_get_buffer ()
PS_get_buffer
********************/

static PyObject *PY_get_buffer(PyObject *self, PyObject *args)
{
    int num;
    long size;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num, &size))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_get_buffer(ps, size);

    return Py_BuildValue("s", "not yet implemented");
}

/********************
PY_restore () Restore previously save context
PS_restore -- 
********************/

static PyObject *PY_restore(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_restore(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_save ()  Save current context
PS_save -- 
********************/

static PyObject *PY_save(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_save(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_get_opaque ()
PS_get_opaque -- Closes a PostScript document
********************/

static PyObject *PY_get_opaque(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_get_opaque(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_endpage ()
PS_endpage -- Closes a PostScript document
********************/

static PyObject *PY_end_page(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_end_page(ps);

    return Py_BuildValue("i", 0);
}
/********************
PY_delete ()
PS_delete --Deletes all resources of a PostScript document
********************/
static PyObject *PY_delete(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_delete(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_boot ()
PS_boot -- Initialize library
********************/

static PyObject *PY_boot(PyObject *self, PyObject *args)
{
    PS_boot();

    return Py_BuildValue("i", 0);
}

/********************
PY_get_majorversion ()
PS_get_majorversion -- returns  the  major  version  number  of  the
       PostScript library
********************/

static PyObject *PY_get_majorversion(PyObject *self, PyObject *args)
{
    int ret;
    ret = PS_get_majorversion();

    return Py_BuildValue("i", ret);
}

/********************
PY_get_minorversion ()
PS_get_minorversion -- returns  the  minor  version  number  of  the
       PostScript library
********************/

static PyObject *PY_get_minorversion(PyObject *self, PyObject *args)
{
    int ret;
    ret = PS_get_minorversion();

    return Py_BuildValue("i", ret);
}

/********************
PY_get_subminorversion ()
PS_get_subminorversion -- returns  the  subminor  version  number  of  the
       PostScript library
********************/

static PyObject *PY_get_subminorversion(PyObject *self, PyObject *args)
{
    int ret;
    ret = PS_get_subminorversion();

    return Py_BuildValue("i", ret);
}

/********************
PY_shutdown ()
PS_shutdown -- Final clean up of library
********************/

static PyObject *PY_shutdown(PyObject *self, PyObject *args)
{
    PS_shutdown();

    return Py_BuildValue("i", 0);
}

/********************
PY_fill ()
PS_fill -- Fills current path
********************/

static PyObject *PY_fill(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_fill(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_clip ()
PS_clip -- Clips drawing to current path
********************/

static PyObject *PY_clip(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_clip(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_closepath ()
PS_closepath -- Closes path
********************/

static PyObject *PY_closepath(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_closepath(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_closepath_stroke ()
PS_closepath_stroke -- Closes path and draws
********************/

static PyObject *PY_closepath_stroke(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_closepath_stroke(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_fill_stroke ()
PS_fill_stroke -- Fills and strokes current path
********************/

static PyObject *PY_fill_stroke (PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_fill_stroke(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_stroke ()
********************/
static PyObject *PY_stroke(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_stroke(ps);

    return Py_BuildValue("i", 0);
}

/******************************
 FUNCTIONS with one argument.
 ******************************/

/********************
PY_new () filename
PS_new -- Creating a new PostScript document object
********************/

static PyObject *PY_new(PyObject *self, PyObject *args)
/* Takes a string = filename */
{
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, ""))
        return NULL;

    /* functional block */
    ps = PS_new();

    return Py_BuildValue("i", ps);
}

/********************
PY_close ("l")
PS_close -- Closes a PostScript document
********************/

static PyObject *PY_close(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_close(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_list_parameters ("l") List all set parameters to STDOUT.
PS_list_parameters -- For debugging.
********************/
static PyObject *PY_list_parameters(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_list_parameters(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_list_resources ("l") List all resources
PS_list_resources -- For debugging.
********************/
static PyObject *PY_list_resources(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_list_resources(ps);

    return Py_BuildValue("i", 0);
}

/********************
PY_open_file ("ls") filename
PS_open_file --  Opens a file for output
********************/

static PyObject *PY_open_file(PyObject *self, PyObject *args)
/* Takes a string = filename */
{
    PSDoc *ps;

    char *filename;
    int ret, num;
    
    if (!PyArg_ParseTuple(args, "ls", &num, &filename))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_open_file(ps, filename );

    return Py_BuildValue("i", ret);
}

/********************
PY_open_fp ("lO") File-object
PS_open_fp --  Takes a file-object to write to.
********************/

static PyObject *PY_open_fp(PyObject *self, PyObject *args)
{
    PSDoc *ps;
    int num, ret;
    FILE *c_fp;
    PyObject *file=NULL;

    if (!PyArg_ParseTuple(args, "lO", &num, &file))
        return NULL;

    ps = (PSDoc*) num;    
    c_fp = PyFile_AsFile(file);

    /* functional block */
    ret = PS_open_fp(ps, c_fp);

    return Py_BuildValue("i", ret);
}

/********************
PY_list_values ("l") List all values
PS_list_values -- For debugging.
********************/
static PyObject *PY_list_values(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_list_values(ps);

    return Py_BuildValue("i", 0);
}

/******************************
 FUNCTIONS with two arguments 
 ******************************/

/********************
PY_setpolydash ("l(f)") tupel with floats 
PS_setpolydash -- Sets appearance of a dashed line
********************/
static PyObject *PY_setpolydash(PyObject *self, PyObject *args)
{
/*This is a rather unelegant way to handle a tupel,
  but I didnt understand UnpackTuple at first glance,
  and I wanted to have it up and running quickly.
  Here we have min/max = 2/10.
  0.0 is an identifier, because no one would set 0.0 as 
  length for a black or white part.
*/
    int num;
    float r[10] = {0.0, 0.0, 0.0, 0.0, 0.0, 
    		0.0, 0.0, 0.0, 0.0, 0.0 };
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff|ffffffff", &num, &r[0],
    	&r[1], &r[2],&r[3], &r[4],&r[5], &r[6],&r[7], &r[8], &r[9]))
        return NULL;

    ps = (PSDoc*) num;    
    /* find number of elements */
    int i = 0;
    while (r[i] != 0) i++;
    
    /* functional block */
    PS_setpolydash(ps, r, i);

    return Py_BuildValue("i", 0);
}

/********************
PY_setmiterlimit ("lf") maximum size of miter 
PS_setmiterlimit -- Maximum size of miter.
********************/
static PyObject *PY_setmiterlimit(PyObject *self, PyObject *args)
{
    int num;
    float r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lf", &num, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setmiterlimit(ps, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_setlinewidth ("lf") width
PS_setlinewidth -- Set width of line
********************/
static PyObject *PY_setlinewidth(PyObject *self, PyObject *args)
{
    int num;
    float r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lf", &num, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setlinewidth(ps, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_setlinecap ("li") Linetype
PS_setlinecap --  Sets appearance of line ends
********************/
static PyObject *PY_setlinecap(PyObject *self, PyObject *args)
{
    int num;
    int t;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "li", &num, &t))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setlinecap(ps, t);

    return Py_BuildValue("i", 0);
}

/********************
PY_deletefont ("li") fontid
PS_deletefont --  Delete Font
********************/
static PyObject *PY_deletefont(PyObject *self, PyObject *args)
{
    int num;
    int fontid;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "li", &num, &fontid))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_deletefont(ps, fontid);

    return Py_BuildValue("i", 0);
}

/********************
PY_close_image ("li") imageid
PS_close_image -- Close image an free memory
********************/
static PyObject *PY_close_image(PyObject *self, PyObject *args)
{
    int num;
    int imageid;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "li", &num, &imageid))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_close_image(ps, imageid);

    return Py_BuildValue("i", 0);
}

/********************
PY_setlinejoin ("li") Linetype
PS_setlinejoin --  Sets how connected lines are joined
********************/
static PyObject *PY_setlinejoin(PyObject *self, PyObject *args)
{
    int num;
    int t;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "li", &num, &t))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setlinejoin(ps, t);

    return Py_BuildValue("i", 0);
}

/********************
PY_setgray ("lf") gray value
PS_setgray -- Sets gray value
********************/
static PyObject *PY_setgray(PyObject *self, PyObject *args)
{
    int num;
    float r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lf", &num, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setgray(ps, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_setflat ("lf") No documentation
PS_setflat -- 
********************/
static PyObject *PY_setflat(PyObject *self, PyObject *args)
{
    int num;
    float r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lf", &num, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setflat(ps, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_rotate ("lf") r angle
PS_rotate -- Set rotation
********************/
static PyObject *PY_rotate(PyObject *self, PyObject *args)
{
    int num;
    float r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lf", &num, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_rotate(ps, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_continue_text ("ls") text
PS_continue_text --  Continue text in next line
********************/
static PyObject *PY_continue_text(PyObject *self, PyObject *args)
{
    int num;
    char *line;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "ls", &num, &line))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_continue_text(ps, line);

    return Py_BuildValue("i", 0);
}

/********************
PY_show ("ls") text
PS_show -- Output text at current position
********************/
static PyObject *PY_show(PyObject *self, PyObject *args)
{
    int num;
    char *line;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "ls", &num, &line))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_show(ps, line);

    return Py_BuildValue("i", 0);
}

/******************************
 FUNCTIONS with three arguments 
 ******************************/

/********************
PY_show2 ("lsi") text, length
PS_show2 -- Output text at current position with given length
********************/
static PyObject *PY_show2(PyObject *self, PyObject *args)
{
    int num;
    int xlen;
    char *line;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsi", &num, &line, &xlen))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_show2(ps, line, xlen);

    return Py_BuildValue("i", 0);
}

/********************
PY_continue_text2 ("lsi") text, xlen
PS_continue_text2 --  Continue text in next line with length
********************/
static PyObject *PY_continue_text2(PyObject *self, PyObject *args)
{
    int num;
    int xlen;
    char *line;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsi", &num, &line, &xlen))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_continue_text2(ps, line, xlen);

    return Py_BuildValue("i", 0);
}

/********************
PY_moveto ("lff") x, y coordinates
PS_moveto -- Set current point
********************/
static PyObject *PY_moveto(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_moveto(ps, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_begin_page ("lff") width, height coordinates
PS_begin_page -- Start a new page
********************/
static PyObject *PY_begin_page(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_begin_page(ps, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_setdash ("lff") on, off proportions. 
PS_setdash -- Sets appearance of a dashed line
********************/
static PyObject *PY_setdash(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setdash(ps, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_scale ("lff") x, y coordinates
PS_scale -- Sets scaling factor
********************/
static PyObject *PY_scale(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_scale(ps, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_translate ("lff") x, y coordinates
PS_translate -- Reset coordinate system
********************/
static PyObject *PY_translate(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_translate(ps, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_lineto ("lff") x, y coordinates
PS_lineto -- Draws line
********************/
static PyObject *PY_lineto(PyObject *self, PyObject *args)
{
    int num;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lff", &num, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_lineto(ps, x, y);

    return Py_BuildValue("i", 0);
}


/********************
PY_get_value ("lsf") paramter name, modifier
PS_get_value -- Sets certain parameters
********************/
static PyObject *PY_get_value(PyObject *self, PyObject *args)
{
    int num;
    float x;
    float ret;
    char *parameter;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsf", &num, &parameter, &x))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_get_value(ps, parameter, x);

    return Py_BuildValue("f", ret);
}

/********************
PY_set_border_style ("lsf") style, width
PS_set_border_style -- Sets border style of annotations
********************/
static PyObject *PY_set_border_style(PyObject *self, PyObject *args)
{
    int num;
    float width;
    char *style;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsf", &num, &style, &width))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_border_style(ps, style, width);

    return Py_BuildValue("i", 0);
}

/********************
PY_set_value ("lsf") parameter key, value
PS_set_value -- Sets certain values
********************/
static PyObject *PY_set_value(PyObject *self, PyObject *args)
{
    int num;
    float x;
    char *parameter;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsf", &num, &parameter, &x))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_value(ps, parameter, x);

    return Py_BuildValue("i", 0);
}

/********************
PY_set_info ("lss") key, value
PS_set_info -- Sets information fields of document
********************/
static PyObject *PY_set_info(PyObject *self, PyObject *args)
{
    int num;
    char *key, *value;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lss", &num, &key, &value))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_info(ps, key, value);

    return Py_BuildValue("i", 0);
}

/********************
PY_set_parameter ("lss") parameter, value
PS_set_parameter -- Sets certain parameters
********************/
static PyObject *PY_set_parameter(PyObject *self, PyObject *args)
{
    int num;
    char *parameter, *value;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lss", &num, &parameter, &value))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_parameter(ps, parameter, value);

    return Py_BuildValue("i", 0);
}
/********************
PY_get_parameter ("lsf") parameter, modifier
PS_get_parameter -- Gets certain parameters
********************/
static PyObject *PY_get_parameter(PyObject *self, PyObject *args)
{
    int num;
    float modifier;
    char *name;
    char *ret;
    PSDoc *ps;
    /* Must I do some mallocation here? */

    if (!PyArg_ParseTuple(args, "lsf", &num, &name, &modifier))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = (char*)PS_get_parameter(ps, name, modifier);

    return Py_BuildValue("s", ret);
}

/********************
PY_setfont ("lif") fontid, size
PS_setfont -- Sets font to use for following output
********************/
static PyObject *PY_setfont(PyObject *self, PyObject *args)
{
    int num, fontid;
    float s;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lif", &num, &fontid, &s))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setfont(ps, fontid, s);

    return Py_BuildValue("i", 0);
}

/******************************
 FUNCTIONS with four and more arguments 
 ******************************/

/********************
PY_show_boxed ("lsffffss") text,left,bottom,width,height,hmode,feature
PS_show_boxed -- Output text in a box
********************/
static PyObject *PY_show_boxed(PyObject *self, PyObject *args)
{
    int num, ret;
    char *text, *hmode, *feature;
    float b, w, l, h;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsffffss", &num, &text, 
    	&l, &b, &w, &h, &hmode, &feature))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_show_boxed(ps, text, l, b, w, h, hmode, feature);

    return Py_BuildValue("i", ret);
}

/********************
PY_open_image ("lsssiiiiis") type,source,data,length,width,height,components,params
PS_open_image -- Reads an image for later placement
********************/
static PyObject *PY_open_image(PyObject *self, PyObject *args)
{
    int num, ret;
    char *type, *source, *data, *params;
    int length, width, height, components, bpc;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsssiiiiis", &num, &type, &source, &data, 
    	&length, &width, &height, &components, &bpc, &params))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_open_image(ps, type, source, data, 
    	length, width, height, components, bpc, params);

    return Py_BuildValue("i", ret);
}

/********************
PY_open_image_file ("lsssi") type,filename,params,intparams
PS_open_image_file -- Open image from file
********************/
static PyObject *PY_open_image_file(PyObject *self, PyObject *args)
{
    int num, ret;
    char *type, *filename, *params;
    int intparams;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsssi", &num, &type, &filename,  
    	&params, &intparams))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_open_image_file(ps, type, filename, params,  intparams);

    return Py_BuildValue("i", ret);
}

/********************
PY_show_xy ("lsff") text,x,y
PS_show_xy -- Output text at given position
********************/
static PyObject *PY_show_xy(PyObject *self, PyObject *args)
{
    int num;
    char *text;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsff", &num, &text, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_show_xy(ps, text, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_show_xy2 ("lsiff") text,xlen,x,y
PS_show_xy2 -- Output text at given position with length
********************/
static PyObject *PY_show_xy2(PyObject *self, PyObject *args)
{
    int num, xlen;
    char *text;
    float x, y;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsiff", &num, &text, &xlen, &x, &y))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_show_xy2(ps, text, xlen, x, y);

    return Py_BuildValue("i", 0);
}

/********************
PY_place_image ("lifff") imageid, x, y, scale
PS_place_image -- Places image on the page
********************/
static PyObject *PY_place_image(PyObject *self, PyObject *args)
{
    int num, imageid;
    float x, y, scale;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lifff", &num, &imageid, &x, &y, &scale))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_place_image(ps, imageid, x, y, scale);

    return Py_BuildValue("i", 0);
}

/********************
PY_stringwidth ("lsif") text, fontid, size
PS_stringwidth -- Get width of a string
********************/
static PyObject *PY_stringwidth(PyObject *self, PyObject *args)
{
    int num, fontid;
    char *line;
    float s, ret;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsif", &num, &line, &fontid, &s))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_stringwidth(ps, line, fontid, s);

    return Py_BuildValue("f", ret);
}

/********************
PY_stringwidth2 ("lsiif") text, xlen, fontid, size
PS_stringwidth2 -- Get width of a string
********************/
static PyObject *PY_stringwidth2(PyObject *self, PyObject *args)
{
    int num, fontid, xlen;
    char *line;
    float s, ret;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsiif", &num, &line, &xlen, &fontid, &s))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_stringwidth2(ps, line, xlen, fontid, s);

    return Py_BuildValue("f", ret);
}

/********************
PY_add_bookmark ("lsii") text, parent, open
PS_add_bookmark --
********************/
static PyObject *PY_add_bookmark(PyObject *self, PyObject *args)
{
    int num, parent, open;
    char *line;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lsii", &num, &line, &parent, &open))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_bookmark(ps, line, parent, open);

    return Py_BuildValue("i", 0);
}
/********************
PY_add_note ("lffffsssi") 

PS_add_note(PSDoc *psdoc, float llx, float lly, float urx, float ury, const char
 *contents, const char *title, const char *icon, int open);

********************/
static PyObject *PY_add_note(PyObject *self, PyObject *args)
{
    int num, open;
    float llx, lly, urx, ury;
    char *contents, *title, *icon;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffsssi", &num, &llx, &lly, &urx, &ury,
    	&contents, &title, &icon, &open))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_note(ps, llx, lly, urx, ury, contents, title, icon, open);

    return Py_BuildValue("i", 0);
}

/********************
PY_string_geometry ("lsiif") text, xlen, fontid, size (, dimension no longer)
PS_string_geometry -- Get geometry of a string
Signature differs from C-Version.
Returns length and dimension in kinda tupel.
********************/
static PyObject *PY_string_geometry(PyObject *self, PyObject *args)
{
    int num, fontid, xlen;
    char *line;
    float s, ret;
    float dim[3];
    PSDoc *ps;
    
    if (!PyArg_ParseTuple(args, "lsiif", &num, &line, &xlen, &fontid, &s))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_string_geometry(ps, line, xlen, fontid, s, dim);

    return Py_BuildValue("ffff", ret, dim[0], dim[1], dim[2]);
}

/********************
PY_circle ("lfff") x, y, r coordinates
PS_circle -- Draws a circle
********************/
static PyObject *PY_circle(PyObject *self, PyObject *args)
{
    int num;
    float x, y, r;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lfff", &num, &x, &y, &r))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_circle(ps, x, y, r);

    return Py_BuildValue("i", 0);
}

/********************
PY_set_border_dash ("lff") black, white
PS_set_border_dash -- Sets length of dashes for border of annotations
********************/
static PyObject *PY_set_border_dash(PyObject *self, PyObject *args)
{
    int num;
    float b, w;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lfff", &num, &b, &w))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_border_dash(ps, b, w);

    return Py_BuildValue("i", 0);
}

/********************
PY_set_border_color ("lfff") rgb
PS_set_border_color -- Sets color of border for annotations
********************/
static PyObject *PY_set_border_color(PyObject *self, PyObject *args)
{
    int num;
    float r,g,b;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lfff", &num, &r, &g, &b))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_set_border_color(ps, r, g, b);

    return Py_BuildValue("i", 0);
}

/********************
PY_findfont ("lssi") fontname, encoding, embed
PS_findfont --  Loads a font
********************/
static PyObject *PY_findfont(PyObject *self, PyObject *args)
{
    int num;
    int embed, ret;
    char *fontname, *encoding;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lssi", &num, &fontname, &encoding, &embed))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    ret = PS_findfont(ps, fontname, encoding, embed);

    return Py_BuildValue("i", ret);
}

/********************
PY_rect ("lffff") x, y, w, h coordinates
PS_rect -- Draws rectangle
********************/
static PyObject *PY_rect(PyObject *self, PyObject *args)
{
    int num;
    float x, y, w, h;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffff", &num, &x, &y, &w, &h))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_rect(ps, x, y, w, h);

    return Py_BuildValue("i", 0);
}
/********************
PY_add_launchlink ("lffffs") llx, lly, urx, ury, filename
PS_add_launchlink --
********************/
static PyObject *PY_add_launchlink(PyObject *self, PyObject *args)
{
    int num;
    float llx, lly, urx, ury;
    char *filename;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffs", &num, &llx, &lly, &urx, &ury, &filename))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_launchlink(ps, llx, lly, urx, ury, filename);

    return Py_BuildValue("i", 0);
}

/********************
PY_add_pdflink ("lffffsis") llx, lly, urx, ury, filename, page, dest
PS_add_pdflink --
********************/
static PyObject *PY_add_pdflink(PyObject *self, PyObject *args)
{
    int num, page;
    float llx, lly, urx, ury;
    char *filename, *dest;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffsis", &num, &llx, &lly, &urx, &ury, 
    	&filename, &page, &dest))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_pdflink(ps, llx, lly, urx, ury, filename, page, dest);

    return Py_BuildValue("i", 0);
}

/********************
PY_add_locallink ("lffffis") llx, lly, urx, ury, page, dest
PS_add_locallink --
********************/
static PyObject *PY_add_locallink(PyObject *self, PyObject *args)
{
    int num, page;
    float llx, lly, urx, ury;
    char *dest;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffis", &num, &llx, &lly, &urx, &ury, 
    	&page, &dest))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_locallink(ps, llx, lly, urx, ury, page, dest);

    return Py_BuildValue("i", 0);
}

/********************
PY_add_weblink ("lffffs") llx, lly, urx, ury, url
PS_add_weblink --
********************/
static PyObject *PY_add_weblink(PyObject *self, PyObject *args)
{
    int num;
    float llx, lly, urx, ury;
    char *url;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffs", &num, &llx, &lly, &urx, &ury, 
    	&url))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_add_weblink(ps, llx, lly, urx, ury, url);

    return Py_BuildValue("i", 0);
}

/********************
PY_arc ("lfffff") x, y, r, a, b coordinates
PS_arc -- Draws an arc
********************/
static PyObject *PY_arc(PyObject *self, PyObject *args)
{
    int num;
    float x, y, r, a, b;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lfffff", &num, &x, &y, 
    						&r, &a,
    						&b))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_arc(ps, x, y, r, a, b);

    return Py_BuildValue("i", 0);
}

/********************
PY_arcn ("lfffff") x, y, r, a, b coordinates
PS_arcn -- Draws an arc
********************/
static PyObject *PY_arcn(PyObject *self, PyObject *args)
{
    int num;
    float x, y, r, a, b;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lfffff", &num, &x, &y, 
    						&r, &a,
    						&b))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_arcn(ps, x, y, r, a, b);

    return Py_BuildValue("i", 0);
}

/********************
PY_curveto ("lffffff") x1, y1, x2, y2, x3, y3 coordinates
PS_curveto -- Draws curve defined by x1-y3
********************/
static PyObject *PY_curveto(PyObject *self, PyObject *args)
{
    int num;
    float x1, y1, x2, y2, x3, y3;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lffffff", &num, &x1, &y1, 
    						&x2, &y2,
    						&x3, &y3))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_curveto(ps, x1, y1, x2, y2, x3, y3);

    return Py_BuildValue("i", 0);
}

/********************
PY_setcolor ("lssffff") type, colorspace, c1-c4 color descriptor
PS_setcolor -- Self explanatory
********************/
static PyObject *PY_setcolor(PyObject *self, PyObject *args)
{
    int num;
    char *type, *colorspace;
    float c1, c2, c3, c4;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "lssffff", &num, &type, &colorspace, 
    						&c1, &c2,
    						&c3, &c4))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_setcolor(ps, type, colorspace, c1, c2, c3, c4);

    return Py_BuildValue("i", 0);
}

/******************************
 FUNCTIONS  not in the original library
 ******************************/

/********************
  convenience function
  does end_page, close, delete
PY_do_close ("l")
PS_do_close -- Closes a PostScript document
********************/
static PyObject *PY_do_close(PyObject *self, PyObject *args)
{
    int num;
    PSDoc *ps;

    if (!PyArg_ParseTuple(args, "l", &num))
        return NULL;

    ps = (PSDoc*) num;    

    /* functional block */
    PS_end_page(ps);
    PS_close(ps);
    PS_delete(ps);

    return Py_BuildValue("i", 0);
}

/********************
Convenience function
PY_do_line (l,x1,y1,x2,y2) 
********************/
static PyObject *PY_do_line(PyObject *self, PyObject *args)
{
    int num, x1, y1, x2, y2;
    PSDoc *ps;
    if (!PyArg_ParseTuple(args, "liiii", &num, &x1, &y1, &x2, &y2))
        return NULL;
    ps = (PSDoc*) num;

    PS_moveto(ps, x1, y1);
    PS_lineto(ps, x2, y2);
    PS_stroke(ps);
    return Py_BuildValue("i", 0);
}

/********************
Convenience function
draw_example (x1,y1,x2,y2) 
Creates an example-Postscript. For debugging only.
********************/
static PyObject *draw_example(PyObject *self, PyObject *args)
{
    PSDoc *ps;

    ps = PS_new();

    if (0 > PS_open_file(ps, "example.ps")) {
            printf("Cannot open PostScript file\n");
            exit(1);
    }

    PS_begin_page(ps, 596, 842);
    PS_set_parameter(ps, "transition", "wipe");
    PS_set_value(ps, "duration", 0.5);
                   
/*    int psfont = PS_findfont(ps, "Helvetica", "", 0);
    PS_setfont(ps, psfont, 12.0);
    PS_show_xy(ps, "hello", 10, 500);
    PS_show(ps, "abcABC");
    PS_symbol(ps, "\\337", "10");
 */   
    PS_moveto(ps, 0, 0);
    PS_lineto(ps, 596, 842);
    PS_stroke(ps);
    PS_moveto(ps, 0, 842);
    PS_lineto(ps, 596, 0);
    PS_stroke(ps);
    PS_end_page(ps);
    PS_close(ps);
    PS_delete(ps);
    return Py_BuildValue("i", 0);
}

/********************
  convenience function for opening a new file 
  does new, open and begin_page
PY_do_new ("s") filename
PS_do_new -- Creating a new PostScript document object
********************/

static PyObject *PY_do_new(PyObject *self, PyObject *args)
/* Takes a string = filename */
{
    PSDoc *ps;

    char *filename;
    if (!PyArg_ParseTuple(args, "s", &filename))
        return NULL;

    /* functional block */
    ps = PS_new();
    if (0 > PS_open_file(ps, filename )) {
            printf("Cannot open PostScript file\n");
            exit(1);
    }
    PS_begin_page(ps, 596, 842);
    
    return Py_BuildValue("i", ps);
}

/********************
********************/

static PyMethodDef pslib_methods[] = {
{ "PS_add_bookmark", PY_add_bookmark  , METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_add_launchlink", PY_add_launchlink, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_add_locallink", PY_add_locallink , METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_add_note", PY_add_note      , METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_add_pdflink", PY_add_pdflink   , METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_add_weblink", PY_add_weblink   , METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_arcn", PY_arcn, METH_VARARGS , " Draws an arc "},
{ "PS_arc", PY_arc, METH_VARARGS , " Draws an arc "},
{ "PS_begin_page", PY_begin_page, METH_VARARGS , " Start a new page "},
{ "PS_boot", PY_boot, METH_VARARGS , " Initialize library "},
{ "PS_circle", PY_circle, METH_VARARGS , " Draws a circle "},
{ "PS_clip", PY_clip, METH_VARARGS , " Clips drawing to current path "},
{ "PS_close_image", PY_close_image, METH_VARARGS , " Close image an free memory "},
{ "PS_closepath", PY_closepath, METH_VARARGS , " Closes path "},
{ "PS_closepath_stroke", PY_closepath_stroke, METH_VARARGS , " Closes path "},
{ "PS_close", PY_close, METH_VARARGS , " Closes a PostScript document "},
{ "PS_continue_text2", PY_continue_text2, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_continue_text", PY_continue_text, METH_VARARGS , " Continue text in next line "},
{ "PS_curveto", PY_curveto, METH_VARARGS , " Draws a curve "},
{ "PS_delete", PY_delete, METH_VARARGS , " nichts passendes. "},
{ "PS_deletefont", PY_deletefont, METH_VARARGS , " Deletes all resources of a PostScript document "},
{ "PS_do_close", PY_do_close, METH_VARARGS , " nichts passendes. "},
{ "PS_do_line", PY_do_line, METH_VARARGS , " nichts passendes. "},
{ "PS_do_new", PY_do_new, METH_VARARGS , " nichts passendes. "},
{ "PS_end_page", PY_end_page, METH_VARARGS , " End a page "},
{ "PS_fill", PY_fill, METH_VARARGS , " Fill the current path "},
{ "PS_fill_stroke", PY_fill_stroke, METH_VARARGS , " Fills and strokes the current path "},
{ "PS_findfont", PY_findfont, METH_VARARGS , " Loads a font "},
{ "PS_get_buffer", PY_get_buffer, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_get_majorversion", PY_get_majorversion, METH_VARARGS , " returns the major version number of the PostScript library "},
{ "PS_get_minorversion", PY_get_minorversion, METH_VARARGS , " returns the minor version number of the PostScript library "},
{ "PS_get_opaque", PY_get_opaque, METH_VARARGS , " Returns pointer which has been passed to PS_new2() "},
{ "PS_get_parameter", PY_get_parameter, METH_VARARGS , " Sets certain parameters "},
{ "PS_get_subminorversion", PY_get_subminorversion, METH_VARARGS , " returns the subminor version number of the PostScript library "},
{ "PS_get_value", PY_get_value, METH_VARARGS , " Sets certain parameters "},
{ "PS_lineto", PY_lineto, METH_VARARGS , " Draws line "},
{ "PS_list_parameters", PY_list_parameters, METH_VARARGS , " List all set parameters "},
{ "PS_list_resources", PY_list_resources, METH_VARARGS , " List all resources "},
{ "PS_list_values", PY_list_values, METH_VARARGS , " List all set values "},
{ "PS_moveto", PY_moveto, METH_VARARGS , " Set current point "},
{ "PS_new", PY_new, METH_VARARGS , " Returns pointer which has been passed to PS_new2() "},
{ "PS_open_file", PY_open_file, METH_VARARGS , " Creating a new PostScript document object "},
{ "PS_open_fp", PY_open_fp, METH_VARARGS , " Use an already open file for output "},
{ "PS_open_image_file", PY_open_image_file, METH_VARARGS , " Creating a new PostScript document object "},
{ "PS_open_image", PY_open_image, METH_VARARGS , " Opens a file for output "},
{ "PS_place_image", PY_place_image, METH_VARARGS , " Open image from file "},
{ "PS_rect", PY_rect, METH_VARARGS , " Reads an image for later placement "},
{ "PS_restore", PY_restore, METH_VARARGS , " Draws a rectangle "},
{ "PS_rotate", PY_rotate, METH_VARARGS , " Restore previously save context "},
{ "PS_save", PY_save, METH_VARARGS , " Sets rotation factor "},
{ "PS_scale", PY_scale, METH_VARARGS , " Save current context "},
{ "PS_scale", PY_scale, METH_VARARGS , " Sets scaling factor "},
{ "PS_set_border_color", PY_set_border_color, METH_VARARGS , " Sets color of border for annotations "},
{ "PS_set_border_dash", PY_set_border_dash, METH_VARARGS , " Sets length of dashes for border of annotations "},
{ "PS_set_border_style", PY_set_border_style, METH_VARARGS , " Sets border style of annotations "},
{ "PS_setcolor", PY_setcolor, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_setdash", PY_setdash, METH_VARARGS , " Sets appearance of a dashed line "},
{ "PS_setflat", PY_setflat, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_setfont", PY_setfont, METH_VARARGS , " Sets font to use for following output "},
{ "PS_setgray", PY_setgray, METH_VARARGS , " Sets gray value "},
{ "PS_set_info", PY_set_info, METH_VARARGS , " Sets information fields of document "},
{ "PS_setlinecap", PY_setlinecap, METH_VARARGS , " Sets appearance of line ends "},
{ "PS_setlinejoin", PY_setlinejoin, METH_VARARGS , " Sets how contected lines are joined "},
{ "PS_setlinewidth", PY_setlinewidth, METH_VARARGS , " Set width of line "},
{ "PS_setmiterlimit", PY_setmiterlimit, METH_VARARGS , " (unbekanntes Thema) "},
{ "PS_setpolydash", PY_setpolydash, METH_VARARGS , " Sets certain parameters "},
{ "PS_set_parameter", PY_set_parameter, METH_VARARGS , " Sets appearance of a dashed line "},
{ "PS_set_value", PY_set_value, METH_VARARGS , " Sets certain values "},
{ "PS_show2", PY_show2, METH_VARARGS , " Output text with given length "},
{ "PS_show_boxed", PY_show_boxed, METH_VARARGS , " Output text in a box "},
{ "PS_show", PY_show, METH_VARARGS , " Output text at current position "},
{ "PS_show_xy2", PY_show_xy2, METH_VARARGS , " Output text at given position "},
{ "PS_show_xy", PY_show_xy, METH_VARARGS , " Output text at given position "},
{ "PS_shutdown", PY_shutdown, METH_VARARGS , " Final clean up of library "},
{ "PS_stringwidth2", PY_stringwidth2, METH_VARARGS , " Get width of a string "},
{ "PS_string_geometry", PY_string_geometry, METH_VARARGS , " Get geometry of a string "},
{ "PS_stringwidth", PY_stringwidth, METH_VARARGS , " Get width of a string "},
{ "PS_stroke", PY_stroke, METH_VARARGS , " Draws the current path "},
{ "PS_translate", PY_translate, METH_VARARGS , " Sets translation "},
{ "draw_example", draw_example, METH_VARARGS , "This draws an example"},
{ NULL, NULL, 0, NULL }
};

PyMODINIT_FUNC
initpslib(void)
{
    (void) Py_InitModule("pslib", pslib_methods);
}
