/*
 * Handler for status changes and messages.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: status.c,v 1.2 1999/05/19 08:22:17 kefka Exp $
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include "x10.h"
#include "status.h"
#include "error.h"

/* Translation table for housecodes. */
char housecode_letter[]={
	'm',
	'e',
	'c',
	'k',
	'o',
	'g',
	'a',
	'i',
	'n',
	'f',
	'd',
	'l',
	'p',
	'h',
	'b',
	'j'
};

/* Translation table for devicecodes. */
int devicecode_number[]={
	13,
	5,
	3,
	11,
	15,
	7,
	1,
	9,
	14,
	6,
	4,
	12,
	16,
	8,
	2,
	10
};

/* Translation table for commands. */
char *command_name[] = {
	"ALL_UNITS_OFF",
	"ALL_LIGHTS_ON",
	"ON",
	"OFF",
	"DIM",
	"BRIGHT",
	"ALL_LIGHTS_OFF",
	"EXTENDED_CODE",
	"HAIL_REQUEST",
	"HAIL_ACKNOWLEDGE",
	"PRESET_DIM1",
	"PRESET_DIM2",
	"EXTENDED_DATA_TRANSFER",
	"STATUS_ON",
	"STATUS_OFF",
	"STATUS_REQUEST"
};


/* 
 * Display a received event.
 *
 * Here are the formats they can take:
 *
 * (time) Housecode a; Device 1,2,3; ON
 * (time) Housecode a; ALLON
 * (time) Housecode a; Device 1,2,3; DIM 5
 * (time) Housecode a; Device 1; EXTENDED 0x40 0x60
 *
 * All the devices may have names after them, like:
 * (time) Housecode a; Device 1 (light1), 2, 3 (light 2); ON
 */
void status_display_event(Event *event) {
	int i;
	char string_buffer[64];
	
	/* Show the timestamp of the event. */
	sprintf(string_buffer, "%s", ctime(&event->time));
	string_buffer[strlen(string_buffer) - 1] = 0;
	printf("(%s) ", string_buffer);
	
	/* Print the housecode of the function. */
	printf("Housecode %c; ", housecode_letter[event->housecode]);
	
	/* If we have devices, print those. */
	if(event->devices) {
		printf("Device ");
		for(i=0; i < event->devices; i++) {
			printf("%i", devicecode_number[event->device[i]]);
			if((i+1) < event->devices) printf(",");
		}
		printf("; ");
	}
	
	/* Print the command. */
	printf("%s", command_name[event->command]);
	
	/* Was it a DIM or BRIGHT? */
	if(event->command == COMMAND_DIM || event->command == COMMAND_BRIGHT) {
		
		/* Print the dim/bright level. */
		printf(" %i", event->extended1);
	}
		
	/* Was it an extended? */
	else if(event->command == COMMAND_EXTENDED_DATA_TRANSFER) {
		
		/* Print the two bytes. */
		printf(" 0x%02x 0x%02x", event->extended1, event->extended2);
	}
	
	/* End the line. */
	printf("\n");
	
	return;		
}
