/* --------------------------------------------------------------------- *
   Linux PIC16C84 programming software

   Supports Russ Reiss's PICSPA84 PIC16C84 programmer board
   and David Tait's PD PIC16C84 programmer schematic and probably any
   custom parallel port programmer.
   
   Uses parallel port to control programmer
   
   Copyright 1994-1998 by Brian C. Lane
  
   No Warranty expressed or implied
   =====================================================================
   05/29/98	Changing Version # to v2.2
   		Finally added the extra outhi() calls to lowlvl.c so that
   		the ITU-1 programmer will work. Testing with my programmer
   		to make sure it works.

   01/01/98	Finishing up the fixes/changes
   		Added display of addresses during read, verify, write
   		
   12/31/97	Changing Version # to v2.1
   		Fixing bugs and adding features
   		x 1. Ian's intel16 code really isn't. Removing and writing
   		     my own intel hex load routines.
   		x 2. Adding a monochrome mode (-m)
   		x 3. Fixing an error with the fuse being programmed wrong?
   		     or maybe recalled wrong from the data file?
   		4. Add a goto memory address command to memory display
   		5. Add home and end keys to memory display

   11/20/97	Dropping support for pictools since it won't work and
   		I cannot find Ian King anywhere. Changing rcfile to be
   		~/.picprgrc

   11/18/97	Well, pictools v0.8 do not work on my machine. I cannot
   		track down Ian, so I'll leave my pictools loading the
   		same as it has been, although it appears that he reverted
   		to the 'old' pictools .cfg format for pictools v0.8,
   		rather than the newer format he was using in pictools v0.7
   		which I don't have anymore either.

   11/15/97	Revising this to work with kernel 2.0.31 for me. Its
   		been a long time since I've even run this program. I
   		have modified my Russ Reiss programmer to act like a
   		David Tait programmer with inverted lines (I forget
   		which of course). I need to add a line test routine
   		so that users can eaily get their programmer configured
   		correctly.
   		Added a -d debugging flag.
		Well, its gets to the color setup for ncurses now. But
		since I'm running in a xterm it fails... ncurses or
		a termcap problem?
		ioperm needed a 1L paramter passed, not a 1. Newer
		GCC must be more strict, or doesn't convert?
		OK, setting TERM=xterm-color lets it run in a xterm
		TODO
		  User defined programmer setup. Specify the DB-25 pin
		  number and the logic state (+ or -)
		  Test program that will toggle the selected lines
		  on and off (like Vpp on, etc.)
		  Support other PIC processors (what's the diff?)
		  Better feedback while reading and writing (user
		  selectable if it slows things down considerably).

 * --------------------------------------------------------------------- */
#include <stdio.h>
#include <ctype.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "picprg.h"

int	        debug;
extern int	optind;			/* Option index			*/

struct _pic	pic;			/* PIC memory map		*/
struct _pconfig	pconfig; 		/* port configuration		*/

/* PIC device information (BAJ 5/18/2002):
   dev_id			 PIC device ID from location 0x2006 of conf mem

   dev_mask			 PIC device ID mask masked again dev_id

   progmax			 PIC device max program addr for device

   datamax			 PIC device max data EPROM addr for device

   algorithm                     PIC device programming algorithm
                                 0:16F84 0x10:16F87X or 16F62X parts

   name				 PIC device name

   confstr			 PIC config points to a list of strings that
                                 encodes config word info.
				 It consists of a list of comma separated 
                                 values. Format:

	"NAME,MASK,OFFSET,WIDTH,V1:NAME1,V2:NAME2,V3:NAME3,..."
	
	NAME     is the name of the config bits
	MASK     is the bit mask applied to the config word. Must be hex.
        OFFSET   is the position of the name/value on the status line
        WIDTH    is the MAX width of the name on the status line
        V1:NAME1 is a pair that describes a bit pattern and the name for that
                 pattern. Multiple value/name pairs can exist for a single
		 mask. Values may be hex,decimal, or octal using normal C
                 notation. Note that values must match the original mask 
		 and are not shifted. The string UNKNOWN is substituted for
		 the output if none of the values match in the list.

	An example that describes the clock bits for a 16F84:
	"CLOCK,0x0003,27,2,0:LP,1:XT,2:HS,3:RC"

	And the 16F84 watchdog timer:
	"WD,0x0004,48,3,0x0004:ON,0x0000:OFF"

   BTW I'm well aware that these strings could have been encoded as 
   structs. But I wanted to have the flexibility of loading everything from
   a file later on. So encoding as strings forces parsing now as opposed to
   later...
*/
char *pic16F84_config_strings[] = {
"CLOCK,0x0003,17,2,0:LP,1:XT,2:HS,3:RC",
"CP,0x0010,40,3,0x0000:ON,0x0010:OFF",
"WD,0x0004,48,3,0x0004:ON,0x0000:OFF",
"PU:0x0008,56,3,0x0008:OFF,0x0000:ON",
NULL
};

char *pic16F628_config_strings[] = {
"CLOCK,0x0013,13,8,0x00:LP,0x01:XT,0x02:HS,0x03:EXTCLK,0x10:INTRCIO," //cont...
"0x11:INTRCCLK,0x12:EXTRCIO,0x13:EXTRCCLK",
"CP,0x3c00,30,4,0x0000:ALL,0x3c00:OFF,0x2800:HALF,0x1400:3/4",
"WD,0x0004,39,3,0x0004:ON,0x0000:OFF",
"PU,0x0008,47,3,0x0008:OFF,0x0000:ON",
"CPD,0x0100,55,3,0x0100:OFF,0x0000:ON",
"LVP,0x0080,64,3,0x0080:ON,0x0000:OFF",
"BOD,0x0040,110,3,0x0040:ON,0x0000:OFF",
"MCLR,0x0020,119,3,0x0020:EXT,0x0000:INT",
NULL
};

char *pic16F877_config_strings[] = {
"CLOCK,0x0003,17,2,0:LP,1:XT,2:HS,3:RC",
"CP,0x3030,30,4,0x0000:ALL,0x1010:HALF,0x2020:TOP,0x3030:OFF",
"WD,0x0004,39,3,0x0004:ON,0x0000:OFF",
"PU,0x0008,47,3,0x0008:OFF,0x0000:ON",
"CPD,0x0100,55,3,0x0100:OFF,0x0000:ON",
"LVP,0x0080,64,3,0x0080:ON,0x0000:OFF",
"BOD,0x0040,110,3,0x0040:ON,0x0000:OFF",
"FLASHWRT,0x0200,119,3,0x0200:ON,0x0000:OFF",
NULL
};

dev_id_t pic_dev[] =
{{0x07c0,0x0fc0,0x0800,0x0080,0x10,"PIC16F628",pic16F628_config_strings},
 {0x09a0,0x0fe0,0x2000,0x0100,0x10,"PIC16F877",pic16F877_config_strings},
 {0x0560,0x0fe0,0x0400,0x0040,0x00,"PIC16F84A",pic16F84_config_strings},
 {0x0000,0x0000,0x0000,0x0000,0x00,NULL}
};

dev_id_p pic_device;

/* -------------------------------------------------------------------------
   Set window to attribute 'attr' and clear it with spaces
   ------------------------------------------------------------------------- */
void attr_clear(WINDOW *win, int width, int height, chtype attr)
{
  int i, j;

  wattrset(win, attr);   	 	/* Set window to attribute 'attr' */
  for (i = 0; i < height; i++) {
    wmove(win, i, 0);
    for (j = 0; j < width; j++)
      waddch(win, ' ');
  }
  wmove( win, 0, 0 );			/* Put cursor at top right */
}


/* -----------------------------------------------------------------------
   Title line
   ----------------------------------------------------------------------- */
void title()
{
  char temp[160], temp1[40], **confstr, *p, *name, *tokens[80];
  char confstrcopy[320],*d=confstrcopy;
  int i,offset=0,width,nt=0;
  uint confword = pic.mem[pconfig.CFaddr], mask, val ;

   debug=0;
  attr_clear( stdscr, 80, 25, COLOR_PAIR( 1 ) );

  attron( A_REVERSE );
  addstr( VERSION );
  attrset( COLOR_PAIR( 1 ) );
    
  // Init status string to all spaces
  for(i=0;i<159;i++)
     temp[i] = ' ';

  temp[159] = '\0';
  
  // Copy the device name first 
  sprintf(temp1,"%-12.12s",pic_device->name);
  for(d=temp+3,p=temp1;*p;d++,p++)
     *d = *p;

  for(confstr=pic_device->confstr;*confstr;confstr++) {
    if (debug)
       fprintf(stderr,"Tokenizing [%s]\n",*confstr);
    // tokenize a copy of the confstring. nt counts the number of tokens
    
    for(nt=0,p=*confstr,d=confstrcopy;*p;p++,d++) {
       tokens[nt++] = d;
       while(*p && *p != ',') *d++ = *p++;
       *d = '\0';
       if (!*p) break;
    }
    tokens[nt] = NULL;
    if (debug) {
       for(i=0;i<nt;i++)
        fprintf(stderr,"token[%02d] = [%s]\n",i,tokens[i]);
    }

     // Decode the static items
     name = tokens[0];
     mask = strtol(tokens[1],NULL,0);
     offset = strtol(tokens[2],NULL,0);
     width = strtol(tokens[3],NULL,0);
     
     if (debug) 
        fprintf(stderr,"name[%s] mask[%04x] offset[%d] width[%d]\n",
           name,mask,offset,width);
   
     // Now process the values
     for(i=4;tokens[i];i++) {
        val=strtol(tokens[i],NULL,0);
        if ((confword & mask) == val)
           break;
     }
   
     if (tokens[i]) {
        p=strchr(tokens[i],':');
        if (p) {
           sprintf(temp1,"%s: %*.*s",name,-width,width,p+1);
        }
        else
           sprintf(temp1,"%s: %*.*s",name,-width,width,"NOVAL");
     }
     else
        sprintf(temp1,"%s: %*.*s",name,-width,width,"UNKNOWN");
   
     for(d=temp+offset,p=temp1;*p;d++,p++)
        *d = *p;
  }

    attrset( COLOR_PAIR( 2 ) );
    mvprintw( 1, 0, "%-160s", temp);
    attrset( COLOR_PAIR( 1 ) );

  refresh();
}


/* -----------------------------------------------------------------------
   Ask user to press any key to continue
   ----------------------------------------------------------------------- */
void press_any()
{
  attron( A_REVERSE );
  mvaddstr( 24, 0, "                           Press any key to continue                            " );
  attrset( COLOR_PAIR( 1 ) );
  move( 24, 0 );
  refresh();
  
  getch();
}


/* -----------------------------------------------------------------------
   Erase the error line and the 'press any key' line.
   ----------------------------------------------------------------------- */
void erase_err()
{
  move( 22, 0 );
  clrtoeol( );
  move( 24, 0 );
  clrtoeol( );
}


/* -----------------------------------------------------------------------
   Show an error string on line 22, 0 in Red on White
   ----------------------------------------------------------------------- */
void show_err( char *fmt, ... )
{
  va_list	ap;
  int		l;
  char		temp[81];
  
  va_start( ap, fmt );
  vsprintf( temp, fmt, ap );
  
  l = strlen( temp );
  attrset( COLOR_PAIR( 6 ) );
  mvprintw( 22, 0, "%*s%s%*s", (80-l) / 2, " ", temp, (80-l) / 2, " " );

  /* If roundoff error, add a space to end of string */
  if ( ( ( ( ( 80-l ) / 2 ) * 2 ) + l ) != 80 )
    addstr( " " );
  attrset( COLOR_PAIR( 1 ) );  
}


/* -----------------------------------------------------------------------
   Read in configuration data from ~/.picprgrc data file
   ----------------------------------------------------------------------- */
void read_config(int do_defaults)
{
  FILE	*fp;
  char	path[255];
    
  /* Defaults: */
  if (do_defaults) {
     pconfig.port = 0;			/* /dev/lp port		*/
     pconfig.VPP_pin = 5;		/* VPP on pin 5		*/
     pconfig.VPP_pol = 1;		/* Normal by default	*/
     pconfig.VDD_pin = 4;		/* Vdd on pin 4		*/
     pconfig.VDD_pol = 1;		/* Normal by default	*/
     pconfig.PGM_pin = 6;		/* PGM on pin 6		*/
     pconfig.PGM_pol = 1;		/* Normal by default	*/
     pconfig.CLK_pin = 3;		/* Clock on pin 3	*/
     pconfig.alt = 0;                   /* Aisha 4/2/00 */
     pconfig.CLK_pol = 1;		/* Normal by default	*/
     pconfig.DATAO_pin = 2;		/* Data to PIC on pin 2	*/
     pconfig.DATAO_pol = 1;		/* Normal by default	*/
     pconfig.DATAI_pin = 10;		/* Data from pic on p10 */
     pconfig.DATAI_pol = 0;		/* Inverted by default	*/
     pconfig.IDaddr = 0x2000;		/* Default ID locaiton	*/
     pconfig.EEaddr = 0x2100;		/* Default EE location	*/
     pconfig.CFaddr = 0x2007;		/* Config Fuses		*/
     pconfig.colors = TRUE;		/* Color by default	*/
  } 

  /* Make a path to the configuration file */
  strcpy( path, getenv( "HOME" ) );
  strcat( path, "/.picprgrc" );

  if ( ( fp = fopen( path, "r" ) ) == NULL )
  {
    return;
  }
  
  /* Read the file 1 line at a time */
  while ( fgets( path, 80, fp ) != NULL )
  {
    /* Ignore comments and blank lines */
    if ( path[0] == '#' || path[0] == '\n' )
      continue;

    /* Set the /dev/lp port to use */
    if ( strncmp( path, "port=", 5 ) == 0 )
    {
      pconfig.port = strtol( &path[5], NULL, 0);                 /*atoi( &path[5] )*/
      continue;
    }

    /* Set the VPP pin number and polarity */
    if( strncmp( path, "vpp=", 4 ) == 0 )
    {
      if( (atoi( &path[5] ) < 26) && (atoi( &path[5] ) > 0) )
      { 
        pconfig.VPP_pin = atoi( &path[5] );
        if( path[4] == '+' )
          pconfig.VPP_pol = 1;
        else if( path[4] == '-' )
          pconfig.VPP_pol = 0;
      }
    }

    /* Set the Vdd pin number and polarity */
    if( strncmp( path, "vdd=", 4 ) == 0 )
    {
      if( (atoi( &path[5] ) < 26) && (atoi( &path[5] ) > 0) )
      { 
        pconfig.VDD_pin = atoi( &path[5] );
        if( path[4] == '+' )
          pconfig.VDD_pol = 1;
        else if( path[4] == '-' )
          pconfig.VDD_pol = 0;
      }
    }

    /* Set the PGM pin number and polarity */
    if( strncmp( path, "pgm=", 4 ) == 0 )
    {
      if( (atoi( &path[5] ) < 26) && (atoi( &path[5] ) > 0) )
      { 
        pconfig.PGM_pin = atoi( &path[5] );
        if( path[4] == '+' )
          pconfig.PGM_pol = 1;
        else if( path[4] == '-' )
          pconfig.PGM_pol = 0;
      }
    }

    /* Set the Clock pin number and polarity */
    if( strncmp( path, "clock=", 6 ) == 0 )
    {
      if( (atoi( &path[7] ) < 26) && (atoi( &path[7] ) > 0) )
      { 
        pconfig.CLK_pin = atoi( &path[7] );
        if( path[6] == '+' )
          pconfig.CLK_pol = 1;
        else if( path[6] == '-' )
          pconfig.CLK_pol = 0;
      }
    }

    /* Set the Data to PIC pin number and polarity */
    if( strncmp( path, "datao=", 6 ) == 0 )
    {
      if( (atoi( &path[7] ) < 26) && (atoi( &path[7] ) > 0) )
      { 
        pconfig.DATAO_pin = atoi( &path[7] );
        if( path[6] == '+' )
          pconfig.DATAO_pol = 1;
        else if( path[6] == '-' )
          pconfig.DATAO_pol = 0;
      }
    }
    /* Set the Data from PIC pin number and polarity */
    if( strncmp( path, "datai=", 6 ) == 0 )
    {
      if( (atoi( &path[7] ) < 26) && (atoi( &path[7] ) > 0) )
      { 
        pconfig.DATAI_pin = atoi( &path[7] );
        if( path[6] == '+' )
          pconfig.DATAI_pol = 1;
        else if( path[6] == '-' )
          pconfig.DATAI_pol = 0;
      }
    }

    /* Set the location in memory map of ID data */    
    if ( strncmp( path, "idaddr=", 7 ) == 0 )
    {
      sscanf( &path[7], "%x", &pconfig.IDaddr );
      continue;
    }
    
    /* Set the location of the EEPROM data */
    if ( strncmp( path, "eeaddr=", 7 ) == 0 )
    {
      sscanf( &path[7], "%x", &pconfig.EEaddr );
      continue;
    }

    /* Set the location of the Config FUSES data */
    if ( strncmp( path, "cfaddr=", 7 ) == 0 )
    {
      sscanf( &path[7], "%x", &pconfig.CFaddr );
      continue;
    }
    
    /* Set the colors variable to TRUE or FALSE */
    if ( strncmp( path, "colors=", 7 ) == 0 )
    {
      sscanf( &path[7], "%d", &pconfig.colors );
      continue;
    }

    /* Set the alt variable to type */
    if ( strncmp( path, "type=", 5 ) == 0 )
    {
      pconfig.alt = strtol( path+5, NULL, 0);  
      printf("Set type/alt to 0x%02x\n",pconfig.alt);
      continue;
    }
  }
  fclose( fp );

}

void config_device(void)
{
  uint dev_id,len;
  char dev_name[64];

  while (1) {
     // Get the id config for the part
     dev_id = get_device_id();
     printf("The part's id is %04x\n",dev_id);

     // Search for the device
     for(pic_device=pic_dev;pic_device->dev_id;pic_device++) {
        if (pic_device->dev_id == (dev_id & pic_device->dev_mask))
           break;
     }

     // break loop if pic_device is found
     if (pic_device->dev_id)
        break;

     // Ask user to install pic_device or give pic_device name
     printf("picprg: Device not located in programmer.\n");
     printf("        Please insert device and press ENTER...\n");
     printf("        ...or type the name of the device...\n");
     printf("        ...or type 'list' to get a list of devices.\n");
     printf("choice> ");
     fflush(stdout);
     fgets(dev_name,63,stdin);

     if (debug)
        fprintf(stderr,"length=%d\n",strlen(dev_name));

     switch (len=strlen(dev_name)) {
        case 1: continue; // Loop around the loop again
        case 5: 
                printf("\nList of available PIC devices:\n");
                printf("--------------------------------\n");
                for(pic_device=pic_dev;pic_device->dev_id;pic_device++)
                   printf("%s\n",pic_device->name);
                printf("--------------------------------\n\n");
                break;
        default:
                for(pic_device=pic_dev;pic_device->dev_id;pic_device++)
                   if (!strncasecmp(pic_device->name,dev_name,
                                    strlen(pic_device->name)))
                      break;
                break;
     }
     if (pic_device->dev_id)
        break;
   }
}

     

/* -----------------------------------------------------------------------
   Write configuration data to ~/.picprgrc data file
   ----------------------------------------------------------------------- */
int write_config()
{
  FILE	*fp;
  char	path[255];
  
  strcpy( path, getenv( "HOME" ) );
  strcat( path, "/.picprgrc" );
  
  if ( ( fp = fopen( path, "w" ) ) == NULL )
  {
    show_err( "Error writing ~/.picprgrc" );
    return FALSE; 
  }

  fprintf( fp, "# Linux PIC programmer configuration v2.0\n");  
  fprintf( fp, "port=0x%x\n",     pconfig.port );      /* Aisha changed %d, 3/21/00*/
  fprintf( fp, "vpp=%c%d\n",    pconfig.VPP_pol ? '+' : '-', pconfig.VPP_pin );
  fprintf( fp, "vdd=%c%d\n",    pconfig.VDD_pol ? '+' : '-', pconfig.VDD_pin );
  fprintf( fp, "pgm=%c%d\n",    pconfig.PGM_pol ? '+' : '-', pconfig.PGM_pin );
  fprintf( fp, "clock=%c%d\n",  pconfig.CLK_pol ? '+' : '-', pconfig.CLK_pin );
  fprintf( fp, "datao=%c%d\n",  pconfig.DATAO_pol ? '+' : '-', pconfig.DATAO_pin );
  fprintf( fp, "datai=%c%d\n",  pconfig.DATAI_pol ? '+' : '-', pconfig.DATAI_pin );
  fprintf( fp, "idaddr=%04X\n", pconfig.IDaddr );
  fprintf( fp, "eeaddr=%04X\n", pconfig.EEaddr );
  fprintf( fp, "cfaddr=%04X\n", pconfig.CFaddr );
  fprintf( fp, "colors=%d\n",   pconfig.colors );
  fprintf( fp, "type=0x%x\n",   pconfig.alt );
  
  fclose( fp );

  return TRUE;
}


/* -----------------------------------------------------------------------
   Show a page of memory information
   ----------------------------------------------------------------------- */
void show_mem( int mem_start )
{
  int x, y; 
  
  /* Print memory info. 8 per line */
  for ( x = 0; x < 20; x++ )
  {
    mvprintw( x+3, 0, "%04X:  ", mem_start+(x*8) );
    for( y = 0; y < 8; y++ )
      printw( "%04X  ", pic.mem[ mem_start+(x*8)+y] );
  }
  refresh();
}


/* -----------------------------------------------------------------------
   Display the PIC memory
   Use up/down arrows, pageup/pagedn
   ----------------------------------------------------------------------- */
void edit_pic()
{
  int	mem_start, ch;

  title();				/* Title/copyright line 	*/

  mem_start = 0;			/* Start at beginning of memory */
    
  attrset( COLOR_PAIR( 3 ) );
  mvaddstr( 24, 0, "  Arrow keys to move                                       [Q]uit to main menu  " );
  attrset( COLOR_PAIR( 1 ) );

  while ( 1 )
  {   
    show_mem( mem_start );
    move( 0, 0 );			/* Put cursor out of the way */
            
    /* Get keypresses. Q quits as does ESC and X. Arrows move directions */
    ch = getch();

    /* Check for exit keys first */
    switch( ch )
    {
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 0x1B:  return;
      		  break;
      		  
      case 0x0D:
      case KEY_DOWN:
		if ( mem_start < ( MAXPICSIZE - 0xA0 ) )
		  mem_start += 0x08;
		break;
		
      case KEY_UP:
      		if ( mem_start > 0x00 )
      		  mem_start -= 0x08;
      		break;

      case ' ':      		
      case KEY_NPAGE:
      		if ( mem_start < ( MAXPICSIZE - 0x140 ) )
      		  mem_start += 0xA0;
      		else
      		  mem_start = MAXPICSIZE - 0x9F;
      		break;
      		
      case KEY_PPAGE:
      		if ( mem_start >= 0xA0 )
      		  mem_start -= 0xA0;
      		else
      		  mem_start = 0x00;
      		break;
      		
      default:	break;
    }
  }
}


/* -----------------------------------------------------------------------
   Initalize all the memory to its blank state (0xFFFF)
   ----------------------------------------------------------------------- */
void init_pic()
{
  int i;

  for(i=0; i <= MAXPICSIZE; i++ )
    pic.mem[i] = 0xffff;
}


/* -----------------------------------------------------------------------
   Load PIC data from a Intel HEX formatted file (8M or 16)
   ----------------------------------------------------------------------- */
int load_hex( char *fname )
{
  int	min, max;
  FILE	*fp;
    
  if ( ( fp = fopen( fname, "rb" ) ) == NULL )
  {
    return FALSE;
  }
  
  min = 0;
  max = MAXPICSIZE;
  if ( readihex( fp, pic.mem, &min, &max ) == FALSE )
  {
    fclose( fp );
    return FALSE;
  }

  fclose( fp );
  return TRUE;
}


/* -------------------------------------------------------------------------
   Prompt the user for the filename
   ------------------------------------------------------------------------- */
void load_file()
{
  char	fname[255];
  
  title();
  
  /* prompt for a filename */
  attrset( COLOR_PAIR( 5 ) );
  mvprintw( 5, 0, "Load an Intel HEX file (IHX16 or IHX8M)" );
  attrset( COLOR_PAIR( 1 ) );
  
  mvprintw( 7, 0, "Enter filename to load : ");
  echo();
  getstr( fname );
  noecho();
  
  mvprintw( 9, 0, "Loading %s", fname );
  refresh();

  /* Load the file */
  if ( load_hex( fname ) == FALSE )
  {
    show_err( "Error loading %s", fname );
    press_any();
    erase_err();
  }
}


/* -----------------------------------------------------------------------
   Write PIC data to a Intel Hex (IHX16) formatted file
   ----------------------------------------------------------------------- */
int write_hex( char *fname )
{
  FILE	*fp;
    
  if ( ( fp = fopen( fname, "wb" ) ) == NULL )
  {
    show_err( "Error writing %s", fname );
    return FALSE;
  }

  /* If it is a PIC16C84 then its only 1k, if its unknown, write it all */
  if( pconfig.PIC_type == PIC16C84 )
  {  
    writeihex16( fp, pic.mem, 0, 0x0400 );

    writeihex16( fp, pic.mem, pconfig.IDaddr, pconfig.IDaddr+8 );

    writeihex16( fp, pic.mem, pconfig.EEaddr, pconfig.EEaddr+64 );
  } else {
    writeihex16( fp, pic.mem, 0, MAXPICSIZE );
  }
  writeiend16( fp );

  fclose( fp );
        
  return TRUE;
}


/* -----------------------------------------------------------------------
   Write PIC data to a file.

   Written as INHX16 file format
  ----------------------------------------------------------------------- */
void write_file()
{
  char	fname[255];
  
  title();
  
  /* prompt for a filename */
  /* Do a little directory maybe? filename completion? */
  attrset( COLOR_PAIR( 5 ) );
  mvprintw( 5, 0, "Write an INHX16 file");
  attrset( COLOR_PAIR( 1 ) );
  
  mvprintw( 7, 0, "Enter filename to write : ");
  echo();
  getstr( fname );
  noecho();
  
  mvprintw( 9, 0, "Writing %s", fname );
  refresh();

  if ( write_hex( fname ) == FALSE )		/* Write INHX16	*/
  {
    press_any();
    erase_err();
  }
}


/* -----------------------------------------------------------------------
   Select the parallel port /dev/lpX
   ----------------------------------------------------------------------- */
int select_port( int ch, int *port, int select )
{
  switch ( ch )
  {
    case '0':	*port = 0;
    		break;
    		
    case '1':	*port = 1;
    		break;
    		
    case '2':	*port = 2;
    		break;

    case 0x0D:
    case 0x0A:
    case KEY_DOWN:
                if( select < 6 )
                  select++;
    		return select;
    		break;
    		
    case KEY_UP:
    		if( select > 0 )
    		  select--;
    		return select;
    		break;
    		
    default:	return 0;
    		break;
  }
  return 0;
}


/* -----------------------------------------------------------------------
   Select the address -- hex, 4 digits max.
   ----------------------------------------------------------------------- */
int select_addr( int ch, int *addr, int select , int len)
{
  int	pos = 0, done = 0;
  char	value[5];
  
  /* Convert the current value to a string */
  sprintf( value, "%0*X", len, *addr );

  while ( !done )
  {  
    switch ( ch )
    {
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 0x1B:  return -1;
      		  break;
      		  
      case 's':
      case 'S':   mvaddstr( 3, 5, "Configuration written to ~/.picprgrc                   " );
                  move( 5+select, 56+pos );
                  if ( write_config() == FALSE )
      		  {
      		    press_any();
      		    erase_err();
      		  }
      		  break;

      case 0x0D:
      case 0x0A:
      case KEY_DOWN:
                  if( select < 6 )
                    select++;
      		  return select;
      		  break;

      case KEY_UP:
                  if( select > 0 )
                    select--;
    		  return select;
    		  break;
    		    		
      case KEY_RIGHT:  if ( pos < len ) pos++;
      		       move( 5+select, 56+pos );
      		       break;
    		  		  
      case KEY_LEFT:   if ( pos > 0 ) pos--;
      		       move( 5+select, 56+pos );
		       break;
		       
      default:	       if ( isxdigit( ch ) )
                       {
                         value[pos] = toupper(ch);
                         if ( pos < len ) pos++;
                         addch( toupper(ch) );
                         move( 5+select, 56+pos );
                         sscanf( value, "%x", addr );
                       }
    		       break;
    		       		  
    } /* switch */
    refresh();
    ch = getch();
  } /* which */
  
  return select;
}


/* -----------------------------------------------------------------------
   Select the pin number 1-25
   ----------------------------------------------------------------------- */
int select_pin( int ch, int *pin, int *pol, int *state, int select )
{
  int	pos = 0, done = 0, tmp_pin;
  char	pin_ch[3];
  
  sprintf( pin_ch, "%02d", *pin );

  while ( !done )
  {  
    switch ( ch )
    {
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 0x1B:  return -1;
      		  break;
      		  
      case 's':
      case 'S':   mvaddstr( 3, 5, "Configuration written to ~/.picprgrc                   " );
                  move( 5+select, 56+pos );
                  if ( write_config() == FALSE )
      		  {
      		    press_any();
      		    erase_err();
      		  }
      		  break;

      /* Exit and return the next selection number */
      case 0x0D:
      case 0x0A:
      case KEY_DOWN:
                  if( select < 6 )		/* 7 is the last one 	*/
      		    select++;
      		  return select;
      		  break;

      /* Exit and return the previous selection */
      case KEY_UP:
                  if( select > 0 )
                    select--;
    		  return select;
    		  break;
    		    		
      case KEY_RIGHT:  if ( pos == 0 )
                       {
                         pos = 1;
      		         move( 5+select, 56+pos );
      		       }
      		       break;
    		  		  
      case KEY_LEFT:   if ( pos > 0 )
                       {
                         pos = 0;
      		         move( 5+select, 56+pos );
      		       }
		       break;

      case '-':        *pol = 0;
                       move( 5+select, 54+pos );
                       addch( '-' );
                       move( 5+select, 56+pos );
                       break;
                       
                       
      case '+':        *pol = 1;
                       move( 5+select, 54+pos );
                       addch( '+' );
                       move( 5+select, 56+pos );
                       break;

      /* Turn on the selected bit, using current setting */
      case 'f':
      case 'F':        if( state != NULL )
                       {
                         switch( select )
                         {
                           case 1:	vppoff();
                         		break;
                         		
                           case 2:	vddoff();
                         		break;
                         		
                           case 3:	pgmoff();
                         		break;
                         		
                           case 4:	clklo();
                         		break;
                         
                           case 5:	outlo();
                           		mvaddstr( 6+select, 62, 
                           		          inbit() ? "on " : "off" );
                         		break;
                         		
                           default:	break;
                         }
                         *state = 0;
                         mvaddstr( 5+select, 62, "off" );
                         move( 5+select, 56+pos );
                       }
                       break;
      
      /* Turn off the selected bit, using current setting */
      case 'o':
      case 'O':        if( state != NULL )
                       {
                         switch( select )
                         {
                           case 1:	vppon();
                         		break;
                         		
                           case 2:	vddon();
                         		break;
                         		
                           case 3:	pgmon();
                         		break;
                         		
                           case 4:	clkhi();
                         		break;
                         		
                           case 5:	outhi();
                           		mvaddstr( 6+select, 62, 
                           		          inbit() ? "on " : "off" );
                         		break;
                         
                           default:	break;
                         }
                         *state = 1;
                         mvaddstr( 5+select, 62, "on " );
                         move( 5+select, 56+pos );
                       }
                       break;

      default:	       if ( isdigit( ch ) )
                       {
                         pin_ch[pos] = ch;
                         if ( pos == 0 )
                           pos++;
                         addch( ch );
                         move( 5+select, 56+pos );
                         /* Check for a valid pin # first */
                         sscanf( pin_ch, "%02d", &tmp_pin );
                         if( tmp_pin < 18 )
                           *pin = tmp_pin;
                       }
    		       break;
    		       		  
    } /* switch */

    /* Update the display with the new info */
    refresh();
    
    /* Get the next key */
    ch = getch();
  } /* which */
  
  return select;
}


/* -----------------------------------------------------------------------
   Configure the programmer -- Programmer type, ID, EEPROM locations, port
   
   Allow the user to specify the DB-25 pin number for each pin and the
   logic state for its on state.
   
     VPP control
     Vdd control
     Clock to PIC
     Data to PIC
     Data from PIC
   
   Display help for the line that you are on.
   
   ----------------------------------------------------------------------- */
void config()
{
  int	ch, select=0, done=0, line, indent,
  	vpp_state=0,
  	vdd_state=0,
  	pgm_state=0,
  	clk_state=0,
  	out_state=0;
  
  /* Clear the screen and show the title bar */
  title();

  while ( !done )
  {
    line = 5;
    indent = 15;
    /* Select port 0-3 or whatever are recognized */
    mvprintw( line++, indent,
              "Printer Port                         :  [%01x]",
              pconfig.port
            );

    /* Show the current Pin settings */
    mvprintw( line++, indent,
              "Vpp control pin #                    : %c[%02d]   %s ",
              pconfig.VPP_pol ? '+' : '-',
              pconfig.VPP_pin,
              vpp_state ? "on" : "off"
            );

    mvprintw( line++, indent,
              "Vdd control pin #                    : %c[%02d]   %s ",
              pconfig.VDD_pol ? '+' : '-',
              pconfig.VDD_pin,
              vdd_state ? "on" : "off"
            );
              
    mvprintw( line++, indent,
              "PGM control pin #                    : %c[%02d]   %s ",
              pconfig.PGM_pol ? '+' : '-',
              pconfig.PGM_pin,
              vdd_state ? "on" : "off"
            );
              
    mvprintw( line++, indent,
              "Clock pin #                          : %c[%02d]   %s ",
              pconfig.CLK_pol ? '+' : '-',
              pconfig.CLK_pin,
              clk_state ? "on" : "off"
            );

    mvprintw( line++, indent,
              "Data to PIC pin #                    : %c[%02d]   %s ",
              pconfig.DATAO_pol ? '+' : '-',
              pconfig.DATAO_pin,
              out_state ? "on" : "off"
            );

    mvprintw( line++, indent,
              "Data from PIC pin #                  : %c[%02d]   %s ",
              pconfig.DATAI_pol ? '+' : '-',
              pconfig.DATAI_pin,
              inbit() ? "on" : "off"
            );
#if 0
    /* Select start memory locations ID/FUSES, EEPROM */  
    mvprintw( line++, indent,
              "ID memory location in HEX file (HEX) :  [%04X]",
              pconfig.IDaddr
            );
    
    mvprintw( line++, indent,
              "EEPROM memory location in file (HEX) :  [%04X]",
              pconfig.EEaddr
            );

    mvprintw( line++, indent,
              "Config FUSES location in file (HEX)  :  [%04X]",
              pconfig.CFaddr
            );

    mvprintw( line++, indent,
              "Programming Type (0: 16F84,10:16F87x):  [%02X]",
              pconfig.alt
            );
#endif

    /* Show special options */
    attrset( COLOR_PAIR( 5 ) );
    switch( select )
    {
      case 0:	mvaddstr( 17, 5, "Enter parallel port 0-2                                      " );
                mvaddstr( 18, 5, "                                            ");
      		break;
      		
      case 1:	mvaddstr( 17, 5, "Enter VPP pin number and select polarity with +/-           " );
                mvaddstr( 18, 5, "Use [O]n and o[F]f to toggle the output line");
		break;
		
      case 2:	mvaddstr( 17, 5, "Enter Vdd pin number and select polarity with +/-            " );
                mvaddstr( 18, 5, "Use [O]n and o[F]f to toggle the output line");
		break;
		
      case 3:	mvaddstr( 17, 5, "Enter PGM pin number and select polarity with +/-            " );
                mvaddstr( 18, 5, "Use [O]n and o[F]f to toggle the output line");
		break;
		
      case 4:	mvaddstr( 17, 5, "Enter Clock pin number and select polarity with +/-          " );
                mvaddstr( 18, 5, "Use [O]n and o[F]f to toggle the output line");
		break;
		
      case 5:	mvaddstr( 17, 5, "Enter Data to PIC pin number and select polarity with +/-    " );
                mvaddstr( 18, 5, "Use [O]n and o[F]f to toggle the output line");
		break;

      case 6:	mvaddstr( 17, 5, "Enter Data from PIC pin number and select polarity with +/-  " );
                mvaddstr( 18, 5, "                                            ");
		break;
		
#if 0
      case 7:	mvaddstr( 17, 5, "Enter starting address in HEX                                " );
      		break;
      		
      case 8:	mvaddstr( 17, 5, "Enter starting address in HEX                                " );
      		break;

      case 9:	mvaddstr( 17, 5, "Enter starting address in HEX                                " );
      		break;
            		
      case 10:	mvaddstr( 17, 5, "Enter 0 for 16F84. Enter 10 for 16F87X or 16F62X             " );
      		break;
            		
#endif
      default:
      		break;
    }
    
    attrset( COLOR_PAIR( 3 ) );
    mvaddstr( 24, 0, "  Arrow keys to move          [S]ave configuration         [Q]uit to main menu  " );
    attrset( COLOR_PAIR( 1 ) );

    /* Move the cursor to the line being edited */
    move( 5+select, 41+indent );
    refresh();
  
    /* Get keypresses. Q quits as does ESC and X. Arrows move directions */
    ch = getch();

    /* Check for exit keys first */
    switch( ch )
    {
      case 'q':
      case 'Q':
      case 'x':
      case 'X':
      case 0x1B:  done = 1;
      		  break;
 		  
      case 's':
      case 'S':   mvaddstr( 3, 5, "Configuration written to ~/.picprgrc                   " );
                  if ( write_config() == FALSE )
      		  {
      		    press_any();
      		    erase_err();
      		  }
      		  break;

      default:
        switch( select )
        {
          /* Selecting the Parallel port -- Allow numbers */
          case 0:  select = select_addr( ch, &pconfig.port, select , 0);
    		   break;

          /* Select a pin numer and polarity -- allow +/- */
          case 1:  select = select_pin( ch, &pconfig.VPP_pin,
                                            &pconfig.VPP_pol,
                                            &vpp_state,
                                            select );
    		   break;

          case 2:  select = select_pin( ch, &pconfig.VDD_pin,
                                            &pconfig.VDD_pol,
                                            &vdd_state,
                                            select );
    		   break;

          case 3:  select = select_pin( ch, &pconfig.PGM_pin,
                                            &pconfig.PGM_pol,
                                            &pgm_state,
                                            select );
    		   break;

          case 4:  select = select_pin( ch, &pconfig.CLK_pin,
                                            &pconfig.CLK_pol,
                                            &clk_state,
                                            select );
    		   break;

          case 5:  select = select_pin( ch, &pconfig.DATAO_pin,
                                            &pconfig.DATAO_pol,
                                            &out_state,
                                            select );
    		   break;

          case 6:  select = select_pin( ch, &pconfig.DATAI_pin,
                                            &pconfig.DATAI_pol,
                                            NULL,
                                            select );
    		   break;

#if 0
          /* Selecting memory locations -- Allow 0-9,A-F */
          case 7:  select = select_addr( ch, &pconfig.IDaddr, select , 4);
    		   break;
    		   
    	  case 8:  select = select_addr( ch, &pconfig.EEaddr, select , 4);
    	  	   break;

    	  case 9:  select = select_addr( ch, &pconfig.CFaddr, select , 4);
    	  	   break;
    		
    	  case 10:  select = select_addr( ch, &pconfig.alt, select , 2);
    	  	   break;
#endif
    		
          default:
    		   break;
        } /* Switch selection */
    } /* Switch ch */

    /* If the select routine wants and exit, give it to them */
    if ( select == -1 )
      done = 1;
  } /* While */

  /* Make sure everything is off when we leave */
  vppoff();
  vddoff();
  clklo();
  outlo();
}


/* -----------------------------------------------------------------------
   Prompt for the fill value for memory. Default to 0x03FF
   ----------------------------------------------------------------------- */
void fill_mem()
{

}


/* -----------------------------------------------------------------------
   Main Menu display. Gets a keypress...
   ----------------------------------------------------------------------- */
int main_menu()
{
  title();

  mvaddstr( 6, 27, "[L]oad a file" );
  mvaddstr( 7, 27, "[W]rite RAM to file");
  mvaddstr( 8, 27, "[D]isply RAM" );
  mvaddstr( 9, 27, "[F]ill RAM" );

  mvaddstr( 11, 27, "[R]ead contents of PIC into RAM" );
  mvaddstr( 12, 27, "[P]rogram PIC from RAM");
  mvaddstr( 13, 27, "[V]erify PIC with RAM" );

  mvaddstr( 15, 27, "[C]onfigure PICPGMR");
  mvaddstr( 16, 27, "[Q]uit");
  move( 0, 0 );
  
  refresh();

  return getch();
}

/* -----------------------------------------------------------------------
   Setup my color pairs
   
   Make these user defined eventually
   ----------------------------------------------------------------------- */ 
void setup_colors()
{
  init_pair( 1, COLOR_WHITE, COLOR_BLUE );
  init_pair( 2, COLOR_WHITE, COLOR_RED );
  init_pair( 3, COLOR_BLACK, COLOR_WHITE );
  init_pair( 4, COLOR_BLUE, COLOR_WHITE );
  init_pair( 5, COLOR_CYAN, COLOR_BLUE );
  init_pair( 6, COLOR_RED, COLOR_WHITE );
}

void setup_mono()
{
  init_pair( 1, COLOR_WHITE, COLOR_BLACK );
  init_pair( 2, COLOR_WHITE, COLOR_BLACK );
  init_pair( 3, COLOR_BLACK, COLOR_WHITE );
  init_pair( 4, COLOR_BLACK, COLOR_WHITE );
  init_pair( 5, COLOR_WHITE, COLOR_BLACK );
  init_pair( 6, COLOR_BLACK, COLOR_WHITE );
}


/* -----------------------------------------------------------------------
   Main code
   ----------------------------------------------------------------------- */ 
int main( int argc, char *argv[] )
{
  int 	done = 0;
  char	ch;
  WINDOW *scrollw;

  /* Read configuration file from ~/.picprgrc */
  read_config(1);

  /* Clear our PIC memory storage area */
  init_pic();

  
  /* process command line arguments -- Override config and file settings */
  ch = 0;
  debug = 0;
  
  while ( ch != -1 )
  {
    ch = getopt( argc, argv, "?map:i:e:t:d" );
    
    switch ( ch )
    {
      /* Aisha -a option 4/2/00 */
      case 'a':
	        pconfig.alt=0x10;
		fprintf(stderr, "changed alt to 0x10\n");
		break;
    /* Turn on debugging info */
      case 'd':
      		debug = 1;
      		break;
     
      		
      /* Set the port to 0-2 */
      case 'p':                    /*Aisha changed 3/21/00 atoi( optarg )*/
	        pconfig.port = strtol( optarg, NULL, 0);  
      		break;

      /* Set the ID address in hex */      		
      case 'i':
      		sscanf( optarg, "%x", &pconfig.IDaddr );
      		break;
      		
      /* Set the EEPROM address in hex */
      case 'e':
      		sscanf( optarg, "%x", &pconfig.EEaddr );
      		break;
      		
      case '?':
      		printf("\n%s\n", VERSION );
      		printf("\nUsage: %s -p[0-2] -i[hex] -e[hex] <filename>\n\n", argv[0] );
      		printf("  -p[0-2]  =  Parallel port to use\n");
      		printf("  -i[hex]  =  HEX value of ID data in memory map.\n");
      		printf("  -e[hex]  =  HEX value of EEPROM data in memory map.\n");
		printf("\n\n");
      		return 0;
      		break;
      		
      case 'm':
      		pconfig.colors = FALSE;
      		break;
      		
      default:	break;
      
    } /* Switch */
  } /* While */


  /* pass lp number on command line */
  if ( !init_port() )
  {
    printf("Failed to initalize /dev/lp%d\n", pconfig.port );
    exit( -1 );
  }

  /* Initalize all the control lines to off */
  vppoff();
  vddoff();
  pgmoff();
  clklo();
  outlo();

  config_device();

  if ( optind < argc )
  {
    /* Load the file from disk */
    if ( load_hex( argv[optind] ) == FALSE )
    {
      fprintf( stderr, "Error loading %s\n", argv[optind] );
      exit( -1 );
    }

    /* Program the device, show errors on stderr */
    program_pic( NULL );
        
    exit( 1 );
  }


  /* Initalize Ncurses and color if available */
  if ( initscr() == NULL  )
  {
    pconfig.colors = FALSE;
  }

  if( pconfig.colors == TRUE )
  {
    if( has_colors() )
    {
      start_color();
      setup_colors();
    }
  } else {
    setup_mono();
  }
  
  noecho();
  cbreak();  
  keypad( stdscr, TRUE );
  

  /* If no valid command line parameters are passed, show the main menu */
  while( !done )
  {
    switch( toupper( main_menu() ) )
    {
      case 'P' :  title();
      		  mvaddstr( 3, 30, "Programming PIC" );
      		  refresh();
      		  scrollw = newwin( 0, 0, 4, 0 );
                  attr_clear( scrollw, 80, 23, COLOR_PAIR( 1 ) );
/*                  touchwin( scrollw ); */
		  refresh();
		  scrollok( scrollw, TRUE );
                  program_pic( scrollw );
		  press_any();
                  delwin( scrollw );
                  touchwin( stdscr );
      		  break;

      case 'R' :  mvaddstr( 3, 32, "Reading PIC" );
      		  refresh();
      		  init_pic();
                  read_pic( stdscr );
                  break;
                  
      case 'D' :  edit_pic();
      		  break;
      
      /* Clear our PIC memory storage area */		  
      case 'F' :  attrset( COLOR_PAIR( 5 ) );
                  mvaddstr( 18, 15, "Are you sure you want to erase RAM?");
                  ch = getch();
                  if( toupper( ch ) == 'Y' )
                  init_pic();
                  attrset( COLOR_PAIR( 1 ) );
                  mvaddstr( 17, 15, "                                   ");
                  break;
      		  
      case 'L' :  load_file();
      		  break;
      		  
      case 'V' :  title();
                  mvaddstr( 3, 30, "Verifying PIC" );
                  refresh();
      		  scrollw = newwin( 0, 0, 4, 0 );
                  attr_clear( scrollw, 80, 23, COLOR_PAIR( 1 ) );
/*                  touchwin( scrollw ); */
		  refresh();
		  scrollok( scrollw, TRUE );
                  verify_pic( scrollw );
		  press_any();
                  delwin( scrollw );
                  touchwin( stdscr );
      		  break;
      		  
      case 'W' :  write_file();
      		  break;

      case 'C' :  config();
      		  break;
      		              
      case 'Q' :  done = 1;
      		  break;

      default: break;
    }
  }

  /* Make sure everything is off when we leave */
  vppoff();
  vddoff();
  clklo();
  outlo();

  endwin();
  
  return 0;
}
