const std = @import("std");
const fs = std.fs;
const io = std.io;
const posix = std.posix;
const mem = std.mem;
const unicode = std.unicode;
const clap = @import("clap");
const ziglyph = @import("ziglyph");
const build_options = @import("build-options");
const docs = @import("docs.zig");
const ps = fs.path.sep_str;

const default_wordlist = if (build_options.embed_wordlist) @embedFile("en.txt") else {};

pub fn main() !void {
    const std_in = io.getStdIn();
    const std_out = io.getStdOut();
    var buffered_std_out = io.bufferedWriter(std_out.writer());
    var buffered_writer = buffered_std_out.writer();

    var gpa = std.heap.GeneralPurposeAllocator(.{}){};
    var arena = std.heap.ArenaAllocator.init(gpa.allocator());
    defer arena.deinit();
    const allocator = arena.allocator();

    const parsers = .{
        .num = clap.parsers.int(u8, 10),
        .str = clap.parsers.string,
    };
    const res = try clap.parse(
        clap.Help,
        &docs.params,
        parsers,
        .{ .allocator = allocator },
    );

    if (res.args.help != 0) {
        return clap.help(
            io.getStdErr().writer(),
            clap.Help,
            &docs.params,
            .{},
        );
    }

    var seed_bytes: [std.rand.DefaultCsprng.secret_seed_length]u8 = undefined;
    std.crypto.random.bytes(&seed_bytes);
    var csprng = std.rand.DefaultCsprng.init(seed_bytes);
    var rand = csprng.random();
    const stat = try std_in.stat();

    if (res.args.password != 0) {
        try writePassword(&buffered_writer, &rand, res.args.length orelse 24);
    } else {
        const file = while (true) {
            if (stat.kind == .character_device) {
                if (res.args.file) |f|
                    break try mapFile(f);

                if (@TypeOf(default_wordlist) == void)
                    @panic("no wordlist available")
                else
                    break default_wordlist;
            }
            break try std_in.readToEndAlloc(
                allocator,
                std.math.maxInt(usize),
            );
        };

        try writePassphrase(
            &buffered_writer,
            &rand,
            file,
            .{
                .num_words = res.args.words orelse 4,
                .add_symbol = res.args.symbol != 0,
                .add_digit = res.args.digit != 0,
                .capitalize = res.args.capitalize != 0,
                .separator = res.args.separator orelse " ",
            },
        );
    }

    if ((try std_out.stat()).kind == .character_device)
        try buffered_writer.writeByte('\n');

    try buffered_std_out.flush();
}

fn mapFile(path: []const u8) ![]align(mem.page_size) u8 {
    const file = try fs.cwd().openFile(path, .{});
    defer file.close();
    const stat = try file.stat();

    if (stat.size == 0)
        return error.InvalidFormat;

    return posix.mmap(
        null,
        stat.size,
        posix.PROT.READ,
        .{ .TYPE = .PRIVATE, .ANONYMOUS = true },
        file.handle,
        0,
    );
}

pub const PassphraseOptions = struct {
    num_words: u8 = 4,
    add_symbol: bool = false,
    add_digit: bool = false,
    capitalize: bool = false,
    separator: []const u8 = " ",
};

pub fn writePassphrase(
    writer: anytype,
    prng: anytype,
    words: []const u8,
    options: PassphraseOptions,
) !void {
    try writeRandomWords(
        writer,
        prng,
        words,
        options,
    );
}

pub fn writePassword(writer: anytype, prng: anytype, letters: u8) !void {
    for (0..letters) |_| {
        _ = try writer.writeByte(prng.intRangeAtMost(u8, '!', '~'));
    }
}

pub fn writeRandomWords(
    writer: anytype,
    prng: anytype,
    words: []const u8,
    options: PassphraseOptions,
) !void {
    const dw = if (options.add_digit)
        prng.uintAtMost(usize, options.num_words - 1)
    else
        null;
    const sw = if (options.add_symbol)
        prng.uintAtMost(usize, options.num_words - 1)
    else
        null;

    for (0..options.num_words) |i| {
        if (i != 0)
            try writer.writeAll(options.separator);

        const word = try findWordSlice(words, prng);

        if (options.capitalize) {
            var buf: [4]u8 = undefined;
            const letter_len = try unicode.utf8ByteSequenceLength(word[0]);
            const first_letter = try unicode.utf8Decode(word[0..letter_len]);
            const upper_len = try unicode.utf8Encode(ziglyph.toUpper(first_letter), &buf);
            try writer.writeAll(buf[0..upper_len]);
            try writer.writeAll(word[letter_len..]);
        } else try writer.writeAll(word);

        if (dw) |p|
            if (p == i)
                try writeRandomDigit(writer, prng);

        if (sw) |p|
            if (p == i)
                try writeRandomSymbol(writer, prng);
    }
}

fn findWordSlice(words: []const u8, prng: anytype) ![]const u8 {
    const i = prng.uintLessThan(usize, words.len);
    const e = i + (mem.indexOfScalar(u8, words[i..], '\n') orelse
        return error.InvalidFormat);

    if (e == 0)
        return error.InvalidFormat;

    const b = if (mem.lastIndexOfScalar(u8, words[0..e], '\n')) |li| li + 1 else 0;

    return words[b..e];
}

fn writeRandomSymbol(writer: anytype, prng: anytype) !void {
    const symbols = [_]u8{
        '!', '"', '#', '$', '%', '&', '\'', '(',
        ')', '*', '+', ',', '-', '.', '/',  ':',
        ';', '<', '=', '>', '?', '@', '[',  '\\',
        ']', '^', '_', '`', '{', '|', '}',  '~',
    };
    const i = prng.uintLessThan(usize, symbols.len);

    try writer.writeByte(symbols[i]);
}

fn writeRandomDigit(writer: anytype, prng: anytype) !void {
    try writer.writeByte(prng.intRangeAtMost(u8, '0', '9'));
}
