/*
 *		     Author:  Nick Spence
 *	      Copyright (C) 1999 All Rights Reserved
 *
 *			      NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#define FALSE 0
#define TRUE !FALSE

#define	LINELENGTH 80

struct rec_hdr {
	int		type;
	int		id;
	unsigned long	offset;
	long		length;
	long		count;
} *rec_hdrs = NULL;

int		resources;
int		indent;
int		curpos;

int		res;
int		bufsz;
unsigned char	*buffer;
int		offset;


int		write_bitmaps	= TRUE;
int		write_code	= TRUE;
int		write_resources	= TRUE;
int		write_header	= TRUE;
int		verbose		= FALSE;
int		indent_size	= 2;
int		pass;

const char	rev[] = "1.0";
const char	*progname;
const char	*filename;
char		hdr_file[256];
char		bin_file[256];
char		res_file[256];
char		dir_name[256];
char		base_name[256];
char		bmap_file[256];

FILE		*ifp;
FILE		*ofp = stdout;
FILE		*hfp = NULL;
FILE		*cfp = NULL;
FILE		*rfp = NULL;


struct res_type {
	char	type[5];
	char	*name;
	void	(*print)(void);
};

void print_msg_resource(void);
void print_string_resource(void);
void print_code_resource(void);
void print_data_resource(void);
void print_msg_resource(void);
void print_word_resource(void);
void print_string_resource(void);
void print_categories_resource(void);
void print_alert_resource(void);
void print_menu_resource(void);
void print_form_resource(void);
void print_bitmap_resource(void);
void print_unknown_resource(void);

#define MAXRESTYPES 20

char unknown[] = "** unknown **";

struct res_type  res_types[MAXRESTYPES] = {
	{ "code", "code",		print_code_resource },
	{ "data", "data",		print_data_resource },
	{ "APPL", "APPLICATION",	print_msg_resource  },
	{ "tver", "VERSION",		print_msg_resource  },
	{ "TRAP", "TRAP",		print_word_resource },
	{ "tAIN", "APPLICATIONICONNAME",print_msg_resource  },
//	{ "tAIS", "CATEGORIES",		print_categories_resource },
	{ "tSTR", "STRING",		print_string_resource },
	{ "Talt", "ALERT",		print_alert_resource },
	{ "MBAR", "MENU",		print_menu_resource },
	{ "tFRM", "FORM",		print_form_resource },
	{ "Tbmp", "BITMAP",		print_bitmap_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
	{ ""    , unknown,		print_unknown_resource },
};

char *control_name[] = {
 "BUTTON", "PUSHBUTTON", "CHECKBOX", "POPUPTRIGGER", "SELECTORTRIGGER", "REPEATBUTTON"
};



void
usage(void)
{
	fprintf(stderr,"\n\nPalm OS Program splitter - rev %s\n\n",rev);
//	fprintf(stderr,"Usage : %s [-help] [-i N] [-nb] [-nc] [-nh] [-nr] [-v] [-o output] file\n\n",progname);
	fprintf(stderr,"Usage : %s [-help] [-i N] [-nc] [-nh] [-nr] [-v] [-o output] file\n\n",progname);
	exit(1);
}


void
help(void)
{
	fprintf(stderr,"\n\nPalm OS Program splitter - rev %s\n\n",rev);
	fprintf(stderr,"Usage: %s  [-help]\n", progname);
	fprintf(stderr,"  [-i N]                   # text indent size, default=2\n");
	fprintf(stderr,"  [-nc]                    # don't write out code files\n");
	fprintf(stderr,"  [-nh]                    # don't write out resource file\n");
	fprintf(stderr,"  [-nr]                    # don't write out header file\n");
	fprintf(stderr,"  [-v]                     # verbose header file output\n");
	fprintf(stderr,"  [-o output-file]         # name of file for output\n");
	fprintf(stderr,"  file                     # path to file to be split\n");
    fprintf(stderr,"\n");
    fprintf(stderr," By default this program will analyze the supplied PRC file and produce\n");
    fprintf(stderr,"the following output files:\n");
    fprintf(stderr,"  filename.hdr - summary of PRC file and the resources it contains.\n");
    fprintf(stderr,"  filename.rcp - resource file in same format as PilRC.\n");
    fprintf(stderr,"  filename.bin - code 1 resource which is the main executable.\n");
    fprintf(stderr,"\n");
    fprintf(stderr," If the file contains any code sections other than 0 or 1 then any sections\n");
    fprintf(stderr,"under 32 bytes long will be listed in the header file and any sections over\n");
    fprintf(stderr,"16 bytes long will be output to files with the name:\n");
    fprintf(stderr,"  codeNNNN.bin - where NNNN is the ID numbder of the code section\n");
    fprintf(stderr,"\n");
	exit(0);
}


void
ferr(const char* format, ...)
{
	va_list ap;

	va_start(ap, format);
	fprintf(stderr,"\n\nError - ");
	vfprintf(stderr, format, ap);
	fprintf(stderr,".\n\n");
	va_end(ap);
	exit(1);
}


int
get_word(unsigned char cp[])
{
	return ((cp[0]<<8) + cp[1]);
}


unsigned long
get_lword(unsigned char cp[])
{
	return (unsigned long) ((((unsigned long) cp[0])<<24) + (((unsigned long) cp[1])<<16) + (((unsigned long) cp[2])<<8) + (unsigned long) cp[3]);
}


void
dump(unsigned long offset, long length)
{
	char	asc[20];
	int	c, bytectr, linectr;
	long	ctr = 0;

//	if (verbose == FALSE) return;

	fseek(ifp, offset, SEEK_SET);

	bytectr = linectr = 0;

	fprintf(ofp,"\n");
	while ((c = getc(ifp)) != EOF && ctr++ < length) {
		if (bytectr == 0) fprintf(ofp,"\n%07x0 :",linectr++);
		fprintf(ofp," %02X", c);
		if (c<' ' || c >= 127) c = '.';
		asc[bytectr] = c;
		if (++bytectr == 16) { asc[bytectr] = 0; bytectr=0; fprintf(ofp," : %s",asc); }
	}

	if (bytectr) {
		asc[bytectr] = 0;
		while (bytectr++<16) fprintf(ofp,"   ");
		fprintf(ofp," : %s",asc);
	}
	fprintf(ofp,"\n");

}


char *
preprocess_string(int menu, unsigned char str[])
{
	unsigned char	*tbuffer, ch;
	int	len, len1;

	len = strlen(str);
	tbuffer = (unsigned char*) malloc(4*len+1);
	if (tbuffer == NULL) ferr("cannot allocate memory for string data");

	for (len = len1 = 0; (ch = str[len]) != 0; len++) {
		if (menu && ((int) ch) == 0x85 && str[len+1] == '\0') {
			len1 += sprintf((char*)&str[len1], "...");
		} else if (ch < ' ' || ch == '\\' || ch == '"' || ch >= 0x7f) {
			tbuffer[len1++] = '\\';
			if (ch >= ' ' && ch < 0x7f) tbuffer[len1++] = ch;
			else if (ch ==0x09) tbuffer[len1++] = 't';
			else if (ch ==0x0a) tbuffer[len1++] = 'n';
			else len1 += sprintf((char*)&tbuffer[len1],"%03o",ch);
		} else {
			tbuffer[len1++] = ch;
		}
	}
	tbuffer[len1] = '\0';
	return((char*)tbuffer);
}


int
print_string(int newline, const char* format, ...)
{
	va_list ap;
	int	len;

	va_start(ap, format);
	if (newline) {
		fprintf(ofp, "\n%*s",indent,"");
		curpos = indent;
	}
	len = vfprintf(ofp, format, ap);
	curpos += len;
	va_end(ap);
	return(len);
}


void
print_begin(void) {
	print_string(1,"BEGIN");
	indent += indent_size;
}


void
print_end(void) {
	indent -= indent_size;
	print_string(1,"END");
}


void
print_res_str(int menu, unsigned char *buffer, int *offset)
{
	char *tbuf;

	tbuf = preprocess_string(menu, buffer + (*offset));
	print_string(0," \"%s\"",tbuf);
	(*offset) += strlen(buffer + (*offset)) + 1;
	free(tbuf);
}


void
print_res_lstr(int entries, unsigned char *buffer, int *offset)
{
	while (entries-- > 0)
		print_res_str(0, buffer, offset);
}

void
print_res_sstr(int res, unsigned char *buffer, int *offset)
{
	char tmp[LINELENGTH+1], *tbuf, ch;
	int  len, len1, toffset;
	int  tmpindent;

	tbuf = preprocess_string(0, buffer + (*offset));
	(*offset) += strlen(buffer + (*offset)) + 1;

	tmpindent = indent;
	indent = curpos + 1;
	toffset = 0;
	do {
		len = strlen(tbuf + toffset);
		if (len + curpos + 4 >= LINELENGTH) {
			len1 = LINELENGTH - curpos - 5;
			strncpy(tmp, tbuf + toffset, len1);
			for (len = len1; len && !isspace(tmp[len-1]) && tmp[len-1] != '.'; len--);
			if (len==0) len = len1;
			tmp[len] = '\0';
		}
		else
			strcpy(tmp, tbuf + toffset);
		toffset += len;
		print_string(0, " \"%s\"",tmp);
		if (tbuf[toffset]) {
			fprintf(ofp,"\\\n%*s", indent-1," ");
			curpos = indent - 1;
		}
	} while (tbuf[toffset]);
	indent = tmpindent;
	free(tbuf);
}


void
dump_code_resource(void)
{
	int		ctr;
	FILE		*bfp;

	if (!(bfp = fopen(bin_file,"wb"))) ferr("cannot open file '%s' for writing",bin_file);
	fseek(ifp, rec_hdrs[res].offset, SEEK_SET);
	for (ctr=rec_hdrs[res].length; ctr>0; ctr--) putc(getc(ifp),bfp);
	fclose(bfp);
}


void
print_code_resource(void)
{
	int		ctr;

	if (rec_hdrs[res].id == 0 && bufsz >= 16 && pass == 0)
	{

		if (!verbose) print_string(1,"");
		fprintf(ofp,"\nCode Jump Table:\n");
		fprintf(ofp,"\n  Size above A5            : 0x%08lx\n", get_lword(&buffer[0]));
		fprintf(ofp,"  Size of Globals          : 0x%08lx\n", get_lword(&buffer[4]));
		fprintf(ofp,"  Size of Jump Table       : 0x%08lx\n", get_lword(&buffer[8]));
		fprintf(ofp,"  A5 offset of Jump Table  : 0x%08lx\n", get_lword(&buffer[12]));

		for (ctr=16; ctr<bufsz; ctr+=8)
		{
			fprintf(ofp,"             Jump entry %2d : ", (ctr-16)/8);
			if (bufsz-ctr < 8)
				fprintf(ofp,"Incomplete Entry\n");
			else if (get_word(&buffer[ctr+2]) == 0x3f3c && get_word(&buffer[ctr+6]) == 0xa9f0)
				fprintf(ofp,"Code segment = %d, Offset = 0x%04x\n",
					get_word(&buffer[ctr+4]), get_word(&buffer[ctr]));
			else
				fprintf(ofp,"Invalid Entry,  0x%04x 0x%04x 0x%04x 0x%04x\n",
					get_word(&buffer[ctr]), get_word(&buffer[ctr+2]),
					get_word(&buffer[ctr+4]), get_word(&buffer[ctr+6]));
		}
		print_string(1,"");
	} else if (bufsz < 32 && pass == 0) {
		fprintf(ofp,"\nCode Resource: %d\n", res);
        if (!verbose)
		  dump(rec_hdrs[res].offset, rec_hdrs[res].length);
    }
}


void
print_data_resource(void)
{
	char		asc[20];
	int		bytes, ctr, ectr;
	int		global, bytectr;
	unsigned long	address, start;
	unsigned char	ebuf[256];
	unsigned long   codexref_offset;

	if (pass) return;

	if (!verbose) print_string(1,"");
	codexref_offset = get_lword(&buffer[0]);
	fprintf(ofp,"\nGlobal Data Table:\n");
	fprintf(ofp,"\n  Offset of Code1 xrefs    : 0x%08lx\n", codexref_offset);

	ctr = 4;
	for (global=0; global<3; global++)
	{
		offset = get_lword(&buffer[ctr]);
		start = offset & 0xfffffff0L;
		ectr  = offset & 0xf;
		ctr += 4;
		
		fprintf(ofp,"\n  Global data block %d preloaded at offset : 0x%08lx\n", global, offset);
		if (buffer[ctr])
		{
			while (buffer[ctr] && ctr < codexref_offset)
			{
				if (buffer[ctr] >= 0x80) {
					bytes = buffer[ctr++] - 0x7f;
					while (bytes--) ebuf[ectr++] = buffer[ctr++];
				} else if (buffer[ctr] >= 0x40) {
					bytes = buffer[ctr++] - 0x3f;
					while (bytes--) ebuf[ectr++] = 0;
				} else if (buffer[ctr] >= 0x20) {
					bytes = buffer[ctr++] - 0x1e;
					while (bytes--) ebuf[ectr++] = buffer[ctr];
					ctr++;
       			 } else if (buffer[ctr] >= 0x10) {
       				 bytes = buffer[ctr++] - 0xf;
       				 while (bytes--) ebuf[ectr++] = 0xff;
       			 } else if (buffer[ctr] == 0x04) {
					ebuf[ectr++] = 0xa9;
					ebuf[ectr++] = 0xf0;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = buffer[ctr+1];
					ebuf[ectr++] = buffer[ctr+2];
					ebuf[ectr++] = buffer[ctr+3];
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = buffer[ctr+4];
       				 ctr+=5;
       			 } else if (buffer[ctr] == 0x03) {
					ebuf[ectr++] = 0xa9;
					ebuf[ectr++] = 0xf0;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = buffer[ctr+1];
					ebuf[ectr++] = buffer[ctr+2];
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = buffer[ctr+3];
       				 ctr+=4;
       			 } else if (buffer[ctr] == 0x02) {
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0xff;
					ebuf[ectr++] = buffer[ctr+1];
					ebuf[ectr++] = buffer[ctr+2];
					ebuf[ectr++] = buffer[ctr+3];
       				 ctr+=4;
       			 } else if (buffer[ctr] == 0x01) {
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0x00;
					ebuf[ectr++] = 0xff;
					ebuf[ectr++] = 0xff;
					ebuf[ectr++] = buffer[ctr+1];
					ebuf[ectr++] = buffer[ctr+2];
       				 ctr+=3;
				} else {
					fprintf(ofp,"\n Unknown control command byte 0x%02x\n", buffer[ctr++]);
					break;
				}
	
				bytectr = 0;
				while (ectr >= bytectr + 16) {
					fprintf(ofp,"%08lx :", start);
					do {
						if (start+bytectr < offset) {
							fprintf(ofp," ..");
							asc[bytectr%16] = ' ';
						} else {
							fprintf(ofp," %02x", ebuf[bytectr]);
							asc[bytectr%16] = (ebuf[bytectr]<' ' || ebuf[bytectr]>127 ? '.' : ebuf[bytectr]);
						}
						bytectr++;
					} while (bytectr%16);
					asc[16] = '\0';
					fprintf(ofp," : %s\n",asc);
					start += 16;
				}
				while (ectr > bytectr) {
					ebuf[bytectr%16] = ebuf[bytectr++];
				}
				ectr = ectr%16;
			}
	
			bytectr = 0;
			if (ectr > bytectr) {
				fprintf(ofp,"%08lx :", start);
				do {
					if (start+bytectr < offset) {
						fprintf(ofp," ..");
						asc[bytectr%16] = ' ';
					} else {
						fprintf(ofp," %02x", ebuf[bytectr]);
						asc[bytectr%16] = (ebuf[bytectr]<' ' || ebuf[bytectr]>127 ? '.' : ebuf[bytectr]);
					}
					bytectr++;
				}
				while (ectr > bytectr);
				asc[bytectr] = '\0';
	
				while (ectr++ < 16) fprintf(ofp," ..");
				fprintf(ofp," : %s\n",asc);
			}
		}
		else
			fprintf(ofp,"    Zero length datastream - no data loaded\n");
		ctr++;
	}

	if (ctr + 12 != codexref_offset) {
		fprintf(ofp,"\n\nWARNING - data block did not decompress correctly - aborting block decode.\n\n");
	} else {
		fprintf(ofp,"\n  Data 0 xrefs    : 0x%08lx 0x%08lx 0x%08lx\n", 
		get_lword(&buffer[ctr]), get_lword(&buffer[ctr+4]), get_lword(&buffer[ctr+8]));
		fprintf(ofp,"\n  Code 0 xrefs    : 0x%08lx 0x%08lx 0x%08lx\n", 
		get_lword(&buffer[ctr+12]), get_lword(&buffer[ctr+16]), get_lword(&buffer[ctr+20]));
	}

}


void
print_msg_resource(void)
{
	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_res_str(0, buffer, &offset);
	print_string(1,"");
}


void
print_string_resource(void)
{
	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_res_sstr(0, buffer, &offset);
	print_string(1,"");
}


void
print_word_resource(void)
{
	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_string(0, " %d",get_word(&buffer[0]));
	print_string(1,"");
}


void
print_categories_resource(void)
{
	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_res_lstr(16, buffer, &offset);
	print_string(1,"");
}


void
print_alert_resource(void)
{
	int i;

	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	if (get_word(&buffer[2])) print_string(1,"HELPID %d",get_word(&buffer[2]));
	if (get_word(&buffer[0]) == 0 ) print_string(1,"INFORMATION");
	if (get_word(&buffer[0]) == 1 ) print_string(1,"CONFIRMATION");
	if (get_word(&buffer[0]) == 2 ) print_string(1,"WARNING");
	if (get_word(&buffer[0]) == 3 ) print_string(1,"ERROR");
	offset = 8;
	print_begin();
	print_string(1,"TITLE");
	print_res_str(0, buffer, &offset);
	print_string(1,"MESSAGE");
	print_res_sstr(res, buffer, &offset);
	print_string(1,"BUTTONS");
	for (i = get_word(&buffer[4]); i; i--)
		print_res_str(0, buffer, &offset);
	print_end();
	print_string(1,"");
}


void
print_menu_resource(void)
{
	int pulldown, item, item_ptr;

	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_begin();
	for (pulldown=0; pulldown<get_word(&buffer[26]); pulldown++) {
		print_string(1,"PULLDOWN");
		item_ptr = get_lword(&buffer[62+pulldown*34]);
		offset = get_lword(&buffer[56+pulldown*34]);
		print_res_str(1, buffer, &offset);
		print_begin();
		for (item=0; item<get_word(&buffer[60+pulldown*34]); item++) {
			print_string(1,"MENUITEM");
			offset = get_lword(&buffer[item_ptr+4+8*item]);
			if (strcmp("-",&buffer[offset])) {
				print_res_str(1, buffer, &offset);
				print_string(0,"%*s ID %d", 38-curpos, "", get_word(&buffer[item_ptr+8*item]));
				if (buffer[item_ptr+2+8*item]) print_string(0," \"%c\"", buffer[item_ptr+2+8*item]);
			} else print_string(0," SEPARATOR");
		}
		print_end();
	}
	print_end();
	print_string(1,"");
}


void
print_form_resource(void)
{
	int item, style, entries;

	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_string(0," AT (%d %d %d %d)",get_word(&buffer[10]),get_word(&buffer[12]),get_word(&buffer[14]),get_word(&buffer[16]));
	if (buffer[8] & 0x20) print_string(1,"MODAL");
	else {
		if ((buffer[30] & 0x03) == 2) print_string(1,"BOLDFRAME");
	}
	if ((buffer[30] & 0x03) == 0) print_string(1,"NOFRAME");
	if ((buffer[42] & 0x08) == 0) print_string(1,"NOSAVEBEHIND");
	if (buffer[42] & 0x04) print_string(1,"GRAFFITISHIFT");
	if (get_word(&buffer[58])) print_string(1,"HELPID %d",get_word(&buffer[58]));
	if (get_word(&buffer[60])) print_string(1,"MENUID %d",get_word(&buffer[60]));
	if (get_word(&buffer[56])) print_string(1,"DEFAULTBTNID %d",get_word(&buffer[56]));
	offset = 64;
	print_begin();
	for (item=0; item < get_word(&buffer[62]); item++) {
		offset = get_lword(&buffer[70+item*6]);
		switch(buffer[68+item*6]) {
			case 0 : /* tFLD - Field */
				print_string(1,"FIELD ID %d AT (%d %d %d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]),get_word(&buffer[offset+8]));
				offset += 10;
				if ((buffer[offset] & 0x80) == 0) print_string(0," NONUSABLE");
				if ((buffer[offset] & 0x20) != 0) print_string(0," EDITABLE");
				if ((buffer[offset] & 0x20) == 0) print_string(0," NONEDITABLE");
				if ((buffer[offset] & 0x10) != 0) print_string(0," SINGLELINE");
				if ((buffer[offset] & 0x10) == 0) print_string(0," MULTIPLELINES");
				if ((buffer[offset] & 0x04) != 0) print_string(0," DYNAMICSIZE");
				offset++;
				if ((buffer[offset] & 0xc0) != 0) print_string(0," UNDERLINED");
				if ((buffer[offset] & 0x30) == 0) print_string(0," LEFTALIGN");
				if ((buffer[offset] & 0x30) == 0x10) print_string(0," CENTERALIGN");
				if ((buffer[offset] & 0x30) == 0x20) print_string(0," RIGHTALIGN");
				if ((buffer[offset] & 0x08) != 0) print_string(0," AUTOSHIFT");
				if ((buffer[offset] & 0x04) != 0) print_string(0," SCROLLBAR");
				if ((buffer[offset] & 0x02) != 0) print_string(0," NUMERIC");
				print_string(0," MAXCHARS %d", get_word(&buffer[offset+17]));
				if (buffer[offset+27])
					print_string(0," FONT %d", buffer[offset+27]);
				break;
			case 1 : /* tCTL - Control */
				style = buffer[offset+16];
				print_string(1,"%s", control_name[style]);
				offset += 20;
				print_res_str(0, buffer, &offset);
				offset = get_lword(&buffer[70+item*6]);
				print_string(0," ID %d AT (%d %d %d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]),get_word(&buffer[offset+8]));
				offset += 14;
				if ((buffer[offset] & 0x80) == 0) print_string(0," NONUSABLE");
				if ((buffer[offset] & 0x40) == 0) print_string(0," DISABLED");
				if ((buffer[offset] & 0x08) != 0) print_string(0," LEFTANCHOR");
				if ((buffer[offset] & 0x08) == 0) print_string(0," RIGHTANCHOR");
				if (style == 0 || style == 5) {
					if ((buffer[offset] & 0x03) == 2) print_string(0," BOLDFRAME");
					if ((buffer[offset] & 0x03) == 0) print_string(0," NOFRAME");
				}
				print_string(0," FONT %d", buffer[offset+3]);
				if (buffer[offset+4])
					print_string(0," GROUP %d", buffer[offset+4]);
				if ((buffer[offset] & 0x10) != 0) print_string(0," CHECKED");
				break;
			case 2 : /* tLST - List */
				print_string(1,"LIST");
                entries = get_word(&buffer[offset+16]);
			    offset += (get_word(&buffer[offset+16])*4) + 32;
				print_res_lstr(entries, buffer, &offset);
				offset = get_lword(&buffer[70+item*6]);
				print_string(0," ID %d AT (%d %d %d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]),get_word(&buffer[offset+8]));
				offset += 10;
				if ((buffer[offset] & 0x80) == 0) print_string(0," NONUSABLE");
				if ((buffer[offset] & 0x40) == 0) print_string(0," DISABLED");
				print_string(0," FONT %d", buffer[offset+11]);
				break;
			case 3 : /* tTBL - Table */
				print_string(1,"TABLE ID %d AT (%d %d %d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]),get_word(&buffer[offset+8]));
				if ((buffer[offset+10] & 0x40) == 0) print_string(0," NONEDITABLE");
				print_string(0," ROWS %d COLUMNS %d COLUMNWIDTHS",
				  get_word(&buffer[offset+14]), get_word(&buffer[offset+12]));
				style = get_word(&buffer[offset+12]);
				offset += 34;
				while (style-- > 0) {
					print_string(0, " %d", get_word(&buffer[offset]));
					offset += 18;
				}
				break;
			case 4 : /* tFBM - Bitmap */
				print_string(1,"FORMBITMAP AT (%d %d) BITMAP %d",
				  get_word(&buffer[offset+2]), get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]));
				if ((buffer[offset] & 0x80) == 0) print_string(0," NONUSABLE");
				break;
			case 8 : /* tLBL - Label */
				print_string(1,"LABEL");
				offset += 14;
				print_res_str(0, buffer, &offset);
				offset = get_lword(&buffer[70+item*6]);
				print_string(0," ID %d AT (%d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]));
				if ((buffer[offset+6] & 0x80) == 0) print_string(0," NONUSABLE");
				print_string(0," FONT %d", buffer[offset+8]);
				break;
			case 9 : /* tTTL - Title */
				print_string(1,"TITLE");
				offset += 12;
				print_res_str(0, buffer, &offset);
				break;
			case 10 : /* tPUL - Popup */
				print_string(1,"POPUPLIST ID %d %d", get_word(&buffer[offset]),
				  get_word(&buffer[offset+2]));
				break;
			case 11 : /* tGSI - Graffitistate */
				print_string(1,"GRAFFITISTATEINDICATOR AT (%d %d)",
				  get_word(&buffer[offset]),get_word(&buffer[offset+2]));
				break;
			case 12 : /* tGDT - Gadget */
				print_string(1,"GADGET ID %d AT (%d %d %d %d)", get_word(&buffer[offset]),
				  get_word(&buffer[offset+4]),get_word(&buffer[offset+6]),
				  get_word(&buffer[offset+8]),get_word(&buffer[offset+10]));
				if ((buffer[offset+2] & 0x80) == 0) print_string(0," NONUSABLE");
				break;
			case 13 : /* tSCL - Scrollbar */
				print_string(1,"SCROLLBAR ID %d AT (%d %d %d %d)", get_word(&buffer[offset+8]),
				  get_word(&buffer[offset+2]),get_word(&buffer[offset+4]),
				  get_word(&buffer[offset+6]),get_word(&buffer[offset+8]));
				if ((buffer[offset+10] & 0x80) == 0) print_string(0," NONUSABLE");
				print_string(0,"VALUE %d MIN %d MAX %d  PAGESIZE %d)",
				  get_word(&buffer[offset+12]),get_word(&buffer[offset+14]),
				  get_word(&buffer[offset+16]),get_word(&buffer[offset+18]));
				break;
			case 5 : /* tLIN */
			case 6 : /* tFRA */
			case 7 : /* tREC */
			default : print_string(1,"// UNKNOWN FORM OBJECT %d", buffer[68+item*6]); break;
		}
	}

	print_end();
	print_string(1,"");
}


void print_bitmap_resource(void)
{
	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
}


void print_unknown_resource(void)
{
//	print_string(1, "%s ID %d", res_types[rec_hdrs[res].type].name, rec_hdrs[res].id);
	print_string(1, "// UNKNOWN RESOURCE TYPE - %s ID %d", res_types[rec_hdrs[res].type].type, rec_hdrs[res].id);
}


void
print_resource(void)
{
	int	 i;


	if (pass == 0 && verbose) {
		fprintf(ofp,"\n\nResource %2d :\n", res);
		fprintf(ofp,"  Type   : %s\n", res_types[rec_hdrs[res].type].type);
		fprintf(ofp,"  ID     : %4d\n", rec_hdrs[res].id);
		fprintf(ofp,"  Offset : 0x%08lx\n", rec_hdrs[res].offset);
		fprintf(ofp,"  Length : 0x%08lx\n", rec_hdrs[res].length);
	}

	fseek(ifp, rec_hdrs[res].offset, SEEK_SET);

	if (rec_hdrs[res].type || rec_hdrs[res].id == 0 || rec_hdrs[res].length < 32) {
		buffer = (unsigned char*) malloc(rec_hdrs[res].length+1);
		if (buffer == NULL) ferr("cannot allocate memory for resource data");

		bufsz = fread((void*) buffer, (size_t) 1, (size_t) rec_hdrs[res].length, ifp);
		if (bufsz != rec_hdrs[res].length) ferr("cannot read resource data");

		buffer[rec_hdrs[res].length] = '\0';
		offset = 0;
	}

	indent = 0;
	if (pass || rec_hdrs[res].type < 2 || verbose)
		res_types[rec_hdrs[res].type].print();

	if (rec_hdrs[res].type || rec_hdrs[res].id == 0 || rec_hdrs[res].length < 32) free(buffer);

	if (verbose && (rec_hdrs[res].type || rec_hdrs[res].length < 256))
		dump(rec_hdrs[res].offset, rec_hdrs[res].length);
}


#define HEADER_SIZE	78
#define RES_HDR_SIZE    10

void
split(void)
{
	unsigned char	header[HEADER_SIZE];
	size_t		hdrsz;
	int		ctr, i;
	long		count, next_count;

	pass = 0;

	hdrsz = fread((void*) header, (size_t) 1, (size_t) HEADER_SIZE, ifp);
	if (hdrsz != HEADER_SIZE) ferr("cannot read file header");


	resources = get_word(&header[76]);

	if (write_header) {
		if (!(ofp = fopen(hdr_file,"w"))) ferr("cannot open file '%s' for writing",hdr_file);
		fprintf(ofp,"Header Information\n");
		fprintf(ofp,"  Name             : '");
		for (ctr=0; ctr<32; ctr++) fprintf(ofp,"%c",header[ctr]?header[ctr]:'.');
		fprintf(ofp,"'\n");
		fprintf(ofp,"  Flags            : 0x%04x\n", get_word(&header[32]));
		fprintf(ofp,"  Version          : 0x%04x\n", get_word(&header[34]));
		fprintf(ofp,"  Create Time      : 0x%08lx\n", get_lword(&header[36]));
		fprintf(ofp,"  Mod Time         : 0x%08lx\n", get_lword(&header[40]));
		fprintf(ofp,"  Backup Time      : 0x%08lx\n", get_lword(&header[44]));
		fprintf(ofp,"  Mod Number       : 0x%08lx\n", get_lword(&header[48]));
		fprintf(ofp,"  App Info         : 0x%08lx\n", get_lword(&header[52]));
		fprintf(ofp,"  Sort Info        : 0x%08lx\n", get_lword(&header[56]));
		fprintf(ofp,"  Type             : 0x%08lx = '%c%c%c%c'\n", get_lword(&header[60]), header[60], header[61], header[62], header[63]);
		fprintf(ofp,"  ID               : 0x%08lx = '%c%c%c%c'\n", get_lword(&header[64]), header[64], header[65], header[66], header[67]);
        fprintf(ofp,"  Unique ID Seed   : 0x%08lx\n", get_lword(&header[68]));
		fprintf(ofp,"  Next Record List : 0x%08lx\n", get_lword(&header[72]));
		fprintf(ofp,"  Resources        : %2d\n", resources);
	}

	rec_hdrs = (struct rec_hdr*) calloc(resources,sizeof(struct rec_hdr));
	if (rec_hdrs == NULL) ferr("cannot allocate memory for resource headers");


	fseek(ifp, HEADER_SIZE, SEEK_SET);
	for (res=0;res<resources;res++)
	{
		hdrsz = fread((void*) header, (size_t) 1, (size_t) RES_HDR_SIZE, ifp);
		if (hdrsz != RES_HDR_SIZE) ferr("cannot read resource header");

		for (i=0;res_types[i].type[0];i++) {
			if (strncmp(res_types[i].type,header,4)==0) break;
			if (i+1 == MAXRESTYPES) ferr("too many new resource names");
		}
		if (res_types[i].type[0] == '\0') {
			strncpy(res_types[i].type,header,4);
			res_types[i].type[4] = '\0';
		}
		rec_hdrs[res].type    = i;
		rec_hdrs[res].id      = get_word(&header[4]);
		rec_hdrs[res].offset  = get_lword(&header[6]);
		rec_hdrs[res].count   = (i << 16) + rec_hdrs[res].id;
	}

	fseek(ifp, 0, SEEK_END);
	for (res=0;res<resources;res++)
		rec_hdrs[res].length = (res == resources-1 ? ftell(ifp) : rec_hdrs[res+1].offset) - rec_hdrs[res].offset;

	if (write_header) {
		fprintf(ofp,"\n     Resource   Type                         ID     Offset      Length");
		fprintf(ofp,"\n     ========   ====                        ====  ==========  ==========\n");
		for (res=0;res<resources;res++)
		{
			fprintf(ofp,"      %4d      %-4s = %-19s %5d  0x%08lx  0x%08lx\n",
			  res, res_types[rec_hdrs[res].type].type, res_types[rec_hdrs[res].type].name,
			  rec_hdrs[res].id, rec_hdrs[res].offset, rec_hdrs[res].length);
		}
		if (verbose) dump(0L, HEADER_SIZE + 10 * resources);

		for (res=0;res<resources;res++) print_resource();

		fclose(ofp);

	}

	ctr = 0;
    for (res=0;res<resources;res++) {
		if (write_code && strcmp(res_types[rec_hdrs[res].type].type,"code")==0 &&
           (rec_hdrs[res].id>0 || rec_hdrs[res].length > 512))
            ctr++;
	}

	for (res=0;res<resources;res++) {
		if (write_code && strcmp(res_types[rec_hdrs[res].type].type,"code")==0 &&
           (rec_hdrs[res].id>0 || rec_hdrs[res].length > 512)) {
            if (ctr>1)
                sprintf(bin_file,"%scode%04d.bin",dir_name,rec_hdrs[res].id);
            dump_code_resource();
        }
	}

	if (write_resources) {
		if (!(ofp = fopen(res_file,"w"))) ferr("cannot open file '%s' for writing",res_file);
		pass++;
		verbose = FALSE;
		count = 0;
		while (count >= 0) {
			next_count = -1;
			for (res=0;res<resources;res++) {
				if (rec_hdrs[res].count > count &&
				   (rec_hdrs[res].count < next_count || next_count == -1))
					next_count = rec_hdrs[res].count;
				if (rec_hdrs[res].count == count) print_resource();
			}
			count = next_count;
		}
		fclose(ofp);
	}


}

/* splitprc
  -nb no bitmaps
  -nc no code
  -nr no rcp
  -nh no hdr
  -v verbose
*/

int main(int argc, char *argv[])
{
	char asc[20];
	int c, bytectr, linectr, ctr;

	progname = argv[0];
	hdr_file[0] = '\0';

	while (--argc && **++argv == '-') {
		if (strcmp("-nb", *argv) == 0) {
			write_bitmaps = FALSE;
		} else if (strcmp("-nc", *argv) == 0) {
			write_code = FALSE;
		} else if (strcmp("-nr", *argv) == 0) {
			write_resources = FALSE;
		} else if (strcmp("-nh", *argv) == 0) {
			write_header = FALSE;
		} else if (strcmp("-i", *argv) == 0 && argc) {
			indent_size = atoi(*++argv);
			argc--;
		} else if (strcmp("-v", *argv) == 0) {
			verbose = TRUE;
		} else if (strcmp("-o", *argv) == 0 && argc) {
			strcpy(hdr_file, *++argv);
			argc--;
		} else if (strncmp("-h", *argv, 2) == 0) {
			help();
		} else {
			fprintf(stderr, "%s: bad option: %s\n", progname, *argv);
			usage();
		}
	}

	if (argc != 1) usage();

	if (!write_header) verbose = FALSE;

	filename = *argv;
	strcpy(dir_name, (hdr_file[0] ? hdr_file : filename));
	for (ctr = strlen(dir_name); ctr>=0 && dir_name[ctr] != '/' && dir_name[ctr] != '\\' && dir_name[ctr] != ':'; ctr--);
	strcpy(base_name,&dir_name[ctr+1]);
	dir_name[ctr+1] = '\0';

	if (strlen(base_name)>3 && base_name[strlen(base_name)-4]=='.') {
		for (ctr=0; ctr<4; ctr++) bin_file[ctr]=toupper(base_name[strlen(base_name)-3+ctr]);
		if ((!hdr_file[0] && strcmp(bin_file,"PRC")==0) ||
		    ( hdr_file[0] && strcmp(bin_file,"HDR")==0) ||
		    ( hdr_file[0] && strcmp(bin_file,"RCP")==0) ||
		    ( hdr_file[0] && strcmp(bin_file,"BIN")==0))
			base_name[strlen(base_name)-4]='\0';
	}

	sprintf(hdr_file,"%s%s.hdr",dir_name,base_name);
	sprintf(bin_file,"%s%s.bin",dir_name,base_name);
	sprintf(res_file,"%s%s.rcp",dir_name,base_name);

	if (!(ifp = fopen(filename,"rb"))) ferr("cannot open file '%s' for reading",filename);

	split();

	fclose(ifp);

	return(0);
}


