/* 
 * otpCalc - A one time password calculator.
 *
 * Copyright (C) 2001 Anthony D. Urso
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 */

#include <string.h>

#include "crypto.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef HAVE_MD4
# include <openssl/md4.h>
#else
# include "global.h"
# include "md4.h"
#endif /* HAVE_MD4 */

#ifdef HAVE_MD5
# include <openssl/md5.h>
#else
# include "global.h"
# include "md5.h"
#endif /* HAVE_MD5 */

#ifdef HAVE_RMD160
# include <openssl/ripemd.h>
#else
# include "rmd160.h"
#endif /* HAVE_RMD160 */

#ifdef HAVE_SHA1
# include <openssl/sha.h>
#else
# include "sha1.h"
#endif /* HAVE_SHA1 */



/*
 * Perform an MD5 transform on the passwd+seed, and fold the results into 64
 * bits.
 */

void md4lite(char *message, unsigned int len)
{

#ifndef HAVE_MD4
	MD4_CTX ctx;
#endif
	unsigned char digest[16];
	unsigned short i;


#ifdef HAVE_MD4
	MD4(message, len, digest);
#else
	MD4Init(&ctx);
	MD4Update(&ctx, message, len);
	MD4Final(digest, &ctx);
#endif /* HAVE_LIBCRYPTO */

	/* Fold the 128 bit result to 64 bits */
	for (i = 0; i < 8; i++)
		digest[i] ^= digest[i+8];

	memcpy(message, digest, 8);

}


/*
 * Perform an MD5 transform on the passwd+seed, and fold the results into 64
 * bits.
 */

void md5lite(char *message, unsigned int len)
{

#ifndef HAVE_MD5
	MD5_CTX ctx;
#endif
	unsigned char digest[16];
	unsigned short i;


#ifdef HAVE_MD5
	MD5(message, len, digest);
#else
	MD5Init(&ctx);
	MD5Update(&ctx, message, len);
	MD5Final(digest, &ctx);
#endif /* HAVE_LIBCRYPTO */

	/* Fold the 128 bit result to 64 bits */
	for (i = 0; i < 8; i++)
		digest[i] ^= digest[i+8];

	memcpy(message, digest, 8);

}


/*
 * Perform an RIPEMD-160 transform on the passwd+seed, and fold the results
 * into 64 bits.
 */

void rmd160lite(char *message, unsigned int len)
{

#ifndef HAVE_RMD160
	dword ctx[5];
	dword X[16];
	dword nbytes;
#endif
	unsigned char digest[20];
	unsigned short i;


#ifdef HAVE_RMD160
	RIPEMD160(message, len, digest);
#else
	MDinit(ctx);

	for (nbytes = len; nbytes > 63; nbytes -= 64) {
		for (i = 0; i < 16; i++) {
			X[i] = BYTES_TO_DWORD(message);
			message += 4;
		}
		compress(ctx, X);
	}

	MDfinish(ctx, message, len, 0);

	for (i = 0; i < 160 / 8; i += 4) {
		digest[i] = ctx[i >> 2];
		digest[i + 1] = (ctx[i >> 2] >> 8);
		digest[i + 2] = (ctx[i >> 2] >> 16);
		digest[i + 3] = (ctx[i >> 2] >> 24);
	}
#endif /* HAVE_LIBCRYPTO */

	/* Fold the 160 bit result to 64 bits */
	for (i = 0; i < 8; i++)
		digest[i] ^= digest[i+8];
	for (i = 0; i < 4; i++)
		digest[i] ^= digest[i+16];

	memcpy(message, digest, 8);

}


/*
 * Perform an SHA-1 transform on the passwd+seed, and fold the results
 * into 64 bits.
 */

void sha1lite(char *message, unsigned int len)
{

#ifndef HAVE_SHA1
	SHA1_CTX ctx;
#endif
	unsigned char digest[20];
	unsigned short i;


#ifdef HAVE_SHA1
	SHA1(message, len, digest);
#else
	SHA1Init(&ctx);
	SHA1Update(&ctx, message, len);
	SHA1Final(digest, &ctx);
#endif /* HAVE_SHA1 */

	/* Fold the 160 bit result to 64 bits */
	for (i = 0; i < 8; i++)
		digest[i] ^= digest[i+8];
	for (i = 0; i < 4; i++)
		digest[i] ^= digest[i+16];

	memcpy(message, digest, 8);

}
