package osvscanner

import (
	"testing"

	"github.com/google/osv-scanner/internal/config"
	"github.com/google/osv-scanner/internal/testutility"
	"github.com/google/osv-scanner/pkg/lockfile"
	"github.com/google/osv-scanner/pkg/models"
	"github.com/google/osv-scanner/pkg/osv"
	"github.com/google/osv-scanner/pkg/reporter"
)

func Test_assembleResult(t *testing.T) {
	t.Parallel()
	type args struct {
		r            reporter.Reporter
		packages     []scannedPackage
		vulnsResp    *osv.HydratedBatchedResponse
		licensesResp [][]models.License
		actions      ScannerActions
		config       *config.ConfigManager
	}
	packages := []scannedPackage{
		{
			Name:      "pkg-1",
			Ecosystem: lockfile.Ecosystem("npm"),
			Version:   "1.0.0",
			Source: models.SourceInfo{
				Path: "dir/package-lock.json",
				Type: "lockfile",
			},
		},
		{
			Name:      "pkg-2",
			Ecosystem: lockfile.Ecosystem("npm"),
			Version:   "1.0.0",
			Source: models.SourceInfo{
				Path: "dir/package-lock.json",
				Type: "lockfile",
			},
		},
		{
			Name:      "pkg-3",
			Ecosystem: lockfile.Ecosystem("npm"),
			Version:   "1.0.0",
			Source: models.SourceInfo{
				Path: "other-dir/package-lock.json",
				Type: "lockfile",
			},
		},
	}
	vulnsResp := &osv.HydratedBatchedResponse{
		Results: []osv.Response{
			{Vulns: models.Vulnerabilities([]models.Vulnerability{
				{
					ID:      "GHSA-123",
					Aliases: []string{"CVE-123"},
				}, {
					ID: "CVE-123",
				},
			})},
			{Vulns: models.Vulnerabilities{}},
			{Vulns: models.Vulnerabilities([]models.Vulnerability{
				{
					ID: "GHSA-456",
				},
			})},
		},
	}
	licensesResp := [][]models.License{
		{models.License("MIT"), models.License("0BSD")},
		{models.License("MIT")},
		{models.License("UNKNOWN")},
	}
	makeLicensesResp := func() [][]models.License {
		cpy := make([][]models.License, len(licensesResp))
		copy(cpy, licensesResp)

		return cpy
	}

	callAnalysisStates := make(map[string]bool)

	tests := []struct {
		name string
		args args
	}{{
		name: "group_vulnerabilities",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       false,
					ScanLicensesAllowlist: nil,
				},
				CallAnalysisStates: callAnalysisStates,
			},
			config: &config.ConfigManager{},
		},
	}, {
		name: "group_vulnerabilities_with_all_packages_included",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       true,
					ScanLicensesAllowlist: nil,
				},
				CallAnalysisStates: callAnalysisStates,
			},
			config: &config.ConfigManager{},
		},
	}, {
		name: "group_vulnerabilities_with_licenses",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       true,
					ScanLicensesSummary:   true,
					ScanLicensesAllowlist: nil,
				},
				CallAnalysisStates: callAnalysisStates,
			},
			config: &config.ConfigManager{},
		},
	}, {
		name: "group_vulnerabilities_with_license_allowlist",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       false,
					ScanLicensesAllowlist: []string{"MIT", "0BSD"},
				},
				CallAnalysisStates: callAnalysisStates,
			},

			config: &config.ConfigManager{},
		},
	}, {
		name: "group_vulnerabilities_with_license_allowlist_and_license_override",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       false,
					ScanLicensesAllowlist: []string{"MIT", "0BSD"},
				},
				CallAnalysisStates: callAnalysisStates,
			},
			config: &config.ConfigManager{
				OverrideConfig: &config.Config{
					PackageOverrides: []config.PackageOverrideEntry{
						{
							Name:      "pkg-3",
							Ecosystem: "npm",
							License: config.License{
								Override: []string{"MIT"},
							},
						},
					},
				},
			},
		},
	}, {
		name: "group_vulnerabilities_with_license_allowlist_and_all_packages",
		args: args{
			r:            &reporter.VoidReporter{},
			packages:     packages,
			vulnsResp:    vulnsResp,
			licensesResp: makeLicensesResp(),
			actions: ScannerActions{
				ExperimentalScannerActions: ExperimentalScannerActions{
					ShowAllPackages:       true,
					ScanLicensesAllowlist: []string{"MIT", "0BSD"},
				},
				CallAnalysisStates: callAnalysisStates,
			},
			config: &config.ConfigManager{},
		},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			got := buildVulnerabilityResults(tt.args.r, tt.args.packages, tt.args.vulnsResp, tt.args.licensesResp, tt.args.actions, tt.args.config)
			testutility.NewSnapshot().MatchJSON(t, got)
		})
	}
}
