/* Open Digita Services  --  Camera Device Protocol definitions.
   
   Copyright (C) 1998, 1999 James C. Thompson <jim.thompson@pobox.com>
   Copyright (C) 1998, 1999 Viljo Hakala <viljo.hakala@pcuf.fi>
   
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

/*--------------------------------------------------------------------------
  System include files */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif

/*--------------------------------------------------------------------------
  Local include files */

#include "cameraP.h"
#include "send.h"

/* Start Editing Here: */

ODSResult 
begin_send(ODSCamera camera,
	   unsigned long length,
	   ods_command command)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_presend) 
    {
      return camera->error = kODSModeError;	/* Assignment */
    }

  /* Advance the mode. */
  camera->mode = ods_send;

  /* Set up the variables that track the message size, number of
     bytes sent, etc. */
  camera->message_length = length + 12;	/* Add 12 to account for
					   the words we send. */

  camera->bufbytes = 0;
  camera->bufused = 0;

  /* Make sure we have a buffer to use. */
  ensure_buffer_size(camera, camera->message_length);

    /* Send the size, version, and reserved bytes. */
  send_uint(camera, camera->message_length - 4);
  send_uint(camera, 0);

    /* Send the command and (unused) result. */
  camera->command = command;
  send_ushort(camera, (unsigned short) command);
  send_ushort(camera, 0);

  /* This returns an error if any of the send_ functions encountered an
     error. */
  return camera->error;
}

ODSResult 
send_ushort(ODSCamera camera, short s)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment! */
    }

  swap_short(s);
  return send_bytes(camera, &s, sizeof(s));
}

ODSResult 
send_uint(ODSCamera camera, UInteger u)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment! */
    }

  swap_long(u);
  return send_bytes(camera, &u, sizeof(u));
}

ODSResult 
send_sint(ODSCamera camera, SInteger s)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment! */
    }

  swap_long(s);
  return send_bytes(camera, &s, sizeof(s));
}

ODSResult 
send_fixed(ODSCamera camera, Fixed f)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment */
    }

  swap_long(f);
  return send_bytes(camera, &f, sizeof(f));
}

ODSResult 
send_boolean(ODSCamera camera, Boolean b)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment */
    }

  swap_long(b);
  return send_bytes(camera, &b, sizeof(b));
}

ODSResult 
send_bitflags(ODSCamera camera, BitFlags f)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment */
    }

  swap_long(f);
  return send_bytes(camera, &f, sizeof(f));
}

ODSResult 
send_pname(ODSCamera camera, PName p)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment */
    }

  swap_long(p);
  return send_bytes(camera, &p, sizeof(p));
}

ODSResult 
send_string(ODSCamera camera, const String * s)
{
  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_send) 
    {
      return kODSModeError;
    }

  return send_bytes(camera, s, sizeof(*s));
}

ODSResult 
send_dosname(ODSCamera camera, const DOSName * d)
{
  if (camera == 0) 
    {
      return kODSParamError;
    }

  if (camera->mode != ods_send) 
    {
      return kODSModeError;
    }

  return send_bytes(camera, d, sizeof(*d));
}

ODSResult 
send_bytes(ODSCamera camera, const void *addr, size_t len)
{
  unsigned long avail;
  const char *caddr = addr;
  char *buffer = camera->buffer;

  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if (camera->mode != ods_send) 
    {
      return camera->error = kODSModeError;	/* Assignment! */
    }

  avail = camera->bufsize - camera->bufbytes;

  if (avail < len)
    {
#if defined(DEBUG)
      printf("send_bytes: camera buffer overrun.\n");
#endif

      return kODSIOError;
    }

  /* Copy the supplied buffer into the message buffer. */
  if (caddr != 0)
    {
      memcpy(buffer + camera->bufbytes, caddr, len);
    }

  /* Increment the number of bytes stored. */
  camera->bufbytes += len;

  return kODSNoErr;
}

ODSResult 
send_ptvs(ODSCamera camera, PNameTypeValueStruct * ptvs)
{
  switch (ptvs->Type) 
    {
    case kUInteger:
      swap_long(ptvs->Value.UInt);
      break;

    case kSInteger:
      swap_long(ptvs->Value.Int);
      break;

    case kFixed:
      swap_long(ptvs->Value.fFixed);
      break;

    case kBoolean:
      swap_long(ptvs->Value.fBoolean);
      break;

    case kBitFlags:
      swap_long(ptvs->Value.fBitFlags);
      break;

    case kPName:
      swap_long(ptvs->Value.fPName);
      break;

    case kDOSName:
    case kString:
    case kEnumList:
      break;
    }

  send_pname(camera, (ptvs->Name));
  send_uint(camera, (ptvs->Type));
  send_bytes(camera, &(ptvs->Value), sizeof(PNameValue));

  return kODSNoErr;
}

ODSResult 
send_filename(ODSCamera camera, const FileNameStruct * filename)
{
  send_uint(camera, filename->DriveNo);
  send_string(camera, &(filename->PathName));
  send_dosname(camera, &(filename->DOSName));

  return kODSNoErr;
}

ODSResult 
send_partialtag(ODSCamera camera, const PartialTag * partialtag)
{
  send_uint(camera, partialtag->Offset);
  send_uint(camera, partialtag->Length);
  send_uint(camera, partialtag->Filesize);

  return kODSNoErr;
}

ODSResult 
end_send(ODSCamera camera)
{
  /* Check the passed arguments. */
  if (camera == 0) 
    {
      return kODSParamError;
    }

  /* Ensure that we're in the right mode. */
  if ((camera->mode != ods_send) ||
      (camera->send_message == 0))
    {
      return camera->error = kODSModeError;	/* Assignment! */
    }

  /* Send the message to the camera, and check for resulting errors. */
  if ((*camera->send_message)(camera) != 0)
    {
      return kODSIOError;
    }

  camera->mode = ods_prereceive;

  return kODSNoErr;
}
