#pragma once

// Author: Torarin Hals Bakke (2012)

// Boost Software License - Version 1.0 - August 17th, 2003

// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:

// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.

// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include <type_traits>
#include <utility>
#include <deque>
#include <mutex>
#include <condition_variable>

#include "variant.hpp"
#include "LockBlock.h"
#include "FunctorVisitor.h"

template <typename... Ts>
class Mailbox {
public:
  template <typename T>
  void send(T&& msg) {
    LOCK_BLOCK (messages_lock_) {
      messages_.push_back(std::forward<T>(msg));
      messages_cond_.notify_one();
    }
  }

  template <typename... Funs>
  void receive(Funs&&... funs) {
    for (;;) {
      LOCK_BLOCK (messages_lock_)
        if (messages_.empty()) break;

      auto extract_message = [&]() {
        LOCK_BLOCK (messages_lock_) {
          utils::variant<Ts...> msg(std::move(messages_.front()));
          messages_.pop_front();
          return msg;
        }
        assert(0);
      };

      utils::apply_visitor(functor_visitor<Funs&...>(funs...),
                           extract_message());
    }
  }

  template <typename Rep, typename Period, typename... Funs>
  void receive_wait(const std::chrono::duration<Rep, Period>& rel_time,
                    Funs&&... funs)
  {
    {
      std::unique_lock<std::mutex> lock(messages_lock_);
      if (!messages_cond_.wait_for(lock,
                                   rel_time,
                                   [&]{return !messages_.empty();}))
        return;
    }
    receive(std::forward<Funs>(funs)...);
  }

  void clear() {
    LOCK_BLOCK(messages_lock_)
      messages_.clear();
  }


private:
  std::deque<utils::variant<Ts...>> messages_;
  std::mutex messages_lock_;
  std::condition_variable messages_cond_;
};
