//
// $Id: VectorTest.m,v 1.24 2007/03/28 03:16:53 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "VectorTest.h"
#import "Random.h"
#import <ObjectiveLib/Vector.h>
#import <ObjectiveLib/DataOutStream.h>
#import <ObjectiveLib/DataInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/ObjectInStream.h>
#import "Number.h"
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSString.h>
#import <Foundation/NSData.h>
#import <Foundation/NSArchiver.h>
#if defined(HAVE_KEYED_ARCHIVES)
#import <Foundation/NSKeyedArchiver.h>
#endif
#else
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Text.h>
#endif
#include <stdlib.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-class.h>
#endif

@implementation VectorTest

- (void) testAccessors
{
    OLVector* v;

    v = [[OLVector alloc] init];
    [v pushBack: REAP([OLNumber numberWithInt: 1])];
    [v pushBack: REAP([OLNumber numberWithInt: 2])];
    [v pushBack: REAP([OLNumber numberWithInt: 3])];

    if ([[v at: 0] intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [[v at: 0] intValue]];
    }
    if ([[v at: 1] intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [[v at: 1] intValue]];
    }
    if ([[v at: 2] intValue] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [[v at: 2] intValue]];
    }
    if ([[v front] intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [[v front] intValue]];
    }
    if ([[v back] intValue] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [[v back] intValue]];
    }
    [v RELEASE];
}

- (void) testAssign
{
    OLVector* v;
    OLVector* v2;
    int nums[] = { 1, 2, 3, 4, 5 };
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: REAP([OLNumber numberWithInt: nums[i]])];
    v2 = [v copy];
    [v assign: 2 filledWith: @"one"];
    if ([v size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [v size]];
    }
    if (![[v at: 0] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[v at: 0] cString]];
    }
    if (![[v at: 1] isEqual: @"one"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"one\", but got \"%s\"", [[v at: 1] cString]];
    }
    [v assignFrom: REAP([v2 begin]) to: REAP([v2 end])];
    [v2 RELEASE];
    if ([v size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [v size]];
    }
    for (i = 0; i < 5; i++)
    {
        if ([[v at: i] intValue] != nums[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", nums[i], [[v at: i] intValue]];
        }
    }
    [v assignAt: 2 value: @"doggy"];
    if (![[v at: 2] isEqual: @"doggy"])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"doggy\", but got \"%s\"", [[v at: 2] cString]];
    }
    [v RELEASE];
}

#if !defined(OL_NO_OPENSTEP)
- (void) testCoding
{
    NSMutableData* data;
    NSArchiver* archiver;
    NSData* archData;
    OLVector* v1;
    OLVector* v2;
    OLNumber* num;
    int i;


    v1 = [[OLVector alloc] init];
    for (i = 0; i< 5000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [v1 pushBack: num];
        [num RELEASE];
    }
    data = [[NSMutableData alloc] initWithCapacity: 5000];
    archiver = [[NSArchiver alloc] initForWritingWithMutableData: data];
    [archiver encodeRootObject: v1];
    [archiver RELEASE];
    v2 = [NSUnarchiver unarchiveObjectWithData: data];
    [data RELEASE];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#if defined(HAVE_KEYED_ARCHIVES)
    archData = [NSKeyedArchiver archivedDataWithRootObject: v1];
    v2 = [NSKeyedUnarchiver unarchiveObjectWithData: archData];
    if (![v1 isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
#endif
    [v1 RELEASE];
}
#endif

- (void) testConvenienceAllocators
{
    OLVector* v;
    OLVector* v2;
    OLNumber* num;

    v = REAP([OLVector vector]);
    if (![v IS_MEMBER_OF: [OLVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLVector class])->name, ((Class)[v class])->name];
    }
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 1];
    [v pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 2];
    [v pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 3];
    [v pushBack: num];
    [num RELEASE];
    v2 = REAP([OLVector vectorFrom: REAP([v begin]) to: REAP([v end])]);
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    num = [[OLNumber alloc] initWithInt: 4];
    [v pushBack: num];
    [num RELEASE];
    num = [[OLNumber alloc] initWithInt: 5];
    [v pushBack: num];
    [num RELEASE];
    v2 = REAP([OLVector vectorWithVector: v]);
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    v = REAP([OLVector vectorWithCapacity: 7000]);
    if (![v IS_MEMBER_OF: [OLVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLVector class])->name, ((Class)[v class])->name];
    }
    if ([v capacity] < 7000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected at least 7000, but got %u", [v capacity]];
    }
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector should be empty"];
    }
    num = [[OLNumber alloc] initWithInt: 6];
    v = REAP([OLVector vectorWithSize: 5234 filledWith: num]);
    [num RELEASE];
    if (![v IS_MEMBER_OF: [OLVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"%s\", but got \"%s\"",
            ((Class)[OLVector class])->name, ((Class)[v class])->name];
    }
    if ([v size] != 5234)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5234, but got %u", [v size]];
    }
}

- (void) testErase
{
    OLVector* v;
    OLArrayIterator* itor;
    OLArrayIterator* itor2;

    v = [[OLVector alloc] init];
    [v pushBack: REAP([OLNumber numberWithInt: 1])];
    [v pushBack: REAP([OLNumber numberWithInt: 2])];
    [v pushBack: REAP([OLNumber numberWithInt: 3])];
    [v pushBack: REAP([OLNumber numberWithInt: 4])];
    [v pushBack: REAP([OLNumber numberWithInt: 5])];
    [v popBack];
    if ([v size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [v size]];
    }
    if ([[v back] intValue] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [[v back] intValue]];
    }
    [v pushBack: REAP([OLNumber numberWithInt: 5])];
    itor = REAP([v begin]);
    [itor advance];
    itor = REAP([v erase: itor]);
    if ([v size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [v size]];
    }
    if ([[itor dereference] intValue] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [[itor dereference] intValue]];
    }
    itor2 = [itor copy];
    [itor2 advanceBy: 2];
    itor = REAP([v eraseFrom: itor to: itor2]);
    [itor2 RELEASE];
    if ([v size] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [v size]];
    }
    if ([[itor dereference] intValue] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [[itor dereference] intValue]];
    }
    [itor reverse];
    if ([[itor dereference] intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [[itor dereference] intValue]];
    }
    if (![itor isEqual: REAP([v begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at the beginning"];
    }
    [v clear];
    if ([v size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 0, but got %i", [v size]];
    }
    [v RELEASE];
}

- (void) testInsert
{
    OLVector* v;
    OLVector* v2;
    int i;
    OLArrayIterator* itor;

    v = [[OLVector alloc] init];
    itor = REAP([v insertAt: REAP([v begin]) value: @"one"]);
    if ([v size] != 1 || [v capacity] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size and capacity should be 1."];
    }
    if (![@"one" isEqual: [v at: 0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Wrong object at index 0"];
    }
    if (![itor isEqual: REAP([v begin])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The iterator should be equal to [v begin]"];
    }
    [v RELEASE];

    v = [[OLVector alloc] init];
    [v insertAt: REAP([v begin]) count: 100 filledWith: @"two"];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v size]];
    }
    if ([v capacity] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity should be 100, got %u", [v capacity]];
    }
    for (i = 0; i < 100; i++)
    {
        if (![@"two" isEqual: [v at: i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right object at index %i", i];
        }
    }
    itor = REAP([v begin]);
    [itor advanceBy: 5];
    itor = REAP([v insertAt: itor value: @"three"]);
    if ([v size] != 101)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 101, got %u", [v size]];
    }
    if (![@"three" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"three\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor reverse];
    if (![@"two" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }
    [itor advanceBy: 2];
    if (![@"two" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"two\", but got \"%s\"", [[itor dereference] cString]];
    }

    v2 = [[OLVector alloc] init];
    [v2 pushBack: @"7"];
    [v2 pushBack: @"8"];
    [v2 pushBack: @"9"];
    [v2 pushBack: @"10"];
    if ([v2 size] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, got %u", [v2 size]];
    }
    if (![@"7" isEqual: [v2 at: 0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"7\", got \"%s\"", [[v2 at: 0] cString]];
    }
    if (![@"8" isEqual: [v2 at: 1]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"8\", got \"%s\"", [[v2 at: 1] cString]];
    }
    if (![@"9" isEqual: [v2 at: 2]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"9\", got \"%s\"", [[v2 at: 2] cString]];
    }
    if (![@"10" isEqual: [v2 at: 3]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"10\", got \"%s\"", [[v2 at: 3] cString]];
    }

    [v insertAt: itor from: REAP([v2 begin]) to: REAP([v2 end])];
    [v2 RELEASE];
    if ([v size] != 105)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 105, got %u", [v size]];
    }
    itor = REAP([v begin]);
    [itor advanceBy: 6];
    if (![@"7" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"7\", got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![@"8" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"8\", got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![@"9" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"9\", got \"%s\"", [[itor dereference] cString]];
    }
    [itor advance];
    if (![@"10" isEqual: [itor dereference]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected \"10\", got \"%s\"", [[itor dereference] cString]];
    }
    [v RELEASE];
}

- (void) testInitializers
{
    OLVector* v;
    OLVector* v2;
    int i;

    v = [[OLVector alloc] init];
    if ([v size] != 0 || [v capacity] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size and capacity should be 0."];
    }
    [v RELEASE];

    v = [[OLVector alloc] initWithCapacity: 10];
    if ([v size] != 0)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 0."];
    }
    if ([v capacity] != 10)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity should be 10."];
    }
    [v RELEASE];

    v = [[OLVector alloc] initWithSize: 1 filledWith: @"one"];
    if ([v size] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 1."];
    }
    if ([v capacity] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity should be 1."];
    }
    if (![@"one" isEqual: [v at: 0]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Did not get the right object at index 0"];
    }
    [v RELEASE];

    v = [[OLVector alloc] initWithSize: 100 filledWith: @"two"];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v size]];
    }
    if ([v capacity] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity should be 100, got %u", [v capacity]];
    }
    for (i = 0; i < 100; i++)
    {
        if (![@"two" isEqual: [v at: i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right object at index %i", i];
        }
    }

    v2 = [[OLVector alloc] initWithVector: v];
    if ([v2 size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v2 size]];
    }
    if ([v2 capacity] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity should be 100, got %u", [v2 capacity]];
    }
    for (i = 0; i < 100; i++)
    {
        if (![@"two" isEqual: [v2 at: i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right object at index %i", i];
        }
    }
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [v2 RELEASE];

    v2 = [[OLVector alloc] initFrom: REAP([v begin]) to: REAP([v end])];
    if ([v2 size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size should be 100, got %u", [v2 size]];
    }
    for (i = 0; i < 100; i++)
    {
        if (![@"two" isEqual: [v2 at: i]])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Did not get the right object at index %i", i];
        }
    }
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [v2 RELEASE];
    [v RELEASE];
}

- (void) testIterators
{
    OLVector* v;
    OLArrayIterator* itor;
    OLReverseRandomIterator* ritor;

    v = [[OLVector alloc] init];
    [v pushBack: REAP([OLNumber numberWithInt: 1])];
    [v pushBack: REAP([OLNumber numberWithInt: 2])];
    [v pushBack: REAP([OLNumber numberWithInt: 3])];
    [v pushBack: REAP([OLNumber numberWithInt: 4])];
    [v pushBack: REAP([OLNumber numberWithInt: 5])];
    itor = REAP([v begin]);
    if ([[itor dereference] intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [[itor dereference] intValue]];
    }
    [itor advance];
    if ([[itor dereference] intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [[itor dereference] intValue]];
    }
    [itor advanceBy: 2];
    if ([[itor dereference] intValue] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [[itor dereference] intValue]];
    }
    [itor reverse];
    if ([[itor dereference] intValue] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [[itor dereference] intValue]];
    }
    [itor advanceBy: -2];
    if ([[itor dereference] intValue] != 1)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 1, but got %i", [[itor dereference] intValue]];
    }
    [itor advanceBy: 5];
    if (![itor isEqual: REAP([v end])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at end"];
    }

    ritor = REAP([v rbegin]);
    if ([[ritor dereference] intValue] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [[ritor dereference] intValue]];
    }
    [ritor advance];
    if ([[ritor dereference] intValue] != 4)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 4, but got %i", [[ritor dereference] intValue]];
    }
    [ritor advanceBy: 2];
    if ([[ritor dereference] intValue] != 2)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 2, but got %i", [[ritor dereference] intValue]];
    }
    [ritor reverse];
    if ([[ritor dereference] intValue] != 3)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 3, but got %i", [[ritor dereference] intValue]];
    }
    [ritor advanceBy: -2];
    if ([[ritor dereference] intValue] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 5, but got %i", [[ritor dereference] intValue]];
    }
    [ritor advanceBy: 5];
    if (![ritor isEqual: REAP([v rend])])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected to be at end"];
    }
    [v RELEASE];
}

- (void) testReserveResizeSwap
{
    OLVector* v;
    OLVector* v2;
    OLVector* vCopy;
    OLVector* v2Copy;
    int nums[] = { 1, 2, 3, 4, 5 };
    int i;

    v = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
        [v pushBack: REAP([OLNumber numberWithInt: nums[i]])];
    [v reserve: 1000];
    if ([v capacity] != 1000)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity must be 1000"];
    }
    for (i = 0; i < 5; i++)
    {
        if ([[v at: i] intValue] != nums[i])
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected %i, but got %i", nums[i], [[v at: i] intValue]];
        }
    }
    v2 = [v copy];
    [v resize: 100 filledWith: REAP([OLNumber numberWithInt: 6])];
    if ([v size] != 100)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected 100, but got %i", [v size]];
    }
    for (i = 5; i < 100; i++)
    {
        if ([[v at: i] intValue] != 6)
        {
            [self errInFile: __FILE__ line: __LINE__
                format: "Expected 6, but got %i", [[v at: i] intValue]];
        }
    }
    vCopy = [v copy];
    v2Copy = [v2 copy];
    [v swap: v2];
    if (![v isEqual: v2Copy])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected the vectors to be equal"];
    }
    if (![v2 isEqual: vCopy])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected the vectors to be equal"];
    }
    [v RELEASE];
    [v2 RELEASE];
    [vCopy RELEASE];
    [v2Copy RELEASE];
}

- (void) testProperties
{
    OLVector* v;
    OLVector* v2;
    int nums[] = { 1, 2, 3, 4, 5 };
    int i;

    v = [[OLVector alloc] init];
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector must report that it is empty"];
    }
    for (i = 0; i < 5; i++)
        [v pushBack: REAP([OLNumber numberWithInt: nums[i]])];
    v2 = [[OLVector alloc] init];
    for (i = 0; i < 5; i++)
        [v2 pushBack: REAP([OLNumber numberWithInt: nums[i]])];
    if (![v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors must be equal"];
    }
    if ([v capacity] < 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The capacity must be at least 5"];
    }
    if ([v size] != 5)
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The size must be exactly 5"];
    }
    if ([v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector cannot report that it is empty"];
    }
    [v clear];
    if (![v empty])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vector must report that it is empty"];
    }
    if ([v isEqual: v2])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors must not be equal"];
    }
    if ([v maxSize] != [v2 maxSize])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "All vectors must have the same max size"];
    }
    [self logMessage: "The max size of a vector is %u", [v maxSize]];
    [v RELEASE];
    [v2 RELEASE];
}

- (void) testStreaming
{
    OLDataOutStream* dout;
    OLObjectOutStream* oout;
    OLObjectInStream* oin;
    OLVector* vector;
    OLVector* readVector;
    OLNumber* num;
    int i;

    dout = REAP([OLDataOutStream stream]);
    oout = REAP([OLObjectOutStream streamWithOutStream: dout]);
    vector = [[OLVector alloc] init];
    for (i = 0; i< 10000; i++)
    {
        num = [[OLNumber alloc] initWithInt: OLRandom() % 10000];
        [vector pushBack: num];
        [num RELEASE];
    }
    [oout writeObject: vector];
    oin = REAP([OLObjectInStream streamWithInStream:
        REAP([OLDataInStream streamWithBytes: [dout bytes] count: [dout count]])]);
    readVector = REAP([oin readObject]);
    if (![readVector IS_MEMBER_OF: [OLVector class]])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "Expected OLVector class, but got %s", ((Class)[readVector class])->name];
    }
    if (![readVector isEqual: vector])
    {
        [self errInFile: __FILE__ line: __LINE__
            format: "The vectors should be equal"];
    }
    [vector RELEASE];
}

@end
