// $Id: UnitTest.m,v 1.19 2007/03/28 03:16:53 will_mason Exp $
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "UnitTest.h"
#import "UnitTestRunner.h"
#import "ThreadServices.h"
#if defined(OL_NO_OPENSTEP)
#import <ObjectiveLib/Reaper.h>
#import <ObjectiveLib/Exception.h>
#else
#import <Foundation/NSAutoreleasePool.h>
#import <Foundation/NSException.h>
#import <Foundation/NSString.h>
#endif
#import <ObjectiveLib/Text.h>
#include <stdlib.h>
#import <assert.h>
#if defined(__NEXT_RUNTIME__)
#import <objc/objc-runtime.h>
#else
#import <objc/objc-api.h>
#endif
#import <string.h>

static OLMutex messageLock;

@interface TestRecord :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject
#endif
{
@private
    SEL         sel;
    const char* className;
    char*       qualifiedName;
}

- (id) initWithSelector: (SEL)s className: (const char*)cls;
- (id) initWithTestRecord: (TestRecord*)other;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
- (int) compare: (id)other;
- (id) copy;
- (BOOL) isEqual: (id)object;
- (const char*) name;
- (const char*) qualifiedName;
- (SEL) selector;

@end

@implementation TestRecord

- (id) initWithSelector: (SEL)s className: (const char*)cls
{
    [super init];
    sel = s;
    className = cls;
    return self;
}

- (id) initWithTestRecord: (TestRecord*)other
{
	[super init];
    sel = other->sel;
    className = other->className;
	if (other->qualifiedName != NULL)
		qualifiedName = strdup(other->qualifiedName);
	return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    free(qualifiedName);
    SUPER_FREE;
}

- (int) compare: (id)other
{
    int rc = -1;

    if ([other IS_MEMBER_OF: [TestRecord class]])
        rc = strcmp(SEL_GET_NAME(sel), SEL_GET_NAME(((TestRecord*)other)->sel));
    return rc;
}

- (id) copy
{
	return [[TestRecord alloc] initWithTestRecord: self];
}

- (BOOL) isEqual: (id)obj
{
    return [obj IS_MEMBER_OF: [TestRecord class]] &&
           SEL_EQ(((TestRecord*)obj)->sel, sel) &&
           strcmp(((TestRecord*)obj)->className, className) == 0;
}

- (const char*) name
{
    return SEL_GET_NAME(sel);
}

- (const char*) qualifiedName
{
    char buf[512];
    unsigned len;

    if (qualifiedName == NULL)
    {
        len = snprintf(buf, 512, "[%s %s]", className, SEL_GET_NAME(sel));
        qualifiedName = malloc(len + 1);
        strcpy(qualifiedName, buf);
    }
    return qualifiedName;
}

- (SEL) selector
{
    return sel;
}

@end

@implementation UnitTest

#if defined(OL_NO_OPENSTEP)
+ (id) initialize
#else
+ (void) initialize
#endif
{
    OLCreateMutex(&messageLock);
#if defined(OL_NO_OPENSTEP)
    return self;
#endif
}

- (id) init
{
    return [self initWithOut: stdout err: stderr];
}

- (id) initWithOut: (FILE*)o err: (FILE*)e
{
    Class myClass;
    struct objc_method_list* methods;
    int i;
    SEL theSEL;
    const char* theName;
    TestRecord* rec;
#if defined(__NEXT_RUNTIME__)
    void* itor = 0;
#endif

    assert(sizeof(SEL) == sizeof(int));

    [super init];

    failedTests = [[OLSet alloc] init];
    tests = [[OLVector alloc] initWithCapacity: 10];
    outStream = o;
    errStream = e;

    myClass = [self class];
#if defined(__NEXT_RUNTIME__)
    methods = class_nextMethodList(myClass, &itor);
#else
    methods = myClass->methods;
#endif
    while (methods != NULL)
    {
        for (i = 0; i < methods->method_count; i++)
        {
            theSEL = methods->method_list[i].method_name;
            theName = SEL_GET_NAME(theSEL);
            if (strncmp(theName, "test", 4) == 0 &&
                strlen(theName) > 4
#if defined(__NEXT_RUNTIME__)
               && method_getNumberOfArguments(&methods->method_list[i]) == 2)
#else
               )
#endif
            {
                rec = [[TestRecord alloc] initWithSelector: theSEL
                                          className: ((Class)[self class])->name];
                [tests pushBack: rec];
                [rec RELEASE];
            }
        }
#if defined(__NEXT_RUNTIME__)
        methods = class_nextMethodList(myClass, &itor);
#else
        methods = NULL;
#endif
    }

    return self;
}

- (id) initWithUnitTest: (UnitTest*)other
{
	[super init];
	outStream = other->outStream;
	errStream = other->errStream;
	errorCount = other->errorCount;
	tests = [other->tests copy];
	failedTests = [other->failedTests copy];
	return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    [failedTests RELEASE];
    [tests RELEASE];
    SUPER_FREE;
}

- (int) compare: (id)other
{
    UnitTest* otherTest;
    unsigned i = 0;
    int rc = -1;

    if ([other IS_KIND_OF: [UnitTest class]])
    {
        otherTest = (UnitTest*)other;
        rc = (int)[tests size] - (int)[otherTest->tests size];
        while (rc == 0 && i < [tests size])
            rc = [[tests at: i] compare: [otherTest->tests at: i]];
    }
    return rc;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[[self class] alloc] initWithUnitTest: self];
}
#endif

- (void) errInFile: (const char*)file line: (int)line format: (const char* const)format, ...
{
    char buf[8192];
    va_list args;
    id testInProcess = [UnitTestRunner testInProcess];
    id tmp;

    OLLockMutex(&messageLock);
    assert(errStream != NULL);
    errorCount++;
    assert(testInProcess != nil);
    tmp = [failedTests insert: testInProcess];
    FREE_AUTO(tmp);
    sprintf(buf, "FAIL (%s,%i): %s", file, line, format);
    va_start(args, format);
    vfprintf(errStream, buf, args);
    va_end(args);
    fputc('\n', errStream);
    fflush(errStream);
    OLUnlockMutex(&messageLock);
}

- (FILE*) errStream
{
    return errStream;
}

- (int) errorCount
{
    return errorCount;
}

- (BOOL) hasTestWithName: (const char*)test
{
    id rec = [self lookUpTestWithName: test];
    return (rec != nil) ? YES : NO;
}

- (void) listFailedTestsToStream: (FILE*)stream delimiter: (char)delim
{
    OLAssociativeIterator* cur;
    OLAssociativeIterator* last;

    if ([failedTests size] > 0)
    {
        for (cur = [failedTests begin], last = [[failedTests end] reverse]; ; [cur advance])
        {
            fputs([[cur dereference] qualifiedName], stream);
            if ([cur isEqual: last])
                break;
            fputc(delim, stream);
        }
        FREE_AUTO(cur);
        FREE_AUTO(last);
    }
}

- (void) logMessage: (const char* const)format, ...
{
    va_list args;

    assert(outStream != NULL);
    va_start(args, format);
    OLLockMutex(&messageLock);
    vfprintf(outStream, format, args);
    va_end(args);
    fputc('\n', outStream);
    fflush(outStream);
    OLUnlockMutex(&messageLock);
}

- (id) lookUpTestWithName: (const char*)name
{
    unsigned i;
    id rec;

    for (i = 0; i < [tests size]; i++)
    {
		rec = [tests at: i];
        if (strcmp(name, (const char*)[rec name]) == 0)
            return rec;
    }
    return nil;
}

- (FILE*) outStream
{
    return outStream;
}

- (BOOL) preventAutoLoad
{
    return NO;
}

- (void) runAllTests
{
    unsigned i;

    for (i = 0; i < [tests size]; i++)
        [self runTest: [tests at: i]];
}

- (void) runTest: (id)test
{
#if defined(OL_NO_OPENSTEP)
    OLReaper* reaper;
#else
    NSAutoreleasePool* autoPool;
#endif

    assert(outStream != NULL);
#if defined(OL_NO_OPENSTEP)
    reaper = [[OLReaper alloc] init];
#else
    autoPool = [[NSAutoreleasePool alloc] init];
#endif
    [UnitTestRunner pushTestInProcess: test];
    fprintf(outStream, "========================================================================\nBEGIN: %s\n------------------------------------------------------------------------\n", [test qualifiedName]);


    if ([self setUp])
    {
#if !defined(GNUSTEP)
        TRY
#endif
            [self PERFORM: [test selector]];
#if !defined(GNUSTEP)
        CATCH
            [self errInFile: __FILE__ line: __LINE__ format: "Uncaught exception: %s - %s",
                EXCEPTION_NAME, EXCEPTION_MESSAGE];
        END_CATCH
#endif
        if (![self tearDown])
            fprintf(errStream, "* %s tear-down failed *\n", [test qualifiedName]);
    }
    else
    {
        fprintf(errStream, "* %s set-up failed *\n", [test qualifiedName]);
    }
    [UnitTestRunner popTestInProcess];
    fprintf(outStream, "------------------------------------------------------------------------\n  END: %s\n========================================================================\n", [test qualifiedName]);
#if defined(OL_NO_OPENSTEP)
    [reaper RELEASE];
#else
    [autoPool RELEASE];
#endif
 }

- (BOOL) runTestWithName: (const char*)name
{
    id test = [self lookUpTestWithName: name];

    if (test != nil)
    {
        [self runTest: test];
        return YES;
    }
    else
    {
        return NO;
    }
}

- (void) setStreamOut: (FILE*)o err: (FILE*)e
{
    outStream = o;
    errStream = e;
}

- (BOOL) setUp
{
    return YES;
}

- (BOOL) tearDown
{
    return YES;
}

- (int) testCount
{
    return [tests size];
}

- (OLVector*) testListQualified: (BOOL)flag
{
    OLVector* names = [[OLVector alloc] initWithCapacity: [tests size]];
    unsigned i;
    OLText* text;

    for (i = 0; i < [tests size]; i++)
    {
        text = [[OLText alloc] initWithCString:
            flag ? (const char*)[[tests at: i] qualifiedName] :
                   (const char*)[[tests at: i] name]];
        [names pushBack: text];
        [text RELEASE];
    }
    return AUTORELEASE(names);
}

@end
