//
// $Id: Number.m,v 1.5 2007/03/06 20:42:21 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Number.h"
#import <ObjectiveLib/ObjectInStream.h>
#import <ObjectiveLib/ObjectOutStream.h>
#import <ObjectiveLib/HashFunction.h>
#if !defined(OL_NO_OPENSTEP)
#import <Foundation/NSCoder.h>
#endif
#import <math.h>
#import <string.h>
#import <stdio.h>

#define DO_ARITHMETIC(op) \
    OLNumber* num; \
    OLNumber* result = nil; \
\
    if ([value IS_KIND_OF: [OLNumber class]]) \
    { \
        num = (OLNumber*)value; \
        if (num->type == type) \
        { \
            switch (type) \
            { \
                case NumberType_Bool: \
                    result = [[OLNumber alloc] \
                        initWithBool: (blob.boolValue op num->blob.boolValue) ? YES : NO]; \
                    break; \
\
                case NumberType_Double: \
                    result = [[OLNumber alloc] \
                        initWithDouble: blob.doubleValue op num->blob.doubleValue]; \
                    break; \
\
                case NumberType_Int: \
                    result = [[OLNumber alloc] \
                        initWithInt: blob.intValue op num->blob.intValue]; \
                    break; \
\
                case NumberType_UnsignedInt: \
                    result = [[OLNumber alloc] \
                        initWithUnsignedInt: blob.unsignedIntValue op \
                            num->blob.unsignedIntValue]; \
                    break; \
            } \
            return AUTORELEASE(result); \
        } \
    } \
    return nil;

@implementation OLNumber

+ (id) numberWithBool: (BOOL)b
{
    return AUTORELEASE([[OLNumber alloc] initWithBool: b]);
}

+ (id) numberWithDouble: (double)d
{
    return AUTORELEASE([[OLNumber alloc] initWithDouble: d]);
}

+ (id) numberWithInt: (int)i
{
    return AUTORELEASE([[OLNumber alloc] initWithInt: i]);
}

- (id) initWithBool: (BOOL)bul
{
    [super init];
    blob.boolValue = bul;
    type = NumberType_Bool;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    [super init];
    [decoder decodeValueOfObjCType: @encode(OLNumberType) at: &type];
    switch (type)
    {
    case NumberType_Bool:
        [decoder decodeValueOfObjCType: @encode(BOOL) at: &blob.boolValue];
        break;
    case NumberType_Double:
        [decoder decodeValueOfObjCType: @encode(double) at: &blob.doubleValue];
        break;
    case NumberType_Int:
    case NumberType_UnsignedInt:
        [decoder decodeValueOfObjCType: @encode(int) at: &blob.intValue];
        break;
    }
    return self;
}
#endif

- (id) initWithDouble: (double)dbl
{
    [super init];
    blob.doubleValue = dbl;
    type = NumberType_Double;
    return self;
}

- (id) initWithInt: (int)i
{
    [super init];
    blob.intValue = i;
    type = NumberType_Int;
    return self;
}

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    [super init];
    type = [stream readByte];
    switch (type)
    {
    case NumberType_Bool:
        blob.boolValue = [stream readBool];
        break;
    case NumberType_Double:
        blob.doubleValue = [stream readDouble];
        break;
    case NumberType_Int:
    case NumberType_UnsignedInt:
        blob.intValue = [stream readInt];
        break;
    }
    return self;
}

- (id) initWithUnsignedInt: (unsigned)i
{
    [super init];
    blob.unsignedIntValue = i;
    type = NumberType_UnsignedInt;
    return self;
}

- (id) arithmeticAdd: (id)value
{
    DO_ARITHMETIC(+);
}

- (id) arithmeticDivideBy: (id)value
{
    DO_ARITHMETIC(/);
}

- (id) arithmeticModulus: (id)value
{
    OLNumber* num; 
    OLNumber* result = nil;

    if ([value IS_KIND_OF: [OLNumber class]]) 
    { 
        num = (OLNumber*)value; 
        if (num->type == type) 
        { 
            switch (type) 
            { 
                case NumberType_Bool: 
                    result = [[OLNumber alloc] 
                        initWithBool: (blob.boolValue % num->blob.boolValue) ? YES : NO]; 
                    break; 

                case NumberType_Double: 
                    result = [[OLNumber alloc] 
                        initWithDouble: remainder(blob.doubleValue, num->blob.doubleValue)];
                    break; 

                case NumberType_Int: 
                    result = [[OLNumber alloc] 
                        initWithInt: blob.intValue % num->blob.intValue]; 
                    break; 

                case NumberType_UnsignedInt: 
                    result = [[OLNumber alloc] 
                        initWithUnsignedInt: blob.unsignedIntValue % 
                            num->blob.unsignedIntValue]; 
                    break; 
            } 
            return AUTORELEASE(result); 
        } 
    } 
    return nil;
}

- (id) arithmeticMultiply: (id)value
{
    DO_ARITHMETIC(*);
}

- (id) arithmeticNegate
{
    OLNumber* result = nil;

    switch (type) 
    { 
        case NumberType_Bool: 
            result = [[OLNumber alloc] 
                initWithBool: -blob.boolValue ? YES : NO]; 
            break; 

        case NumberType_Double: 
            result = [[OLNumber alloc] 
                initWithDouble: -blob.doubleValue];
            break; 

        case NumberType_Int: 
            result = [[OLNumber alloc] 
                initWithInt: -blob.intValue];
            break; 

        case NumberType_UnsignedInt: 
            result = [[OLNumber alloc] 
                initWithUnsignedInt: -blob.unsignedIntValue];
            break; 
    } 
    return AUTORELEASE(result); 
}

- (id) arithmeticSubtract: (id)value
{
    DO_ARITHMETIC(-);
}

- (BOOL) boolValue
{
    return blob.boolValue;
}

- (int) compare: (id)other
{
    OLNumber* num;
    double dresult;
    int result = -1;

    if ([other IS_KIND_OF: [OLNumber class]])
    {
        num = (OLNumber*)other;
        if (num->type == type)
        {
            switch (type)
            {
            case NumberType_Bool:
                result = (int)(blob.boolValue - num->blob.boolValue);
                break;

            case NumberType_Double:
                dresult = blob.doubleValue - num->blob.doubleValue;
                if (dresult > 0)
                    result = 1;
                else if (dresult == 0)
                    result = 0;
                break;

            case NumberType_Int:
                result = blob.intValue - num->blob.intValue;
                break;

            case NumberType_UnsignedInt:
                result = (int)(blob.unsignedIntValue - num->blob.unsignedIntValue);
                break;
            }
        }
    }
    return result;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    OLNumber* num = [[OLNumber alloc] init];

    memcpy(&num->blob, &blob, sizeof(blob));
    num->type = type;
    return num;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    OLNumber* num = [[OLNumber allocWithZone: zone] init];

    memcpy(&num->blob, &blob, sizeof(blob));
    num->type = type;
    return num;
}

#endif

- (double) doubleValue
{
    return blob.doubleValue;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    [encoder encodeValueOfObjCType: @encode(OLNumberType) at: &type];
    switch (type)
    {
    case NumberType_Bool:
        [encoder encodeValueOfObjCType: @encode(BOOL) at: &blob.boolValue];
        break;
    case NumberType_Double:
        [encoder encodeValueOfObjCType: @encode(double) at: &blob.doubleValue];
        break;
    case NumberType_Int:
    case NumberType_UnsignedInt:
        [encoder encodeValueOfObjCType: @encode(int) at: &blob.intValue];
        break;
    }
}
#endif

- (unsigned) hash
{
    if (type == NumberType_Double)
        return OLHash((const uint8_t*)&blob.doubleValue, sizeof(double));
    return blob.unsignedIntValue;
}

- (int) intValue
{
    return blob.intValue;
}

- (BOOL) isEqual: (id)other
{
    return [self compare: other] == 0 ? YES : NO;
}

- (OLText*) toText
{
    char buf[256];

    if (type == NumberType_Bool)
        strcpy(buf, blob.boolValue ? "YES" : "NO");
    else if (type == NumberType_Double)
        snprintf(buf, 256, "%g", blob.doubleValue);
    else if (type == NumberType_Int)
        snprintf(buf, 256, "%i", blob.intValue);
    else
        snprintf(buf, 256, "%u", blob.unsignedIntValue);
    return AUTORELEASE([[OLText alloc] initWithCString: buf]);
}

- (unsigned) unsignedIntValue
{
    return blob.unsignedIntValue;
}

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    [stream writeByte: type];
    switch (type)
    {
    case NumberType_Bool:
        [stream writeBool: blob.boolValue];
        break;
    case NumberType_Double:
        [stream writeDouble: blob.doubleValue];
        break;
    case NumberType_Int:
    case NumberType_UnsignedInt:
        [stream writeInt: blob.intValue];
        break;
    }
}

@end
