//
// $Id: Text.m,v 1.18 2007/03/28 03:16:52 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#import "Text.h"
#import "ConfigPrivate.h"
#import "RunTime.h"
#import "Macros.h"
#import "InStreamPackage.h"
#import "OutStreamPackage.h"
#import "ObjectOutStream.h"
#import "ObjectInStream.h"
#import "HashFunction.h"
#import "Synchronization.h"
#if defined(OL_NO_OPENSTEP)
#import "Exception.h"
#else
#import "ConfigPrivate.h"
#import <Foundation/NSString.h>
#import <Foundation/NSException.h>
#endif
#import <iconv.h>
#import <stdlib.h>
#import <string.h>
#import <errno.h>
#import <limits.h>

#if defined(_AIX)
#if !defined(OL_BIG_ENDIAN)
#error AIX is assumed to always be big endian
#endif
#define OL_DEFAULT_ENCODING "ISO8859-1"
#define OL_UNICODE_INTERNAL "UCS-2"
#else
#define OL_DEFAULT_ENCODING ""
#if defined(OL_BIG_ENDIAN)
#define OL_UNICODE_INTERNAL "UTF-16BE"
#else
#define OL_UNICODE_INTERNAL "UTF-16LE"
#endif

#endif

#if !defined(OL_NO_OPENSTEP)
NSString* const UTF8_DATA_KEY = @"OL_UTF8_DATA_KEY";
#endif

typedef struct _OLTextReference
{
    olchar*     characters;
    char*       cString;
    unsigned    length;
    unsigned    refCount;
} OLTextReference;

static OLMutex __refLock;
static OLOnceControl __refLockInit = OL_ONCE_INIT;

static uint8_t* __convertBytes(const uint8_t* bytes, unsigned num, const char* fromEnc, const char* toEnc , unsigned* outCount, BOOL nullTerm, BOOL trimOutBuf)
{
    iconv_t converter;
    ICONV_CONST char* inbuf = (ICONV_CONST char*)bytes;
    size_t inbytesleft = num;
    char* outbuf;
    char* newOutBuf;
    char* tmpOutBuf;
    size_t outbytesleft;
    size_t rc;
    unsigned bytesRightNow;
    unsigned outBufSize;
    BOOL finished = NO;

    converter = iconv_open(toEnc, fromEnc);
    if (converter == ((iconv_t)-1))
        RAISE_EXCEPTION(INVALID_ARGUMENT, @"Unsupported encoding - (%s->%s)", fromEnc, toEnc);
    outBufSize = num * 2;
    outbuf = tmpOutBuf = objc_malloc(outBufSize);
    outbytesleft = outBufSize;
    *outCount = 0;
    do
    {
        bytesRightNow = outbytesleft;
        if (inbytesleft == 0)
        {
            rc = iconv(converter, NULL, NULL, &tmpOutBuf, &outbytesleft);
            finished = YES;
        }
        else
        {
            rc = iconv(converter, &inbuf, &inbytesleft, &tmpOutBuf, &outbytesleft);
        }
        *outCount += bytesRightNow - outbytesleft;
        if (rc == -1 && errno == E2BIG)
        {
            outBufSize *= 2;
            newOutBuf = objc_malloc(outBufSize);
            memcpy(newOutBuf, outbuf, *outCount);
            objc_free(outbuf);
            outbuf = newOutBuf;
            outbytesleft = outBufSize - *outCount;
            tmpOutBuf = outbuf + *outCount;
            rc = 0;
        }
    } while (!(finished || rc));
    iconv_close(converter);
    if (rc == -1)
    {
        objc_free(outbuf);
        RAISE_EXCEPTION(GENERIC_EXCEPTION, @"Error converting text - %s", strerror(errno));
    }
    if (trimOutBuf)
    {
        newOutBuf = objc_malloc(*outCount + (nullTerm ? 1 : 0));
        memcpy(newOutBuf, outbuf, *outCount);
        if (nullTerm)
            newOutBuf[*outCount] = 0;
        objc_free(outbuf);
        return (uint8_t*)newOutBuf;
    }
    else
    {
        return (uint8_t*)outbuf;
    }
}

static inline unsigned __findChar(olchar ch, const olchar* buf, unsigned size, unsigned offset)
{
    for ( ; offset < size; offset++)
    {
        if (buf[offset] == ch)
            return offset;
    }
    return UINT_MAX;
}

static inline unsigned __findFirstNotOf(OLText* text,
                                        const olchar* buf,
                                        unsigned size,
                                        unsigned offset)
{
    for ( ; offset < size; offset++)
    {
        if ([text findChar: buf[offset] fromOffset: 0] == UINT_MAX)
            return offset;
    }
    return UINT_MAX;
}


static inline unsigned __findFirstOf(OLText* text,
                                     const olchar* buf,
                                     unsigned size,
                                     unsigned offset)
{
    for ( ; offset < size; offset++)
    {
        if ([text findChar: buf[offset] fromOffset: 0] != UINT_MAX)
            return offset;
    }
    return UINT_MAX;
}

static inline unsigned __findLastNotOf(OLText* text,
                                       const olchar* buf,
                                       unsigned size,
                                       unsigned offset)
{
    for (offset = MIN(offset, size - 1); offset < UINT_MAX; offset--)
    {
        if ([text findChar: buf[offset] fromOffset: 0] == UINT_MAX)
            return offset;
    }
    return UINT_MAX;
}

static inline unsigned __findLastOf(OLText* text,
                                    const olchar* buf,
                                    unsigned size,
                                    unsigned offset)
{
    for (offset = MIN(offset, size - 1); offset < UINT_MAX; offset--)
    {
        if ([text findChar: buf[offset] fromOffset: 0] != UINT_MAX)
            return offset;
    }
    return UINT_MAX;
}

static inline unsigned __findText(OLText* pattern,
                                  const olchar* buf,
                                  unsigned size,
                                  unsigned offset)
{
    unsigned* array;
    unsigned i;
    unsigned arrayCount;
    unsigned patternLen = [pattern length];
    unsigned patternLenPlusOne = patternLen + 1;
    unsigned curOff;
    olchar minChar = UINT16_MAX;
    olchar maxChar = 0;
    olchar cur;

    for (i = 0; i < patternLen; i++)
    {
        cur = [pattern at: i];
        maxChar = MAX(maxChar, cur);
        minChar = MIN(minChar, cur);
    }
    arrayCount = maxChar - minChar + 1;
    array = objc_malloc(arrayCount * sizeof(unsigned));
    for (i = 0; i < arrayCount; i++)
        array[i] = patternLenPlusOne;
    for (i = 0; i < patternLen; i++)
        array[[pattern at: i] - minChar] = patternLen - i;
    while (offset + patternLen <= size)
    {
        curOff = 0;
        while (curOff < patternLen && [pattern at: curOff] == buf[offset + curOff])
            curOff++;
        if (curOff == patternLen)
        {
            objc_free(array);
            return offset;
        }
        if (offset + patternLen >= size)
        {
            objc_free(array);
            return UINT_MAX;
        }
        i = buf[offset + patternLen] - minChar;
        offset += (i >= arrayCount) ? patternLenPlusOne : array[i];
    }
    objc_free(array);
    return UINT_MAX;
}

static inline int __olcharcmp(const olchar* cp1, const olchar* cp2, unsigned len)
{
    int result = 0;

    while (len > 0 && result == 0)
    {
        result = (int)*cp1++ - (int)*cp2++;
        len--;
    }
    return result;
}

static void __refLockPerformInit()
{
    OLCreateMutex(&__refLock);
}

static inline unsigned __rfindChar(olchar ch, const olchar* buf, unsigned size, unsigned offset)
{
    for (offset = MIN(offset, size - 1); offset < UINT_MAX; offset--)
    {
        if (buf[offset] == ch)
            return offset;
    }
    return UINT_MAX;
}

@implementation OLText

#if defined(OL_NO_OPENSTEP)
+ (id) initialize
#else
+ (void) initialize
#endif
{
    OLOnce(&__refLockInit, __refLockPerformInit);
#if defined(OL_NO_OPENSTEP)
    return self;
#endif
}

+ (id) textWithBytes: (const uint8_t*)bytes count: (unsigned)num encoding: (const char*)enc
{
    OL_BEGIN_AUTO_CTOR(OLText)
        initWithBytes: bytes count: num encoding: enc
    OL_END_AUTO_CTOR;
}

+ (id) textWithCString: (const char*)chs
{
    OL_BEGIN_AUTO_CTOR(OLText)
        initWithCString: chs
    OL_END_AUTO_CTOR;
}

#if !defined(OL_NO_OPENSTEP)
+ (id) textWithNSString: (NSString*)str
{
    OL_BEGIN_AUTO_CTOR(OLText)
        initWithNSString: str
    OL_END_AUTO_CTOR;
}
#endif

+ (id) textWithText: (OLText*)text
{
    OL_BEGIN_AUTO_CTOR(OLText)
        initWithText: text
    OL_END_AUTO_CTOR;
}

- (id) init
{
    return [self initWithChars: NULL count: 0];
}

- (id) initWithBytes: (const uint8_t*)bytes count: (unsigned)num encoding: (const char*)enc
{
    olchar* chars;
    unsigned len;

    [super init];

    OL_DURING

        chars = (olchar*)__convertBytes(bytes,
                                        num,
                                        enc,
                                        OL_UNICODE_INTERNAL,
                                        &len,
                                        NO,
                                        YES);
        reference = objc_malloc(sizeof(OLTextReference));
        reference->characters = chars;
        reference->length = len / sizeof(olchar);
        reference->refCount = 1;
        reference->cString = NULL;

    OL_HANDLER

        OBJ_RELEASE(super);
        [localException raise];

    OL_ENDHANDLER

    return self;
}

- (id) initWithChars: (const olchar*)chs count: (unsigned)num
{
    [super init];
    reference = objc_malloc(sizeof(OLTextReference));
    if (num > 0)
    {
        reference->characters = objc_malloc(num * sizeof(olchar));
        memcpy(reference->characters, chs, num * sizeof(olchar));
    }
    else
    {
        reference->characters = NULL;
    }
    reference->length = num;
    reference->refCount = 1;
    reference->cString = NULL;
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithCoder: (NSCoder*)decoder
{
    unsigned count;
    uint8_t* utf8Data;

    if ([decoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [decoder allowsKeyedCoding])
    {
        utf8Data = (uint8_t*)[decoder decodeBytesForKey: UTF8_DATA_KEY returnedLength: &count];
    }
    else
    {
        utf8Data = (uint8_t*)[decoder decodeBytesWithReturnedLength: &count];
    }
    return [self initWithBytes: utf8Data count: count encoding: "UTF-8"];
}
#endif

#if defined(OL_NO_OPENSTEP)
- (id) initWithConstantString: (OLConstantString*)str
{
    return [self initWithBytes: (uint8_t*)[str cString] count:
        [str length] encoding: OL_ASCII_ENCODING];
}
#endif

- (id) initWithCString: (const char*)chs
{
    unsigned len = strlen(chs);

    [self initWithBytes: (const uint8_t*)chs count: len encoding: OL_DEFAULT_ENCODING];
    reference->cString = objc_malloc(len + 1);
    strcpy(reference->cString, chs);
    return self;
}

#if !defined(OL_NO_OPENSTEP)
- (id) initWithNSString: (NSString*)str
{
    [super init];
    reference = objc_malloc(sizeof(OLTextReference));
    reference->length = [str length];
    if (reference->length > 0)
    {
        reference->characters = objc_malloc(reference->length * sizeof(olchar));
        [str getCharacters: reference->characters];
    }
    else
    {
        reference->characters = NULL;
    }
    reference->refCount = 1;
    reference->cString = NULL;
    return self;
}
#endif

- (id) initWithObjectInStream: (OLObjectInStream*)stream
{
    unsigned count = [stream readInt];
    uint8_t* utf8Data = objc_malloc(count);

    [stream completelyReadBytes: utf8Data count: count];
    [self initWithBytes: utf8Data count: count encoding: "UTF-8"];
    objc_free(utf8Data);
    return self;
}

- (id) initWithText: (OLText*)text
{
    [super init];
    reference = text->reference;
    OLLockMutex(&__refLock);
    reference->refCount++;
    OLUnlockMutex(&__refLock);
    return self;
}

- (id) initWithText: (OLText*)text offset: (unsigned)roff count: (unsigned)num
{
    if (roff == 0 && num == text->reference->length)
        [self initWithText: text];
    else
        [self initWithChars: text->reference->characters + roff count: num];
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    OLLockMutex(&__refLock);
    if (reference != NULL && --reference->refCount == 0)
    {
        objc_free(reference->characters);
        objc_free(reference->cString);
        objc_free(reference);
    }
    OLUnlockMutex(&__refLock);
    SUPER_FREE;
}

- (olchar) at: (unsigned)index
{
    return reference->characters[index];
}

- (uint8_t*) bytesWithEncoding: (const char*)enc returnedCount: (unsigned*)rc
{
    return __convertBytes((const uint8_t*)reference->characters,
                          reference->length * sizeof(olchar),
                          OL_UNICODE_INTERNAL,
                          enc,
                          rc,
                          NO,
                          YES);
}

- (int) compare: (id)other
{
    OLText* text;
#if defined(OL_NO_OPENSTEP)
    OLConstantString* constStr;
    olchar* buf;
    unsigned bufLen;
#else
    NSString* myself;
#endif
    int result = -1;

    if (IS_KIND_OF(other, OLText))
    {
        text = (OLText*)other;
        if (reference == text->reference)
        {
            result = 0;
        }
        else
        {
            result = __olcharcmp(reference->characters,
                                 text->reference->characters,
                                 MIN(reference->length, text->reference->length));
            if (result == 0 && reference->length != text->reference->length)
                result = (reference->length > text->reference->length) ? 1 : -1;
        }
    }
#if defined(OL_NO_OPENSTEP)
    else if (IS_KIND_OF(other, OLConstantString))
    {
        constStr = (OLConstantString*)other;
        buf = (olchar*)__convertBytes((const uint8_t*)[constStr cString],
                                      [constStr length],
                                      OL_ASCII_ENCODING,
                                      OL_UNICODE_INTERNAL,
                                      &bufLen,
                                      NO,
                                      NO);
        bufLen /= sizeof(olchar);
        result = __olcharcmp(reference->characters,
                             buf,
                             MIN(reference->length, bufLen));
        if (result == 0 && reference->length != bufLen)
            result = (reference->length > bufLen) ? 1 : -1;
        objc_free(buf);
    }
#else
    else if (IS_KIND_OF(other, NSString))
    {
        myself = [[NSString alloc] initWithCharacters: reference->characters
            length: reference->length];
        result = [myself compare: other];
        OBJ_RELEASE(myself);
    }
#endif
    return result;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    return [[OLText alloc] initWithText: self];
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    return [[OLText allocWithZone: zone] initWithText: self];
}
#endif

- (const char*) cString
{
    unsigned outCount;

    OLLockMutex(&__refLock);
    if (reference->cString == NULL)
    {
        reference->cString = (char*)__convertBytes((const uint8_t*)reference->characters,
                                                   reference->length * sizeof(olchar),
                                                   OL_UNICODE_INTERNAL,
                                                   OL_DEFAULT_ENCODING,
                                                   &outCount,
                                                   YES,
                                                   YES);
    }
    OLUnlockMutex(&__refLock);
    return reference->cString;
}

- (BOOL) empty
{
    return (reference->length == 0) ? YES : NO;
}

#if !defined(OL_NO_OPENSTEP)
- (void) encodeWithCoder: (NSCoder*)encoder
{
    unsigned count;
    uint8_t* utf8Data = [self bytesWithEncoding: "UTF-8" returnedCount: &count];

    if ([encoder respondsToSelector: @selector(allowsKeyedCoding)] &&
        [encoder allowsKeyedCoding])
    {
        [encoder encodeBytes: utf8Data length: count forKey: UTF8_DATA_KEY];
    }
    else
    {
        [encoder encodeBytes: utf8Data length: count];
    }
    objc_free(utf8Data);
}
#endif

- (unsigned) findChar: (olchar)ch fromOffset: (unsigned)offset
{
    return __findChar(ch, reference->characters, reference->length, offset);
}

- (unsigned) findFirstNotOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findFirstNotOf(text, reference->characters, reference->length, offset);
}

- (unsigned) findFirstOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findFirstOf(text, reference->characters, reference->length, offset);
}

- (unsigned) findLastNotOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findLastNotOf(text, reference->characters, reference->length, offset);
}

- (unsigned) findLastOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findLastOf(text, reference->characters, reference->length, offset);
}

- (unsigned) findText: (OLText*)text fromOffset: (unsigned)offset
{
    return __findText(text, reference->characters, reference->length, offset);
}

- (void) getCharacters: (olchar*)buffer fromOffset: (unsigned)offset count: (unsigned)num
{
    memcpy(buffer, reference->characters + offset, num * sizeof(olchar));
}

- (unsigned) hash
{
    return OLHash((const uint8_t*)reference->characters, reference->length * sizeof(olchar));
}

- (BOOL) isEqual: (id)object
{
    return [self compare: object] == 0 ? YES : NO;
}

- (unsigned) length
{
    return reference->length;
}

- (unsigned) maxSize
{
    return UINT_MAX;
}

- (uint8_t*) nullTerminatedBytesWithEncoding: (const char*)enc
{
    unsigned outCount;

    return __convertBytes((const uint8_t*)reference->characters,
                          reference->length * sizeof(olchar),
                          OL_UNICODE_INTERNAL,
                          enc,
                          &outCount,
                          YES,
                          YES);
}

- (unsigned) rfindChar: (olchar)ch fromOffset: (unsigned)offset
{
    return __rfindChar(ch, reference->characters, reference->length, offset);
}

- (unsigned) size
{
    return reference->length;
}

- (OLText*) substrFromOffset: (unsigned)offset count: (unsigned)num
{
    return OBJ_AUTORELEASE([[OLText alloc]
        initWithText: self offset: offset count: num]);
}

#if !defined(OL_NO_OPENSTEP)
- (NSString*) toNSString
{
    return [NSString stringWithCharacters: reference->characters length: reference->length];
}
#endif

- (void) writeSelfToStream: (OLObjectOutStream*)stream
{
    unsigned numberOfBytes;
    uint8_t* bytes = [self bytesWithEncoding: "UTF-8" returnedCount: &numberOfBytes];

    [stream writeInt: numberOfBytes];
    [stream completelyWriteBytes: bytes count: numberOfBytes];
    objc_free(bytes);
}

@end

@implementation OLTextBuffer

+ (id) textBuffer
{
    OL_BEGIN_AUTO_CTOR(OLTextBuffer)
        init
    OL_END_AUTO_CTOR;
}

+ (id) textBufferWithText: (OLText*)text
{
    OL_BEGIN_AUTO_CTOR(OLTextBuffer)
        initWithText: text
    OL_END_AUTO_CTOR;
}

- (id) init
{
    return [self initWithCapacity: 32];
}

- (id) initWithCapacity: (unsigned)cap
{
    [super init];
    capacity = cap;
    begin = objc_malloc(capacity * sizeof(olchar));
    size = 0;
    return self;
}

- (id) initWithText: (OLText*)text
{
    unsigned len = [text length];

    [self initWithCapacity: len];
    [text getCharacters: begin fromOffset: 0 count: len];
    size = len;
    return self;
}

#if defined(OL_NO_OPENSTEP)
- (id) free
#else
- (void) dealloc
#endif
{
    objc_free(begin);
    SUPER_FREE;
}

- (OLTextBuffer*) appendChar: (olchar)ch
{
    [self reserve: size + 1];
    begin[size++] = ch;
    return self;
}

- (OLTextBuffer*) appendChars: (const olchar*)chars fromOffset: (unsigned)offset count: (unsigned)num
{
    [self reserve: size + num];
    memcpy(begin + size, chars + offset, num * sizeof(olchar));
    size += num;
    return self;
}

- (OLTextBuffer*) appendText: (OLText*)text
{
    unsigned len = [text length];

    [self reserve: size + len];
    [text getCharacters: begin + size fromOffset: 0 count: len];
    size += len;
    return self;
}

- (OLTextBuffer*) appendTextBuffer: (OLTextBuffer*)buffer
{
    return [self appendChars: buffer->begin fromOffset: 0 count: buffer->size];
}

- (OLTextBuffer*) assignAt: (unsigned)index character: (olchar)ch
{
    begin[index] = ch;
    return self;
}

- (olchar) at: (unsigned)index
{
    return begin[index];
}

- (unsigned) capacity
{
    return capacity;
}

- (void) clear
{
    size = 0;
}

- (int) compare: (id)other
{
    OLTextBuffer* buffer;
    int result = -1;

    if (IS_KIND_OF(other, OLTextBuffer))
    {
        buffer = (OLTextBuffer*)other;
        result = __olcharcmp(begin, buffer->begin, MIN(size, buffer->size));
        if (result == 0 && size != buffer->size)
            result = (size > buffer->size) ? 1 : -1;
    }
#if !defined(OL_NO_OPENSTEP)
    if (result < 0)
        result = NSOrderedAscending;
    else if (result > 0)
        result = NSOrderedDescending;
#endif
    return result;
}

#if defined(OL_NO_OPENSTEP)
- (id) copy
{
    OLTextBuffer* buf = [[OLTextBuffer alloc] initWithCapacity: capacity];
    [buf appendTextBuffer: self];
    return buf;
}

#else

- (id) copyWithZone: (NSZone*)zone
{
    OLTextBuffer* buf = [[OLTextBuffer allocWithZone: zone] initWithCapacity: capacity];
    [buf appendTextBuffer: self];
    return buf;
}
#endif

- (BOOL) empty
{
    return (size == 0) ? YES : NO;
}

- (OLTextBuffer*) eraseAt: (unsigned)index
{
    return [self eraseFromOffset: index count: 1];
}

- (OLTextBuffer*) eraseFromOffset: (unsigned)offset count: (unsigned)num
{
    if (offset + num < size)
        memcpy(begin + offset, begin + offset + num, (size - offset - num) * sizeof(olchar));
    size -= num;
    return self;
}

- (unsigned) findChar: (olchar)ch fromOffset: (unsigned)offset
{
    return __findChar(ch, begin, size, offset);
}

- (unsigned) findFirstNotOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findFirstNotOf(text, begin, size, offset);
}

- (unsigned) findFirstOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findFirstOf(text, begin, size, offset);
}

- (unsigned) findLastNotOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findLastNotOf(text, begin, size, offset);
}

- (unsigned) findLastOf: (OLText*)text fromOffset: (unsigned)offset
{
    return __findLastOf(text, begin, size, offset);
}

- (unsigned) findText: (OLText*)text fromOffset: (unsigned)offset
{
    return __findText(text, begin, size, offset);
}

- (void) getCharacters: (olchar*)buffer fromOffset: (unsigned)offset count: (unsigned)num
{
    memcpy(buffer, begin + offset, num * sizeof(olchar));
}

- (OLTextBuffer*) insertChar: (olchar)ch atOffset: (unsigned)offset
{
    return [self insertChars: &ch atOffset: offset count: 1];
}

- (OLTextBuffer*) insertChars: (const olchar*)chars atOffset: (unsigned)offset count: (unsigned)num
{
    [self reserve: size + num];
    if (offset < size)
        memmove(begin + offset + num, begin + offset, (size - offset) * sizeof(olchar));
    memcpy(begin + offset, chars, num * sizeof(olchar));
    size += num;
    return self;
}

- (OLTextBuffer*) insertText: (OLText*)text atOffset: (unsigned)offset
{
    unsigned len = [text length];

    [self reserve: size + len];
    if (offset < size)
        memmove(begin + offset + len, begin + offset, (size - offset) * sizeof(olchar));
    [text getCharacters: begin + offset fromOffset: 0 count: len];
    size += len;
    return self;
}

- (BOOL) isEqual: (id)object
{
    return [self compare: object] == 0 ? YES : NO;
}

- (unsigned) length
{
    return size;
}

- (OLTextBuffer*) replaceFromOffset: (unsigned)offset count: (unsigned)num withText: (OLText*)text
{
    unsigned textLen = [text length];

    if (textLen > num)
    {
        [self reserve: textLen - num];
        if (size > offset + num)
        {
            memmove(begin + offset + textLen,
                    begin + offset + num,
                    (size - (offset + num)) * sizeof(olchar));
        }
        size += textLen - num;
    }
    else if (textLen < num)
    {
        [self eraseFromOffset: offset + textLen count: num - textLen];
    }
    if (textLen > 0)
        [text getCharacters: begin + offset fromOffset: 0 count: textLen];
    return self;
}

- (void) reserve: (unsigned)cap
{
    olchar* newBegin;

    if (cap > capacity)
    {
        capacity = MAX(cap, capacity * 2);
        newBegin = objc_malloc(capacity * sizeof(olchar));
        memcpy(newBegin, begin, size * sizeof(olchar));
        objc_free(begin);
        begin = newBegin;
    }
}

- (OLTextBuffer*) resize: (unsigned)newLength filledWith: (olchar)ch
{
    unsigned i;

    if (newLength > size)
    {
        [self reserve: newLength];
        for (i = size; i < newLength; i++)
            begin[i] = ch;
    }
    size = newLength;
    return self;
}

- (OLTextBuffer*) reverse
{
    unsigned fwd;
    unsigned bkwd;

    if (size > 1)
    {
        fwd = 0;
        bkwd = size;
        for ( ; fwd != bkwd && fwd != --bkwd; fwd++)
        {
            // The following line MUST be within curly braces!!!
            // (Check out the OL_FAST_SWAP macro if you don't believe me.)
            OL_FAST_SWAP(begin[fwd], begin[bkwd]);
        }
    }
    return self;
}

- (unsigned) rfindChar: (olchar)ch fromOffset: (unsigned)offset
{
    return __rfindChar(ch, begin, size, offset);
}

- (unsigned) size
{
    return size;
}

- (OLText*) substrFromOffset: (unsigned)offset count: (unsigned)num
{
    return OBJ_AUTORELEASE([[OLText alloc] initWithChars: begin + offset count: num]);
}

- (OLText*) text
{
    return [self substrFromOffset: 0 count: size];
}

@end

#if defined(OL_NO_OPENSTEP)

#if defined(__NEXT_RUNTIME__)
struct objc_class _OLConstantStringClassReference;
#endif

@implementation OLConstantString

#if defined(__NEXT_RUNTIME__)
+ (id) class
{
    return &_OLConstantStringClassReference;
}

+ (void) load
{
    memcpy(&_OLConstantStringClassReference,
           self,
           sizeof(_OLConstantStringClassReference));
    objc_addClass(&_OLConstantStringClassReference);
}

#endif

- (id) free
{
    return self;
}

#if defined(__NEXT_RUNTIME__)
- (id) class
{
    return &_OLConstantStringClassReference;
}
#endif

- (int) compare: (id)other
{
    int result = -1;

    if (IS_KIND_OF(other, OLConstantString))
    {
        result = strcmp(c_string, ((OLConstantString*)other)->c_string);
    }
    else if (IS_KIND_OF(other, OLText))
    {
        result = -[(OLText*)other compare: self];
    }
    return result;
}

- (id) copy
{
    return self;
}

- (const char*) cString
{
    return c_string;
}

- (unsigned) hash
{
    return OLHash((const uint8_t*)c_string, len);
}

- (BOOL) isEqual: (id)other
{
    return [self compare: other] == 0 ? YES : NO;
}

- (unsigned) length
{
    return len;
}

@end

#endif
