//
// $Id: Synchronization.h,v 1.2 2007/04/15 23:32:57 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(SYNCHRONIZATION_OL_GUARD)
#define SYNCHRONIZATION_OL_GUARD

#include "ConfigPrivate.h"

typedef void (*OLInitFunction)(void);

#if defined(OL_POSIX_THREADS)

#include <pthread.h>

typedef pthread_once_t  OLOnceControl;
typedef pthread_key_t   OLThreadDataKey;
typedef pthread_mutex_t OLMutex;

#define OL_ONCE_INIT PTHREAD_ONCE_INIT

static inline int OLOnce(OLOnceControl* onceControl, OLInitFunction init)
{
    return pthread_once(onceControl, init);
}

static inline int OLCreateThreadDataKey(OLThreadDataKey* key)
{
    return pthread_key_create(key, NULL);
}

static inline int OLSetThreadData(OLThreadDataKey key, const void* value)
{
    return pthread_setspecific(key, value);
}

static inline void* OLGetThreadData(OLThreadDataKey key)
{
    return pthread_getspecific(key);
}

static inline int OLCreateMutex(OLMutex* mutex)
{
    return pthread_mutex_init(mutex, NULL);
}

static inline int OLLockMutex(OLMutex* mutex)
{
    return pthread_mutex_lock(mutex);
}

static inline int OLUnlockMutex(OLMutex* mutex)
{
    return pthread_mutex_unlock(mutex);
}

#elif defined(OL_WIN32_THREADS)

#include <windows.h>

#if defined(OL_HAVE_WIN32_ONCE)
typedef INIT_ONCE   OLOnceControl;
#else
typedef int         OLOnceControl;
#endif
typedef DWORD       OLThreadDataKey;
typedef HANDLE      OLMutex;

#if defined(OL_HAVE_WIN32_ONCE)
#define OL_ONCE_INIT NULL
#else
#define OL_ONCE_INIT 0
#endif

int OLOnce(OLOnceControl* onceControl, OLInitFunction init);

static inline int OLCreateThreadDataKey(OLThreadDataKey* key)
{
    *key = TlsAlloc();
    return (*key == TLS_OUT_OF_INDEXES) ? 1 : 0;
}

static inline int OLSetThreadData(OLThreadDataKey key, const void* value)
{
    return (TlsSetValue(key, (void*)value) == 0) ? 1 : 0;
}

static inline void* OLGetThreadData(OLThreadDataKey key)
{
    return TlsGetValue(key);
}

static inline int OLCreateMutex(OLMutex* mutex)
{
    *mutex = CreateMutex(NULL, FALSE, NULL);
    return (*mutex == NULL) ? 1 : 0;
}

static inline int OLLockMutex(OLMutex* mutex)
{
    return (WaitForSingleObject(*mutex, INFINITE) == WAIT_FAILED) ? 1 : 0;
}

static inline int OLUnlockMutex(OLMutex* mutex)
{
    return (ReleaseMutex(*mutex) == 0) ? 1 : 0;
}

#endif

#endif
